 /**************************************************************************
 *                                                                         *
 *                                 RasTop                                  *
 *                                                                         *
 *                 Molecular Graphics Visualisation Tool                   *
 *                                                                         *
 *                           by Philippe Valadon                           *
 *                 P.O. Box 8064, La Jolla, CA 92192-8064, USA,            *
 *                       www.geneinfinity.org/rastop                       *
 *                                                                         *
 *                                *******                                  *
 *                                                                         *
 *                   Based on RasMol 2.6 by Roger Sayle                    *
 * Biomolecular Structures Group, Glaxo Wellcome Research & Development,   *
 *                      Stevenage, Hertfordshire, UK                       *
 *         Version 2.6, August 1995, Version 2.6.4, December 1998          *
 *                   Copyright (C) Roger Sayle 1992-1999                   *
 *                                                                         *
 *                                 and                                     *
 *                                                                         *
 * Arne Mueller       RasMol 2.6x1   May 1998   (C) Arne Mueller 1998      *
 * Gary Grossman and  RasMol 2.5-ucb Nov 1995   (C) UC Regents/ModularCHEM *
 * Marco Molinaro     RasMol 2.6-ucb Nov 1996       Consortium 1995, 1996  *
 * Herbert J.         RasMol 2.7.0   Mar 1999   (C) Herbert J. Bernstein   * 
 * Bernstein          RasMol 2.7.1   Jun 1999       1998-2001              *
 *                    RasMol 2.7.1.1 Jan 2001                              *
 *                    RasMol 2.7.2   Aug 2000                              *
 *                    RasMol 2.7.2.1 Apr 2001                              *
 * Philippe Valadon   RasTop 2.0.2   Oct 2002   (C) Philippe Valadon       *
 *                                                  2000-2002              *
 *                                                                         *
 * Please read the file NOTICE for important notices which apply to this   *
 * package. If you are not going to make changes to Rasmol, you are not    *
 * only permitted to freely make copies and distribute them, you are       *
 * encouraged to do so, provided you do the following:                     *
 *   * 1. Either include the complete documentation, especially the file   *
 *     NOTICE, with what you distribute or provide a clear indication      *
 *     where people can get a copy of the documentation; and               *
 *   * 2. Please give credit where credit is due citing the version and    *
 *     original authors properly; and                                      *
 *   * 3. Please do not give anyone the impression that the original       *
 *     authors are providing a warranty of any kind.                       *
 *                                                                         *
 * If you would like to use major pieces of Rasmol in some other program,  *
 * make modifications to Rasmol, or in some other way make what a lawyer   *
 * would call a "derived work", you are not only permitted to do so, you   *
 * are encouraged to do so. In addition to the things we discussed above,  *
 * please do the following:                                                *
 *   * 4. Please explain in your documentation how what you did differs    *
 *     from this version of Rasmol; and                                    *
 *   * 5. Please make your modified source code available.                 *
 *                                                                         *
 * This version of Rasmol is not in the public domain, but it is given     *
 * freely to the community in the hopes of advancing science. If you make  *
 * changes, please make them in a responsible manner, and please offer us  *
 * the opportunity to include those changes in future versions of Rasmol.  *
 ***************************************************************************/


/* prefer.c
 */

#include "rasmol.h"

#ifdef IBMPC
#include <windows.h>
#include <shellapi.h>
#include <malloc.h>
#endif
#ifdef APPLEMAC
#include <Types.h>
#include <Errors.h>
#ifdef __CONDITIONALMACROS__
#include <Printing.h>
#else
#include <PrintTraps.h>
#endif
#endif
#ifndef sun386
#include <stdlib.h>
#endif

#include <string.h>
#if defined(IBMPC) || defined(VMS) || defined(APPLEMAC)
#include "string_case.h"
#else
#include <strings.h>
#endif
#include <ctype.h>
#include <stdio.h>

#if !defined(IBMPC) && !defined(VMS) && !defined(APPLEMAC)
#include <pwd.h>
#endif

#define PREFER
#include "prefer.h"
#include "command.h"
#include "tokens.h"
#include "molecule.h"
/* #include "infile.h"   */
/* #include "abstree.h"  */
/* #include "transfor.h" */
#include "cmndline.h"
#include "render.h"
/* #include "pixutils.h" */
/* #include "repres.h"   */
#include "graphics.h"
/* #include "outfile.h"  */
/* #include "script.h"   */
#include "multiple.h"
/* #include "vector.h"   */
/* #include "brotate.h"  */

#include <direct.h> 

#define IsIdentChar(x)  ((isalnum(x))||((x)=='_')||((x)=='$'))


#ifndef VMS
#ifdef IBMPC
#define DirChar  '\\'
#else
#define DirChar  '/'
#endif
#endif

#define MAXPREFLEN      1024

#define MAXERROR        20

char CurPrefLine[MAXPREFLEN];

static int PrefTokenLength;
static Long PrefTokenValue;
static char PrefTokenIdent[128];
static char *PrefTokenStart;
static char *PrefTokenPtr;
static int CurPrefToken;


/*===============================*/
/*  Prefs Line Lexical Analysis  */
/*===============================*/

static int FetchPrefToken( void )
{
    register char ch;

    CurPrefToken = 0;
    ch = *PrefTokenPtr++;
    while( ch && (ch!='#') )
    {   if( isspace(ch) )
        {   ch = *PrefTokenPtr++;
            continue;
        }

        PrefTokenStart = PrefTokenPtr-1;
        if( isalpha(ch) )
         {   PrefTokenLength = 1;
             *PrefTokenIdent = ToUpper(ch);
             while( IsIdentChar(*PrefTokenPtr) && (PrefTokenLength<32) )
             {   ch = *PrefTokenPtr++;
                 PrefTokenIdent[PrefTokenLength++] = ToUpper(ch);
             }
             if( PrefTokenLength==32 )
             {   return(0);
             } else PrefTokenIdent[PrefTokenLength] = '\0';
             return( CurPrefToken = LookUpKeyword(PrefTokenIdent) );

         } else if( isdigit(ch) )
         {   PrefTokenValue = ch-'0';
             while( isdigit(*PrefTokenPtr) )
                 PrefTokenValue = 10*PrefTokenValue + (*PrefTokenPtr++)-'0';
             return( CurPrefToken = NumberTok );

         } else if( (ch=='\'') || (ch=='\"') || (ch=='`') )
         {   PrefTokenLength = 0;
             while( *PrefTokenPtr && (PrefTokenLength<128) && (*PrefTokenPtr!=ch) )
                 PrefTokenIdent[PrefTokenLength++] = *PrefTokenPtr++;

             if( ch != *PrefTokenPtr )
             {   return( 0 );
             } else PrefTokenPtr++;

             PrefTokenIdent[PrefTokenLength]='\0';
             return( CurPrefToken = StringTok );
         } else if( ispunct(ch) )
             return( CurPrefToken = ch );

         ch = *PrefTokenPtr++;
    }
    PrefTokenPtr--;
    return 0;
}


static int NextIf( int tok, int err )
{
    if( FetchPrefToken() != tok )
        return True;
    else return False;
}


static void FetchFloat( Long value, int scale )
{
    register int count;
    register int mant;

    if( !value && !isdigit(*PrefTokenPtr) )
    {   PrefTokenValue = 0;
        return;
    }

    mant = 0;
    count = 1;
    while( isdigit(*PrefTokenPtr) )
    {   if( count < scale )
        {   mant = 10*mant + (*PrefTokenPtr-'0');
            count *= 10;
        }
        PrefTokenPtr++;
    }

    mant = (scale*mant)/count;
    PrefTokenValue = value*scale + mant;
}



/*===========================================*/
/*  Generic PrefCommand Parsing & Execution  */
/*===========================================*/


static int ExecutePrefCommand( void )
{
    PrefTokenPtr = CurPrefLine;
    if( !FetchPrefToken() )
    {   PrefTokenPtr = NULL;
        return True;
    }


    switch( CurPrefToken )
    {    
        case(ClipboardTok):
                           FetchPrefToken();
                           if( CurPrefToken == NumberTok )
                           {   ClipBoardFactor = PrefTokenValue;
                               if( ClipBoardFactor<1 )
                                   ClipBoardFactor = 1;
                               if( ClipBoardFactor> MAXCLIPBOARDFACTOR )
                                   ClipBoardFactor = MAXCLIPBOARDFACTOR;
                           } else return( False );
                           break;
               
        case(HelpTok):     FetchPrefToken();
                           if( CurPrefToken==HTMLTok )
                           {   FetchPrefToken();
                               if( !CurPrefToken )
                                   *HelpHTMLSource = '\0';
                               else if( CurPrefToken==StringTok )
                                   strncpy(HelpHTMLSource,PrefTokenIdent,246);
                               else
                                   strncpy(HelpHTMLSource,PrefTokenStart,246);
                               CurPrefToken = 0;
                           } else return( False );                          
                           break;

        case(ScriptTok):   FetchPrefToken();
                           if( CurPrefToken==DefaultTok )
                           {   FetchPrefToken();
                               if( !CurPrefToken )
                                   *DefaultScriptName = '\0';
                               else if( CurPrefToken==StringTok )
                                   strncpy(DefaultScriptName,PrefTokenIdent,246);
                               else
                                   strncpy(DefaultScriptName,PrefTokenStart,246);
                               CurPrefToken = 0;
                           } else return( False );                              
                           break;

        case(SetTok):      FetchPrefToken();
                           if( CurPrefToken==MouseTok )
                           {   FetchPrefToken();
                               if( CurPrefToken==RasMolTok )
                               {   if( Interactive )
                                       SetMouseMode( MMRasMol );
                               } else if( CurPrefToken==InsightTok )
                               {   if( Interactive )
                                       SetMouseMode( MMInsight );
                               } else if( CurPrefToken==QuantaTok )
                               {   if( Interactive )
                                       SetMouseMode( MMQuanta );
                               } else if( CurPrefToken==SybylTok )
                               {   if( Interactive )
                                       SetMouseMode( MMSybyl );
                               } else return( False );
                          } else if( CurPrefToken==SequenceTok )
                          {    FetchPrefToken();
                               if( CurPrefToken==FalseTok )
                               {   SeqFormat = False;
                               } else if( CurPrefToken==TrueTok )
                               {   SeqFormat = True;
                               } else return( False );
                          } else if( CurPrefToken == DepthCueTok )
                          {    FetchPrefToken();
                               if( CurPrefToken==NumberTok )
                               {   if( (PrefTokenValue>0) && (PrefTokenValue<=4))
                                       DepthCueMode = PrefTokenValue;
                               }
                          } else if( CurPrefToken==CoordTok )
                          {    FetchPrefToken();
                               if( CurPrefToken==MoleculeTok )
                               {   SysCoor = MolCoor;
                               } else if( CurPrefToken==WorldTok )
                               {   SysCoor = WorldCoor;
                               } else if( CurPrefToken==ScreenTok )
                               {      SysCoor = ScreenCoor;
                               }  else if( CurPrefToken==AbsoluteTok )
                               {      SysCoor = AbsCoor;
                               } else return( False );
                          }                          
                          break;


        case(ExitTok):    break;
        default:          return( False );
    }

    if( CurPrefToken )
        if( FetchPrefToken() )
            return( False );
    PrefTokenPtr = NULL;
    return True;
}


/*======================*/
/*  Preference Writing  */
/*======================*/

void WriteScriptPrefs(FILE *fp)
{
    fprintf(fp,"#!rasmol -preferences\n");
    fprintf(fp,"# Creator: %s Version %s\n\n", VER_NAME, VERSION);

    /* Mouse mode */
    if( MouseMode == MMRasMol )
    {    fputs("set mouse rasmol\n",fp);
    } else if( MouseMode == MMInsight )
    {    fputs("set mouse insight\n",fp);
    } else if( MouseMode == MMQuanta )
    {    fputs("set mouse quanta\n",fp);
    } else if( MouseMode == MMSybyl )
    {    fputs("set mouse sybyl\n",fp);
    }
    fputc('\n',fp);

    /* Help HTML source */
    fprintf(fp,"help HTML %s\n",HelpHTMLSource);
    fputc('\n',fp);

    /* Default template source */
    fprintf(fp,"script default %s\n",DefaultScriptName);
    fputc('\n',fp);

    /* Clipboard zoom factor */
    fprintf(fp,"clipboard %d\n",ClipBoardFactor);
    fputc('\n',fp);

    /* Sequence ouptput format */
    if( SeqFormat==True)
    {    fputs("set sequence true\n",fp);
    } else if( SeqFormat==False)
    {    fputs("set sequence false\n",fp);
    }
    fputc('\n',fp);

    /* DepthCue Mode */
    fprintf(fp,"set depthcue %d\n",DepthCueMode);
    fputc('\n',fp);

    /* Coordinate System */
    if( SysCoor == MolCoor )
        fputs("set coordinate molecule\n",fp);
    else if( SysCoor == WorldCoor )
        fputs("set coordinate world\n",fp);
    else if( SysCoor == ScreenCoor )
        fputs("set coordinate screen\n",fp);
    else if( SysCoor == AbsCoor )
        fputs("set coordinate absolute\n",fp);
    fputc('\n',fp);

    fputs("exit\n",fp);
    fclose(fp);
}



/*================*/
/*  Read & Write  */
/*================*/


static void getPrefFileName( void )
{
#if defined(IBMPC)
    char *path = 0;
	char *c;
	HKEY key;
    DWORD type, size;

    if( RegOpenKeyEx(
            HKEY_CURRENT_USER,
            "Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders",
            0,
            KEY_READ,
            &key) != 0)
        return;

    if( RegQueryValueEx(key, "Desktop", 0, &type, 0, &size) == 0 && type == REG_SZ )
    {	path = (char*) malloc(size);
		if (RegQueryValueEx(key, "Desktop", 0, 0, (unsigned char*)path, &size) != 0)
        {	free(path);
			path = 0;
        }
    }
	RegCloseKey(key);
    
    c = strrchr(path, '\\');
    if (c == NULL) return;
    *c = '\0';
	if( path )
	{	strcpy(PrefFileName, path);
		free(path);
	}
#endif   

// PV no need to create a folder in windows, but the PrefFileName
// "C:/Documents and Settings/userdir/rasmol/rasmolprefs" might be better.

    // create folder here
	if( *PrefFileName )
		strcat(PrefFileName,"\\rasmolprefs");
}

static int LoadPrefFile( FILE *fp )
{
    register int ch,len;
    register int stat;
    register int errorcount;

    errorcount = 0;
    do {
        len = 0;
        ch = getc(fp);
        while( (ch!='\n') && (ch!='\r') &&  (ch!=EOF) )
        {   if( len<MAXPREFLEN )
                CurPrefLine[len++] = ch;
            ch = getc(fp);
        }

        if( ch == '\r' )
        {   ch = getc(fp);
            if( ch != '\n' )
                ungetc(ch,fp);
        }

        if( len<MAXPREFLEN )
        {   CurPrefLine[len] = '\0';
            stat = ExecutePrefCommand();
            if( !stat )
            {   errorcount++;
                if( errorcount>MAXERROR )
                {   fclose( fp );
                    return False;
                }
            }
         } else 
             return False;
    } while( ch!=EOF );
    fclose( fp );
    return True;
}


void ReadPrefs( void )
{   
#ifdef PREFERENCES
	FILE *fp;
    
	if( !*PrefFileName )
    {   WriteString("Preferences not found!\n");
    } else
    {   CurPrefToken = 0;
        PrefTokenPtr = NULL;
        fp = fopen(PrefFileName,"rb");
        if( fp )
        {   if( !LoadPrefFile(fp) )
            {   WriteString("Error: Resetting default preferences!\n");
                PrefsModified = True;
                WritePrefs( );
            } else
            {   /*WriteString("Preferences loaded\n");*/
                PrefsModified = False;
            }
        } else
        {   PrefsModified = True;
            WritePrefs( );
            if( !PrefsModified )
                WriteString("Creating Preference file.\n");
        }
    }
#endif
}


void WritePrefs( void )
{    
#if defined PREFERENCES
	FILE *fp;
    
	if( !PrefsModified || !*PrefFileName )
        return;

    fp = fopen(PrefFileName,"w");

    if( fp )
    {   WriteScriptPrefs(fp);
        PrefsModified = False;
    }    
#endif
}


void InitialisePrefs( void )
{
    *PrefFileName = '\0';
    PrefsModified= False;

#if defined PREFERENCES
    getPrefFileName();
#endif   
}
