 /**************************************************************************
 *                                                                         *
 *                                 RasTop                                  *
 *                                                                         *
 *                 Molecular Graphics Visualisation Tool                   *
 *                                                                         *
 *                           by Philippe Valadon                           *
 *                 P.O. Box 8064, La Jolla, CA 92192-8064, USA,            *
 *                       www.geneinfinity.org/rastop                       *
 *                                                                         *
 *                                *******                                  *
 *                                                                         *
 *                   Based on RasMol 2.6 by Roger Sayle                    *
 * Biomolecular Structures Group, Glaxo Wellcome Research & Development,   *
 *                      Stevenage, Hertfordshire, UK                       *
 *         Version 2.6, August 1995, Version 2.6.4, December 1998          *
 *                   Copyright (C) Roger Sayle 1992-1999                   *
 *                                                                         *
 *                                 and                                     *
 *                                                                         *
 * Arne Mueller       RasMol 2.6x1   May 1998   (C) Arne Mueller 1998      *
 * Gary Grossman and  RasMol 2.5-ucb Nov 1995   (C) UC Regents/ModularCHEM *
 * Marco Molinaro     RasMol 2.6-ucb Nov 1996       Consortium 1995, 1996  *
 * Herbert J.         RasMol 2.7.0   Mar 1999   (C) Herbert J. Bernstein   * 
 * Bernstein          RasMol 2.7.1   Jun 1999       1998-2001              *
 *                    RasMol 2.7.1.1 Jan 2001                              *
 *                    RasMol 2.7.2   Aug 2000                              *
 *                    RasMol 2.7.2.1 Apr 2001                              *
 * Philippe Valadon   RasTop 2.0     Jan 2002   (C) Philippe Valadon       *
 *                                                  2000-2002              *
 *                                                                         *
 * Naoum Salam       RasTop 2.0.1-VF May 2002  (C) Naoum Salam 2002      *
 *                                                                         *
 * Please read the file NOTICE for important notices which apply to this   *
 * package. If you are not going to make changes to Rasmol, you are not    *
 * only permitted to freely make copies and distribute them, you are       *
 * encouraged to do so, provided you do the following:                     *
 *   * 1. Either include the complete documentation, especially the file   *
 *     NOTICE, with what you distribute or provide a clear indication      *
 *     where people can get a copy of the documentation; and               *
 *   * 2. Please give credit where credit is due citing the version and    *
 *     original authors properly; and                                      *
 *   * 3. Please do not give anyone the impression that the original       *
 *     authors are providing a warranty of any kind.                       *
 *                                                                         *
 * If you would like to use major pieces of Rasmol in some other program,  *
 * make modifications to Rasmol, or in some other way make what a lawyer   *
 * would call a "derived work", you are not only permitted to do so, you   *
 * are encouraged to do so. In addition to the things we discussed above,  *
 * please do the following:                                                *
 *   * 4. Please explain in your documentation how what you did differs    *
 *     from this version of Rasmol; and                                    *
 *   * 5. Please make your modified source code available.                 *
 *                                                                         *
 * This version of Rasmol is not in the public domain, but it is given     *
 * freely to the community in the hopes of advancing science. If you make  *
 * changes, please make them in a responsible manner, and please offer us  *
 * the opportunity to include those changes in future versions of Rasmol.  *
 ***************************************************************************/

////////////////////////////////////////////////////////////////////////
// rasdoc.cpp 
////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "RasTop.h"
#include <limits.h>
#include <direct.h>
#include <stdlib.h>

#include "dialog.h"
class CRasView;	//Forward declaration
#include "mainfrm.h"
#include "rasdoc.h"
#include "rasview.h"
#include "dialog.h"
#include "graphics.h"

static char snamebuf[1024];
static char fnamebuf[1024];

#define STOPTIMER		GETFRAME \
						pFrame->m_wndCommandBar.KillTimerPannel();
#define STARTTIMER		pFrame->m_wndCommandBar.SetTimerPannel();

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
// CRasDoc

IMPLEMENT_DYNCREATE(CRasDoc, CDocument)

BEGIN_MESSAGE_MAP(CRasDoc, CDocument)
	//{{AFX_MSG_MAP(CRasDoc)
	ON_COMMAND(ID_SAVE_MOLECULARFILE, OnSaveMolecularfile)
	ON_UPDATE_COMMAND_UI(ID_SAVE_MOLECULARFILE, OnUpdateSaveMolecularfile)
	ON_COMMAND(ID_SAVE_IMAGE, OnSaveImage)
	ON_UPDATE_COMMAND_UI(ID_SAVE_IMAGE, OnUpdateSaveImage)
	ON_COMMAND(ID_SAVE_SCRIPT, OnSaveScript)
	ON_UPDATE_COMMAND_UI(ID_SAVE_SCRIPT, OnUpdateSaveScript)
	ON_COMMAND(ID_FILE_LOAD_RASMOLSCRIPT, OnFileLoadRasmolscript)
	ON_COMMAND(ID_FILE_LOAD_MOLECULARFILE, OnLoadMolecularfile)
	ON_COMMAND(ID_FILE_OPEN_RASTOP, OnFileOpenFromDoc)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE, OnUpdateFileSave)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE_AS, OnUpdateFileSaveAs)
	ON_COMMAND(ID_FILE_SAVE, OnFileSave)
	ON_COMMAND(ID_FILE_SAVE_AS, OnFileSaveAs)
	ON_COMMAND(ID_FILE_MOLECULE_AS, OnFileMoleculeAs)
	ON_UPDATE_COMMAND_UI(ID_FILE_MOLECULE_AS, OnUpdateFileMoleculeAs)
	ON_COMMAND(ID_FILE_ADD, OnFileAdd)
	ON_UPDATE_COMMAND_UI(ID_FILE_ADD, OnUpdateFileAdd)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CRasDoc construction/destruction

CRasDoc::CRasDoc()
{
	ViewActivationFlag = 2;
	m_PixMap = NULL;
	m_DocCopyNum = 1;
}


CRasDoc::~CRasDoc()	{}


/////////////////////////////////////////////////////////////////////////////
// Switching documents

// Manage the switch of RasMol between documents
void CRasDoc::ManageDisplay(BOOL ActivationFlag)
{	
	GETFRAME
	
	/* Activation of the view */
	if( ActivationFlag )	
	{	if( ViewActivationFlag != 1 )
			ViewActivationFlag = 1;
		else
			return;

		CheckActivationState();
		TRACE1( "TRACE\tView Activation\t%s\n", GetTitle() );

		if( m_PixMap )
		{	DeleteObject(m_PixMap);
			m_PixMap = NULL;
		}

		m_DataDisplay.RegenerateData();
		ResetUserSetBox( );
		if ( Database )
			ReDrawFlag |= RFRefresh | RFColour | RFInitial;

		GETVIEW
		if( pView!=NULL )
			pView->RefreshView();

	/* Deactivation of the view */
	} else		
	{	// Only one deactivation
		if( ViewActivationFlag != 0 )
			ViewActivationFlag = 0;
		else
			return;

		TRACE1( "TRACE\tView Deactivation\t%s\n", GetTitle() );

		if( Database )
		{	FixRange();
			HDC hDC = ::GetDC(NULL);
			if( PixMap )
			{	LPSTR Buffer = (LPSTR)::GlobalLock((HGLOBAL) FBufHandle); 
	
				m_PixMap = CreateDIBitmap( hDC, (BITMAPINFOHEADER __far *)BitInfo, 
				           CBM_INIT, Buffer, BitInfo, DIB_RGB_COLORS);
				if( m_PixMap==NULL )
       
// NS May 2002

					RasMolFatalExit ("Erreur: Memoire insuffisante!");
				GlobalUnlock(Buffer);
			}
			::ReleaseDC(NULL,hDC);
		}

		m_DataDisplay.SaveData();
		pFrame->m_wndCommandBar.ResetSpinMode();
		ZapDatabase(False);
	}
}

void CRasDoc::CheckActivationState()
{	/* Ensure that all Rasmol data are deactivated before activating
	   the current one */

	CRasDoc* pDoc;
	CDocTemplate* pDocTemp;
	pDocTemp = GetDocTemplate( );
	POSITION posdoc = pDocTemp->GetFirstDocPosition();
	while( posdoc != 0 )
	{	pDoc = (CRasDoc *)pDocTemp->GetNextDoc(posdoc);
		if( pDoc->ViewActivationFlag && (pDoc != this) )
			pDoc->ManageDisplay(false);
	}
}

void CRasDoc::FixRange()
{	/* Initialize Range according to saved XRange and YRange.
	   Done systematically before zapping the database.	*/

	XRange = m_DataDisplay.Data_XRange;
	YRange = m_DataDisplay.Data_YRange;
	Range =  MinFun(XRange,YRange);
	ZRange = Range;
	HRange = YRange>>1;
	WRange = XRange>>1;
	if( UseStereo )
	    SRange = WRange>>1;
	else
		SRange = WRange;
}

/////////////////////////////////////////////////////////////////////////////
// CRasDoc diagnostics

#ifdef _DEBUG
void CRasDoc::AssertValid() const
{	CDocument::AssertValid();	}

void CRasDoc::Dump(CDumpContext& dc) const
{	CDocument::Dump(dc);		}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////

/* ========= */
/* Documents */
/* ========= */

BOOL CRasDoc::OnNewDocument()
{	if( !CDocument::OnNewDocument() )
		return false;
	GETFRAME
	pFrame->ActivateCommandBar();
	pFrame->m_wndCommandBar.UpdatePannel();
	return true;
}

CRasDoc* CRasDoc::GetEmptyDoc()
{	
	GETFRAME

	// Check if active document is empty; if not make sure all are deactivated
	CRasFrame *pChild = (CRasFrame *) pFrame->GetActiveFrame();
	if( !pChild )
		return NULL;
	CRasDoc *pDoc = (CRasDoc *)pChild->GetActiveDocument( );
	if( pDoc )
	{	if( !MoleculeList && (pDoc->m_DataDisplay.Data_MoleculeList==NULL)  )
		{	return pDoc;
		} else
		{	pDoc->CheckActivationState();
			pDoc->ManageDisplay(false);
		}
	}
	
	// Check if any empty document
	POSITION tpos = AfxGetApp( )->GetFirstDocTemplatePosition( );
	CDocTemplate* pDocTemp = AfxGetApp( )->GetNextDocTemplate(tpos);
	POSITION posdoc = pDocTemp->GetFirstDocPosition( );
	int countdoc = 0;
	while( posdoc!=NULL )
	{	pDoc = (CRasDoc *)pDocTemp->GetNextDoc(posdoc);
		if( pDoc->m_DataDisplay.Data_MoleculeList!=NULL )
			pDoc = NULL;
		else
			break;
	}
	if( pDoc )
	{	// Activate the unique view		
		POSITION posview = pDoc->GetFirstViewPosition( );
		CRasView *pView = (CRasView *)pDoc->GetNextView( posview );
		ASSERT( posview == 0 && pView != NULL );
		CMDIChildWnd *pChild = (CMDIChildWnd *)pView->GetParentFrame( );
		pChild->MDIActivate();
		return pDoc;
	}

	// At last, create a new document
	((CRasTopApp*) AfxGetApp( ))->OnFileNewRasDoc();
	pChild = (CRasFrame *) pFrame->GetActiveFrame();
	pDoc = (CRasDoc *)pChild->GetActiveDocument( );		
	if( pDoc==NULL || MoleculeList) 
       
// NS May 2002

	{	WriteString("Erreur: Impossible de creer une nouvelle fenetre!\n");
		return NULL;
	}
	return pDoc;
}


void CRasDoc::OnCloseDocument() 
{	
	ManageDisplay(true);
	if( m_PixMap )
		DeleteObject(m_PixMap);
	CloseDisplay();
		
	if( MoleculeList )
	{	FixRange();
		ZapDatabase(True);

		// Free all rasmol memory allocations
		int mem = 0;
		CDocTemplate* pDocTemp = GetDocTemplate( );
		POSITION posdoc = pDocTemp->GetFirstDocPosition();
		while( posdoc )
		{	CRasDoc* pDoc = (CRasDoc *)pDocTemp->GetNextDoc(posdoc);
			if( pDoc->m_DataDisplay.Data_MoleculeList!=NULL )
				mem = 1;
		}
		if( !mem )
		{	ResetVoxelData();
			PurgeDatabase();
			UpdateRender(2);
			UpdateAbstree();
			UpdateRepres();
		}
	}
	
	SetPickMode(PickIdent);

	CDocument::OnCloseDocument();
}


/* ======== */
/* Openings */
/* ======== */

CRasDoc* CRasDoc::OpenRSMFile(CString& PathName)
{
	CRasDoc* pDoc = CRasDoc::GetEmptyDoc();
	if( pDoc == NULL )
		return NULL;
	if( !pDoc->ManageOpening(PathName) )
	{	WriteString("Warning: Failure in loading data!\n");
		return NULL;
	}
	return( pDoc );
}
// NS May 2002
void CRasDoc::OnFileOpenFromDoc() 
{		
	char BASED_CODE Filter[] = 
		 "Fichiers (*.rsm;*.pdb;*.cif;*.scr;*.top)|*.rsm; *.pdb; *.cif; *.scr; *.top|All Files (*.*)|*.*||";
 
	CFileDialog fdlg(true, ".rsm", NULL, OFN_HIDEREADONLY | OFN_FILEMUSTEXIST | 
		 OFN_OVERWRITEPROMPT , Filter, NULL);

	if( fdlg.DoModal()==IDOK )
		OpenRSMFile(fdlg.GetPathName()); 
}

int CRasDoc::CountDuplicates( void )
{	
	CDocTemplate* pDocTemp = GetDocTemplate( );
	POSITION posdoc = pDocTemp->GetFirstDocPosition( );
	int countdoc = 0;
	int max = 0;
	while( posdoc!=NULL )
	{	CRasDoc* pOtherDoc = (CRasDoc *)pDocTemp->GetNextDoc(posdoc);
		ASSERT(pOtherDoc->IsKindOf(RUNTIME_CLASS(CRasDoc)));
		if( pOtherDoc == this )
		{	countdoc++;
			continue;
		}
		if( !strcmp(SpaceName,pOtherDoc->m_DataDisplay.Data_SpaceName) )
		{	countdoc++;
			if( pOtherDoc->m_DocCopyNum > max )
				max = pOtherDoc->m_DocCopyNum;
		}
	}
	ASSERT(countdoc>=1);
	if( max >= countdoc )
		countdoc = max + 1;
	m_DocCopyNum = countdoc;
	return(countdoc);
}

int CRasDoc::ManageOpening(CString& PathName)
{	GETFRAME
	GETVIEW
	if( pView==NULL )
		return false;

    /* Get Extension */
    char path_buffer[_MAX_PATH];
    char ext[_MAX_EXT];
	strcpy(path_buffer,PathName);
    _splitpath(path_buffer,NULL,NULL,NULL,ext);

	/*=== Opening file according to extension ===*/
    CString Command;
	BeginWait();
    if( !strcmp(ext,".pdb") || !strcmp(ext,".cif") )
	    Command = "load " + PathName + "\n";
    else
	    Command = "script " + PathName + "\n";
	pView->ExecuteCommand(DisplayOff,Command);
	EndWait();

	if( Database==NULL )
		return false;
	
	if( CurMolSet && (WorldScriptFormat==RSMFormat) )
	{	
		/* Compatibility RasTop 1.3 Format */
		if( !*(CurMolSet->title) )
		{	strcpy(CurMolSet->title,SpaceName);
			SetCanvasTitle( );
		}

		FileModified = False;
	} else
		FileModified = True;

    // Add to recent FileList
	if( CountDuplicates()==1 )
		AfxGetApp()->AddToRecentFileList(PathName);

	// Tric to keep the pathname while not changing the title on screen
	SetPathName(( LPCTSTR)PathName, false);
	SetCanvasTitle( );

	// Force background drawing in 32 bits
	FBClear = false;
	pView->RefreshView();
	return true;
}

void CRasDoc::OnFileAdd() 
{	
// NS May 2002
//	char BASED_CODE Filter[] = 
//		 "RSM Scripts (*.rsm)|*.rsm|All Files (*.*)|*.*||";
	char BASED_CODE Filter[] = 
		 "Fichiers (*.rsm;*.pdb;*.cif)|*.rsm; *.pdb; *.cif|All Files (*.*)|*.*||";

	CFileDialog fdlg(true, ".rsm", NULL, OFN_HIDEREADONLY | OFN_FILEMUSTEXIST | 
		 OFN_OVERWRITEPROMPT , Filter, NULL);

	if( fdlg.DoModal()!=IDOK )
		return;
	
	GETFRAME
	GETVIEW
	if( pView==NULL )
		return;
	
	BeginWait();
	CString Command = "add " + fdlg.GetPathName() + "\n";
	pView->ExecuteCommand(DisplayOff,Command);
	EndWait();
}

void CRasDoc::OnUpdateFileAdd(CCmdUI* pCmdUI) 
{	pCmdUI->Enable( true );		}


/* =============== */
/* Saving Routines */
/* =============== */

int CRasDoc::GetSaveAsFilename( CString &Filename, CString &NewFilename )
{
	char BASED_CODE Filter[] = "All Files (*.*)|*.*||";
 
	Filename.LockBuffer();
	CFileDialog fdlg(false, NULL, (LPCSTR) Filename, OFN_HIDEREADONLY | 
		OFN_NOREADONLYRETURN | OFN_OVERWRITEPROMPT , Filter, NULL);
	Filename.UnlockBuffer();

	if( fdlg.DoModal()!=IDOK )
		return false;
	NewFilename = fdlg.GetPathName();
	return true;
}

void CRasDoc::OnFileSave() 
{	
	if( FileModified==False )
	{	WriteString("No modification to save!\n");
		return;
	} else if( WorldScriptFormat!=RSMFormat )
	{	OnFileSaveAs();
		return;
	} else if( m_DocCopyNum>1 ) 
	{	OnFileSaveAs();
		return;
    }
    
	GETFRAME
	GETVIEW
	if( pView==NULL )
       
// NS May 2002

	{	RasMolFatalExit("Erreur: enregistrement impossible!\n");
		return;
	}
	pView->ExecuteCommand(DisplayOff,"save\n");

}

void CRasDoc::OnUpdateFileSave(CCmdUI* pCmdUI) 
{	pCmdUI->Enable( CurMolSet!=NULL);	}

void CRasDoc::OnFileSaveAs() 
{	CString prompt, name, newname;
	
	STOPTIMER
	GETVIEW
	if( pView==NULL )
       
// NS May 2002

	{	RasMolFatalExit("Erreur: Enregistrement impossible!\n");
		STARTTIMER
		return;
	}

	if( m_DocCopyNum>1 )
		name.Format("%s copy%d",SpaceName,m_DocCopyNum);
	else
		name = SpaceName;
	AfxFormatString1(prompt, AFX_IDP_ASK_TO_SAVE, name);
	name += ".rsm";
	newname.Empty();
	
	if( !GetSaveAsFilename(name, newname) )
	{	STARTTIMER
		return;
	}				
	if( !newname.IsEmpty() )
	{	AfxGetApp()->AddToRecentFileList(newname);
	    newname = "save " + newname;
		newname += "\n";
		pView->ExecuteCommand(DisplayOff,newname);
    	CountDuplicates();
        SetCanvasTitle( );
    }
	STARTTIMER
}

void CRasDoc::OnUpdateFileSaveAs(CCmdUI* pCmdUI) 
{	pCmdUI->Enable( MoleculeList!=NULL );		}

BOOL CRasDoc::SaveModified() 
{	CString prompt, name, newname;
	
	GETFRAME
	GETVIEW
// NS May 2002
	if( pView==NULL )       
	{	RasMolFatalExit("Erreur: Enregistrement impossible!\n");
		return true;
	}

	/* Called when closing a document - Bypass MFCs */

	/* Empty document */
	if( !MoleculeList || !Database )
	{	return true;
	} else 

	/* Ongoing Script */
	if( FileDepth > -1 )
	{	return true;
	} else 

    /* Probably from a previous script  */
    if( CurMolSet && !CurMolSet->molfile )
	{	return true;
	} else 

	/* Unmodified Document, not a duplicat */
	if( !FileModified && (m_DocCopyNum==1) )
	{	return true;
	} else 

	/* Modified RSM , no duplicat */
	if( (WorldScriptFormat==RSMFormat) && (m_DocCopyNum==1) )
	{	STOPTIMER
		AfxFormatString1(prompt, AFX_IDP_ASK_TO_SAVE, SpaceName);
		switch (AfxMessageBox(prompt, MB_YESNOCANCEL, AFX_IDP_ASK_TO_SAVE))
		{
			case IDCANCEL:
				STARTTIMER
				return false;
	
			case IDYES:
				pView->ExecuteCommand(DisplayOff,"save\n");
				break;

			case IDNO:
				break;

			default:
				ASSERT(FALSE);
				break;
		}
		STARTTIMER
		return true;
	} else

	/* Duplicats and data files */
	if( (m_DocCopyNum>1) || (WorldScriptFormat==DataFormat))
	{	
		STOPTIMER
		if( m_DocCopyNum>1 )
			name.Format("%s copy%d",SpaceName,m_DocCopyNum);
		else
			name = SpaceName;
		AfxFormatString1(prompt, AFX_IDP_ASK_TO_SAVE, name);
		switch (AfxMessageBox(prompt, MB_YESNOCANCEL, AFX_IDP_ASK_TO_SAVE))
		{
			case IDCANCEL:
				STARTTIMER
				return false;
	
			case IDYES:
				name += ".rsm";
				newname.Empty();
				if( !GetSaveAsFilename(name, newname) )
				{	STARTTIMER
					return false;
				}				
				if( !newname.IsEmpty() )
				{	AfxGetApp()->AddToRecentFileList(newname);
				    newname = "save " + newname;
					newname += "\n";
					pView->ExecuteCommand(DisplayOff,newname);
				}
				break;

			case IDNO:
				break;

			default:
				ASSERT(FALSE);
				break;
		}
		STARTTIMER
		return true;
	} 

	return true;
}

// NS May 2002

void CRasDoc::OnFileMoleculeAs() 
{	CString prompt, name, newname;
	
	STOPTIMER
	GETVIEW
	if( pView==NULL )
	{	RasMolFatalExit("Erreur: enregistrement impossible!\n");
		STARTTIMER
		return;
	}

	if( CurMolSet->copynum>1 )
		name.Format("%s copy%d",CurMolSet->title,CurMolSet->copynum);
	else
		name = CurMolSet->title;
	AfxFormatString1(prompt, AFX_IDP_ASK_TO_SAVE, name);
	name += ".rsm";
	newname.Empty();
	
	if( !GetSaveAsFilename(name, newname) )
	{	STARTTIMER
		return;
	}				
	if( !newname.IsEmpty() )
	{	newname = "save molecule " + newname;
		newname += "\n";
		pView->ExecuteCommand(DisplayOff,newname);
	}
	STARTTIMER
}

void CRasDoc::OnUpdateFileMoleculeAs(CCmdUI* pCmdUI) 
{	pCmdUI->Enable( (MoleculeList!=NULL) && MoleculeList->next );		}


/* =================== */
/* Mol. File & Scripts */
/* =================== */

// NS May 2002

void CRasDoc::OnLoadMolecularfile() 
{	GETFRAME
	GETVIEW
	if( !pView )
		return;

	BOOL newdoc = (MoleculeList == NULL);

	WriteString(" \n");
	WriteString("Chargement de la molcule...\n");
	if ( !LoadInputFile(FormatPDB) )
	{	WriteString("Erreur : Chargement impossible\n");
		return;
	}

	if( newdoc )
		if( CountDuplicates()>1 )
			SetCanvasTitle( );

	pView->RefreshView();
	FileModified = true;
}

// NS May 2002

void CRasDoc::OnFileLoadRasmolscript() 
{	char ifilters[128];
    char* dst;
	*fnamebuf = '\0';

	GETFRAME
	GETVIEW
	if( pView==NULL )
		return;

	WriteString("Chargement du script...\n");

	dst = ifilters; 
	dst = RegisterFormat(dst,"RasMol Script Files (*.scr)","*.scr");
	dst = RegisterFormat(dst,"Top Script Files (*.top)","*.top");
	dst = RegisterFormat(dst,"All Files (*.*)","*.*");
	*dst = '\0';

	CFileDialog fdlg(true, NULL, NULL, 
					  OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT, NULL,  NULL);

	fdlg.m_ofn.lStructSize=sizeof(OPENFILENAME);
	fdlg.m_ofn.hwndOwner = NULL;
	fdlg.m_ofn.lpstrFilter = ifilters;
	fdlg.m_ofn.lpstrCustomFilter = NULL;
	fdlg.m_ofn.nFilterIndex = 1;
	fdlg.m_ofn.nMaxFile = 128;
	fdlg.m_ofn.lpstrFileTitle = NULL;
	fdlg.m_ofn.lpstrInitialDir = NULL;
	fdlg.m_ofn.lpstrTitle = "Choisir le fichier de script";
	fdlg.m_ofn.Flags |= OFN_FILEMUSTEXIST;
	fdlg.m_ofn.lpstrFile = fnamebuf;
	fdlg.m_ofn.lpstrDefExt = "scr";
	
	if( fdlg.DoModal()!=IDOK )
	{	WriteString("Chargement du script interrompu!\n");
		return;
	}

	CString Command = fdlg.GetPathName();
	Command ="script " + Command + "\n";
	pView->ExecuteCommand(DisplayOn,Command);
	
	if( !IsPaused )
		WriteString("Script execute!\n");
}


/* ====== */
/* Export */
/* ====== */

/* Return Adjusted ClipBoardFactor */
int CRasDoc::GetMaxCBFactor( void )
{
    register Real maxzoom, dialzoom;
    register int cbfactor;

    cbfactor = ClipBoardFactor;
    while( 1 )
    {   int done = 1;
        /* Check MaxZoom superior to one */
        maxzoom = 0.336*WorldSize/(Range*cbfactor);
        if( maxzoom < 1.0 )
            done = 0;
        /* Check DialValue[DialZoom] does not change */
        if( DialValue[3]>0.0 )
        {   dialzoom = DialValue[3];
            dialzoom *= MaxZoom/maxzoom;
            if( dialzoom>1.0 )
                done = 0;
        }
        if( !done )
            cbfactor -= 1;
        else
            break;
    }

// NS May 2002

    if( !cbfactor )
    {   WriteString("Erreur : chargement de l'image impossible!\n");
        return 0;
    }

    if( cbfactor!=ClipBoardFactor )
        WriteString("Attention : image trop grande, reduction de la taille par le zoom!\n");

    return cbfactor;
}


void CRasDoc::OnSaveImage() 
{	
    STOPTIMER
	CDialogImage dlg;
	dlg.m_SizeX = m_DataDisplay.Data_XRange;
	dlg.m_SizeY = m_DataDisplay.Data_YRange;
	if( dlg.DoModal()!=IDOK )
	{	STARTTIMER
		return;
	}

	CString ext;
	int num;
	switch( dlg.m_Format )
	{	case(0)	:	ext = "bmp";  num=1;	break;
		case(1)	:	ext = "gif";  num=2;	break;
		case(2)	:	ext = "ps";	  num=3;	break;
		case(3)	:	ext = "ps";	  num=4;	break;
		case(4)	:	ext = "ps";	  num=5;	break;
		case(5)	:	ext = "ppm";  num=6;	break;
		case(6)	:	ext = "ppm";  num=7;	break;
		case(7)	:	ext = "pic";  num=8;	break;
		case(8)	:	ext = "rgb";  num=9;	break;
		case(9)	:	ext = "ras";  num=10;	break;
		case(10):	ext = "ras";  num=11;	break;
	}

	char filename[128];
	CString fn = GetTitle();
	if( fn.Find('.')!= -1 )		fn = fn.Left(fn.Find('.'));
	if( fn.IsEmpty() )			fn = "notitle";
	if( fn.GetLength()>24 )		fn = fn.Left(24);
	if( !ext.IsEmpty() )		fn += "." + ext;
	fn.MakeLower( );
	strcpy(filename,fn.LockBuffer());
	fn.UnlockBuffer();

// NS May 2002

	char ofilters[512];
	char* dst;
	dst = ofilters;
    dst = RegisterFormat(dst,"Microsoft Bitmap","*.bmp");
    dst = RegisterFormat(dst,"CompuServe GIF","*.gif");
    dst = RegisterFormat(dst,"Colour PostScript","*.ps;*.eps");
    dst = RegisterFormat(dst,"Mono PostScript","*.ps;*.esp");
    dst = RegisterFormat(dst,"Vector PostScript","*.ps;*.eps");
    dst = RegisterFormat(dst,"Raw Portable Pixmap","*.ppm");
    dst = RegisterFormat(dst,"ASCII Portable Pixmap","*.ppm");
    dst = RegisterFormat(dst,"Apple Macintosh PICT","*.pic");
    dst = RegisterFormat(dst,"Silicon Graphics RGB","*.rgb");
    dst = RegisterFormat(dst,"RLE Sun Rasterfile","*.ras;*.im8");
    dst = RegisterFormat(dst,"Sun Rasterfile","*.ras");
    dst = RegisterFormat(dst,"Tous (*.*)","*.*");
    *dst = '\0';

	CFileDialog fdlg(false, NULL, (LPCSTR) fn, OFN_HIDEREADONLY | 
		OFN_NOREADONLYRETURN | OFN_OVERWRITEPROMPT ,  NULL,  NULL);

// NS May 2002

    fdlg.m_ofn.lStructSize=sizeof(OPENFILENAME);
    fdlg.m_ofn.hwndOwner = NULL;
    fdlg.m_ofn.lpstrFilter = ofilters;
    fdlg.m_ofn.lpstrCustomFilter = NULL;
    fdlg.m_ofn.nFilterIndex = num;
    fdlg.m_ofn.lpstrDefExt = ext;
    fdlg.m_ofn.nMaxFile = 128;
    fdlg.m_ofn.lpstrFileTitle = filename;
    fdlg.m_ofn.lpstrInitialDir = NULL;
    fdlg.m_ofn.lpstrTitle = "Choisir le repertoire et le nom de fichier";
    fdlg.m_ofn.Flags |= OFN_PATHMUSTEXIST;

	if (fdlg.DoModal()!=IDOK)
	{	STARTTIMER
		return;
	}

	if( (XRange != dlg.m_SizeX) || (YRange != dlg.m_SizeY) ) 
    {   Real maxzoom, dialzoom;
        int dx,xrange,yrange,range;

		xrange = dlg.m_SizeX;
		if( dx = xrange%4 )
			xrange += 4-dx;
		yrange = dlg.m_SizeY;
		range =  MinFun(xrange,yrange);

// NS May 2002

        /* Check MaxZoom superior to one */
        maxzoom = 0.336*WorldSize/range;
        if( maxzoom < 1.0 )
        {   WriteString("Erreur : Image trop grande!\n");
            return;
        }

// NS May 2002

        /* Check DialValue[DialZoom] does not change */
        if( DialValue[3]>0.0 )
        {   dialzoom = DialValue[3];
            dialzoom *= MaxZoom/maxzoom;
            if( dialzoom>1.0 )
            {   WriteString("Erreur : Image trop grande!\n");
                return;
            }
        }
    }

	BeginWait();

	BOOL changedim = false;
	if( (XRange != dlg.m_SizeX) || (YRange != dlg.m_SizeY) ) 
	{	changedim = true;
		XRange = dlg.m_SizeX;
		int dx;
		if( dx = XRange%4 )
			XRange += 4-dx;
		YRange = dlg.m_SizeY;
		Range =  MinFun(XRange,YRange);
		ZRange = Range;
		HRange = YRange>>1;
		WRange = XRange>>1;
		ReDrawFlag |= RFReSize;
	    RefreshState = True;
		ClearImage();
		RefreshScreen();
	}

	fn = fdlg.GetPathName();
	CString sext = fdlg.GetFileExt();
	int sresult = 0;
	if( dlg.m_Format == (int)(fdlg.m_ofn.nFilterIndex -1) )
	{	switch( dlg.m_Format )
		{   case(0):  sresult = WriteBMPFile((LPTSTR)fn.LockBuffer( ));             break;
		    case(1):  sresult = WriteGIFFile((LPTSTR)fn.LockBuffer( ));             break;
		    case(2):  sresult = WriteEPSFFile((LPTSTR)fn.LockBuffer( ),True,True);  break;
		    case(3):  sresult = WriteEPSFFile((LPTSTR)fn.LockBuffer( ),False,True); break;
		    case(4):  sresult = WriteVectPSFile((LPTSTR)fn.LockBuffer( ));          break;
		    case(5):  sresult = WritePPMFile((LPTSTR)fn.LockBuffer( ),True);        break;
		    case(6):  sresult = WritePPMFile((LPTSTR)fn.LockBuffer( ),False);       break;
		    case(7):  sresult = WritePICTFile((LPTSTR)fn.LockBuffer( ));            break;
		    case(8):  sresult = WriteIRISFile((LPTSTR)fn.LockBuffer( ));            break;
		    case(9):  sresult = WriteRastFile((LPTSTR)fn.LockBuffer( ),True);       break;
		    case(10): sresult = WriteRastFile((LPTSTR)fn.LockBuffer( ),False);      break;
		}
	} else if( sext == "bmp" )
	{	sresult = WriteBMPFile((LPTSTR)fn.LockBuffer( ));  
	} else if( sext == "gif" )
	{	sresult = WriteGIFFile((LPTSTR)fn.LockBuffer( )); 
	} else if( sext == "pic" )
	{	sresult = WritePICTFile((LPTSTR)fn.LockBuffer( ));   
	} else if( !((fdlg.m_ofn.nFilterIndex == 10) && sext.IsEmpty()) )
	{	switch(fdlg.m_ofn.nFilterIndex -1)
		{   case(0):  sresult = WriteBMPFile((LPTSTR)fn.LockBuffer( ));             break;
		    case(1):  sresult = WriteGIFFile((LPTSTR)fn.LockBuffer( ));             break;
		    case(2):  sresult = WriteEPSFFile((LPTSTR)fn.LockBuffer( ),true,true);  break;
		    case(3):  sresult = WriteEPSFFile((LPTSTR)fn.LockBuffer( ),false,true); break;
		    case(4):  sresult = WriteVectPSFile((LPTSTR)fn.LockBuffer( ));          break;
		    case(5):  sresult = WritePPMFile((LPTSTR)fn.LockBuffer( ),true);        break;
		    case(6):  sresult = WritePPMFile((LPTSTR)fn.LockBuffer( ),false);       break;
		    case(7):  sresult = WritePICTFile((LPTSTR)fn.LockBuffer( ));            break;
		    case(8):  sresult = WriteIRISFile((LPTSTR)fn.LockBuffer( ));            break;
		    case(9):  sresult = WriteRastFile((LPTSTR)fn.LockBuffer( ),true);       break;
		    case(10): sresult = WriteRastFile((LPTSTR)fn.LockBuffer( ),false);      break;
		}
	} else
		sresult = 0;

// NS May 2002

	if( !sresult )
		WriteString("Attention : Enregistrement impossible!\n");

	fn.UnlockBuffer( );

	if( changedim ) 
	{	FixRange();
		ReDrawFlag |= RFReSize;
        RefreshState = True;
		ClearImage();
		RefreshScreen();
	}
	EndWait();
	STARTTIMER
}

void CRasDoc::OnUpdateSaveImage(CCmdUI* pCmdUI) 
{	pCmdUI->Enable( (int)Database );	}

void CRasDoc::OnSaveMolecularfile() 
{	STOPTIMER
	CDialogSaveFile dlg;
	if( dlg.DoModal()!=IDOK )
	{	STARTTIMER
		return;
	}
	CString ext;
	int num;
	switch( dlg.m_Format )
	{	case(0)	:	ext = "pdb";	num = 1;	break;
		case(1)	:	ext = "mdl";	num = 2;	break;
		case(2)	:	ext = "xyz";	num = 3;	break;
		case(3)	:	ext = "cif";	num = 4;	break;
	}

	CString Filename = GetTitle();
	int d = Filename.Find('.');
	if(d != -1) Filename = Filename.Left(d);
	if(Filename.IsEmpty()) Filename = "notitle";
	if(Filename.GetLength()>24) Filename = Filename.Left(24);
	Filename += "." + ext;

// NS May 2002
	
	char ofilters[512];
	char* dst;
	dst = ofilters;
    dst = RegisterFormat(dst,"Brookhaven Databank (*.pdb;*.ent)","*.pdb;*.ent");
    dst = RegisterFormat(dst,"MDL Mol File Format (*.mdl;*.mol)","*.mdl;*.mol");
    dst = RegisterFormat(dst,"MSC (XMol) XYZ Format (*.xyz)","*.xyz");
    dst = RegisterFormat(dst,"Crystallographic Information File (*.cif)","*.cif");
    dst = RegisterFormat(dst,"Tous (*.*)","*.*");
    *dst = '\0';

	CFileDialog fdlg(false, NULL, (LPCSTR) Filename, OFN_HIDEREADONLY | 
		OFN_NOREADONLYRETURN | OFN_OVERWRITEPROMPT ,  NULL,  NULL);

// NS May 2002

    fdlg.m_ofn.lStructSize=sizeof(OPENFILENAME);
    fdlg.m_ofn.hwndOwner = NULL;
    fdlg.m_ofn.lpstrFilter = ofilters;
    fdlg.m_ofn.lpstrCustomFilter = NULL;
    fdlg.m_ofn.nFilterIndex = num;
    fdlg.m_ofn.lpstrDefExt = ext;
    fdlg.m_ofn.nMaxFile = 128;
    fdlg.m_ofn.lpstrFileTitle = NULL;
    fdlg.m_ofn.lpstrInitialDir = NULL;
    fdlg.m_ofn.lpstrTitle = "Choisir le repertoire et le nom de fichier";
    fdlg.m_ofn.Flags |= OFN_PATHMUSTEXIST;

	if( fdlg.DoModal()!=IDOK )
	{	STARTTIMER
		return;
	}

	Filename = fdlg.GetPathName();
	CString sext = fdlg.GetFileExt();
	int sresult = 0;
	if( sext == "pdb" )
		sresult = SavePDBMolecule(( LPTSTR) Filename.LockBuffer());
	else if( sext == "mdl" )
		sresult = SaveMDLMolecule(( LPTSTR) Filename.LockBuffer());
	else if( sext == "xyz" )
		sresult = SaveXYZMolecule(( LPTSTR) Filename.LockBuffer()); 
	else if( sext == "cif" )
		sresult = SaveCIFMolecule(( LPTSTR) Filename.LockBuffer());
	else if( (sext != "pdb")&&(sext != "mdl")&&(sext != "xyz")&&(sext != "cif") )
	{	switch(dlg.m_Format)
		{	case(0)	:	sresult = SavePDBMolecule(( LPTSTR) Filename.LockBuffer());	break;
			case(1)	:	sresult = SaveMDLMolecule(( LPTSTR) Filename.LockBuffer()); break;
			case(2)	:	sresult = SaveXYZMolecule(( LPTSTR) Filename.LockBuffer()); break;
			case(3)	:	sresult = SaveCIFMolecule(( LPTSTR) Filename.LockBuffer()); break;
		}
	}

// NS May 2002
	
	if( !sresult )
		WriteString("Attention : Enregistrement impossible!\n");
	Filename.UnlockBuffer( );
	STARTTIMER
}

void CRasDoc::OnUpdateSaveMolecularfile(CCmdUI* pCmdUI) 
{	pCmdUI->Enable( (int)Database );	}

void CRasDoc::OnSaveScript() 
{	STOPTIMER
	CDialogScript dlg;
	if( dlg.DoModal()!=IDOK )
	{	STARTTIMER
		return;
	}
	CString ext;
	int num;
	switch( dlg.m_Format )
	{	case(0)	:	ext = "scr";	num = 1;	break;
		case(1)	:	ext = "mol";	num = 2;	break;
		case(2)	:	ext = "kin";	num = 3;	break;
		case(3)	:	ext = "pov";	num = 4;	break;
		case(4)	:	ext = "pov";	num = 5;	break;
		case(5)	:	ext = "vrm";	num = 6;	break;
		case(6)	:	ext = "txt";	num = 7;	break;
		case(7)	:	ext = "txt";	num = 8;	break;
		case(8)	:	ext = "txt";	num = 9;	break;
	}
	
	CString Filename = GetTitle();
	int d = Filename.Find('.');
	if( d != -1 )
		Filename = Filename.Left(d);
	if( Filename.IsEmpty() )
		Filename = "notitle";
	if( Filename.GetLength()>24 )
		Filename = Filename.Left(24);
	Filename += "." + ext;

// NS May 2002

	char ofilters[512];
	char* dst;
	dst = ofilters;
    dst = RegisterFormat(dst,"Rasmol Script file (*.scr)","*.scr");
    dst = RegisterFormat(dst,"MolScript file (*.mol)","*.mol");
    dst = RegisterFormat(dst,"Kinemage file (*.kin)","*.kin");
    dst = RegisterFormat(dst,"POVRay2 file (*.pov)","*.pov");
    dst = RegisterFormat(dst,"POVRay3 file (*.pov)","*.pov");
    dst = RegisterFormat(dst,"VMRL file (*.vmr)","*.vmr");
    dst = RegisterFormat(dst,"Phipsi (*.txt)","*.txt");
    dst = RegisterFormat(dst,"Ramachandran file (*.txt)","*.txt");
    dst = RegisterFormat(dst,"Ramachandran plot (*.txt)","*.txt");
    dst = RegisterFormat(dst,"Tous (*.*)","*.*");
    *dst = '\0';

// NS May 2002

	CFileDialog fdlg(false, NULL, (LPCSTR) Filename, OFN_HIDEREADONLY | OFN_NOREADONLYRETURN | 
	OFN_OVERWRITEPROMPT ,  NULL,  NULL);

    fdlg.m_ofn.lStructSize=sizeof(OPENFILENAME);
    fdlg.m_ofn.hwndOwner = NULL;
    fdlg.m_ofn.lpstrFilter = ofilters;
    fdlg.m_ofn.lpstrCustomFilter = NULL;
    fdlg.m_ofn.nFilterIndex = num;
    fdlg.m_ofn.lpstrDefExt = ext;
    fdlg.m_ofn.nMaxFile = 128;
    fdlg.m_ofn.lpstrFileTitle = NULL;
    fdlg.m_ofn.lpstrInitialDir = NULL;
    fdlg.m_ofn.lpstrTitle = "Choisir le repertoire et le nom de fichier";
    fdlg.m_ofn.Flags |= OFN_PATHMUSTEXIST;

	if( fdlg.DoModal() != IDOK )
	{	STARTTIMER
		return;
	}

	Filename = fdlg.GetPathName();
	CString sext = fdlg.GetFileExt();
	int sresult = 0;
	if( sext == "scr" )
		sresult = WriteScriptFile((LPTSTR) Filename.LockBuffer(),false,false);
	else if( sext == "mol" )
		sresult = WriteMolScriptFile((LPTSTR) Filename.LockBuffer());
	else if( sext == "kin" )
		sresult = WriteKinemageFile((LPTSTR) Filename.LockBuffer()); 
	else if( sext == "vrm" )
		sresult = WriteVRMLFile((LPTSTR) Filename.LockBuffer()); 
	else
	{	
	switch( dlg.m_Format )
	{	
	case(0)	:	sresult = WriteScriptFile((LPTSTR) Filename.LockBuffer(),false,false);
				break;
	case(1)	:	sresult = WriteMolScriptFile((LPTSTR) Filename.LockBuffer());
				break;
	case(2)	:	sresult = WriteKinemageFile((LPTSTR) Filename.LockBuffer());
				break;
	case(3)	:	sresult = WritePOVRayFile((LPTSTR) Filename.LockBuffer());
				break;
	case(4)	:	sresult = WritePOVRay3File((LPTSTR) Filename.LockBuffer());
				break;
	case(5)	:	sresult = WriteVRMLFile((LPTSTR) Filename.LockBuffer());
				break;
	case(6)	:	sresult = WritePhiPsiAngles((LPTSTR) Filename.LockBuffer(), false);
				break;
	case(7)	:	sresult = WritePhiPsiAngles((LPTSTR) Filename.LockBuffer(), 1);
				break;
	case(8)	:	sresult = WritePhiPsiAngles((LPTSTR) Filename.LockBuffer(), -1);
				break;
	}
	}

// NS May 2002
	
	if( !sresult )
		WriteString("Attention : Enregistrement impossible!\n");
	Filename.UnlockBuffer();
	STARTTIMER
}

void CRasDoc::OnUpdateSaveScript(CCmdUI* pCmdUI) 
{	pCmdUI->Enable( (int)Database );	}


///////////////////////////////////////////////////////////////////////////////////

char *RegisterFormat( char *buffer, char *desc, char *ext )
{
    while( *buffer++ = *desc++ );
    while( *buffer++ = *ext++ );
    return( buffer );
}

BOOL LoadInputFile(int format )
{	static char ifilters[512];
	register char *ext;
    register int num;
    char* dst;
    *fnamebuf = '\0';

    switch( format )
    {   case(FormatPDB):      ext = "PDB";  num = 1;  break;
		case(FormatAlchemy):  ext = "MOL";  num = 2;  break;
		case(FormatMol2):     ext = "MOL";  num = 3;  break;
		case(FormatMDL):      ext = "MOL";  num = 4;  break;
		case(FormatXYZ):      ext = "XYZ";  num = 5;  break;
		case(FormatCharmm):   ext = "CHM";  num = 6;  break;
        case(FormatMOPAC):    ext = "MOP";  num = 7;  break;
		case(FormatCIF):      ext = "CIF";  num = 8;  break;
    }

   /* filter for file opening */

// NS May 2002

    dst = ifilters;      
    dst = RegisterFormat(dst,"Brookhaven Databank (*.pdb;*.ent)","*.pdb;*.ent");
    dst = RegisterFormat(dst,"Alchemy File Format (*.alc;*.mol)","*.alc;*.mol");
    dst = RegisterFormat(dst,"Sybyl MOL2 Format (*.syb;*.mol)","*.syb;*.mol");
    dst = RegisterFormat(dst,"MDL Mol File Format (*.mdl;*.mol)","*.mdl;*.mol");
    dst = RegisterFormat(dst,"MSC (XMol) XYZ Format (*.xyz)","*.xyz");
    dst = RegisterFormat(dst,"CHARMm File Format (*.chm)","*.chm");
    dst = RegisterFormat(dst,"MOPAC File Format (*.mop)","*.mop");
    dst = RegisterFormat(dst,"CIF File Format  (*.cif)","*.cif");
    dst = RegisterFormat(dst,"Tous (*.*)","*.*");
    *dst = '\0';

	CFileDialog fdlg(true,NULL,NULL,OFN_HIDEREADONLY | 
	OFN_OVERWRITEPROMPT, NULL, NULL);

// NS May 2002

    fdlg.m_ofn.lStructSize=sizeof(OPENFILENAME);
    fdlg.m_ofn.hwndOwner = NULL;
    fdlg.m_ofn.lpstrFilter = ifilters;
    fdlg.m_ofn.lpstrCustomFilter = NULL;
    fdlg.m_ofn.nFilterIndex = num;
    fdlg.m_ofn.nMaxFile = 128;
    fdlg.m_ofn.lpstrFileTitle = NULL;
    fdlg.m_ofn.lpstrInitialDir = NULL;
    fdlg.m_ofn.lpstrTitle = "Choisir le fichier";
    fdlg.m_ofn.Flags |= OFN_FILEMUSTEXIST;
    fdlg.m_ofn.lpstrFile = fnamebuf;

    fdlg.m_ofn.lpstrDefExt = ext;

    RefreshState = False;
	if (fdlg.DoModal()!=IDOK)
    {   RefreshState = True;
    	return false;
    }
    RefreshState = True;

	BeginWait( );

	ScriptFileFormat = DataFormat;
	CString cext = fdlg.GetFileExt();
	int Success;
	if( (cext == "pdb")||(cext == "ent") )
	{	Success = FetchFile(FormatPDB,true,fnamebuf);
	} else if( cext == "alc" )
	{	Success = FetchFile(FormatAlchemy,true,fnamebuf);
	} else if( cext == "syb" )
	{	Success = FetchFile(FormatMol2,true,fnamebuf);
	} else if( cext == "mdl" )
	{	Success = FetchFile(FormatMDL,true,fnamebuf); 
	} else if( cext == "xyz" )
	{	Success = FetchFile(FormatXYZ,true,fnamebuf);
	} else if( cext == "chm" )
	{	Success = FetchFile(FormatCharmm,true,fnamebuf); 
	} else if( cext == "mop" )
	{	Success = FetchFile(FormatMOPAC,true,fnamebuf);
	} else if( cext == "cif" )
	{	Success = FetchFile(FormatCIF,true,fnamebuf);
	} else 
	{	switch( fdlg.m_ofn.nFilterIndex )
		{   case(1): Success = FetchFile(FormatPDB,true,fnamebuf);		break;
		    case(2): Success = FetchFile(FormatAlchemy,true,fnamebuf);	break;
		    case(3): Success = FetchFile(FormatMol2,true,fnamebuf);		break;
		    case(4): Success = FetchFile(FormatMDL,true,fnamebuf);		break;
		    case(5): Success = FetchFile(FormatXYZ,true,fnamebuf);		break;
		    case(6): Success = FetchFile(FormatCharmm,true,fnamebuf);	break;
		    case(7): Success = FetchFile(FormatMOPAC,true,fnamebuf);	break;
		    case(8): Success = FetchFile(FormatCIF,true,fnamebuf);		break;
			default: Success = FetchFile(FormatPDB,true,fnamebuf);    
		}
	}

	EndWait( );
	DefaultRepresentation();
	return Success;
}
