package edu.sdsc.sirius.contact;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import edu.sdsc.sirius.io.Entry;
import edu.sdsc.sirius.io.FragmentEntry;
import edu.sdsc.sirius.io.StructureEntry;
import edu.sdsc.sirius.util.*;
import edu.sdsc.mbt.*;
import edu.sdsc.sirius.viewers.*;
import edu.sdsc.mbt.viewables.*;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.*;
import java.util.prefs.Preferences;

public class ContactMapViewer extends JDialog implements Viewer, StructureStylesEventListener {
	
	private Manager parent;
	private StructureViewer viewer;
	
	private int width = 550;
	private int height = 600;
	
	private JPanel basePanel;
	
	private Structure structure1;
	private Structure structure2;
	
	private ContactPanel panel1;
	private ContactPanel panel2;
	private ContactPanel difPanel;
	
	private JTabbedPane tabbedPane = new JTabbedPane();
	
	private Vector panels = new Vector();//ContactPanel's
	
	private JCheckBoxMenuItem viewZoomTwo;
	private JCheckBoxMenuItem viewZoomThree;
	private JCheckBoxMenuItem viewZoomFour;
	private JCheckBoxMenuItem viewZoomFive;
	private JCheckBoxMenuItem viewZoomTen;
	
	public ContactMapViewer(Manager m){
		
		super(m.getApplicationFrame(), "Contact Map Viewer", false);
		
		this.parent = m;
		viewer = parent.getStructureViewer();
		
		basePanel = new JPanel();
		basePanel.setBounds(0,0,width-5, height-5);
		basePanel.setLayout(new BorderLayout());
		
		tabbedPane.setBounds(0,0,width-5, height-5);
		basePanel.add(tabbedPane, BorderLayout.CENTER);
		
		tabbedPane.addChangeListener(new ChangeListener(){
			public void stateChanged(ChangeEvent e){
				int index = tabbedPane.getSelectedIndex();
				
				if (index < 0) return;
				
//				System.out.println("tab event from " + index + ", title = " + tabbedPane.getTitleAt(index));
				int ratio = ((ContactPanel)panels.get(index)).getPixelRatio();
				if (ratio == 2){
					viewZoomTwo.setSelected(true);
					viewZoomThree.setSelected(false);
					viewZoomFour.setSelected(false);
					viewZoomFive.setSelected(false);
					viewZoomTen.setSelected(false);
				}
				else if (ratio == 3){
					viewZoomTwo.setSelected(false);
					viewZoomThree.setSelected(true);
					viewZoomFour.setSelected(false);
					viewZoomFive.setSelected(false);
					viewZoomTen.setSelected(false);
				}
				else if (ratio == 4){
					viewZoomTwo.setSelected(false);
					viewZoomThree.setSelected(false);
					viewZoomFour.setSelected(true);
					viewZoomFive.setSelected(false);
					viewZoomTen.setSelected(false);
				}
				else if (ratio == 5){
					viewZoomTwo.setSelected(false);
					viewZoomThree.setSelected(false);
					viewZoomFour.setSelected(false);
					viewZoomFive.setSelected(true);
					viewZoomTen.setSelected(false);
				}
				else if (ratio == 10){
					viewZoomTwo.setSelected(false);
					viewZoomThree.setSelected(false);
					viewZoomFour.setSelected(false);
					viewZoomFive.setSelected(false);
					viewZoomTen.setSelected(true);
				}
			}
		});
		
		//menu
		getContentPane().setLayout(new BorderLayout());
		getContentPane().add(createMenuBar(), BorderLayout.NORTH);
		getContentPane().add(basePanel, BorderLayout.CENTER);
		
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				destructor();
			}
		});
		

		
		//position the window
		Dimension dim = getToolkit().getScreenSize();
		
		//position the panel next to the main window on the right
		//if it ends up being wider than screen, fit it to the width of the screen
		if ((parent.getApplicationFrame().getX() + parent.getApplicationFrame().getWidth() + width) > dim.getWidth()){
			setBounds(dim.width - width, parent.getApplicationFrame().getY(), width, height);
		}
		else{
			setBounds(parent.getApplicationFrame().getWidth(), parent.getApplicationFrame().getY(), width, height);
		}
		
		setSize(width,height);
		setVisible(true);
		
	}
	
	
	
	
	public void setStructures(Structure s1, Structure s2){
		
		//check whether there is already graph for one of the structures
		if (structure1 == s1 && structure2 == s2) return;//everything is already done
		
		tabbedPane.removeAll();
		panels.clear();
		
		structure1 = s1;
		structure2 = s2;
		
		if (structure1 != null){
			panel1 = new ContactPanel(this);
			panel1.setSize(width-5, height-5);
			panel1.setStructures(structure1, null);
			
			panels.add(panel1);
			
			tabbedPane.addTab(" " + viewer.getStructureName(structure1) + " ", panel1);
			
			structure1.getStructureMap().getStructureStyles().addStructureStylesEventListener(this);
		}
		
		if (structure2 != null){
			//create the second panel and difference panel
			panel2 = new ContactPanel(this);
			panel2.setSize(width-5, height-5);
			panel2.setStructures(structure2, null);
			
			panels.add(panel2);
			
			tabbedPane.addTab(" " + viewer.getStructureName(structure2) + " ", panel2);
			
			structure2.getStructureMap().getStructureStyles().addStructureStylesEventListener(this);
			
			
			//create another tab for difference graph
			
			difPanel = new ContactPanel(this);
			difPanel.setSize(width-5, height-5);
			Thread runner = new Thread(){
				public void run(){
					difPanel.setStructures(structure1, structure2);
				}
			};
			runner.start();
			panels.add(difPanel);
			
			tabbedPane.addTab(" " + viewer.getStructureName(structure1) + " - " + viewer.getStructureName(structure2) + " ", difPanel);
			
			
		}
		
	}

	private JMenuBar createMenuBar(){
		
		JMenuBar bar = new JMenuBar();
		
		JMenu file = new JMenu("File");
		file.setMnemonic('f');
		JMenuItem fileSave = new JMenuItem("Save graph...");
		fileSave.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				
			}
		});
		
		JMenuItem fileClose = new JMenuItem("Close");
		fileClose.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){

			}
		});
		
		file.add(fileSave);
		file.add(fileClose);
		
		JMenu view = new JMenu("View");
		view.setMnemonic('v');
		JMenu viewZoom = new JMenu("Data point size");
		viewZoomTwo = new JCheckBoxMenuItem("2 pixels");
		viewZoomTwo.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				viewZoomThree.setSelected(false);
				viewZoomFour.setSelected(false);
				viewZoomFive.setSelected(false);
				viewZoomTen.setSelected(false);
				setPixelRatio(2);
			}
		});
		viewZoomThree = new JCheckBoxMenuItem("3 pixels");
		viewZoomThree.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				viewZoomTwo.setSelected(false);
				viewZoomFour.setSelected(false);
				viewZoomFive.setSelected(false);
				viewZoomTen.setSelected(false);
				setPixelRatio(3);
			}
		});
		viewZoomFour = new JCheckBoxMenuItem("4 pixels");
		viewZoomFour.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				viewZoomThree.setSelected(false);
				viewZoomTwo.setSelected(false);
				viewZoomFive.setSelected(false);
				viewZoomTen.setSelected(false);
				setPixelRatio(4);
			}
		});
		viewZoomFive = new JCheckBoxMenuItem("5 pixels");
		viewZoomFive.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				viewZoomThree.setSelected(false);
				viewZoomFour.setSelected(false);
				viewZoomTwo.setSelected(false);
				viewZoomTen.setSelected(false);
				setPixelRatio(5);
			}
		});
		viewZoomTen = new JCheckBoxMenuItem("10 pixels");
		viewZoomTen.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				viewZoomThree.setSelected(false);
				viewZoomFour.setSelected(false);
				viewZoomFive.setSelected(false);
				viewZoomTwo.setSelected(false);
				setPixelRatio(10);
			}
		});
		
		viewZoom.add(viewZoomTwo);
		viewZoom.add(viewZoomThree);
		viewZoom.add(viewZoomFour);
		viewZoom.add(viewZoomFive);
		viewZoom.add(viewZoomTen);
		
		view.add(viewZoom);

		bar.add(file);
		bar.add(view);
		
		return bar;
	}
	
	private void setPixelRatio(int ratio){
		ContactPanel p = (ContactPanel)tabbedPane.getSelectedComponent();
		p.setPixelRatio(ratio);
	}
	
	public void processStructureDocumentEvent(StructureDocumentEvent event){
		
		//structure added events are ignored, since opening this panel doesn't have to happen as soon as
		//structure is loaded
		//however, once the structure is deleted, the corresponding data structure should be cleared to avoid 
		//memory leaks (or at least having unnecessary data in memory), plus remove the corresponding tab
		
		int type = event.type;
		int change = event.change;

		
		if ( type == StructureDocumentEvent.TYPE_DATA ){
			Entry e = event.entry;
			Structure structure = null;
			if (e instanceof StructureEntry){
				StructureEntry entry = (StructureEntry)event.entry;
				structure = entry.getStructure();
			}
			else if (e instanceof FragmentEntry){
				FragmentEntry entry = (FragmentEntry)event.entry;
				structure = entry.getStructure();
			}
			
			if (structure == null){
				return;
			}
			
			if (structure.getStructureMap() == null || structure.getStructureMap().getAtomCount() == 0){
				return;
			}
			
			if ( change == StructureDocumentEvent.CHANGE_REMOVED ){
				
				//remove the tab and any data on this structure
				if (structure1 == structure){
					//get rid of the first tab and difference tab, if any
					if (tabbedPane.getTabCount() == 3){
						ContactPanel panel = (ContactPanel)panels.get(2);
						panel.destructor();
						tabbedPane.removeTabAt(2);
						panels.remove(2);
						panel = null;
					}
					
					ContactPanel panel = (ContactPanel)panels.get(0);
					panel.destructor();
					tabbedPane.removeTabAt(0);
					structure1.getStructureMap().getStructureStyles().removeStructureStylesEventListener(this);
					panels.remove(0);
					panel = null;
					structure1 = null;
					tabbedPane.revalidate();
					if (tabbedPane.getTabCount() > 0) tabbedPane.setSelectedIndex(0);
				}
				else if (structure2 == structure){
					//remove second tab and difference tab
					if (tabbedPane.getTabCount() == 3){
						ContactPanel panel = (ContactPanel)panels.get(2);
						panel.destructor();
						tabbedPane.removeTabAt(2);
						panels.remove(2);
						panel = null;
					}
					
					ContactPanel panel = (ContactPanel)panels.get(panels.size()-1);
					panel.destructor();
					tabbedPane.removeTabAt(panels.size()-1);
					structure2.getStructureMap().getStructureStyles().removeStructureStylesEventListener(this);
					panels.remove(panels.size()-1);
					panel = null;
					structure2 = null;
					tabbedPane.revalidate();
					if (tabbedPane.getTabCount() > 0) tabbedPane.setSelectedIndex(0);
				}
				
				if (tabbedPane.getTabCount() == 0){
					//close this whole thing
					panels.clear();
					structure1 = null;
					structure2 = null;

					dispose();
				}
			}
			else if (change == StructureDocumentEvent.CHANGE_RENAMED){
				if (structure1 == structure){
					//rename tab 1
					

				}
			}
			
			
		}
		else if (type == StructureDocumentEvent.TYPE_GLOBAL){
			if (change == StructureDocumentEvent.CHANGE_RELOAD){
				//ignore: this is for sequence viewer
			}
		}

		
	}
	
	public void processStructureStylesEvent(StructureStylesEvent event){
		
		//process only selection events at atom and residue levels
		StructureComponent structureComponent = event.structureComponent;
		StructureStyles styles = event.structureStyles;
		
		if (structureComponent == null){
			
		}
		
		
	}
	
	public void destructor(){
		if (structure1 != null && structure2 != null){
			
		}
		else if (structure1 != null){
			
			structure1.getStructureMap().getStructureStyles().removeStructureStylesEventListener(this);
			panels.remove(0);
		}
		else if (structure2 != null){
			structure1.getStructureMap().getStructureStyles().removeStructureStylesEventListener(this);
			panels.remove(panels.size()-1);
		}
		
		tabbedPane.removeAll();
		panels.clear();
		structure1 = null;
		structure2 = null;
		setVisible(false);
		dispose();
	}

	protected HashMap alignStructures(Structure structure1, Structure structure2){
		
		HashMap out = new HashMap();//Structure -> String sequence from the alignment
		
		//get the location of the TM directory
		Preferences root = Preferences.userRoot();
		final Preferences node = root.node("/edu/sdsc/sirius/config");
		String path = node.get("installDir", null);

		if (path == null){
			//use sequence alignment instead
			
			
			
			
			
			
			return out;
		}
		
		String tm = path + File.separator + "tm" + File.separator;
		String name1 = tm + viewer.getStructureName(structure1) + ".pdb";
		String name2 = tm + viewer.getStructureName(structure2) + ".pdb";

		String os = System.getProperty("os.name");
		String arch = System.getProperty("os.arch");
		
		String rmsd = null;
		
		String binary = null;
		if (os.startsWith("Win")){
			binary = "TMalignWin.exe";
		}
		else if (os.startsWith("Linux")){
			binary = "TMalign_32";
		}
		else if (os.startsWith("Mac")){
			if (arch.startsWith("x86")){
				binary = "TMalignIntel";
			}
			else{
				binary = "TMalignPPC";
			}
		}
		
		if (binary == null){
			//fall back to sequence alignment
			//if that fails, bail out
			
			
			
			return out;
		}
		
		File output = null;
		
		try{
			IOHandler.savePdb(structure1, name1, parent, null, false, false);
			IOHandler.savePdb(structure2, name2, parent, null, false, false);
		
			String[] c = new String[5];
			c[0] = tm + binary;
			c[1] = name1;
			c[2] = name2;
			c[3] = "-o";
			c[4] = tm + "tm.sup";
						
			Process process = Runtime.getRuntime().exec(c);
			
			InputStream in = process.getInputStream();
			BufferedReader reader1 = new BufferedReader(new InputStreamReader(in));
			String linea = null;
			String[] dump = new String[2];
			boolean append = false;
			boolean passed = false;
			int i = 0;
			while ((linea = reader1.readLine()) != null){
				if (append){
					dump[i] = linea;
					i++;
					append = false;
					passed = true;
					continue;
				}
				if (linea.startsWith("(")) append = true;
				if (passed) append = true;
			}
	        in.close();
	        
					
			output = new File(tm + "tm.sup");

			String seq1 = dump[0];
			String seq2 = dump[1];
			
			//the following is to be done if sequence viewer should be updated
			parent.openSequence();
			Entry e1 = viewer.getEntry(structure1);
			Entry e2 = viewer.getEntry(structure2);
			
			HashMap set = new HashMap();
			set.put(e1, seq1);
			set.put(e2, seq2);
			
			Vector order = new Vector();
			order.add(e1);
			order.add(e2);
			
			parent.getSequenceViewer().setSequences(set, order);
			
			//use the sequence viewer mechanism for restoring # entries, and get sequences from there
			Vector sequences = parent.getSequenceViewer().getSequenceObjects();
			HashMap entries = parent.getSequenceViewer().getEntries();
			for (int j = 0; j < sequences.size(); j++){
				Sequence ss = (Sequence)sequences.get(j);
				Entry ee = (Entry)entries.get(ss);
				
				if (ee == e1){
					//this is the first sequence
					out.put(structure1, ss.getSequence());
				}
				if (ee == e2){
					//this is the first sequence
					out.put(structure2, ss.getSequence());
				}
			}

			//update the structures, as well
			//read the content
			BufferedReader reader = new BufferedReader(new FileReader(output));
			String liner = null;
			boolean read = false;
			
			HashMap map = new HashMap();//residue number (String) -> double[] coordinates of CA
			
			String[] numbers = new String[4];
			Vector result = new Vector();//resulting coordinates
			int j = 0;
			while ((liner = reader.readLine()) != null){
				if (liner.startsWith("REMARK Aligned")){
					
					//read rmsd
					int p = liner.indexOf("RMSD");
					String mm = liner.substring(p);
					rmsd = mm.substring(mm.indexOf("=")+1, mm.indexOf(",")).trim();
					
					
					
					read = true;
					continue;
				}
				
				if (liner.startsWith("CONECT")){
					break;
				}
				
				if (j == 4) break;
				
				if (read){
					//read in the record for the current CA
					if (liner.startsWith("ATOM")){
						String number = liner.substring(22,26);
						String xs = liner.substring(30,38);
						String ys = liner.substring(38,46);
						String zs = liner.substring(46,52);
						
						try{
							double x = Double.parseDouble(xs);
							double y = Double.parseDouble(ys);
							double z = Double.parseDouble(zs);
							
							numbers[j] = number.trim();
							result.add(new double[]{ x, y, z });
							j++;
						}
						catch (NumberFormatException ex){
							System.out.println("error parsing atom record");
						}
					}
				}
			}
			
			reader.close();

			//generate the three vectors and make the corresponding vectors from the same atoms before
			//the transformation
			Vector initial = new Vector();//initial coordinates
			
			//scan through all residues in the initial structure to make sure the correct CA's are chosen
			int found = 0;//how many residues have been matched
			for (int n = 0; n < structure1.getStructureMap().getResidueCount(); n++){
				Residue r = structure1.getStructureMap().getResidue(n);
				int num = r.getResidueId();
				
//				System.out.println("r = " + r.getCompoundCode() + ", num = " + num);
				
				String Num = (new Integer(num)).toString();
				for (int k = 0; k < numbers.length; k++){
					if (Num.equals((String)numbers[k])){
						found++;
						Atom ca = r.getAlphaAtom();
						if (ca == null) break;
//						System.out.println("found");
						initial.add(k, ca.coordinate);
						break;
					}
				}
				
				if (found == 4) break;
					
					
			}
			
			//now get the three vectors to be used to determine the transformation matrix.
			//0 -> 1, 0 -> 2, 0 -> 3
			Vector initialVectors = new Vector();
			double[] start1 = (double[])initial.get(0);
			for (int k = 1; k < initial.size(); k++){
				double[] end = (double[])initial.get(k);
				double[] v = new double[]{ end[0] - start1[0], end[1] - start1[1], end[2] - start1[2] };
				initialVectors.add(v);
			}
			
			Vector resultVectors = new Vector();
			double[] start2 = (double[])result.get(0);
			for (int k = 1; k < result.size(); k++){
				double[] end = (double[])result.get(k);
				double[] v = new double[]{ end[0] - start2[0], end[1] - start2[1], end[2] - start2[2] };
				resultVectors.add(v);
			}
			
			//now with this set of vectors, calculate the matrix
			Location start = new Location((double[])initialVectors.get(0), (double[])initialVectors.get(1), start1);
			Location target = new Location((double[])resultVectors.get(0), (double[])resultVectors.get(1), start2);
			
			StructureMath.transformStructure(start, target, structure1);
			
			viewer.resetView();
			viewer.updateAppearance();

			
			
		}
		catch (Exception ex){
			//fall back to sequence alignment
		}
		finally{
			if (output != null) output.delete();
			
			File n1 = new File(name1);
			n1.delete();
			
			File n2 = new File(name2);
			n2.delete();

		}
		
		return out;
		
	}

}