package edu.sdsc.sirius.contact;

import javax.swing.*;
import java.awt.event.*;
import edu.sdsc.mbt.*;
import javax.swing.event.*;

import java.util.*;
import java.awt.*;

/**
 * This class encapsulates the image panel used in a tab of the tabbed pane of the contact map viewer.
 * The panel consists of a JLayeredPane that includes the actual image layer and the selection layer used to
 * draw selection markers
 * 
 * @author Sasha Buzko
 *
 */
public class ContactPanel extends JPanel {
	
	private ContactMapViewer viewer;
	
	private JLayeredPane layeredPane;
	private boolean difference = false;//whether this panel contains the difference map (different color scheme)
	
	private ContactData data;
	
	private ImagePanel imagePanel;
	private JPanel selectionPanel;
	
	private JScrollPane scrollPane;
	
	private Structure structure1;
	private Structure structure2;
	
	private int residueCount = 0;
	
	private final int leftMargin = 40;
	private final int topMargin = 30;
	private final int rightMargin = 20;
	private final int bottomMargin = 20;
	
	private boolean clear = false;
	
	private int pixelRatio = 2;
	private int count = 0;
	
	private int width = 0;
	private int height = 0;
	
	private Color green = new Color(70,200,70);
	private Color blue = new Color(70,70,230);
	
	private boolean wait = false;
	
	private Residue[] pair = new Residue[2];//holder to speed up data exchange by avoiding creating arrays every time mouse moves
	
	public ContactPanel(ContactMapViewer v){
		
		viewer = v;
		
		setLayout(new BorderLayout());
		
		layeredPane = new JLayeredPane();
		layeredPane.setLayout(new BorderLayout());
		
		imagePanel = new ImagePanel();
		selectionPanel = new SelectionPanel();
		
		layeredPane.add(imagePanel, 1);
		layeredPane.add(selectionPanel, 0);
		
		scrollPane = new JScrollPane(layeredPane);
		scrollPane.setBackground(Color.WHITE);
		scrollPane.getViewport().setBackground(Color.WHITE);
		
		add(scrollPane, BorderLayout.CENTER);
		
		this.addComponentListener(new ComponentAdapter(){
			public void componentResized(ComponentEvent e){
				scrollPane.setBounds(0,0, getWidth(), getHeight());
				revalidate();
				repaint();
			}
		});
		
	}
	
	/**
	 * This method sets the display of a residue interaction map for a specific structure (one structure argument)
	 * or a difference map (if two structures are given)
	 * @param structure1
	 * @param structure2
	 */
	public void setStructures(Structure structure1, Structure structure2){
		
//		System.out.println("structure1 = " + structure1);
		
		this.structure1 = structure1;
		this.structure2 = structure2;
		
		if (structure1 != null && structure2 == null){
			data = new ContactData(structure1);

			//it's a single structure case
			//compute dimensions of the image
			count = data.getResidueCount();
			
			//let's assume pixel ratio is 4:
			if (count*4 <= 500){
				pixelRatio = 4;
			}
			else if (count*3 <= 500){
				pixelRatio = 3;
			}
			else{
				pixelRatio = 2;
			}
			
			width = leftMargin + count*pixelRatio + rightMargin;
			height = topMargin + count*pixelRatio + bottomMargin;
			layeredPane.setBounds(0,0, width, height);
			layeredPane.setPreferredSize(new Dimension(width, height));
			imagePanel.setBounds(0,0, width, height);
			imagePanel.setPreferredSize(new Dimension(width, height));
			
			
			imagePanel.repaint();
			selectionPanel.repaint();
			
			
		}
		else if (structure1 != null && structure2 != null){
			//it's a difference map case
//			computeInteractionMap(structure2);
			difference = true;
			
			//run structure alignment to get the correspondence of residues
			//first, paint a message about data being generated, just in case the user gets impatient
			
			wait = true;
			selectionPanel.repaint();
			
			final Structure s1 = structure1;
			final Structure s2 = structure2;
			
			Thread runner = new Thread(){
				public void run(){
					
					//generate the data using structure alignment in the background (no structure superposition)
					//use the corresponding sequence alignment to determine which residue from each structure
					//corresponds to which one in the other (two hashes Residue -> Residue)
					//once done, set wait to false
					HashMap aligned = viewer.alignStructures(s1, s2);
					
					String seq1 = (String)aligned.get(s1);
					String seq2 = (String)aligned.get(s2);
					
					//now create the lookup hashes for Residue objects
					//save only residue pairs where both sequences have an amino acid (no dashes or non-amino acids)
					//create the data object storing differences in distances
					
					int length = seq1.length() < seq2.length() ? seq1.length() : seq2.length();
					
					Vector residues1 = new Vector();//reference from the structure1
					Vector residues2 = new Vector();//reference from the structure2
					
					char[] sequence1 = seq1.toCharArray();
					char[] sequence2 = seq2.toCharArray();
					
					StructureMap map1 = s1.getStructureMap();
					StructureMap map2 = s2.getStructureMap();
					
					int counter1 = -1;//residue index in sequence1
					int counter2 = -1;//residue index in sequence2
					for (int i = 0; i < length; i++){
						char a = sequence1[i];
						char b = sequence2[i];
						boolean apass = false;
						boolean bpass = false;
						if (a != '-' && a != '*'){
							//it's a residue
							counter1++;
							if (a != '#') apass = true;
						}
						if (b != '-' && b != '*'){
							//it's a residue
							counter2++;
							if (b != '#') bpass = true;
						}
						
						if (apass && bpass){
							//it's a pair of residues
							
							Residue r1 = map1.getResidue(counter1);
							Residue r2 = map2.getResidue(counter2);
							
//							System.out.println(counter1 + ": r1 = " + r1.getCompoundCode() + ", " + counter2 + ": r2 = " + r2.getCompoundCode());
							
							residues1.add(r1);
							residues2.add(r2);
						}
						
						
					}
					
					data = new ContactData(residues1, residues2);
					
					wait = false;
				}
			};
			runner.start();
			
			while (true){
				if (!wait){
					break;
				}
				
				try{
					Thread.sleep(100);
				}
				catch (InterruptedException e){}
			}
			
			count = data.getResidueCount();
			
			//let's assume pixel ratio is 4:
			if (count*4 <= 500){
				pixelRatio = 4;
			}
			else if (count*3 <= 500){
				pixelRatio = 3;
			}
			else{
				pixelRatio = 2;
			}
			
			width = leftMargin + count*pixelRatio + rightMargin;
			height = topMargin + count*pixelRatio + bottomMargin;
			layeredPane.setBounds(0,0, width, height);
			layeredPane.setPreferredSize(new Dimension(width, height));
			imagePanel.setBounds(0,0, width, height);
			imagePanel.setPreferredSize(new Dimension(width, height));
			
			
			imagePanel.repaint();
			selectionPanel.repaint();
			
		}
		else{
			//both are null
			return;
		}
		
		
		setVisible(true);
	}
	
	
	class ImagePanel extends JPanel {
		
		public ImagePanel(){
			setOpaque(true);
		}
		
		protected void paintComponent(Graphics g){
			
			g.setColor(Color.white);
			g.fillRect(0,0, width, height);
			
			//draw the rectangle
			g.setColor(Color.black);
			g.drawRect(leftMargin, topMargin, count*pixelRatio+1, count*pixelRatio+1);
			
			//draw the data
			for (int i = 0; i < count; i++){
				for (int j = 0; j < count; j++){
					float value = data.getValue(i, j);
					
					Color color = null;
					if (difference){
						if (value > 5){
							color = Color.RED;
						}
						else if (value >= 3 && value < 5){
							color = Color.ORANGE;
						}
						else if (value >=2 && value < 3){
							color = Color.YELLOW;
						}
						else if (value >= 1 && value < 2){
							color = green;
						}
						else if (value >= 0.5 && value < 1){
							color = blue;
						}
						else{
							continue;//paint nothing if it's too far
						}
					}
					else{
						if (value <= 0){
							color = Color.GRAY;
						}
						else if (value < 3){
							color = Color.RED;
						}
						else if (value >= 3 && value < 5){
							color = Color.ORANGE;
						}
						else if (value >=5 && value < 7){
							color = Color.YELLOW;
						}
						else if (value >= 7 && value < 10){
							color = green;
						}
						else{
							continue;//paint nothing if it's too far
						}
					}
					
					g.setColor(color);
					int x = leftMargin + i*pixelRatio + 1;
					int y = topMargin + j*pixelRatio + 1;
					g.fillRect(x, y, pixelRatio, pixelRatio);

				
					if (i > 0 && j > 0){
						if (i%50 == 0 && j%50 == 0){
							//draw the grid marks and numbers
							g.setColor(Color.BLACK);
							
							//x mark
							g.drawLine(leftMargin+i*pixelRatio+1, topMargin-5, leftMargin+i*pixelRatio+1, topMargin);
							g.drawLine(leftMargin-5, topMargin+j*pixelRatio+1, leftMargin, topMargin+j*pixelRatio+1);
							
							String xx = new Integer(data.getResidueId(i)).toString();
							String yy = new Integer(data.getResidueId(j)).toString();
							
							if (xx.equals("-1") || yy.equals("-1")) continue;
							
							g.drawString(xx, leftMargin+i*pixelRatio, topMargin-10);
							g.drawString(yy, 5, topMargin+j*pixelRatio+5);
							
						}
					}
				
				}
				
			}
			
			
		}
		
	}
	
	class SelectionPanel extends JPanel {
		
		private String status;//contains names of residues
		
		public SelectionPanel(){
			setOpaque(false);
			
			MouseInputAdapter adapter = new MouseInputAdapter(){
				public void mouseClicked(MouseEvent e){
					int x = e.getX();
					int y = e.getY();
					
					//get the indices and residues
					int i = (x - leftMargin)/pixelRatio;
					int j = (y - topMargin)/pixelRatio;
					
					if (difference){
						data.getResidues(pair, i);
						
						Residue rx1 = pair[0];
						Residue ry1 = pair[1];
						
						data.getResidues(pair, j);
						Residue rx2 = pair[0];
						Residue ry2 = pair[1];
						
						//all of these should be selected
						//deselect any other residues, unless Ctrl is pressed
						if (!e.isControlDown()){
							structure1.getStructureMap().getStructureStyles().selectNone(true, true);
							structure2.getStructureMap().getStructureStyles().selectNone(true, true);
						}
					
						rx1.structure.getStructureMap().getStructureStyles().setSelected(rx1, true, true, true);
						ry1.structure.getStructureMap().getStructureStyles().setSelected(ry1, true, true, true);
						rx2.structure.getStructureMap().getStructureStyles().setSelected(rx2, true, true, true);
						ry2.structure.getStructureMap().getStructureStyles().setSelected(ry2, true, true, false);
						
						Vector chains = new Vector();
						Chain c1 = rx1.structure.getStructureMap().getChain(rx1.getAtom(0));
						chains.add(c1);
						Chain c2 = ry1.structure.getStructureMap().getChain(ry1.getAtom(0));
						if (!chains.contains(c2)) chains.add(c2);
						Chain c3 = rx2.structure.getStructureMap().getChain(rx2.getAtom(0));
						if (!chains.contains(c3)) chains.add(c3);
						Chain c4 = ry2.structure.getStructureMap().getChain(ry2.getAtom(0));
						if (!chains.contains(c4)) chains.add(c4);
						
						for (int n = 0; n < chains.size(); n++){
							Chain cc = (Chain)chains.get(n);
							cc.structure.getStructureMap().getStructureStyles().updateChain(cc, true);
						}
						
					}
					else{
						Residue rx = data.getResidue(i);
						Residue ry = data.getResidue(j);
						
						if (rx == null || ry == null){
							
							repaint();
							return;
						}
						
						
						//deselect any other residues, unless Ctrl is pressed
						if (!e.isControlDown()){
							rx.structure.getStructureMap().getStructureStyles().selectNone(true, true);
						}
						
						//select the two clicked residues
						rx.structure.getStructureMap().getStructureStyles().setSelected(rx, true, true, true);
						ry.structure.getStructureMap().getStructureStyles().setSelected(ry, true, true, false);
						
						Chain c1 = rx.structure.getStructureMap().getChain(rx.getAtom(0));
						Chain c2 = ry.structure.getStructureMap().getChain(ry.getAtom(0));
						
						if (c1 == c2){
							ry.structure.getStructureMap().getStructureStyles().updateChain(c1, true);
						}
						else{
							ry.structure.getStructureMap().getStructureStyles().updateChain(c1, true);
							ry.structure.getStructureMap().getStructureStyles().updateChain(c2, true);
							
						}
					}
					
				}
				
				public void mouseMoved(MouseEvent e){
					int x = e.getX();
					int y = e.getY();
					
					if (x <= leftMargin || x > leftMargin+width || y <= topMargin || y > topMargin+height){
						status = null;
						repaint();
						return;
					}

					int i = (x - leftMargin)/pixelRatio;
					int j = (y - topMargin)/pixelRatio;
					
					
					Residue rx = data.getResidue(i);
					Residue ry = data.getResidue(j);
					
					if (rx == null || ry == null){
						status = null;
						repaint();
						return;
					}
					
					if (rx == ry){
						status = rx.getCompoundCode() + rx.getResidueId();
					}
					else{
						status = rx.getCompoundCode() + rx.getResidueId() + " - " + ry.getCompoundCode() + ry.getResidueId();
					}
					
					repaint();
					
				}
			};
			
			this.addMouseListener(adapter);
			this.addMouseMotionListener(adapter);
			
		}
		
		
		protected void paintComponent(Graphics g){

			g.setColor(Color.BLACK);
			if (status != null) g.drawString(status, 10,20);
			
			if (clear){
				clear = false;
				return;
			}
			
			//selection handling
			
			if (wait){
				g.setColor(Color.GRAY);
				g.fillRect(0,0, getWidth(), getHeight());
				g.setColor(Color.WHITE);
				g.setFont(new Font("Helvetica", Font.PLAIN, 18));
				g.drawString("Computing difference graph...", 50,50);
			}
			
			
		}

	}
	
	public void setPixelRatio(int ratio){
		pixelRatio = ratio;
		width = leftMargin + count*pixelRatio + rightMargin;
		height = topMargin + count*pixelRatio + bottomMargin;
		layeredPane.setBounds(0,0, width, height);
		layeredPane.setPreferredSize(new Dimension(width, height));
		imagePanel.setBounds(0,0, width, height);
		imagePanel.setPreferredSize(new Dimension(width, height));
		
		scrollPane.revalidate();
		imagePanel.repaint();
		scrollPane.repaint();
//		selectionPanel.repaint();
	}
	
	public int getPixelRatio(){
		return pixelRatio;
	}
	
	public void clearSelection(){
		clear = true;
		selectionPanel.repaint();
	}
	
	/**
	 * Deletes the contents of the panel
	 *
	 */
	public void destructor(){
		wait = false;
		structure1 = null;
		structure2 = null;
		data = null;
	}
	
	
}