package edu.sdsc.sirius.viewers;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.event.*;
import java.net.URL;
import java.util.*;
import java.awt.image.*;

import edu.sdsc.mbt.*;
import edu.sdsc.mbt.util.*;
import edu.sdsc.mbt.viewables.*;

import edu.sdsc.sirius.util.Manager;
import edu.sdsc.sirius.util.StringUtils;
import edu.sdsc.sirius.util.TabListCellRenderer;
import edu.sdsc.sirius.viewers.RamachandranViewerImpl.*;

public class RamachandranViewer extends JPanel implements Viewer, StructureStylesEventListener, StructureComponentEventListener {
	
	private Structure structure;
	private Manager parent;
	private StructureDocument structureDocument;
	
	private JLayeredPane pane = new JLayeredPane();
	private JPanel imagePanel;
	private JPanel gridPanel;
	private GraphPanel graphPanel;
	
	private HashMap coordinates = new HashMap();//Rectangle -> Residue
	private HashMap positions = new HashMap();//Residue -> Rectangle
	private HashMap points = new HashMap();
	
	private HashMap residueIndices = new HashMap();//index -> Residue
	private int structureIndex = -1;
	
	private StructureStyles styles;
	private StructureMap map;
	
	private JPanel statsPanel;
	
	private JMenuBar menuBar;
	private String[] results;
	private JList statsList;
	
	private JScrollPane scroll;
	private Image ramaImage;
	private Image ramaGrid;

	private JCheckBoxMenuItem menuViewDock;
		
	int elementSize = 5;//default
	
	int glycineCount = 0;
	int prolineCount = 0;
	int totalCount = 0;
	int rest = 0;
	
	//residue count in different areas
	int aCount = 0;
	int bCount = 0;
	int cCount = 0;
	int dCount = 0;
	
	private RamachandranViewerDialog dialog;

	
	public RamachandranViewer(Structure structure){
		
		this.structure = structure;
		this.styles = this.structure.getStructureMap().getStructureStyles();
		this.structure.getStructureMap().getStructureStyles().addStructureStylesEventListener( this );
		this.structure.getStructureMap().getStructureStyles().addStructureComponentEventListener( this );
		
		map = structure.getStructureMap();
		
//		System.out.println("RV structure = " + this.structure);

		
		//construct the panel
		setLayout(new BorderLayout());
		setPreferredSize(new Dimension(590,810));
		
		//get the image
		URL url = getClass().getResource("/edu/sdsc/sirius/viewers/RamachandranViewerImpl/rama_colors.png");//don't translate
		ImageIcon imageIcon = new ImageIcon(url);
		ramaImage = imageIcon.getImage();
		JLabel label = new JLabel(imageIcon);
		label.setBounds(0,0, 590, 573);
		
		imagePanel = new JPanel();
		imagePanel.setBounds(0,0, 590, 573);
		imagePanel.setLayout(null);
		imagePanel.add(label);
		
		//get the image
		URL url2 = getClass().getResource("/edu/sdsc/sirius/viewers/RamachandranViewerImpl/rama_lines.png");//don't translate
		ImageIcon imageIcon2 = new ImageIcon(url2);
		ramaGrid = imageIcon2.getImage();
		JLabel label2 = new JLabel(imageIcon2);
		label2.setBounds(0,0, 590, 573);
		
		gridPanel = new JPanel();
		gridPanel.setOpaque(false);
		gridPanel.setBounds(0,0, 590, 573);
		gridPanel.setLayout(null);
		gridPanel.add(label2);

		
		//create the statistics panel
		statsPanel = new JPanel();
		
		results = new String[11];
		results[0] = "Residues in most favored regions";
		results[1] = "Residues in additional allowed regions";
		results[2] = "Residues in generously allowed regions";
		results[3] = "Residues in disallowed regions";
		results[4] = "    ";
		results[5] = "Number of non-glycine and non-proline residues";
		results[6] = "Number of end-residues";
		results[7] = "Number of glycine residues";
		results[8] = "Number of proline residues";
		results[9] = "    ";
		results[10] = "Total number of residues";
		
		statsList = new JList(results);
		
		TabListCellRenderer renderer = new TabListCellRenderer();
		renderer.setTabs(new int[]{ 400, 450, 500});
		statsList.setCellRenderer(renderer);
		statsList.setEnabled(false);
		scroll = new JScrollPane();
		scroll.setPreferredSize(new Dimension(590, 180));
		scroll.getViewport().add(statsList);

		computeMap();

		
		graphPanel = new GraphPanel();
		graphPanel.setBounds(0,0, 590, 573);
		graphPanel.repaint();

		
		pane = new JLayeredPane();
		pane.setPreferredSize(new Dimension(590, 573));
		pane.setBounds(0,0, 590, 573);
		pane.add(imagePanel, new Integer(10));
		pane.add(gridPanel, new Integer(20));
		pane.add(graphPanel, new Integer(30));
		
		pane.setVisible(true);
		
		menuBar = createMenuBar();
		add(menuBar, BorderLayout.NORTH);

		add(pane, BorderLayout.CENTER);
		
		
		add(scroll, BorderLayout.SOUTH);
	}
	
	public void processStructureComponentEvent(StructureComponentEvent e){
		if (e.changeType == StructureComponentEvent.TYPE_GEOMETRY_CHANGE){
			try{
				
				Residue r = (Residue)e.structureComponent;
				
				if (r.structure != this.structure) return;//it's a different structure
				
				if (!points.containsKey(r)){
					return;
				}
				
				//this residue has been changed
				double[] angles = map.getPsiPhi(r);
				if (angles == null) return;
				
				double phiValue = angles[0];
				double psiValue = angles[1];
				
				//calculate new point position
				int phiOffset = (int)(phiValue*256/180);//for now, hard-coded offset from the center line
				int psiOffset = (int)(psiValue*256/180);
				
				int phiPosition = 304 + phiOffset;
				int psiPosition = 542 - (256 + psiOffset);//to convert to offset from the top
				
				if (phiPosition < 50) phiPosition = 50;
				if (phiPosition > 557) phiPosition = 557;
				
				if (psiPosition < 33) psiPosition = 33;
				if (psiPosition > 540) psiPosition = 540;
			
				Point point  = new Point(phiPosition, psiPosition);
				
				points.put(r, point);
				
				repaint();
				
			}
			catch (Exception ex){
				ex.printStackTrace();
				return;
			}
		}
	}
	
	//when undo is selected, disconnect from the current structure and create a hash of
	public void disconnect(int structureIndex){
		this.structureIndex = structureIndex;//index of the associated structure in StructureViewer
		this.styles.removeStructureStylesEventListener( this );
		this.styles.removeStructureComponentEventListener( this );
		scroll.repaint();
	}
	
	public void connect(Structure structure, int index){
		if (index == structureIndex){
			//reconnect
			
			this.structure = structure;
			this.styles = this.structure.getStructureMap().getStructureStyles();
			this.styles.addStructureStylesEventListener( this );
			this.styles.addStructureComponentEventListener( this );
			
			map = structure.getStructureMap();
			map.computePhiPsi();
			HashMap replacement = new HashMap();
			
			HashMap newIndices = new HashMap();
			
			Set keys = residueIndices.keySet();
			Iterator it = keys.iterator();
			while (it.hasNext()){
				Integer n = (Integer)it.next();
				Residue oldR = (Residue)residueIndices.get(n);
				
				//this residue should be replaced with new
				Residue newR = map.getResidue(n.intValue());
				
				replacement.put(oldR, newR);
				
				newIndices.put(n, newR);
			}
			
			residueIndices.clear();
			residueIndices = newIndices;
			
			//now walk through the hashes and get the new Residues in place
			HashMap newPoints = new HashMap();
			Set set = points.keySet();
			Iterator ii = set.iterator();
			while (ii.hasNext()){
				Residue r = (Residue)ii.next();
				newPoints.put(replacement.get(r), points.get(r));
			}
			
			points.clear();
			points = newPoints;
			
			structureIndex = -1;//reset back once the connection is reestablished
			
			scroll.repaint();
		}
	}
	
	public void processStructureDocumentEvent(StructureDocumentEvent event){
		try{
			
			int type = event.type;
			int change = event.change;
			
			if ( type == StructureDocumentEvent.TYPE_VIEWER ){
				if ( change == StructureDocumentEvent.CHANGE_ADDED )
				{
					structureDocument = event.structureDocument;
					graphPanel.repaint();
				}
				else if ( change == StructureDocumentEvent.CHANGE_REMOVED )
				{
					structureDocument = null;
					setVisible(false);
				}
			}
		
		}
		catch (Exception ex){
			structureDocument.parent.displayExceptionMessage("Exception processing StructureDocumentEvent", ex);
		}

	}
	
	public void processStructureStylesEvent(StructureStylesEvent event){
		graphPanel.repaint();
	}
	
	public void destructor(){
		//remove itself from the structureDocument and unload all data
		this.structure.getStructureMap().getStructureStyles().removeStructureStylesEventListener( this );
		structureDocument.removeViewer(this);
	}
	
	public Structure getStructure(){
		return structure;
	}
	
	private JMenuBar createMenuBar(){
		
		JMenuBar menuBar = new JMenuBar();
		JMenu menuFile = new JMenu("File");
		JMenuItem menuPrint = new JMenuItem("Print...");
		menuPrint.setEnabled(false);
		
		JMenuItem menuSave = new JMenuItem("Save graphics...");
		menuSave.setEnabled(false);
		
		JMenuItem menuClose = new JMenuItem("Close display");
		menuClose.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				destructor();
			}
		});
		
		menuFile.add(menuSave);
		menuFile.add(menuPrint);
		menuFile.add(new JSeparator());
		menuFile.add(menuClose);

		
		JMenu menuView = new JMenu("View");
		
		menuViewDock = new JCheckBoxMenuItem("Dock to main window");
		menuViewDock.setSelected(StylesPreferences.dockRama);
		menuViewDock.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				if (menuViewDock.isSelected()){
					StylesPreferences.dockRama = true;
					dialog.dockDialog();
				}
				else{
					StylesPreferences.dockRama = false;
				}
			}
		});
		
		JMenu menuViewSize = new JMenu("Select icon size");
		final JCheckBoxMenuItem menuViewSizeThree = new JCheckBoxMenuItem("3 points");
		final JCheckBoxMenuItem menuViewSizeFive = new JCheckBoxMenuItem("5 points");
		final JCheckBoxMenuItem menuViewSizeSeven = new JCheckBoxMenuItem("7 points");
		
		menuViewSizeThree.setSelected(false);
		menuViewSizeThree.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				elementSize = 3;
				menuViewSizeFive.setSelected(false);
				menuViewSizeSeven.setSelected(false);
				repaint();
			}
		});
		
		menuViewSizeFive.setSelected(true);
		menuViewSizeFive.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				elementSize = 5;
				menuViewSizeThree.setSelected(false);
				menuViewSizeSeven.setSelected(false);
				repaint();
			}
		});
		
		menuViewSizeSeven.setSelected(false);
		menuViewSizeSeven.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				elementSize = 7;
				menuViewSizeThree.setSelected(false);
				menuViewSizeFive.setSelected(false);
				repaint();
			}
		});
		
		menuViewSize.add(menuViewSizeThree);
		menuViewSize.add(menuViewSizeFive);
		menuViewSize.add(menuViewSizeSeven);
	
		
		
		
		
		JMenuItem menuViewGly = new JMenuItem("Show GLY residues");
		menuViewGly.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				//select gly and draw an extra squares around them to make them better visible
				styles.selectNone(true, false);
				for (int i = 0; i < map.getResidueCount(); i++){
					Residue r = map.getResidue(i);
					if (r.getCompoundCode().equals("GLY")) styles.setSelected(r, true, true, false);
				}
				
				//update chains involved
				for (int i = 0; i < styles.getStructureMap().getChainCount(); i++){
					styles.updateChain(styles.getStructureMap().getChain(i), true);
//					styles.registerChainSelection(styles.getStructureMap().getChain(i), true);
				}
			}
		});
		
		JMenuItem menuViewPro = new JMenuItem("Show PRO residues");
		menuViewPro.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				//select pro and draw an extra squares around them to make them better visible
				styles.selectNone(true, false);
				for (int i = 0; i < map.getResidueCount(); i++){
					Residue r = map.getResidue(i);
					if (r.getCompoundCode().equals("PRO")) styles.setSelected(r, true, true, false);
				}
				//update chains involved
				for (int i = 0; i < styles.getStructureMap().getChainCount(); i++){
					styles.updateChain(styles.getStructureMap().getChain(i), true);
//					styles.registerChainSelection(styles.getStructureMap().getChain(i), true);
				}
			}
		});
		
		JMenuItem menuViewEnd = new JMenuItem("Show end residues");
		menuViewEnd.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				//select gly and draw an extra squares around them to make them better visible
				Vector ends = map.getEndResidues();
				styles.selectNone(true, false);
				if (ends == null || ends.size() == 0) return;
				for (int i = 0; i < ends.size(); i++){
					Residue r = (Residue)ends.get(i);
					styles.setSelected(r, true, true, false);
				}
				//update chains involved
				for (int i = 0; i < styles.getStructureMap().getChainCount(); i++){
					styles.updateChain(styles.getStructureMap().getChain(i), true);
//					styles.registerChainSelection(styles.getStructureMap().getChain(i), true);
				}
			}
		});
		
		menuView.add(menuViewSize);
		menuView.add(new JSeparator());
		menuView.add(menuViewGly);
		menuView.add(menuViewPro);
		menuView.add(menuViewEnd);
		menuView.add(new JSeparator());
		menuView.add(menuViewDock);
		
		JMenu menuHelp = new JMenu("Help");
		JMenuItem menuHelpAbout = new JMenuItem("About...");
		menuHelpAbout.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				structureDocument.parent.displayMessage("Ramachandran diagram representation is based on work by Laskowsky et al.\n" +
						"            J. Appl. Cryst. (1993) Vol.26, 283-291.");
				
			}
		});
		
		menuHelp.add(menuHelpAbout);
		
		menuBar.add(menuFile);
		menuBar.add(menuView);
		menuBar.add(menuHelp);
		
		return menuBar;
		
	}
	
	private void computeMap(){
		
		HashMap phi = map.getPhiMap();
		HashMap psi = map.getPsiMap();
		
		Vector endResidues = map.getEndResidues();
		
		if (phi.size() == 0 || psi.size() == 0){
			boolean r = map.computePhiPsi();
			if (!r){
				structureDocument.parent.displayErrorMessage("Ramachandran plot cannot be calculated");
				destructor();
			}
		}
		
		boolean progress = false;
		float percentDone = 1.0f;
		int onePercent = 1;

		if (map.getResidueCount() > 100){
			progress = true;
			onePercent = ( map.getResidueCount()/ 100);
			if ( onePercent <= 0 ) onePercent = 1;
			percentDone = 0.0f;
			Status.progress( percentDone, "Calculating Ramachandran plot");
		}


		for (int i = 0; i < map.getResidueCount(); i++){
			Residue r = map.getResidue(i);
			
			residueIndices.put(new Integer(i), r);
			
			if ( progress && i % onePercent == 0 )
			{
				percentDone = (float) i / (float) map.getResidueCount();
				Status.progress( percentDone, "Calculating Ramachandran plot" );
			}

			if (phi.containsKey(r) && psi.containsKey(r)){
				double phiValue = ((Double)phi.get(r)).doubleValue();
				double psiValue = ((Double)psi.get(r)).doubleValue();
				
//				System.out.println(r.getCompoundCode() + r.getResidueId() + ": phi = " + phiValue + ", psi = " + psiValue);
				
				//convert these values into coordinates in pixels (2 degrees = 3 pixels)
				int phiOffset = (int)(phiValue*256/180);//for now, hard-coded offset from the center line
				int psiOffset = (int)(psiValue*256/180);
				
				int phiPosition = 304 + phiOffset;
				int psiPosition = 542 - (256 + psiOffset);//to convert to offset from the top
				
				if (phiPosition < 50) phiPosition = 50;
				if (phiPosition > 557) phiPosition = 557;
				
				if (psiPosition < 33) psiPosition = 33;
				if (psiPosition > 540) psiPosition = 540;
				
				if (!r.getCompoundCode().equals("GLY") && !r.getCompoundCode().equals("PRO")){
					rest++;
					//check color at this position
					try{
						int[] pixels = new int[1];
						PixelGrabber grabber = new PixelGrabber(ramaImage, phiPosition, psiPosition, 1, 1, pixels, 0, 1);
						if (grabber.grabPixels()){
							//take the center
							int alpha = (pixels[0] >> 24) & 0xff;
							int red   = (pixels[0] >> 16) & 0xff;
							int green = (pixels[0] >>  8) & 0xff;
							int blue  = (pixels[0]      ) & 0xff;
	
							if (red == 255 && green == 89 && blue == 0){
								aCount++;
							}
							else if (red == 255 && green == 191 && blue == 0){
								bCount++;
							}
							else if (red == 255 && green == 255 && blue == 133){
								cCount++;
							}
							else if (red == 255 && green == 255 && blue == 255){
								dCount++;
							}
							else {
								dCount++;
							}
						}
					}
					catch (Exception ex){
						ex.printStackTrace();
					}
				}
				
				Point point  = new Point(phiPosition, psiPosition);
				
				points.put(r, point);
				

				if (r.getCompoundCode().equals("GLY")) glycineCount++;
				if (r.getCompoundCode().equals("PRO")) prolineCount++;
				
				totalCount++;
				
			}
		}
		
		if (progress)Status.progress( 1.0f, null );

		
		double aPercent = 100.0*aCount/rest;
		double bPercent = 100.0*bCount/rest;
		double cPercent = 100.0*cCount/rest;
		double dPercent = 100.0*dCount/rest;
		
		
		
		//take care of the statistics
		results[0] = "Residues in most favored regions\t" + aCount + "\t" + StringUtils.getFormattedNumber(aPercent, 2) + "%";
		results[1] = "Residues in additional allowed regions\t" + bCount + "\t" + StringUtils.getFormattedNumber(bPercent, 2) + "%";
		results[2] = "Residues in generously allowed regions\t" + cCount + "\t" + StringUtils.getFormattedNumber(cPercent, 2) + "%";
		results[3] = "Residues in disallowed regions\t" + dCount + "\t" + StringUtils.getFormattedNumber(dPercent, 2) + "%";

		results[5] = "Number of non-glycine and non-proline residues\t" + rest + "\t100%";
		results[6] = "Number of end-residues\t" + endResidues.size();
		results[7] = "Number of glycine residues\t" + glycineCount;
		results[8] = "Number of proline residues\t" + prolineCount;
		results[9] = "    ";
		results[10] = "Total number of residues\t" + totalCount;

		statsList.setListData(results);
		scroll.repaint();

	}
	
	public void setDialog(RamachandranViewerDialog d){
		dialog = d;
	}
	
	
	class GraphPanel extends JPanel {
		
		public GraphPanel(){
			setOpaque(false);
			
			this.setBorder(BorderFactory.createLineBorder(Color.BLACK, 1));
			
			//add a mouse listener to listen to pick events
			addMouseListener( new MouseInputAdapter(){
				
				public void mouseClicked(MouseEvent e){
					//get the coordinates
					int x = e.getX();
					int y = e.getY();
					
					//scan the rectangles to determine which one is intersected by this point
					boolean found = false;
					
					Set keys = coordinates.keySet();
					Iterator it = keys.iterator();
					
					Vector residues = new Vector();//in case several of them share the same area
					boolean select = false;
					while (it.hasNext()){
						Rectangle rect = (Rectangle)it.next();
						if (rect.contains(x, y)){
							found = true;
							Residue r = (Residue)coordinates.get(rect);
							residues.add(r);
							select = r.structure.getStructureMap().getStructureStyles().isSelected(r);
						}
					}
					if (!found){
						//deselect everything
						styles.selectNone(false, false);
						structureDocument.parent.updateView();
					}
					else{
						if (!e.isControlDown()){
							styles.selectNone(false, true);
							
							if (select) return;//it's already been deselected
						}
						
						Vector chains = new Vector();
						for (int j = 0; j < residues.size(); j++){
							Residue r = (Residue)residues.get(j);
							if (!chains.contains(styles.getStructureMap().getChain(r.getAtom(0)))) chains.add(styles.getStructureMap().getChain(r.getAtom(0)));
							styles.setSelected(r, !styles.isSelected(r), true, false);
							select = styles.isSelected(r);
						}
						
						for (int i = 0; i < chains.size(); i++){
							styles.updateChain((Chain)chains.get(i), true);
//							styles.registerChainSelection((Chain)chains.get(i), select);
						}
					}
					
				}
			});
			
		}

		protected void paintComponent(Graphics graphics){
			
			graphics.setColor(Color.BLACK);//for now
			Color selectionColor = new Color(StylesPreferences.selectionColor[0], StylesPreferences.selectionColor[1], StylesPreferences.selectionColor[2]);

			String label = structure.getUrlString();
			
			if (label.length() > 10) label = label.substring(0, 9);
			
			//draw the label at the top right of the plot
			graphics.drawString(label, 520, 20);

			//walk over all residues and draw their rectangles

			float[] temp = new float[3];
			
			Color gray = new Color(0.7f, 0.7f, 0.7f);
			
			Vector selectedResidues = new Vector();
			
			positions.clear();
			coordinates.clear();
			
			int offset = elementSize/2;
			Set keys = points.keySet();
			
			Iterator it = keys.iterator();
			while (it.hasNext()){
				Residue r = (Residue)it.next();
				
				
				if (r == null) continue;

				Point p = (Point)points.get(r);
				
				Rectangle rect = new Rectangle(p.x-offset, p.y-offset, elementSize, elementSize);
				positions.put(r, rect);
				coordinates.put(rect, r);

					
				//draw the square at this position, and fill it with the color of the residue, if different from default
				//otherwise, leave it blank
				
				//check whether the residue or one of its atoms in the structure are selected
				boolean selected = false;
				if (styles.isSelected(r)){
					selected = true;
					selectedResidues.add(r);
				}
				else{
					//check the atoms
					for (int j = 0; j < r.getAtomCount(); j++){
						Atom a = r.getAtom(j);
						if (styles.isSelected(a)){
							selected = true;
							selectedResidues.add(r);
							break;
						}
					}
				}
				if (selected){
					graphics.setColor(selectionColor);
					graphics.fillRect(rect.x, rect.y, rect.width, rect.height);
				}
				else{
					ResidueColor c = styles.getResidueColor(r);
					if (c instanceof ResidueColorByElement || c instanceof ResidueColorDefault){
						graphics.setColor(gray);
						graphics.fillRect(rect.x, rect.y, rect.width, rect.height);
					}
					else{
						styles.getResidueColor(r).getResidueColor(r, temp);
						Color color = new Color(temp[0], temp[1], temp[2]);
						graphics.setColor(color);
						graphics.fillRect(rect.x, rect.y, rect.width, rect.height);
					}
				}
				
				graphics.setColor(Color.BLACK);
				graphics.drawRect(rect.x, rect.y, rect.width, rect.height);
			
			}
			
			//check for specific residue information
			if (selectedResidues.size() == 1){
				
				Residue rrr = (Residue)selectedResidues.get(0);
				//draw the information above the graph
				double phiV = ((Double)map.getPhiMap().get(rrr)).doubleValue();
				double psiV = ((Double)map.getPsiMap().get(rrr)).doubleValue();
				
				//create the string
				StringBuffer buffer = new StringBuffer();
				buffer.append(rrr.getCompoundCode());
				buffer.append(rrr.getResidueId());
				buffer.append(": phi = ");
				
				buffer.append(StringUtils.getFormattedNumber(phiV, 3));
				buffer.append(", psi = ");
				buffer.append(StringUtils.getFormattedNumber(psiV, 3));

				graphics.drawString(buffer.toString(), 50, 20);
			}
			
		}
		
		
	}
	
}