package edu.sdsc.sirius.viewers;

import edu.sdsc.mbt.*;
import edu.sdsc.mbt.util.*;
import edu.sdsc.mbt.viewables.*;
import edu.sdsc.mbt.image.*;
import edu.sdsc.sirius.dialogs.*;
import edu.sdsc.sirius.io.*;
import edu.sdsc.sirius.util.*;
import edu.sdsc.sirius.viewers.SequenceViewerImpl.*;
//import edu.sdsc.sirius.viewers.TreeViewer.StructureBrowser;

import javax.swing.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.io.*;
import java.awt.print.*;
import java.awt.image.*;

/**
 * The SequenceViewer is a panel that contains a scale area and the sequence display area
 * The display area is painted as a separate component
 * @author Sasha Buzko
 *
 */
public class SequenceViewer extends JPanel implements Viewer, StructureStylesEventListener, SequenceStylesEventListener, StructureComponentEventListener {
	
	public StructureDocument structureDocument;
	private Manager parent;
	
	private SequenceViewer thisViewer;
	
	private Vector sequences = new Vector();//vector of Cell[] arrays
	private Vector backupSequences = new Vector();
	private HashMap entries = new HashMap();//Sequence -> Entry
	private Vector sequenceObjects = new Vector();//order of Sequence objects
	private HashMap styleMap = new HashMap();//Sequence -> StructureStyles (if any)
	
	private HashMap residueCellMapping = new HashMap();//Sequence -> map (Residue -> Cell)
	private HashMap cellResidueMapping = new HashMap();//Sequence -> map (Cell -> Residue)
	
	private Vector hiddenSequences = new Vector();
	
	private JPanel base;
	private SequenceScalePanel scalePanel;
	private SequencePanel sequencePanel;
	
	private boolean undoMode = false;
	
	private StyleManager styleManager;
	
	private JScrollPane scrollPane;
	private JScrollBar scrollBarH;
	private JScrollBar scrollBarV;
	private JPanel scrollBarPanel;
	private JPanel scrollBarFiller;
	private JLabel messageLabel = new JLabel();
	
	private Vector selectedColumns = new Vector();
	private Vector selectedCells = new Vector();
	
	private String alignmentData;
	
	private JMenuBar menuBar;
	private int residueMax = 0;//maximum number of residues in the displayed sequences
	
	private File file;
	
	public final int buttonWidthCells = 7;
	
	private boolean editMode = false;
	private JCheckBoxMenuItem menuEditMode;
	
	private JMenuItem menuEditReset;
	private DisplayParameters parameters = new DisplayParameters();
	
	private String alignmentOutput;
//	private AlignmentSettings alignmentSettings = new AlignmentSettings();
	
	private JCheckBoxMenuItem menuOptionsSs;
	private JMenuItem menuSequenceAlignmentDownload;
	
	private String replacedResidue;
	
	public PageFormat mPageFormat;
	public Book book;

	public boolean showSs = false;
	
	public static final int ssHeight = 5;
	
	
	
	public SequenceViewer(Manager parent){
		
		this.parent = parent;
		thisViewer = this;
		
		styleManager = new StyleManager(this);
		
		this.setLayout(new BorderLayout());
		
		//get the screen resolution to arrive at the correct estimate of visible column number
		Dimension dim = getToolkit().getScreenSize();
		int screenWidth = dim.width;
		
		scalePanel = new SequenceScalePanel(this, screenWidth);
		scalePanel.updateParameters();
		
		sequencePanel = new SequencePanel(this, screenWidth, sequences, sequenceObjects);
		
		scrollPane = new JScrollPane(sequencePanel);
		add( BorderLayout.CENTER, scrollPane );

		//set up the scrollbar
		scrollBarH = new JScrollBar( JScrollBar.HORIZONTAL );
		scrollBarH.setMinimum( 0 );
		scrollBarH.setMaximum( 0 );
		AdjustmentListener adjustmentListenerH = new AdjustmentListener( ){
			public void adjustmentValueChanged( AdjustmentEvent ae ){
				if (residueMax == 1){
					return;
				}
				setStartResidue( ae.getValue() );

			}
		};
		scrollBarH.addAdjustmentListener( adjustmentListenerH );

		//set up the scrollbar
		scrollBarV = new JScrollBar( JScrollBar.VERTICAL );
		scrollBarV.setMinimum( 0 );
		scrollBarV.setMaximum( 0 );
		AdjustmentListener adjustmentListenerV = new AdjustmentListener( ){
			public void adjustmentValueChanged( AdjustmentEvent ae ){
				if (sequences.size() == 0){
					return;
				}
				int v = ae.getValue();
				setStartSequence(v);
			}
		};
		scrollBarV.addAdjustmentListener( adjustmentListenerV );

		scrollBarPanel = new JPanel();
		scrollBarPanel.setLayout(new BorderLayout());
		scrollBarFiller = new JPanel();
		scrollBarFiller.setLayout(new BorderLayout());

		/**
		 * use the filler as a process status area (alignments, e.g.)
		 */
		
		int width = parameters.cellWidth*6;
		scrollBarFiller.setPreferredSize(new Dimension(parameters.cellWidth*buttonWidthCells, scrollBarH.getHeight()));
		scrollBarFiller.setMinimumSize(new Dimension(parameters.cellWidth*buttonWidthCells, scrollBarH.getHeight()));
		scrollBarFiller.setMaximumSize(new Dimension(parameters.cellWidth*buttonWidthCells, scrollBarH.getHeight()));
//		scrollBarFiller.setBorder(new BevelBorder(BevelBorder.RAISED));
		
		JPanel scrollBarCorner = new JPanel();
		scrollBarCorner.setPreferredSize(new Dimension(scrollBarH.getHeight(), scrollBarH.getHeight()));
		scrollBarCorner.setMinimumSize(new Dimension(scrollBarH.getHeight(), scrollBarH.getHeight()));
		scrollBarCorner.setMaximumSize(new Dimension(scrollBarH.getHeight(), scrollBarH.getHeight()));

		
		scrollBarPanel.add(scrollBarFiller, BorderLayout.WEST);
		scrollBarPanel.add(scrollBarH, BorderLayout.CENTER);
		scrollBarPanel.add(scrollBarCorner, BorderLayout.EAST);
		
		scrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
		scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER);
		
		base = new JPanel();
		base.setLayout(new BorderLayout());
		this.add(createMenuBar(), BorderLayout.NORTH);
		
		base.add(scalePanel, BorderLayout.NORTH);
		base.add(scrollPane, BorderLayout.CENTER);
		base.add(scrollBarPanel, BorderLayout.SOUTH );
		base.add(scrollBarV, BorderLayout.EAST);
		
		this.add(base, BorderLayout.CENTER);
		
		addComponentListener(new ComponentAdapter(){
			public void componentResized(ComponentEvent e){
				scalePanel.updateParameters();
				if (sequences.size() == 0) return;
				sequencePanel.updateView(false);
				
				//update the height
				int height = scrollPane.getHeight();
				int ch = parameters.cellHeight;
				if (showSs){
					ch += SequenceViewer.ssHeight;
				}
				int seqs = (int)height/ch;
				parameters.visibleSequences = seqs;
				
				//check whether the window got resized wider than the displayed sequence
				int tempStart = residueMax - parameters.visibleColumns + 10;
				if(tempStart < 0) tempStart = 0;
				if (tempStart < sequencePanel.getStartResidue()){
					adjustHScrollBar(false);
					setStartResidue(tempStart);
				}
				
				//check whether the window got resized higher than the displayed sequences list
				tempStart = sequences.size() - hiddenSequences.size() - parameters.visibleSequences;
				if(tempStart < 0) tempStart = 0;
				if (tempStart < sequencePanel.getStartSequence()){
					setStartSequence(tempStart);
				}

				adjustVScrollBar(false);
				
				scrollPane.revalidate();
				scrollPane.getViewport().updateUI();
				repaint();
			}
		});

		//print-related stuff
		PrinterJob pj = PrinterJob.getPrinterJob();
		mPageFormat = pj.defaultPage();
		mPageFormat.setOrientation(PageFormat.LANDSCAPE);

		
		setVisible(true);
		
	}

	public void processStructureDocumentEvent( StructureDocumentEvent structureDocumentEvent ){
		
		try{
		
			int type = structureDocumentEvent.type;
			int change = structureDocumentEvent.change;
			
			if ( type == StructureDocumentEvent.TYPE_VIEWER ){
				if ( change == StructureDocumentEvent.CHANGE_ADDED ){
					structureDocument = structureDocumentEvent.structureDocument;
					viewerAdded( structureDocumentEvent.viewer );
				}
				else if ( change == StructureDocumentEvent.CHANGE_REMOVED ){
					viewerRemoved( structureDocumentEvent.viewer );
					structureDocument = null;
				}
			}
			else if ( type == StructureDocumentEvent.TYPE_DATA ){
			
				Entry e = structureDocumentEvent.entry;
				if (e == null) return;
								
				if ( change == StructureDocumentEvent.CHANGE_ADDED ){
					entryAdded(e, true);
				}
				else if ( change == StructureDocumentEvent.CHANGE_REMOVED ){
					entryRemoved(e);
				}
				else if (change == StructureDocumentEvent.CHANGE_RENAMED){
					Set keys = entries.keySet();
					Iterator it = keys.iterator();
					while (it.hasNext()){
						Sequence s = (Sequence)it.next();
						Entry entry = (Entry)entries.get(s);
						
						if (e == entry){
							s.setName(entry.getName());
							sequencePanel.repaint();
							break;
						}
					}

				}
				
			}
			else if (type == StructureDocumentEvent.TYPE_GLOBAL){
				
				if (change == StructureDocumentEvent.CHANGE_RELOAD){
					
					//reread all loaded entries
					//remove listeners
					for (int i = 0; i < sequenceObjects.size(); i++){
						Sequence s = (Sequence)sequenceObjects.get(i);
						s.getSequenceStyles().removeSequenceStylesEventListener(this);
						s.getSequenceStyles().removeStructureComponentEventListener(this);
						
						StructureStyles styles = (StructureStyles)styleMap.get(s);
						if (styles != null){
							styles.removeStructureComponentEventListener(this);
							styles.removeStructureStylesEventListener(this);
						}
					}
					
					
					sequences.clear();
					sequenceObjects.clear();
					backupSequences.clear();
					styleMap.clear();
					entries.clear();
					alignmentData = null;
					menuSequenceAlignmentDownload.setEnabled(false);

					hiddenSequences.clear();
					cellResidueMapping = new HashMap();
					residueCellMapping = new HashMap();
					scrollBarH.setMinimum( 0 );
					scrollBarH.setMaximum( 0 );
					scrollBarV.setMinimum( 0 );
					scrollBarV.setMaximum( 0 );
					
					menuEditReset.setEnabled(false);
					
					//now reread everything
					for ( int i = 0; i < structureDocument.getEntryCount(); i++ )
					{
						Entry entry = structureDocument.getEntry(i);
						entryAdded( entry, false );
					}
					
					computeResidueMax(false);
					sequencePanel.updateView(true);
					scrollPane.revalidate();
					revalidate();
					repaint();
				}
			}
		
		}
		catch (Exception ex){
			structureDocument.parent.displayExceptionMessage("Exception processing StructureDocumentEvent", ex);
		}
	}
	
	private void viewerAdded( Viewer viewer )
	{
		if ( viewer != this ) return;
		
		for ( int i = 0; i < structureDocument.getEntryCount(); i++ )
		{
			Entry entry = structureDocument.getEntry(i);
			entryAdded( entry, false );
		}
		
		scalePanel.setResidueCount(residueMax);
		sequencePanel.updateView(true);
		scrollPane.revalidate();
		revalidate();
		repaint();
	}

	/**
	 *  A Viewer was just removed from a StructureDocument.
	 */
	private void viewerRemoved( Viewer viewer )
	{
		if ( viewer != this ) return;

		for ( int i = 0; i < structureDocument.getEntryCount(); i++ )
		{
			Entry entry = structureDocument.getEntry(i);
			entryRemoved( entry );
		}
	}
	
	public void entryAdded(Entry entry, boolean process){
		
		if (entry.TYPE == Entry.SEQUENCE_ENTRY){
			//just the sequence, no structure
			//simply extract symbols, create Cells and instantiate the array
			SequenceEntry e = (SequenceEntry)entry;
			if (e.getSequenceCount() == 1){
				Sequence sequence = e.getSequence();
				String seq = sequence.getSequence();
				
				if (seq == null || seq.length() == 0) return;
				
				Cell[] cells = sequence.getCells();
				sequences.add(cells);
				sequenceObjects.add(sequence);
				entries.put(sequence, entry);
				
				sequence.getSequenceStyles().addSequenceStylesEventListener(this);
				sequence.getSequenceStyles().addStructureComponentEventListener(this);
				
				if (cells.length > residueMax) residueMax = cells.length;
			}
			else{
				//it's a composite file with more than one sequence
				for (int i = 0; i < e.getSequenceCount(); i++){
					Sequence sequence = e.getSequence(i);
					String seq = sequence.getSequence();
					String name = sequence.getName();
					if (seq == null || seq.length() == 0) continue;

					Cell[] cells = sequence.getCells();
					if (cells.length > residueMax) residueMax = cells.length;
					sequences.add(cells);
					sequenceObjects.add(sequence);
					entries.put(sequence, entry);
					sequence.getSequenceStyles().addSequenceStylesEventListener(this);
					sequence.getSequenceStyles().addStructureComponentEventListener(this);
				}
			}
		}
		else if (entry.TYPE == Entry.STRUCTURE_ENTRY || entry.TYPE == Entry.FRAGMENT_ENTRY){
			//get the sequence out of the structure
			
			Structure structure = ((StructureEntry)entry).getStructure();
			StructureMap map = structure.getStructureMap();
			
			StructureStyles styles = map.getStructureStyles();
			
			HashMap residueCell = new HashMap();
			HashMap cellResidue = new HashMap();
			
			StringBuffer buffer = new StringBuffer();
			Cell[] cells = new Cell[map.getResidueCount()];
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (ProteinProperties.isAminoAcid(r.getCompoundCode())){
					cells[i] = new Cell(i, AminoAcidInfo.getLetterFromCode(r.getCompoundCode()), r);

					residueCell.put(r, cells[i]);
					cellResidue.put(cells[i], r);
					buffer.append(AminoAcidInfo.getLetterFromCode(r.getCompoundCode()));
				}
				else if (DNAProperties.isNucleicAcid(r.getCompoundCode())){
					if (r.getCompoundCode().length() > 1){
						cells[i] = new Cell(i, DNAProperties.getSymbolFromThreeLetters(r.getCompoundCode()), r);
					}
					else{
						cells[i] = new Cell(i, r.getCompoundCode(), r);
					}
					residueCell.put(r, cells[i]);
					cellResidue.put(cells[i], r);
					buffer.append(r.getCompoundCode());
				}
				else if (r.getCompoundCode().equals("HOH")){
					cells[i] = new Cell(i, "*", r);
					residueCell.put(r, cells[i]);
					cellResidue.put(cells[i], r);
					buffer.append("*");
				}
				else{
					cells[i] = new Cell(i, "#", r);
					residueCell.put(r, cells[i]);
					cellResidue.put(cells[i], r);
					buffer.append("#");
				}
				
				cells[i].residueId = cells[i].residue.getResidueId();
			}
			
			
			if (cells.length == 0){
//				System.out.println("no cells");
				return;
			}
			
			Sequence sequence = new Sequence(entry.getName(), new String(buffer.toString()));
			sequenceObjects.add(sequence);
			entries.put(sequence, entry);
			
			sequence.setCells(cells);
			
			//set the Sequence object in Cells
			for (int i = 0; i < cells.length; i++){
				((Cell)cells[i]).sequence = sequence;
			}
			
			if (cells.length > residueMax) residueMax = cells.length;
			sequences.add(cells);

			residueCellMapping.put(sequence, residueCell);
			cellResidueMapping.put(sequence, cellResidue);
			
			styles.addStructureStylesEventListener(this);
			styles.addStructureComponentEventListener(this);
			styleMap.put(sequence, styles);
		}
		
		//update the height
		int height = scrollPane.getHeight();
		int ch = parameters.cellHeight;
		if (showSs){
			ch += SequenceViewer.ssHeight;
		}
		int seqs = (int)height/ch;
		parameters.visibleSequences = seqs;

		
		if (process){
			scalePanel.setResidueCount(residueMax);
			sequencePanel.updateView(true);
			scrollPane.revalidate();
			revalidate();
			repaint();
		}
	}
	
	public void entryRemoved(Entry entry){
		
//		System.out.println("entry removed with undoMode = " + undoMode);
		
		alignmentData = null;
		menuSequenceAlignmentDownload.setEnabled(false);
		
		//remove all sequences from this entry
		for (int i = sequenceObjects.size()-1; i >= 0 ; i--){
			Sequence sequence = (Sequence)sequenceObjects.get(i);
			if (entries.get(sequence) == entry){
				//a child sequence
				sequences.remove(i);
				sequenceObjects.remove(i);
				entries.remove(sequence);
				styleMap.remove(sequence);
				residueCellMapping.remove(sequence);
				cellResidueMapping.remove(sequence);
				sequencePanel.matchedSequences.remove(sequence);
				
				if (styleMap.containsKey(sequence)){
					((StructureStyles)styleMap.get(sequence)).removeStructureComponentEventListener(this);
					styleMap.remove(sequence);
				}
				sequence.getSequenceStyles().removeSequenceStylesEventListener(this);
				sequence.getSequenceStyles().removeStructureComponentEventListener(this);
			}
		}
		
		//update the height
		int height = scrollPane.getHeight();
		int ch = parameters.cellHeight;
		if (showSs){
			ch += SequenceViewer.ssHeight;
		}
		int seqs = (int)height/ch;
		parameters.visibleSequences = seqs;

		
		if (!undoMode){
			computeResidueMax(false);
			scalePanel.setResidueCount(residueMax);
		
			revalidate();
			scalePanel.repaint();
			sequencePanel.updateView(true);
		}
	}
	
	public Entry getEntry(Sequence sequence){
		return (Entry)entries.get(sequence);
	}
	
	public void sequenceRemoved(Sequence sequence){
		
		//get the order from the entries vector and remove the corresponding index in the data model
		int index = sequenceObjects.indexOf(sequence);
		if (index == -1) return;
		
//		System.out.println("sequenceRemoved with undoMode = " + undoMode);
		
		if (((Cell[])sequences.get(index)).length == residueMax){
			//find another longest sequence to replace this one
			if (!undoMode) computeResidueMax(false);
		}
		
		adjustVScrollBar(false);
		
		Entry entry = (Entry)entries.get(sequence);
		if (entry instanceof StructureEntry){
			structureDocument.removeEntry(entry.getId());//this is just a sequence trace of a loaded structure
		}
		else if (entry instanceof SequenceEntry){
			SequenceEntry se = (SequenceEntry)entry;
			if (se.getSequenceCount() == 1){
				structureDocument.removeEntry(entry.getId());
			}
			else{
				scalePanel.setResidueCount(residueMax);
				sequences.remove(index);
				
				//this may be a removal of a single sequence Entry.
				//or an alignment
				//remove just this sequence from the Entry
				se.removeSequence(sequence);
				
				sequenceObjects.remove(sequence);
				styleMap.remove(sequence);
				residueCellMapping.remove(sequence);
				cellResidueMapping.remove(sequence);
				
				if (!undoMode){
					revalidate();
					sequencePanel.updateView(true);
					scalePanel.repaint();
//					repaint();
				}
			}
		}
	}
	
	public void computeResidueMax(boolean repaint){
		residueMax = 0;
		for (int i = 0; i < sequences.size(); i++){
			Cell[] cells = (Cell[])sequences.get(i);
			if (cells.length > residueMax) residueMax = cells.length;
		}
		adjustHScrollBar(repaint);
		scalePanel.setResidueCount(residueMax);
	}
	
	
	public void adjustHScrollBar(boolean repaint){
		int margin = 10;
		if (scrollBarV.isVisible()){
			margin += 15;
		}
		
		int span = (int)((scrollPane.getWidth()-margin-this.buttonWidthCells*parameters.cellWidth) / parameters.cellWidth);
//		System.out.println("span = " + span);
		scrollBarH.setMaximum(residueMax);
		scrollBarH.setVisibleAmount(span);
		scrollBarH.setBlockIncrement(span/2);
		
		
		if (repaint){
			revalidate();
			repaint();
		}
	}
	
	public void adjustVScrollBar(boolean repaint){
		
		int ch = parameters.cellHeight;
		if (showSs){
			ch += SequenceViewer.ssHeight;
		}
		
		int vspan = (int)(scrollPane.getHeight()/ch);
		scrollBarV.setValue(sequencePanel.getStartSequence());
		
//		System.out.println("vspan = " + vspan);
		
//		System.out.println("visible sequences = " + parameters.visibleSequences);
		
		scrollBarV.setMaximum(sequences.size() - hiddenSequences.size());
		scrollBarV.setVisibleAmount(vspan);
		scrollBarV.setBlockIncrement(vspan/2);
		
		if (repaint){
			revalidate();
			repaint();
		}

	}
	
	public void updateView(){
		scalePanel.updateParameters();
		scalePanel.repaint();
		sequencePanel.showTT(StylesPreferences.showResidueTips);
		adjustVScrollBar(false);
		computeResidueMax(true);
	}
	
	public boolean isStructure(Sequence sequence){
		return styleMap.containsKey(sequence);
	}

	public void processStructureComponentEvent(StructureComponentEvent event){
		
		StructureComponent structureComponent = event.structureComponent;
		if (event.changeType == StructureComponentEvent.TYPE_REMOVE){
			
			if (structureComponent == null){
				//it came from SequenceStyles
				Cell cell = event.cell;
				
				int currentIndex = cell.index;
				
				sequencePanel.backupDataModel();
				
				Cell[] cells = cell.sequence.getCells();

				Cell[] cc = new Cell[cells.length-1];
				int counter = 0;
				int n = 1;
				for (int i = 0; i < cells.length; i++){
					if (cells[i] == cell){
						continue;
					}
					cc[counter] = cells[i];
					cc[counter].residueId = n;
					n++;
					counter++;
				}

				sequences.remove(cells);

				sequences.add(sequenceObjects.indexOf(cell.sequence), cc);
				cell.sequence.setCells(cc);
				computeResidueMax(false);
				adjustVScrollBar(false);
				
				//modify the corresponding Sequence object as well
				String seq = cell.sequence.getSequence();
				StringBuffer buffer = new StringBuffer();
				buffer.append(seq.substring(0, currentIndex));
				buffer.append(seq.substring(currentIndex+1));
				cell.sequence.setSequence(buffer.toString());
				revalidate();
				repaint();
				setUndo(true);
				
				structureDocument.parent.updateView();
				
				
				
				return;
			}
			
			if (structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				
				//find out which sequence this residue belongs to
				//iterate over styleMap to get the right Sequence object
				
				Residue r = (Residue)structureComponent;
				Sequence sequence = null;
				for (int i = 0; i < sequenceObjects.size(); i++){
					Sequence seq = (Sequence)sequenceObjects.get(i);
					if (r.structure.getStructureMap().getStructureStyles() == styleMap.get(seq)){
						sequence = seq;
						break;
					}
				}
				
				if (sequence == null) return;
				
				HashMap rc = (HashMap)residueCellMapping.get(sequence);
				HashMap cr = (HashMap)cellResidueMapping.get(sequence);
				
				rc.remove(r);
				
				int index = -1;
				
				//delete the specified residue
				Cell[] cells = (Cell[])sequences.get(sequenceObjects.indexOf(sequence));
				Cell[] cc = new Cell[cells.length-1];
				int counter = 0;
				for (int i = 0; i < cells.length; i++){
					if (cells[i].residue != null && cells[i].residue == r){
						cr.remove(cells[i]);
						index = i;
						continue;
					}
					cc[counter] = cells[i];
					counter++;
				}
				sequences.remove(cells);
				sequences.add(sequenceObjects.indexOf(sequence), cc);
//				computeResidueMax(false);
				
				//modify the corresponding Sequence object as well
				String seq = sequence.getSequence();
				StringBuffer buffer = new StringBuffer();
				buffer.append(seq.substring(0, index));
				buffer.append(seq.substring(index+1));
				sequence.setSequence(buffer.toString());
				
				if (!event.batch){
					revalidate();
					repaint();
				}
				
			}
			
		}
		else if (event.changeType == StructureComponentEvent.TYPE_REPLACE){
			
			if (structureComponent == null){
				//it must have been sent by SequenceStyles
				if (event.cell == null) return;//nothing to do
				
				Cell cell = event.cell;
				sequencePanel.replaceResidue(cell.sequence, cell.index, event.data);
				replacedResidue = event.data;
				setUndo(true);
				return;
			}

			if (structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				
				Residue r = (Residue)structureComponent;
				String data = event.data;
				
				Sequence sequence = null;
				for (int i = 0; i < sequenceObjects.size(); i++){
					Sequence seq = (Sequence)sequenceObjects.get(i);
					if (r.structure.getStructureMap().getStructureStyles() == styleMap.get(seq)){
						sequence = seq;
						break;
					}
				}
				
				if (sequence == null) return;
				
				HashMap residueCell = (HashMap)residueCellMapping.get(sequence);
				Cell cell = (Cell)residueCell.get(r);
				String symbol = null;
				if (ProteinProperties.isAminoAcid(r.getCompoundCode())){
					symbol = ProteinProperties.getSingleLetterAminoAcidName(r.getCompoundCode());
				}
				else if (DNAProperties.isNucleicAcid(r.getCompoundCode())){
					symbol = r.getCompoundCode();
				}
				else{
					symbol = "#";
				}
				cell.symbol = symbol;
				
				sequencePanel.replaceResidue(sequence, (Cell)residueCell.get(r), AminoAcidInfo.getLetterFromCode(data));
				replacedResidue = data;
				setUndo(true);
			}
		}
		else if (event.changeType == StructureComponentEvent.TYPE_UNDO_REPLACE){

			Residue r = (Residue)structureComponent;
			String data = replacedResidue;
			
			Sequence sequence = null;
			for (int i = 0; i < sequenceObjects.size(); i++){
				Sequence seq = (Sequence)sequenceObjects.get(i);
				if (r.structure.getStructureMap().getStructureStyles() == styleMap.get(seq)){
					sequence = seq;
					break;
				}
			}
			
			if (sequence == null) return;
			
			HashMap residueCell = (HashMap)residueCellMapping.get(sequence);
			Cell cell = (Cell)residueCell.get(r);
			
			sequencePanel.replaceResidue(sequence, (Cell)residueCell.get(r), AminoAcidInfo.getLetterFromCode(data));

			setUndo(false);
		}
		else if (event.changeType == StructureComponentEvent.TYPE_ADD){
			if (event.structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				
				Residue r = (Residue)structureComponent;
				//append the new residue to the existing sequence
				//find the sequence
				Sequence sequence = null;
				for (int i = 0; i < sequenceObjects.size(); i++){
					Sequence seq = (Sequence)sequenceObjects.get(i);
					if (r.structure.getStructureMap().getStructureStyles() == styleMap.get(seq)){
						sequence = seq;
						break;
					}
				}
				
				if (sequence == null) return;
				Cell[] cells = sequence.getCells();
				
				//create a new Cell
				String symbol = null;
				if (ProteinProperties.isAminoAcid(r.getCompoundCode())){
					symbol = ProteinProperties.getSingleLetterAminoAcidName(r.getCompoundCode());
				}
				else if (DNAProperties.isNucleicAcid(r.getCompoundCode())){
					symbol = r.getCompoundCode();
				}
				else{
					symbol = "#";
				}
				Cell cell = new Cell(sequence, cells.length, symbol, r);
				
				HashMap rc = (HashMap)residueCellMapping.get(sequence);
				HashMap cr = (HashMap)cellResidueMapping.get(sequence);
				
				rc.put(r, cell);
				cr.put(cell, r);
				
				//update the cells array
				Cell[] newCells = new Cell[cells.length + 1];
				for (int i = 0; i < cells.length; i++){
					newCells[i] = cells[i];
				}
				
				newCells[cells.length] = cell;
				
				sequences.remove(cells);
				sequences.add(sequenceObjects.indexOf(sequence), newCells);
				sequence.setCells(newCells);
				
				String seq = sequence.getSequence();
				seq += symbol;
				
			}
		}
	}
	
	public void processStructureStylesEvent(StructureStylesEvent event){
		
		if (event.property == StructureStyles.PROPERTY_RENDERING) return;
//		if (!StylesPreferences.commonColoring && !StylesPreferences.commonSelection) return;//no common action needed
		
		StructureComponent structureComponent = event.structureComponent;
		StructureStyles styles = event.structureStyles;
		
		//get the Sequence
		int index = -1;
		for (int i = 0; i < sequenceObjects.size(); i++){
			Sequence s = (Sequence)sequenceObjects.get(i);
			if (styleMap.get(s) == styles){
				index = i;
				break;
			}
		}
		
		if (index == -1) return;//the event doesn't apply to anything loaded now
		
		
		try{
		
			StructureMap map = styles.getStructureMap();
			HashMap mapping = (HashMap)residueCellMapping.get(sequenceObjects.get(index));
			
			if (structureComponent == null){
				//the event applies to the entire Sequence that corresponds to this style
				//process all residues in this structure
				if (event.attribute == StructureStyles.ATTRIBUTE_SELECTION){
					boolean state = (event.flag != StructureStyles.FLAG_NONE);
					if (state){
						for (int i = 0; i < map.getResidueCount(); i++){
							Cell cell = (Cell)mapping.get(map.getResidue(i));
							cell.selected = state;
							if (!selectedCells.contains(cell)) selectedCells.add(cell);
						}
					}
					else{
						//deselect the current selection
						for (int i = 0; i < selectedCells.size(); i++){
							((Cell)selectedCells.get(i)).selected = state;
						}
						selectedCells.clear();
						selectedColumns.clear();
						
					}
					if (!event.batch) this.updateView();
				}
			}
			else{
				if (event.attribute == StructureStyles.ATTRIBUTE_SELECTION){
					
					//check by type of structure component
					if (structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
						Residue r = map.getResidue((Atom)structureComponent);

						//if at least one atom in the residue is selected, select the cell
						boolean state = false;
						for (int i = 0; i < r.getAtomCount(); i++){
							if (styles.isSelected((Atom)r.getAtom(i))){
								state = true;
								break;
							}
						}
						
//						System.out.println("r = " + r + " for atom " + ((Atom)structureComponent).name);
						
						if (r == null) return;
						
						//now apply selection to the cell that corresponds to this residue
						Cell cell = (Cell)mapping.get(r);
						if (cell.selected == state) return;
						cell.selected = state;
						
						if (state){
							if (!selectedCells.contains(cell))selectedCells.add(cell);
						}
						else{
							if (selectedCells.contains(cell)){
								selectedCells.remove(cell);
								selectedColumns.remove(new Integer(cell.index));
							}
						}
					}
					else if ( structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_BOND){
						Residue r = map.getResidue(((Bond)structureComponent).getAtom(0));
						//if at least one atom in the residue is selected, select the cell
						boolean state = false;
						for (int i = 0; i < r.getAtomCount(); i++){
							if (styles.isSelected((Atom)r.getAtom(i))){
								state = true;
								break;
							}
						}
						
						Cell cell = (Cell)mapping.get(r);
						if (cell.selected == state) return;
						cell.selected = state;
						
						if (state){
							if (!selectedCells.contains(cell))selectedCells.add(cell);
						}
						else{
							if (selectedCells.contains(cell)){
								selectedColumns.remove(new Integer(cell.index));
								selectedCells.remove(cell);
							}
						}
					}
					else if ( structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
						Residue r = (Residue)structureComponent;
						
						//if at least one atom in the residue is selected, select the cell
						boolean state = styles.isSelected(r);
						Cell cell = (Cell)mapping.get(r);
						if (cell.selected == state) return;
						cell.selected = state;
						if (state){
							if (!selectedCells.contains(cell))selectedCells.add(cell);
						}
						else{
							if (selectedCells.contains(cell)){
								selectedCells.remove(cell);
								//just in case there was column selection
								selectedColumns.remove(new Integer(cell.index));
							}
						}
	
					}
					else if ( structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
						
						if (event.property == -10) return;
						//process all residues in this chain
						Chain chain = (Chain)structureComponent;
						boolean state = styles.isSelected(chain);
						for (int i = 0; i < chain.getResidueCount(); i++){
							Residue r = chain.getResidue(i);
							Cell cell = (Cell)mapping.get(r);
							cell.selected = state;
							if (state){
								if (!selectedCells.contains(cell))selectedCells.add(cell);
							}
							else{
								if (selectedCells.contains(cell)){
									selectedCells.remove(cell);
									selectedColumns.remove(new Integer(cell.index));
								}
							}
	
						}
					}
					
					if (!event.batch) sequencePanel.updateView(false);
				}
				else if (event.attribute == StructureStyles.ATTRIBUTE_STYLE){
					if (event.property == StructureStyles.PROPERTY_COLOR){
						
						if ( structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
							
							Residue r = (Residue)structureComponent;
							Atom a = r.getAtom(0);
							if (styles.getAtomColor(a) instanceof AtomColorByElement){
								((Cell)mapping.get(r)).color = styleManager.getDefaultColor();
							}
							else{
								float[] c = new float[3];
								styles.getResidueColor(r).getResidueColor(r, c);
								((Cell)mapping.get(r)).color = styleManager.getColor(c);
								if (((Cell)mapping.get(r)).selected){
									((Cell)mapping.get(r)).selected = false;
									selectedCells.remove((Cell)mapping.get(r));
								}
							}
							
							if (!event.batch) sequencePanel.updateView(false);
							
						}
						else if ( structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
	//						System.out.println("chain event for " + structureComponent);
							//process all residues in this chain
							Chain chain = (Chain)structureComponent;
							float[] c = new float[3];
							for (int i = 0; i < chain.getResidueCount(); i++){
								Atom a = chain.getResidue(i).getAtom(0);
								if (a == null) continue;
								styles.getAtomColor(a).getAtomColor(a, c);
								((Cell)mapping.get(chain.getResidue(i))).color = styleManager.getColor(c);
								if (((Cell)mapping.get(chain.getResidue(i))).selected){
									((Cell)mapping.get(chain.getResidue(i))).selected = false;
									selectedCells.remove((Cell)mapping.get(chain.getResidue(i)));
								}
							}
							sequencePanel.updateView(false);
						}
						
					}
				}
				
				
			}
		
		}
		catch (Exception e){
			System.out.println("exception processing styles event in sequence viewer: " + e);
			e.printStackTrace();
		}
		
	}
	
	private JMenuBar createMenuBar(){
		
		JMenuBar menuBar = new JMenuBar();
		JMenu menuFile = new JMenu("File");

		JMenuItem menuFileNew = new JMenuItem("New sequence...");
		menuFileNew.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				//display dialog and load the new entry
				if (structureDocument.parent.getDisplayDialogStatus()){
					return;
				}
				
				Thread runner = new Thread(){
					public void run(){
						try{
							DisplayDialog dialog = structureDocument.parent.getDisplayDialog();
							dialog = new CreateSequenceDialog(structureDocument.parent.getApplicationFrame(), parent);
						}
						catch (Exception ex){
							structureDocument.parent.displayExceptionMessage("Exception opening create sequence dialog: ", ex);
						}
					}
				};
				runner.start();				


			}
		});
		

		
		
		
		JMenuItem menuFileOpen = new JMenuItem("Open sequence file...");
		menuFileOpen.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, KeyEvent.CTRL_MASK, false));
		menuFileOpen.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				//display the dialog
				Thread runner = new Thread(){
					public void run(){
						JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
						chooser.setDialogTitle("Open sequence file");
						
						chooser.setDialogType(JFileChooser.OPEN_DIALOG);

						String loaders[] = EntryFactory.getFileEntryLoaderNames( );
						javax.swing.filechooser.FileFilter defaultFilter = null;
						int counter = -1;
						for ( int i=0; i<loaders.length; i++ ){
							final FileEntryLoader loader = EntryFactory.getFileEntryLoader( loaders[i] );
							if (loader.getDataType() != Entry.DATA_SEQUENCE) continue;
							
							counter++;
							final String loaderName = loader.getLoaderName( );

//							for (int j = 0; j < 2; j++){
							javax.swing.filechooser.FileFilter sequenceFilter = new javax.swing.filechooser.FileFilter( ){
									private FileEntryLoader fsLoader = null;
									private String fslName = null;
				
									// public FileFilter( ) // anonymous constructor
									{
										fsLoader = loader;
										fslName = loaderName;
									}
				
									public boolean accept( File file ){
										if ( loader.canLoad( file ) ) return true;
										if ( file.isDirectory() ) return true;
										return false;
									}
									
									public String getDescription( ){
										return loaderName;
									}
								};
								
								chooser.addChoosableFileFilter( sequenceFilter );
//							}
							if (counter == 0){
								defaultFilter = sequenceFilter;
							}
							
						}
						chooser.setFileFilter(defaultFilter);

						chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
						int result = chooser.showOpenDialog(parent.getApplicationFrame());
						if (result == JFileChooser.CANCEL_OPTION){
							return;
						}
						File filename = chooser.getSelectedFile();
						openFile(filename);
						
						//save the used directory
						String dir = filename.getParentFile().toString();
						parent.setLastUsedDirectory(dir);

					}
				};
				runner.start();
				
				
			}
		});
		
		JMenuItem menuFileSave = new JMenuItem("Save sequences...");
		menuFileSave.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, KeyEvent.CTRL_MASK, false));
		menuFileSave.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						IOHandler.saveAlignment(sequences, parent);
					}
				};
				runner.start();
			}
		});
				
//		JMenuItem jMenuFileExport = new JMenuItem("Export alignment...");
//		JMenuItem jMenuFileImage = new JMenuItem("Export graphics...");
		
		JMenuItem menuFilePrintSetup = new JMenuItem("Print setup...");
		menuFilePrintSetup.setEnabled(true);
		menuFilePrintSetup.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				printSetupHandler();
			}
		});
		
		JMenuItem menuFilePrint = new JMenuItem("Print...");
		menuFilePrint.setEnabled(true);
		menuFilePrint.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				printHandler(false);
			}
		});
		
		JMenuItem menuFilePrintSelection = new JMenuItem("Print selection...");
		menuFilePrintSelection.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
//				printHandler(true);
			}
		});
		
		JMenuItem menuFileClose = new JMenuItem("Close all entries");
		menuFileClose.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, KeyEvent.CTRL_MASK, false));
		menuFileClose.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				clearEntries();
			}
		});

		menuFile.add(menuFileNew);
		menuFile.add(menuFileOpen);
		menuFile.add(menuFileSave);
		menuFile.add(new JSeparator());
		menuFile.add(menuFilePrintSetup);
		menuFile.add(menuFilePrint);
		menuFile.add(new JSeparator());
		menuFile.add(menuFileClose);
		
		
		
		JMenu menuSequence = new JMenu("Sequence");
		
		JMenuItem menuSequenceFind = new JMenuItem("Find sequence...");
		menuSequenceFind.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				Thread runner = new Thread(){
					public void run(){
						if (sequences.size() == 0) return;
						
						//if we find the sequence that is hidden, mark it and show it
						String input = (String)JOptionPane.showInputDialog(parent.getApplicationFrame(), 
								"Enter name of sequence", "Find sequence by name", JOptionPane.INFORMATION_MESSAGE, null, null, "");
						
						if (input == null || input.length() == 0){
							return;
						}
						
						//find the sequence
						Vector matches = new Vector();//indices of matches
						for (int i = 0; i < sequenceObjects.size(); i++){
							Sequence s = (Sequence)sequenceObjects.get(i);
							if (s.getName().toUpperCase().startsWith(input.trim().toUpperCase())){
								//this is it
								matches.add(s);
								if (hiddenSequences.contains(s)){
									setPanelVisibility(s, true, false);
								}
							}
						}
						
						if (matches.size() == 0){
							parent.displayErrorMessage("No matches found");
							return;
						}
						
						sequencePanel.setMatchedSequences(matches);
						
						//scroll to the first match
						int index = sequenceObjects.indexOf(matches.get(0));
						
						int topIndex = sequences.size() - hiddenSequences.size() - parameters.visibleSequences;//max start index

						setStartSequence(Math.min(topIndex, index));
						adjustVScrollBar(false);
						
						updateView();

					}
				};
				runner.start();
			}
		});

		JMenuItem menuSequenceDistribution = new JMenuItem("Side chain distribution...");
		menuSequenceDistribution.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				Thread runner = new Thread(){
					public void run(){
						computeDistribution();
					}
				};
				runner.start();
			}
		});
		
		
		JMenuItem menuSequenceSignature = new JMenuItem("Sequence signature...");
		menuSequenceSignature.setEnabled(false);
		menuSequenceSignature.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
//				ProfileLoader loader = new ProfileLoader(sequenceFrame, this, sequences.size());
			}
		});
		
		
		JMenuItem menuSequenceMotif = new JMenuItem("Find residue motif...");
		menuSequenceMotif.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (sequences.size() == 0){
							parent.displayErrorMessage("No sequences are loaded");
							return;
						}
						MotifDialog d = new MotifDialog(parent.getApplicationFrame(), thisViewer, sequences.size());
					}
				};
				runner.start();
			}
		});
		
		
		JMenuItem menuSequenceTranslate = new JMenuItem("Translate DNA into protein");
		menuSequenceTranslate.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (sequences.size() == 0){
							parent.displayErrorMessage("No sequences are loaded");
							return;
						}
						if (sequencePanel.getSelectedSequences().size() == 0 && sequences.size() > 1){
							parent.displayErrorMessage("First select sequence to be translated and try again");
							return;
						}

						translateDNA();
					}
				};
				runner.start();
			}
		});
		
		
		
		
		
		
		
		menuSequence.add(menuSequenceFind);
		menuSequence.add(menuSequenceMotif);
		menuSequence.add(new JSeparator());
		menuSequence.add(menuSequenceDistribution);
		menuSequence.add(menuSequenceSignature);
		menuSequence.add(new JSeparator());
//		menuSequence.add(menuSequenceTranslate);
		
		
		
		
		JMenu menuEdit = new JMenu("Edit");

		
		final JMenuItem menuEditInsertColumn = new JMenuItem("Insert gapped column");
		menuEditInsertColumn.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				//get the index of the first selected column
				if (selectedColumns.size() == 0){
					structureDocument.parent.displayErrorMessage("No columns are selected. Select one column and try again");
					return;
				}
				Collections.sort(selectedColumns);
				sequencePanel.insertGappedColumn(((Integer)selectedColumns.get(0)).intValue());
			
			}
		});
		
		final JMenuItem menuEditDeleteColumn = new JMenuItem("Delete gapped column");
		menuEditDeleteColumn.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				if (selectedColumns.size() == 0){
					structureDocument.parent.displayErrorMessage("No columns are selected. Select one column and try again");
					return;
				}
				sequencePanel.deleteGappedColumn(((Integer)selectedColumns.get(0)).intValue());
			}
		});
		
		final JMenuItem menuEditRemoveGappedColumns = new JMenuItem("Remove all gapped columns");
		menuEditRemoveGappedColumns.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				sequencePanel.removeGappedColumns();
			}
		});
		
		final JMenuItem menuEditRemoveAllGaps = new JMenuItem("Remove all gaps");
		menuEditRemoveAllGaps.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				sequencePanel.removeAllGaps();
			}
		});
		
		menuEditReset = new JMenuItem("Undo sequence change");
		menuEditReset.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				sequencePanel.restoreDataModel();
			}
		});
		
		menuEditInsertColumn.setEnabled(editMode);
		menuEditDeleteColumn.setEnabled(editMode);
		menuEditRemoveGappedColumns.setEnabled(editMode);
		menuEditRemoveAllGaps.setEnabled(editMode);
		menuEditReset.setEnabled(false);
		
		menuEditMode = new JCheckBoxMenuItem("Editing mode");
		menuEditMode.setSelected(editMode);
		menuEditMode.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				editMode = menuEditMode.isSelected();
				sequencePanel.clearClipboard();
				sequencePanel.resetEditMenu();
				
				menuEditInsertColumn.setEnabled(menuEditMode.isSelected());
				menuEditDeleteColumn.setEnabled(menuEditMode.isSelected());
				menuEditRemoveGappedColumns.setEnabled(menuEditMode.isSelected());
				menuEditRemoveAllGaps.setEnabled(menuEditMode.isSelected());
			}
		});
		
		
		JMenuItem menuSequenceAlign = new JMenuItem("Align all loaded sequences");
		menuSequenceAlign.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						performAlignment();
					}
				};
				runner.start();
			}
		});
		
		
		JMenuItem menuSequenceAlignSet = new JMenuItem("Align selected sequence to others");
		menuSequenceAlignSet.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						
						//check for identical names
						HashMap nameCheck = new HashMap();
						
						HashMap map = new HashMap();//name -> sequence for new sequences
						Vector names = new Vector();
						
						//get the selected sequence
						if (sequencePanel.getSelectedSequences().size() != 1){
							parent.displayErrorMessage("You must select only one sequence in order to run one-to-set alignment.");
							return;
						}
						
						Sequence query = (Sequence)sequencePanel.getSelectedSequences().get(0);
						
						if (query == null) return;
						
						

						//concatenate the string representations of the loaded sequences
						StringBuffer buffer = new StringBuffer();
						StringBuffer qBuffer = new StringBuffer();//for query
						
						for (int i = 0; i < sequenceObjects.size(); i++){
							Sequence s = (Sequence)sequenceObjects.get(i);
							
							if (s == query){
								qBuffer.append(">");
								qBuffer.append(s.getName());
								nameCheck.put(s.getName(), null);
								qBuffer.append("\n");
								String seq = s.getSequence();
								//compress it if necessary

								Pattern pattern = Pattern.compile("-");
								Matcher matcher = pattern.matcher(seq);
								String temp = matcher.replaceAll("");
								
								Pattern pattern2 = Pattern.compile("#");
								Matcher matcher2 = pattern2.matcher(temp);
								qBuffer.append(matcher2.replaceAll(""));
								qBuffer.append("\n");
								
								continue;
							}
							
							buffer.append(">");
							buffer.append(s.getName());
							nameCheck.put(s.getName(), null);
							buffer.append("\n");
							String seq = s.getSequence();
							//compress it if necessary

							Pattern pattern = Pattern.compile("-");
							Matcher matcher = pattern.matcher(seq);
							String temp = matcher.replaceAll("");
							
							Pattern pattern2 = Pattern.compile("#");
							Matcher matcher2 = pattern2.matcher(temp);
							buffer.append(matcher2.replaceAll(""));
							buffer.append("\n");
							
							map.put(s.getName(), new StringBuffer());
							names.add(s.getName());
						}
						
						if (nameCheck.size() < sequenceObjects.size()){
							//there are duplicate names
							parent.displayErrorMessage("Two or more loaded sequences have identical names, which may affect parsing of the results. Please rename and rerun.");
							return;
						}


						//check whether there is clustalw directory under the installation tree
						String path = parent.getInstallationHome();
						File dir = new File(path + File.separator + "clustalw");
						
						
						
						
					}
				};
				runner.start();
			}
		});
		
		menuSequenceAlignmentDownload = new JMenuItem("Save alignment output...");
		menuSequenceAlignmentDownload.setEnabled(false);
		menuSequenceAlignmentDownload.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				saveAlignmentOutput();
			}
		});
		
		menuSequence.add(menuSequenceAlign);
//		menuSequence.add(menuSequenceAlignSet);
//		menuSequence.add(new JSeparator());
		menuSequence.add(menuSequenceAlignmentDownload);
		

		menuEdit.add(menuEditMode);
		menuEdit.add(new JSeparator());
		menuEdit.add(menuEditInsertColumn);
		menuEdit.add(menuEditDeleteColumn);
		menuEdit.add(menuEditRemoveGappedColumns);
		menuEdit.add(menuEditRemoveAllGaps);
//		menuEdit.add(new JSeparator());
//		menuEdit.add(menuEditReset);
		
		
		JMenu menuOptions = new JMenu("Display");
		menuOptions.setMnemonic('d');
		
		menuOptionsSs = new JCheckBoxMenuItem("Show secondary structure");
		menuOptionsSs.setSelected(StylesPreferences.showSs);
		menuOptionsSs.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				showSs = menuOptionsSs.isSelected();
//				adjustVScrollBar(true);
				sequencePanel.updateView(true);
			}
		});
		
		JMenu menuOptionsColor = new JMenu("Set residue color");
		JMenuItem menuOptionsColorSelect = new JMenuItem("Select solid color...");
		menuOptionsColorSelect.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setResidueColor(StyleManager.SOLID_COLOR);
					}
				};
				runner.start();
				
			}
		});
		
		JMenuItem menuOptionsColorType = new JMenuItem("By residue type");
		menuOptionsColorType.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setResidueColor(StyleManager.TYPE_COLOR);
					}
				};
				runner.start();
			}
		});	
			
		JMenuItem menuOptionsColorHydro = new JMenuItem("By hydrophobicity");
		menuOptionsColorHydro.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setResidueColor(StyleManager.HYDROPHOBIC_COLOR);
					}
				};
				runner.start();
			}
		});	
			
		JMenu menuOptionsColorConserv = new JMenu("By position conservation");
		
		
		JMenuItem menuOptionsColorConservStrict = new JMenuItem("Unique residues");
		menuOptionsColorConservStrict.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setResidueColor(StyleManager.CONSERVATION_COLOR_STRICT);
					}
				};
				runner.start();
			}	
		});
		
		JMenuItem menuOptionsColorConservHomology = new JMenuItem("Homologous residues");
		menuOptionsColorConservHomology.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setResidueColor(StyleManager.CONSERVATION_COLOR_HOMOLOGY);
					}
				};
				runner.start();
			}	
		});
		
		menuOptionsColorConserv.add(menuOptionsColorConservStrict);
		menuOptionsColorConserv.add(menuOptionsColorConservHomology);
		
		
		
		JMenuItem menuOptionsColorChain = new JMenuItem("By chain");
		menuOptionsColorChain.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setResidueColor(StyleManager.CHAIN_COLOR);
					}
				};
				runner.start();
			}	
		});
		
		JMenuItem menuOptionsColorDefault = new JMenuItem("Set to default");
		menuOptionsColorDefault.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setDefaultResidueColors();
					}
				};
				runner.start();
			}
		});
		
		//background color options
/*		JMenu menuOptionsBkgColor = new JMenu("Set background color");
		JMenuItem menuOptionsBkgColorSelect = new JMenuItem("Select solid color...");
		menuOptionsBkgColorSelect.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setBackgroundSolidColor();
					}
				};
				runner.start();
			}
		});
		
		JMenuItem menuOptionsBkgColorType = new JMenuItem("By residue type");
		menuOptionsBkgColorType.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setBackgroundTypeColor();
					}
				};
				runner.start();
			}
		});	
			
		JMenuItem menuOptionsBkgColorHydro = new JMenuItem("By hydrophobicity");
		menuOptionsBkgColorHydro.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setBackgroundHydroColor();
					}
				};
				runner.start();
			}
		});	
			
		JMenuItem menuOptionsBkgColorConserv = new JMenuItem("By position conservation");
		menuOptionsBkgColorConserv.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						setBackgroundConservColor();
					}
				};
				runner.start();
			}	
		});
		
		JMenuItem menuOptionsBkgColorDefault = new JMenuItem("Set to default");
		menuOptionsBkgColorDefault.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						resetDefaultBackgroundColors();
					}
				};
				runner.start();
			}	
		});
*/		
		//protein
		JMenu menuOptionsSelect = new JMenu("Select by residue");
		JMenuItem menuOptionsSelectA = new JMenuItem("Ala");
		menuOptionsSelectA.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("A");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectC = new JMenuItem("Cys");
		menuOptionsSelectC.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("C");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectD = new JMenuItem("Asp");
		menuOptionsSelectD.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("D");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectE = new JMenuItem("Asn");
		menuOptionsSelectE.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("N");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectF = new JMenuItem("Phe");
		menuOptionsSelectF.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("F");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectG = new JMenuItem("Gly");
		menuOptionsSelectG.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("G");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectH = new JMenuItem("His");
		menuOptionsSelectH.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("H");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectI = new JMenuItem("Ile");
		menuOptionsSelectI.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("I");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectK = new JMenuItem("Lys");
		menuOptionsSelectK.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("K");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectL = new JMenuItem("Leu");
		menuOptionsSelectL.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("L");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectM = new JMenuItem("Met");
		menuOptionsSelectM.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("M");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectN = new JMenuItem("Gln");
		menuOptionsSelectN.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("Q");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectP = new JMenuItem("Pro");
		menuOptionsSelectP.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("P");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectQ = new JMenuItem("Glu");
		menuOptionsSelectQ.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("E");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectR = new JMenuItem("Arg");
		menuOptionsSelectR.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("R");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectS = new JMenuItem("Ser");
		menuOptionsSelectS.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("S");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectT = new JMenuItem("Thr");
		menuOptionsSelectT.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("T");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectV = new JMenuItem("Val");
		menuOptionsSelectV.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("V");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectW = new JMenuItem("Trp");
		menuOptionsSelectW.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("W");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectY = new JMenuItem("Tyr");
		menuOptionsSelectY.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("Y");
				sequencePanel.updateView(false);
			}
		});
		
		menuOptionsSelect.add(menuOptionsSelectA);
		menuOptionsSelect.add(menuOptionsSelectC);
		menuOptionsSelect.add(menuOptionsSelectD);
		menuOptionsSelect.add(menuOptionsSelectE);
		menuOptionsSelect.add(menuOptionsSelectF);
		menuOptionsSelect.add(menuOptionsSelectG);
		menuOptionsSelect.add(menuOptionsSelectH);
		menuOptionsSelect.add(menuOptionsSelectI);
		menuOptionsSelect.add(menuOptionsSelectK);
		menuOptionsSelect.add(menuOptionsSelectL);
		menuOptionsSelect.add(menuOptionsSelectM);
		menuOptionsSelect.add(menuOptionsSelectN);
		menuOptionsSelect.add(menuOptionsSelectP);
		menuOptionsSelect.add(menuOptionsSelectQ);
		menuOptionsSelect.add(menuOptionsSelectR);
		menuOptionsSelect.add(menuOptionsSelectS);
		menuOptionsSelect.add(menuOptionsSelectT);
		menuOptionsSelect.add(menuOptionsSelectV);
		menuOptionsSelect.add(menuOptionsSelectW);
		menuOptionsSelect.add(menuOptionsSelectY);

		
		JMenu menuOptionsSelectDNA = new JMenu("Select by DNA base");
		JMenuItem menuOptionsSelectDNAA = new JMenuItem("Adenine");
		menuOptionsSelectDNAA.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("A");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectDNAT = new JMenuItem("Thymine");
		menuOptionsSelectDNAT.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("T");
				sequencePanel.updateView(false);
		}
		});
		JMenuItem menuOptionsSelectDNAC = new JMenuItem("Cytosine");
		menuOptionsSelectDNAC.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("C");
				sequencePanel.updateView(false);
			}
		});
		JMenuItem menuOptionsSelectDNAG = new JMenuItem("Guanine");
		menuOptionsSelectDNAG.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				selectResidue("G");
				sequencePanel.updateView(false);
			}
		});
		
		menuOptionsSelectDNA.add(menuOptionsSelectDNAA);
		menuOptionsSelectDNA.add(menuOptionsSelectDNAT);
		menuOptionsSelectDNA.add(menuOptionsSelectDNAC);
		menuOptionsSelectDNA.add(menuOptionsSelectDNAG);
		
		

		menuOptionsColor.add(menuOptionsColorSelect);
		menuOptionsColor.add(new JSeparator());
		menuOptionsColor.add(menuOptionsColorType);
		menuOptionsColor.add(menuOptionsColorHydro);
		menuOptionsColor.add(menuOptionsColorConserv);
		menuOptionsColor.add(menuOptionsColorChain);
		menuOptionsColor.add(new JSeparator());
		menuOptionsColor.add(menuOptionsColorDefault);

		menuOptions.add(menuOptionsSs);
		menuOptions.add(new JSeparator());
		menuOptions.add(menuOptionsColor);
		menuOptions.add(new JSeparator());
		menuOptions.add(menuOptionsSelect);
		menuOptions.add(menuOptionsSelectDNA);

		
		menuBar.add(menuFile);
		menuBar.add(menuEdit);
		menuBar.add(menuSequence);
		menuBar.add(menuOptions);
		
		
		return menuBar;
		
		
	}
	
	private void openFile(File filename){
		
		file = filename;
		
		Thread runner = new Thread(){
			public void run(){
				Entry entry = null;
				try{
					entry = EntryFactory.load( file, -1 );

				}
				catch (NumberFormatException e){
					parent.displayErrorMessage("Incorrect file format: " + file.toString());
					return;
				}
				
				if (entry == null){
					parent.displayErrorMessage("Unable to load specified file: " + file.toString());
					return;
				}
		
				parent.addEntry(entry, true, false);
			}
		};
		runner.start();
	}
	
	private void clearEntries(){

		parent.clearEntries(true);
		if (sequences.size() == 0){
			sequencePanel.updateView(true);
			scalePanel.resetPanel();
			scrollBarH.setMinimum( 0 );
			scrollBarH.setMaximum( 0 );
			revalidate();
			repaint();
			
			menuEditReset.setEnabled(false);
		}
	}
	
	public DisplayParameters getDisplayParameters(){
		return parameters;
	}
	
	public void setDisplayParameters(DisplayParameters pars){
		parameters = pars;
	}
	
	private void setStartResidue(int index){
		scalePanel.setStartResidue(index);
		sequencePanel.setStartResidue(index);
		sequencePanel.updateView(false);
	}
	
	private void setStartSequence(int index){
		sequencePanel.setStartSequence(index);
		sequencePanel.updateView(false);
	}
	
	public Vector getSelectedColumns(){
		return selectedColumns;
	}
	
	public Vector getSelectedSequences(){
		return sequencePanel.getSelectedSequences();
	}
	
	/**
	 * This method rename the specified sequence with the provided name. fire boolean specifies whether this change should be
	 * propagated to the structure viewer (true) or the structure viewer initiated it in the first place (false)
	 * @param sequence
	 * @param name
	 * @param fire
	 */
	public void renameSequence(Sequence sequence, String name){
		if (name != null && name.length() > 0){
			if (styleMap.containsKey(sequence)){
				Entry entry = (Entry)entries.get(sequence);
				structureDocument.parent.renameEntry(entry, name);
			}
			else{
				//it's a sequence object, use SequenceStyles
				sequence.getSequenceStyles().renameSequence(name);
			}
		}
	}
	
	public StructureDocument getStructureDocument(){
		return structureDocument;
	}
	
	public void movePanel(Sequence sequence, int position){
		
		if (position == sequenceObjects.indexOf(sequence)) return;
		
		//move the panel in the data model to the requested position
		int index = sequenceObjects.indexOf(sequence);
		
		sequenceObjects.remove(sequence);
		Cell[] cells = (Cell[])sequences.get(index);
		sequences.remove(cells);
		if (position == -1){
			sequenceObjects.add(sequence);
			sequences.add(cells);
		}
		else{
			sequenceObjects.add(position, sequence);
			sequences.add(position, cells);
		}
		sequencePanel.updateView(true);
	}
	
	public void setPanelVisibility(Sequence sequence, boolean visible, boolean update){
		if (sequence == null){
			//apply to all panels
			if (visible){
				hiddenSequences.clear();
			}
			else{
				for (int i = 0; i < sequences.size(); i++){
					hiddenSequences.add(sequenceObjects.get(i));
				}
			}
			if (update) sequencePanel.updateView(true);
			return;
		}
		
		if (visible){
			if (hiddenSequences.contains(sequence)){
				hiddenSequences.remove(sequence);
			}
		}
		else{
			if (!hiddenSequences.contains(sequence)){
				hiddenSequences.add(sequence);
			}
		}
		if (update) sequencePanel.updateView(true);
	}
	
	public void removeSequence(Sequence sequence){
		
		if (sequence == null) return;
		if (!sequenceObjects.contains(sequence)) return;
		
		sequenceRemoved(sequence);
	}

	public final Vector getHiddenSequences() {
		return hiddenSequences;
	}

	public final void setHiddenSequences(Vector hiddenSequences) {
		this.hiddenSequences = hiddenSequences;
	}

	public final SequenceScalePanel getScalePanel() {
		return scalePanel;
	}

	public final void setScalePanel(SequenceScalePanel scalePanel) {
		this.scalePanel = scalePanel;
	}
	
	public void processMotifSearch(String motif){
		
		sequencePanel.processMotif(motif);
		
	}
	
	public void computeDistribution(){
		
		//get the currently selected columns
		if (selectedColumns.size() != 1 ){
			structureDocument.parent.displayErrorMessage("Distribution can be calculated only for one selected column");
			return;
		}
		
		int index = ((Integer)selectedColumns.get(0)).intValue();
		
		final HashMap distribution = new HashMap();//aa -> organism -> count
		
		//initialize distribution
		for (int i = 0; i < ProteinProperties.getAminoacids().size(); i++){
			String aa = (String)ProteinProperties.getAminoacids().get(i);
			distribution.put(aa, new HashMap());
		}
		
		distribution.put("-", new HashMap());
		
		for (int i = 0; i < sequences.size(); i++){
			Cell[] cells = (Cell[])sequences.get(i);
			if (index >= cells.length) continue;
			String aa = cells[index].symbol;
			
			if ( aa.equals("X") || aa.equals("#") || aa.equals("*")) continue;
			
			//get the entry and see if there is organism info
			Entry entry = (Entry)entries.get(sequenceObjects.get(i));
			String organism = entry.getOrganism();
			if (organism == null){
				organism = "Other";
			}
			
			HashMap inner = (HashMap)distribution.get(aa);
//			System.out.println("distribution: organism = " + organism);
			if (inner == null){
				inner = new HashMap();
				inner.put(organism, new Integer(1));
				distribution.put(aa, inner);
			}
			else{
				Integer value = (Integer)inner.get(organism);
				if (value == null){
					inner.put(organism, new Integer(1));
				}
				else{
					int count = value.intValue();
					count++;
					inner.put(organism, new Integer(count));
				}
			}
			
		}
		
		//at this point, the distribution hash is ready to be passed to the viewer
		Thread runner = new Thread(){
			public void run(){
				DistributionViewer dv = new DistributionViewer(structureDocument.parent.getApplicationFrame(), structureDocument, distribution);
			}
		};
		runner.start();

		
	}
	
	/**
	 * This method speeds up selection, since it narrows down the search for the Cell to
	 * the supplied Sequence object
	 * @param cell
	 * @param sequence
	 * @param selected
	 */
	public void selectCell(Cell cell, Sequence sequence, boolean selected, boolean ctrl){
		
		cell.selected = selected;
		if (selected){
			if (!selectedCells.contains(cell)) selectedCells.add(cell);
		}
		else{
			if (selectedCells.contains(cell)) selectedCells.remove(cell);
		}
		
		if (sequence.getSequenceStyles() != null) sequence.getSequenceStyles().setSelected(cell, selected, true, true);

		StructureStyles styles = (StructureStyles)styleMap.get(sequence);
		SequenceStyles sequenceStyles = sequence.getSequenceStyles();
		
		if (!ctrl){
			//clear everything else
			if (styles != null){
				styles.selectNone(false, true);
			}
			else{
				sequenceStyles.selectNone(true, true);
			}
		}
		
		if (styles == null){
			sequenceStyles.setSelected(cell, selected, true, true);
//			StructureBrowser sb = structureDocument.parent.getStructureBrowser();
//			if (sb != null) sb.expandToNode(cell);
		}
		else{
			if (cell.residue != null){
				//update the chain where the change was made
				Chain chain = styles.getStructureMap().getChain(cell.residue.getAtom(0));
				styles.setSelected(cell.residue, selected, true, false);
				styles.updateChain(chain, true);

//				StructureBrowser sb = structureDocument.parent.getStructureBrowser();
//				if (sb != null) sb.expandToNode(cell.residue);

			}
		}
		

		
		structureDocument.parent.updateView();
	}
	
	public void selectCells(Sequence sequence, Vector cells){
		
		StructureStyles styles = null;
		SequenceStyles sequenceStyles = sequence.getSequenceStyles();
		StructureMap map = null;
		Vector chains = new Vector();
		if (StylesPreferences.commonSelection){
			styles = (StructureStyles)styleMap.get(sequence);
			if (styles != null){
				map = styles.getStructureMap();
				
				//clear everything else
				styles.selectNone(false, true);
			}
		}
		
		for (int i = 0; i < cells.size(); i++){
			Cell cell = (Cell)cells.get(i);
			if (!selectedCells.contains(cell)){
				selectedCells.add(cell);
				cell.selected = true;
			}
			try{
				if (StylesPreferences.commonSelection && styles != null){
					if (!chains.contains(map.getChain(cell.residue.getAtom(0)))) chains.add(map.getChain(cell.residue.getAtom(0)));
					styles.setSelected(cell.residue, true, true, false);
				}
			}
			catch (Exception e){
				continue;
			}
			if (sequenceStyles != null) sequenceStyles.setSelected(cell, true, true, true);
		}
		
		//update the chain involved
		if (StylesPreferences.commonSelection && styles != null){
			//update only those chains that contain changed residues
			
			for (int i = 0; i < chains.size(); i++){
				Chain chain = (Chain)chains.get(i);
				styles.updateChain(chain, true);
			
				//add the chain to selectedChains in StructureViewer
//				styles.registerChainSelection(chain, true);
			}
			
//			StructureBrowser sb = structureDocument.parent.getStructureBrowser();
//			if (sb != null) sb.expandToNode(((Cell)cells.get(0)).residue);
			
		}
		else{
//			StructureBrowser sb = structureDocument.parent.getStructureBrowser();
//			if (sb != null) sb.expandToNode((Cell)cells.get(0));

		}
		
		structureDocument.parent.updateView();
	}
	
	public void selectColumn(int index, boolean selected, boolean update){
		
		boolean superimpose = parent.getSuperimpose();
		DisplayDialog d = null;
		
		if (superimpose){
			d = parent.getDisplayDialog();
		}
		
		Vector components = new Vector();
		
		HashMap reference = null;
		StructureStyles styles = null;
		SequenceStyles sequenceStyles = null;
		for (int i = 0; i < sequenceObjects.size(); i++){
			Sequence s = (Sequence)sequenceObjects.get(i);
			sequenceStyles = s.getSequenceStyles();
			Cell[] ss = (Cell[])sequences.get(i);
			if (index >= ss.length) continue;
			
			Cell cell = ((Cell[])sequences.get(i))[index];
			cell.selected = selected;
			sequenceStyles.setSelected(cell, selected, true, true);
			selectedCells.add(cell);
			if (StylesPreferences.commonSelection){
				styles = (StructureStyles)styleMap.get(s);
				if (styles == null) continue;
				
				reference = (HashMap)cellResidueMapping.get(s);
				Residue r = (Residue)reference.get(((Cell[])sequences.get(i))[index]);
				styles.setSelected(r, selected, true, true);
				components.add(r);
			}
			else{
				components.add(cell);
			}
			
			if (superimpose){
				Residue r = (Residue)reference.get(cell);
				if (r == null) continue;
				if (sequences.size() == 2){
					//send the atom
					Atom a = r.getAlphaAtom();
					if (a == null) a = r.getAtom(0);
//					System.out.println("a = " + a);
					if (d != null) d.processPick(DataService.componentToLabel(a), a);
				}
				else if (sequences.size() > 2){
					if (sequencePanel.getSelectedSequences().contains(s)){
						//send the atom
						Atom a = r.getAlphaAtom();
						if (a == null) a = r.getAtom(0);
						if (d != null) d.processPick(DataService.componentToLabel(a), a);
					}
				}
			}
			
/*			if (components.size() > 0){
				StructureBrowser sb = structureDocument.parent.getStructureBrowser();
				if (sb != null) sb.expandToNodes(components);
			}
*/		}
		if (update) parent.updateAppearance();
	}
	
	public void updateAppearance(){
		parent.updateAppearance();
	}
	
	
	
	public void selectResidue(String symbol){
		
		Sequence sequence = null;
		HashMap reference = null;
		StructureStyles styles = null;
		SequenceStyles sequenceStyles = null;
		
		for (int i = 0; i < sequences.size(); i++){
			Cell[] cells = (Cell[])sequences.get(i);
			if (StylesPreferences.commonSelection){
				sequence = (Sequence)sequenceObjects.get(i);
				reference = (HashMap)cellResidueMapping.get(sequence);
				styles = (StructureStyles)styleMap.get(sequence);
				sequenceStyles = sequence.getSequenceStyles();
				
				if (styles != null) styles.selectNone(false, true);
				if (sequenceStyles != null) sequenceStyles.selectNone(true, true);
				
			}
			for (int j = 0; j < cells.length; j++){
				if (cells[j].symbol.equals(symbol)){
					selectedCells.add(cells[j]);
					if (StylesPreferences.commonSelection){
						if (styles != null){
							styles.setSelected((Residue)reference.get(cells[j]), true, true, false);
						}
						else if (sequenceStyles != null){
							sequenceStyles.setSelected(cells[j], true, true, true);
						}
					}
					cells[j].selected = true;
				}
			}
			
			if (StylesPreferences.commonSelection && styles != null){
				StructureMap map = styles.getStructureMap();
				for (int j = 0; j < map.getChainCount(); j++){
					Chain chain = map.getChain(j);
//					styles.registerChainSelection(chain, true);
					styles.updateChain(chain, true);
				}
				
			}
		}
//		sequencePanel.updateView(false);
		
	}
	
	public void setResidueColor(int style){
		if (sequences.size() == 0) return;
		
		Sequence sequence = null;
		HashMap reference = null;
		StructureStyles styles = null;
		Vector conservation = null;

		Color color = null;
		if (style == StyleManager.SOLID_COLOR){
			Cell t = ((Cell[])sequences.get(0))[0];
			Color current = t.color;
			Color newColor = JColorChooser.showDialog(null, "Choose residue color", current);
			
		
			if (newColor == null){
				parent.displayMessage("Color was not read correctly. Please try again");
				return;
			}
			
			color = styleManager.getColor(newColor);
		}
		
		if (style == StyleManager.CONSERVATION_COLOR_STRICT){
			conservation = this.getConservationStrict();
//			System.out.println("conservation = " + conservation);
		}
		else if (style == StyleManager.CONSERVATION_COLOR_HOMOLOGY){
			conservation = this.getConservationHomology();
		}
		else if (style == StyleManager.CHAIN_COLOR){
			//selection is ignored, and the entire display is colored, where a corresponding structure
			//with chain records is available
			
			if (StylesPreferences.undoEnabled) structureDocument.parent.getStructureViewer().saveStateForUndo();
			
			for (int i = 0; i < sequenceObjects.size(); i++){
				sequence = (Sequence)sequenceObjects.get(i);
				if (!styleMap.containsKey(sequence)) continue;
				
				styles = (StructureStyles)styleMap.get(sequence);
				StructureMap map =  styles.getStructureMap();
				reference = (HashMap)cellResidueMapping.get(sequence);
				Cell[] cells = (Cell[])sequences.get(i);
				int counter = 0;
				ResidueColor residueColor = ResidueColorFactory.getResidueColor(StylesPreferences.chainColors[counter]);
				RibbonColor ribbonColor = RibbonColorFactory.getRibbonColor(StylesPreferences.chainColors[counter]);
				String lastChain = null;
				for (int j = 0; j < cells.length; j++){
					if (cells[j] == null || cells[j].residue == null) continue;
					String chain = cells[j].residue.getChainId();
					if (chain.equals(lastChain)){
						//simply apply the current color
						styles.setResidueColor(cells[j].residue, residueColor, true, true);
						if (StylesPreferences.commonRibbonColoring){
							styles.setRibbonColor(cells[j].residue, ribbonColor, false, true);
						}
					}
					else{
						if (lastChain == null){
							styles.setResidueColor(cells[j].residue, residueColor, true, true);
							if (StylesPreferences.commonRibbonColoring){
								styles.setRibbonColor(cells[j].residue, ribbonColor, false, true);
							}
							lastChain = cells[j].residue.getChainId();
						}
						else{
							//chain has switched
							lastChain = cells[j].residue.getChainId();
							counter++;
							if (counter >= StylesPreferences.chainColors.length) counter = 0;
							residueColor = ResidueColorFactory.getResidueColor(StylesPreferences.chainColors[counter]);
							ribbonColor = RibbonColorFactory.getRibbonColor(StylesPreferences.chainColors[counter]);
							styles.setResidueColor(cells[j].residue, residueColor, true, true);
							if (StylesPreferences.commonRibbonColoring){
								styles.setRibbonColor(cells[j].residue, ribbonColor, false, true);
							}

						}
					}
				}
				
				//update the chains
				for (int j = 0; j < map.getChainCount(); j++){
					Chain chain = map.getChain(j);
					styles.updateChain(chain, true);
				}
			}
			
			parent.updateAppearance();
			return;
		}
		
		if (StylesPreferences.undoEnabled) structureDocument.parent.getStructureViewer().saveStateForUndo();
		
		if (selectedCells.size() > 0){
			boolean totalUpdate = false;//whether there is at least one structure residue affected
			//color selected cells and deselect them when done
			for (int j = selectedCells.size()-1; j >= 0; j--){
				Cell cell = (Cell)selectedCells.get(j);
				if (cell == null) continue;
				
				if (StylesPreferences.commonColoring && cell.residue != null){
					totalUpdate = true;
					ResidueColor residueColor = null;
					if (style == StyleManager.TYPE_COLOR){
						residueColor = ResidueColorByType.create();
					}
					else if (style == StyleManager.HYDROPHOBIC_COLOR){
						residueColor = ResidueColorByHydrophobicity.create();
					}
					else if (style == StyleManager.SOLID_COLOR){
						residueColor = ResidueColorFactory.getResidueColor(color);
					}
					else if (style == StyleManager.CONSERVATION_COLOR_STRICT || style == StyleManager.CONSERVATION_COLOR_HOMOLOGY){
						residueColor = ResidueColorFactory.getResidueColor(StyleManager.getConservationColor(((Double)conservation.get(cell.index)).doubleValue()));
					}
					cell.residue.structure.getStructureMap().getStructureStyles().setSelected(cell.residue, false, false, true);
					cell.residue.structure.getStructureMap().getStructureStyles().setResidueColor(cell.residue, residueColor, true, true);
					if (StylesPreferences.commonRibbonColoring){
						cell.residue.structure.getStructureMap().getStructureStyles().setRibbonColor(cell.residue, residueColor, false, true);
					}
				}
				else{
					if (style == StyleManager.TYPE_COLOR){
						color = StyleManager.getTypeColor(cell.symbol);
					}
					else if (style == StyleManager.CONSERVATION_COLOR_STRICT || style == StyleManager.CONSERVATION_COLOR_HOMOLOGY){
						color = StyleManager.getConservationColor(((Double)conservation.get(cell.index)).doubleValue());
					}
					if (style == StyleManager.HYDROPHOBIC_COLOR){
						color = StyleManager.getHydrophobicityColor(cell.symbol);
					}
					cell.color = color;
					cell.selected = false;
					
					cell.sequence.getSequenceStyles().setCellColor(cell, color, true, true);
					
				}
			}
			
			selectedCells.clear();
			selectedColumns.clear();
			
			if (totalUpdate){
				parent.updateAppearance();
			}
			else{
				updateView();
			}

			
		}
		else{
			//for color by conservation, walk by column instead of by sequence, to calculate the
			//extent of conservation and map the color immediately
			//color everything
//			System.out.println("coloring by total conservation");
			boolean totalUpdate = false;
			for (int i = 0; i < sequences.size(); i++){
				Cell[] cells = (Cell[])sequences.get(i);
				if (StylesPreferences.commonColoring && styleMap.containsKey(sequenceObjects.get(i))){
					styles = (StructureStyles)styleMap.get(sequenceObjects.get(i));
					totalUpdate = true;
					ResidueColor residueColor = null;
					AtomColor atomColor = null;
					if (style == StyleManager.CONSERVATION_COLOR_STRICT || style == StyleManager.CONSERVATION_COLOR_HOMOLOGY){
						//for conservation, walk residue by residue and set the color
						for (int j = 0; j < cells.length; j++){
							if (cells[j] == null){
//								System.out.println("cells[" + j + "] = null");
								continue;
							}
							residueColor = ResidueColorFactory.getResidueColor(StyleManager.getConservationColor(((Double)conservation.get(cells[j].index)).doubleValue()));
							styles.setResidueColor(cells[j].residue, residueColor, true, true);
							styles.setRibbonColor(cells[j].residue, residueColor, true, true);
//							System.out.println("set colors");
						}
					}
					else{
						if (style == StyleManager.SOLID_COLOR){
							residueColor = ResidueColorFactory.getResidueColor(color);
							atomColor = AtomColorFactory.getAtomColor(color);
						}
						else if (style == StyleManager.HYDROPHOBIC_COLOR){
							residueColor = ResidueColorByHydrophobicity.create();
						}
						else if (style == StyleManager.TYPE_COLOR){
							residueColor = ResidueColorByType.create();
							//atomColor remains null
						}
						styles.setStructureColor(residueColor, atomColor);
						if (StylesPreferences.commonRibbonColoring){
							//color the ribbons
							for (int j = 0; j < styles.getStructureMap().getResidueCount(); j++){
								Residue r = styles.getStructureMap().getResidue(j);
								styles.setRibbonColor(r, residueColor, false, true);
							}
						}
					}
					for (int j = 0; j < styles.getStructureMap().getChainCount(); j++){
						styles.updateChain(styles.getStructureMap().getChain(j), true);
					}
					
					
				}
				else{
					for (int j = 0; j < cells.length; j++){
						if (cells[j] == null) continue;
						if (style == StyleManager.TYPE_COLOR){
							color = StyleManager.getTypeColor(cells[j].symbol);
						}
						else if (style == StyleManager.HYDROPHOBIC_COLOR){
							color = StyleManager.getHydrophobicityColor(cells[j].symbol);
						}
						else if (style == StyleManager.CONSERVATION_COLOR_STRICT || style == StyleManager.CONSERVATION_COLOR_HOMOLOGY){
							color = StyleManager.getConservationColor(((Double)conservation.get(cells[j].index)).doubleValue());
						}
						cells[j].color = color;
					}
					
					Sequence s = (Sequence)sequenceObjects.get(i);
					s.getSequenceStyles().setSequenceColor(color, true, true);
				}

			}
			
			if (totalUpdate){
				parent.updateAppearance();
			}
			else{
				updateView();
			}
		}
		
		
	}
	
	/**
	 * In this viewer, coloring and selection are simplified and are handled directly through the Cell objects for performance reasons.
	 * This event is handled fully in other viewers, such as the tree viewer
	 * @param e
	 */
	public void processSequenceStylesEvent(SequenceStylesEvent e){
		if (!e.batch) updateView();
	}
		
	public void setDefaultResidueColors(){
		
		if (StylesPreferences.undoEnabled) structureDocument.parent.getStructureViewer().saveStateForUndo();

		Color color = styleManager.getDefaultColor();
		//color everything
		for (int i = sequences.size()-1; i >= 0; i--){
			Cell[] cells = (Cell[])sequences.get(i);
			if (StylesPreferences.commonColoring && styleMap.containsKey(sequenceObjects.get(i))){
				StructureStyles styles = (StructureStyles)styleMap.get(sequenceObjects.get(i));
				styles.selectNone(false, false);
				styles.setStructureColor(ResidueColorByElement.create(), AtomColorByElement.create());
				
				if (StylesPreferences.commonRibbonColoring){
					RibbonColor rc = RibbonColorDefault.create();
					//color the ribbons
					for (int j = 0; j < styles.getStructureMap().getResidueCount(); j++){
						Residue r = styles.getStructureMap().getResidue(j);
						styles.setRibbonColor(r, rc, false, true);
					}
					
					for (int j = 0; j < styles.getStructureMap().getChainCount(); j++){
						styles.updateChain(styles.getStructureMap().getChain(j), true);
					}

				}

			}
			else{
				for (int j = 0; j < cells.length; j++){
					if (cells[j] == null) continue;
					cells[j].color = color;
				}
			}
		}

		parent.updateAppearance();
		
	}

	public final Vector getSelectedCells() {
		return selectedCells;
	}

	public final void setSelectedCells(Vector selectedCells) {
		this.selectedCells = selectedCells;
	}
	
	public void clearCellSelection(boolean updateView){
		
		for (int j = 0; j < selectedCells.size(); j++){
			((Cell)selectedCells.get(j)).selected = false;
		}
	
		//fire an event if needed
		for (int i = 0; i < sequenceObjects.size(); i++){
			if (StylesPreferences.commonSelection && styleMap.containsKey(sequenceObjects.get(i))){
				StructureStyles styles = (StructureStyles)styleMap.get(sequenceObjects.get(i));
				styles.selectNone(false, true);
			}
			else{
				((Sequence)sequenceObjects.get(i)).getSequenceStyles().selectNone(true, true);
			}
		}

		selectedCells.clear();
		selectedColumns.clear();
		sequencePanel.matchedSequences.clear();
		parent.updateView();
		
	}
	
	/**
	 * TODO conservation by homology
	 * @return
	 */
	public Vector getConservationStrict(){
		
		//this method calculates simple conservation (fraction of the most abundant residue) at each
		//position and returns the fractional values as a Vector of the same length as the alignment.
		//Positions with more than 50% dashes have conservation = 0
		
		//first, the occurrence of each residue is calculated and abundance hashes are filled
		//The main data structure is hash of hashes: index -> residue -> number
		//populate it with zeros
		
		//in the case of homology-based computation, each residue is checked against several types
		//with a number of members, therefore not the residue, but type occurrence is calculated
		
		HashMap abundance = new HashMap();
		
		Vector result = new Vector();
		Vector aas = ProteinProperties.getAminoacids();
		for (int i = 0; i < residueMax; i++){
			HashMap temp = new HashMap();
			
			for (int j = 0; j < aas.size(); j++){
				temp.put(aas.get(j), new Integer(0));
			}
			temp.put("-", new Integer(0));
			
			abundance.put(new Integer(i), temp);
		}
		
		//scan the alignment and write abundances into the hash
		for (int i = 0; i < residueMax; i++){
			Vector slice = getSlice(i);
			
//			System.out.println(slice);
			
			int skipped = 0;//residues not used for calculation
			
			for (int j = 0; j < slice.size(); j++){
				String resid = (String)slice.get(j);
				if (resid.equals("X") || resid.equals("#") || resid.equals("*")){
					skipped++;
					continue;
				}
				HashMap tmp = (HashMap)abundance.get(new Integer(i));
//				System.out.println("resid = " + resid);
				Integer currentNumber = (Integer)tmp.get(resid);
				
				if (currentNumber == null) continue;
//				System.out.println("currentNumber = " + currentNumber);

				int n = currentNumber.intValue();
				n++;
				tmp.put(resid, new Integer(n));
				abundance.put(new Integer(i),tmp);
			}
			
			//now look at the results
			HashMap tmp = (HashMap)abundance.get(new Integer(i));
			int sliceSize = slice.size() - skipped;

			Set keys = tmp.keySet();
			Iterator iterator = keys.iterator();
			Integer maxNumber = new Integer(0);//initial maximum abundance
			String maxResidue = "";
			while (iterator.hasNext()){
				String key = (String)iterator.next();//residue type or residue
				Integer num = (Integer)tmp.get(key);
				int numValue = num.intValue();
				if (numValue > maxNumber.intValue()){
					maxNumber = new Integer(numValue);
					maxResidue = key;
				}
			}
			
			if (maxResidue.equals("-")){
				double dashRatio = ((double)maxNumber.intValue())/((double)sliceSize);
				if (dashRatio >= 0.5){
					//indicate zero
					result.add(new Double(0.0));
					continue;
				}
				else{
					//the dashes had been calculated as main residue
					//the search must be re-run ignoring dashes
					
					Iterator itt = keys.iterator();
					maxNumber = new Integer(0);//reset maxNumber
					while (itt.hasNext()){
						String key = (String)itt.next();
						if (key.equals("-")){
							continue;
						}
						else{
							Integer n = (Integer)tmp.get(key);
							int nValue = n.intValue();
							if (nValue > maxNumber.intValue()){
								maxNumber = new Integer(nValue);
							}
						}
					}
					
					
					
					//done rescanning
				}
			}
	
			//calculate the conservation coefficient
			double coeff = (double)(maxNumber.intValue())/sliceSize;
//			System.out.println(coeff);
			result.add(new Double(coeff));//add the coefficient
		}
		
		return result;
	}
	
	public Vector getConservationHomology(){
		
		//this method calculates simple conservation (fraction of the most abundant residue) at each
		//position and returns the fractional values as a Vector of the same length as the alignment.
		//Positions with more than 50% dashes have conservation = 0
		
		//first, the occurrence of each residue is calculated and abundance hashes are filled
		//The main data structure is hash of hashes: index -> residue -> number
		//populate it with zeros
		
		//in the case of homology-based computation, each residue is checked against several types
		//with a number of members, therefore not the residue, but type occurrence is calculated
		
		//residue types are: LIVM, STC, FYHW, AG, DENQ, KR, P
		
		HashMap abundance = new HashMap();
		
		HashMap lookup = new HashMap();//residue type lookup hash
		lookup.put("A", "AG");
		lookup.put("C", "STC");
		lookup.put("D", "DENQ");
		lookup.put("E", "DENQ");
		lookup.put("F", "FYHW");
		lookup.put("G", "AG");
		lookup.put("H", "FYHW");
		lookup.put("I", "LIVM");
		lookup.put("K", "KR");
		lookup.put("L", "LIVM");
		lookup.put("M", "LIVM");
		lookup.put("N", "DENQ");
		lookup.put("P", "P");
		lookup.put("Q", "DENQ");
		lookup.put("R", "KR");
		lookup.put("S", "STC");
		lookup.put("T", "STC");
		lookup.put("V", "LIVM");
		lookup.put("W", "FYHW");
		lookup.put("Y", "FYHW");
		
		for (int i = 0; i < residueMax; i++){
			HashMap temp = new HashMap();
			temp.put("LIVM", 0);
			temp.put("STC", 0);
			temp.put("FYHW", 0);
			temp.put("AG", 0);
			temp.put("DENQ", 0);
			temp.put("KR", 0);
			temp.put("P", 0);
			temp.put("-", 0);
			
			abundance.put(new Integer(i), temp);
		}
		
		Vector result = new Vector();
		
		//scan the alignment and write abundances into the hash
		for (int i = 0; i < residueMax; i++){
			Vector slice = getSlice(i);
			
//			System.out.println(slice);
			
			int skipped = 0;//residues not used for calculation
			
			for (int j = 0; j < slice.size(); j++){
				String resid = (String)slice.get(j);
				if (resid.equals("X") || resid.equals("#") || resid.equals("*") || resid.equals("U")){
					skipped++;
					continue;
				}
				
				String type = (String)lookup.get(resid);
				
				HashMap tmp = (HashMap)abundance.get(i);
//				System.out.println("resid = " + resid);
				Integer currentNumber = (Integer)tmp.get(type);
				
				if (currentNumber == null) continue;
//				System.out.println("currentNumber = " + currentNumber);

				int n = currentNumber.intValue();
				n++;
				tmp.put(type, new Integer(n));
				abundance.put(new Integer(i),tmp);
			}
			
			//now look at the results
			HashMap tmp = (HashMap)abundance.get(new Integer(i));
			int sliceSize = slice.size() - skipped;

			Set keys = tmp.keySet();
			Iterator iterator = keys.iterator();
			Integer maxNumber = new Integer(0);//initial maximum abundance
			String maxResidue = "";
			while (iterator.hasNext()){
				String key = (String)iterator.next();//residue type or residue
				Integer num = (Integer)tmp.get(key);
				int numValue = num.intValue();
				if (numValue > maxNumber.intValue()){
					maxNumber = new Integer(numValue);
					maxResidue = key;
				}
			}
			
			if (maxResidue.equals("-")){
				double dashRatio = ((double)maxNumber.intValue())/((double)sliceSize);
				if (dashRatio >= 0.5){
					//indicate zero
					result.add(new Double(0.0));
					continue;
				}
				else{
					//the dashes had been calculated as main residue
					//the search must be re-run ignoring dashes
					
					Iterator itt = keys.iterator();
					maxNumber = new Integer(0);//reset maxNumber
					while (itt.hasNext()){
						String key = (String)itt.next();
						if (key.equals("-")){
							continue;
						}
						else{
							Integer n = (Integer)tmp.get(key);
							int nValue = n.intValue();
							if (nValue > maxNumber.intValue()){
								maxNumber = new Integer(nValue);
							}
						}
					}
					
					
					
					//done rescanning
				}
			}
	
			//calculate the conservation coefficient
			double coeff = (double)(maxNumber.intValue())/sliceSize;
//			System.out.println(coeff);
			result.add(new Double(coeff));//add the coefficient
		}
		
		return result;
	}
	
	
	private Vector getSlice(int index){
		Vector out = new Vector();
		for (int i = 0; i < sequences.size(); i++){
			Cell[] cells = (Cell[])sequences.get(i);
			if (index > cells.length-1 || cells[index] == null) continue;
			out.add(cells[index].symbol);
		}
		
		return out;
	}
	
	/**
	 * This method relies on a local installation of clustalw that is distributed with the native installer
	 * and is called only if clustalw directory is present
	 *
	 */
	private void alignLocal(String input){
		
		
		String path = parent.getInstallationHome();
		String data = path + File.separator + "clustalw" + File.separator + "data.fst";
		
//		System.out.println("data = " + data);
		
		//now buffer contains concatenated sequences in fasta format
		try{
			IOHandler.saveTextFile(input, data);
		}
		catch (Exception ex){
			ex.printStackTrace();
			parent.displayExceptionMessage("Exception running alignment", ex);
		}
		
		//get the name of the specific binary: clustalw.exe on windows, or clustalw on linux and mac
		String clustalw = null;
		File binary = new File(path + File.separator + "clustalw" + File.separator + "clustalw.exe");
		if (binary.exists()){
			clustalw = path + File.separator + "clustalw" + File.separator + "clustalw.exe";
		}
		else{
			clustalw = path + File.separator + "clustalw" + File.separator + "clustalw";
		}
				
		String[] c = new String[2];
		c[0] = clustalw;
		c[1] = data;
		
//		System.out.println("Starting the process: " + c[0] + ", " + c[1]);
		
		//now start the external process
		Process process = null;
		try{
			process = Runtime.getRuntime().exec(c);
			
			InputStream in = process.getInputStream();
			BufferedReader reader1 = new BufferedReader(new InputStreamReader(in));
			String linea = null;
			String dump = null;
			while ((linea = reader1.readLine()) != null){
				dump = linea;
//				System.out.println("dump = " + dump);
			}
	        in.close();
	        
	        //check for output file
			File output = new File(path + File.separator + "clustalw" + File.separator + "data.aln");
			if (output.exists()){
				//read the content
				BufferedReader reader = new BufferedReader(new FileReader(output));
				String liner = null;
				StringBuffer buf = new StringBuffer();
				while ((liner = reader.readLine()) != null){
					buf.append(liner);
					buf.append("\n");
				}
				
				//parse out the alignment and update the view in the sequence viewer
				alignmentData = buf.toString();
				
				menuSequenceAlignmentDownload.setEnabled(true);
				
				processAlignmentOutput(alignmentData);
				
				File datafile = new File(data);
				datafile.delete();
				
				reader.close();
				output.delete();
				
				
				File dnd = new File(path + File.separator + "clustalw" + File.separator + "data.dnd");
				dnd.delete();
				
			}
			else{
				parent.displayErrorMessage("Alignment produced no results");
			}
			
			process.destroy();
			
			
		}
		catch (Exception e){
			parent.displayErrorMessage("Unable to run alignment");
			e.printStackTrace();
			process.destroy();
		}
		
	}
	
	public void processAlignmentOutput(String content){
		
		HashMap map = new HashMap();//name -> sequence for new sequences
		Vector names = new Vector();
		

		//concatenate the string representations of the loaded sequences
		StringBuffer buffer = new StringBuffer();
		for (int i = 0; i < sequenceObjects.size(); i++){
			Sequence s = (Sequence)sequenceObjects.get(i);
			
			buffer.append(">");
			buffer.append(s.getName());
			buffer.append("\n");
			String seq = s.getSequence();
			//compress it if necessary

			Pattern pattern = Pattern.compile("-");
			Matcher matcher = pattern.matcher(seq);
			String temp = matcher.replaceAll("");
			
			Pattern pattern2 = Pattern.compile("#");
			Matcher matcher2 = pattern2.matcher(temp);
			buffer.append(matcher2.replaceAll(""));
			buffer.append("\n");
			
			map.put(s.getName(), new StringBuffer());
			names.add(s.getName());
		}

		//parse the output
		StringTokenizer tok = new StringTokenizer(content, "\n", false);
		String testName = null;
		while (tok.hasMoreTokens()){
			String token = tok.nextToken().trim();
			try{
				String tempName = token.substring(0, token.indexOf(" ")).trim();
				for (int i = 0; i < names.size(); i++){
					String name = (String)names.get(i);
					if (tempName.equals(name)){
						String temp = token.substring(token.indexOf(" ")+1).trim();
						((StringBuffer)map.get(name)).append(temp);
						testName = name;
					}
				}
			}
			catch (Exception ex){
				continue;
			}
		}
		
		//keep track of all symbols for each sequence - waters and ligands get deleted by clustalw and
		//will need to be left in place
		int[] symbolCount = new int[sequences.size()];
		
		//preprocess the new alignment in case there are some X's embedded in the sequence
		//in this case, each sequence in this column should be given a dash to keep the alignment correct
		//go column by column and check for X's. If an X is found, insert dashes in all sequences that
		//don't have it. Residue linking to that cell will be done at the final step, so this
		//is only needed to keep the alignment even
		StringBuffer[] buffers = new StringBuffer[names.size()];
		for (int i = 0; i < names.size(); i++){
			buffers[i] = (StringBuffer)map.get(names.get(i));
			
//			System.out.println(i + ": " + buffers[i]);
		}
		
		
		
		//just in case this is a follow-up alignment, compress the loaded sequences to remove
		//existing dashes
		Vector ligands = new Vector();//intended to keep track of position of ligand cells in sequence
		Vector ligandIndices = new Vector();//indices with respect to non-dash count
		
		for (int i = 0; i < sequences.size(); i++){
			Cell[] cells = (Cell[])sequences.get(i);
			Vector ligs = new Vector();
			Vector ind = new Vector();
			
			//count gaps first
			int counter = 0;
			for (int j = 0; j < cells.length; j++){
				if (cells[j] == null) continue;
				if (cells[j].symbol.equals("-"))counter++;
			}
			
			Cell[] cc = new Cell[cells.length - counter];
			int k = 0;
			StringBuffer buffer1 = new StringBuffer();
			for (int j = 0; j < cells.length; j++){
				if (cells[j] == null) continue;
				if (cells[j].symbol.equals("-")) continue;
				if (cells[j].symbol.equals("#")){
					buffer1.append("X");
					ligs.add(cells[j]);
					ind.add(new Integer(k));
					continue;
				}
				else{
					buffer1.append(cells[j].symbol);
				}
				cc[k] = cells[j];
				k++;
			}
			
			sequences.remove(cells);
			sequences.add(i, cc);
			
			symbolCount[i] = buffer1.length();
			
			ligands.add(ligs);
			ligandIndices.add(ind);
			
			((Sequence)sequenceObjects.get(i)).setSequence(buffer1.toString());
			((Sequence)sequenceObjects.get(i)).setCells(cc);
		}
		
		
		//go through each sequence and restore position of "#", if any
		for (int i = 0; i < buffers.length; i++){
			StringBuffer b = buffers[i];
//			System.out.println(b);
			Vector indices = (Vector)ligandIndices.get(i);
			if (indices.size() == 0) continue;//no ligands in this sequence
			int[] set = new int[indices.size()];
			for (int j = 0; j < indices.size(); j++){
				set[j] = ((Integer)indices.get(j)).intValue();
//				System.out.println("index = " + set[j]);
			}
			
			//now go through the buffer and insert X whenever the correct index is encountered
			//we can't number straight indices, because buffer contains also dashes, so only 
			//non-dash symbols have to be counted
			//we use regular expressions to insert X at every needed position, after each time
			//incrementing index by 1
			int factor = 0;
			for (int j = 0; j < set.length; j++){
				int index = set[j];
				index += factor;
				String pat = "^((\\-*?\\w\\-*?){" + index + "})(.*)";
				Pattern pattern = Pattern.compile(pat);
				Matcher matcher = pattern.matcher(b);
				if (matcher.matches()){
					String start = matcher.group(1);
					String end = matcher.group(3);
					//insert X between start and end and replace the buffer with the new value
					b = new StringBuffer();
					b.append(start);
					b.append("X");
					b.append(end);
					factor++;
				}
			}
//			System.out.println(b);
			buffers[i] = b;
		}
		
		//now insert dashes at X positions in all sequences but those that have X's
		int n = 0;
		
		
		while (true){//stop when all buffers are out
			boolean x = false;
			boolean stop = true;
			for (int j = 0; j < names.size(); j++){
				if (n < buffers[j].length()) stop = false;//even one is sufficient to keep going
				if (n >= buffers[j].length()) continue;
				
				char ch = buffers[j].charAt(n);
				if (ch == 'X'){
					x = true;
					break;
				}
			}
			
			if (stop) break;
			
			if (x){
				//walk over the buffers again and insert dashes wherever there is no X at this position
				for (int j = 0; j < buffers.length; j++){
					
					if (n >= buffers[j].length()){
						//append a dash
						buffers[j].append("-");
					}
					else{
						char ch = buffers[j].charAt(n);
						if (ch != 'X'){
							buffers[j].insert(n, "-");
						}
					}
				}
			}
			
			n++;
		}
		
//		System.out.println(buffers[0]);
//		System.out.println(buffers[1]);

		
		//update sequences
		Vector temp = new Vector();//temporary space for new Cell arrays, to avoid concurrent modification
		//exception
		
		//if the new sequence only contains dashes, leave the original intact (in case it's an organic molecule)
		
		for (int i = 0; i < sequences.size(); i++){
			Cell[] cells = (Cell[])sequences.get(i);
//			System.out.println("length of original cells = " + cells.length);
			Sequence sequence = (Sequence)sequenceObjects.get(i);
			
			Vector ligs = (Vector)ligands.get(i);
//			System.out.println("size of ligs = " + ligs.size());
			
			String name = sequence.getName();
			String line = buffers[i].toString();
			
			Pattern pattern = Pattern.compile("^\\-+$");
			Matcher matcher = pattern.matcher(line);
			if (matcher.matches()){
				//all dashes, don't replace sequence (there were no amino acids)
				temp.add(sequences.get(i));
				continue;
			}
			
			
			//compare the length of the new line with the previous character count
			int oldCount = symbolCount[i];
			int diff = oldCount - line.length();
			if (diff < 0) diff = 0;
			
//			System.out.println("diff = " + diff);
			
			char[] array = line.toCharArray();
			Cell[] cc = new Cell[array.length + diff];//new array
			
//			System.out.println("cc length = " + cc.length + ", array length = " + array.length);
			
			int counter = 0;//keeps position in the original array
			int j = 0;
			int ligCount = 0;
			StringBuffer sequenceBuffer = new StringBuffer();
			for (j = 0; j < array.length; j++){
				if (array[j] == '-'){
					cc[j] = new Cell(sequence, j, "-", null);
					sequenceBuffer.append("-");
				}
				else if (array[j] == 'X'){
					Cell cell = (Cell)ligs.get(ligCount++);
					cc[j] = cell;
					sequenceBuffer.append(cell.symbol);
				}
				else{
					cc[j] = cells[counter];
					if (cells[counter] == null){
//						System.out.println("cc[j] == " + cc[j] + " at j = " + j);
						continue;
					}
					cells[counter].index = j;
					sequenceBuffer.append(cc[j].symbol);
					counter++;
				}
				
//				System.out.print(cc[j].symbol);
			}
			
//			System.out.println("");
			
//			System.out.println("last j = " + j + ", last counter = " + counter);
			
			
			//keep counting in case there were extra symbols in the original sequence
			if (diff > 0){
				try{
					counter++;
					for (int k = 0; k < diff; k++){
						cc[j] = cells[counter];
						cells[counter].index = j;
						sequenceBuffer.append(cc[j].symbol);
						counter++;
						j++;
					}
				}
				catch (Exception e){
					//do nothing, just get done with it
				}
			}
			
			sequence.setSequence(sequenceBuffer.toString());
			sequence.setCells(cc);
			temp.add(cc);
		}
		
		sequences.clear();
		for (int i = 0; i < temp.size(); i++){
			sequences.add(temp.get(i));
		}
		
		adjustVScrollBar(false);
		computeResidueMax(false);
		revalidate();
		repaint();
		
		menuEditReset.setEnabled(true);
		
	}
	
	/**
	 * This method matches the entry to those currently loaded and replaces the sequences in the viewer
	 * The method is used to update view in the event of an externally derived sequence alignment,
	 * such as during the structure alignment. Very similar to actions performed after a sequence alignment
	 * with the new one
	 * @param entry
	 * @param sequence
	 */
	public void setSequences(HashMap list, Vector order){
		
		//check whether all entries are actually present
		//test whether this Entry is in the loaded set
		for (int i = 0; i < order.size(); i++){
			if (!entries.containsValue(order.get(i))){
				structureDocument.parent.displayErrorMessage("Unable to update sequences. Please try again");
				return;
			}
		}
		
		//create buffers array, similar to that used in sequence alignment post-processing
		StringBuffer[] buffers = new StringBuffer[order.size()];
		int m = 0;
		for (int i = 0; i < sequenceObjects.size(); i++){
			Entry e = (Entry)entries.get(sequenceObjects.get(i));
			if (!list.containsKey(e)) continue;
			String seq = (String)list.get(e);
			StringBuffer buffer = new StringBuffer(seq);
			buffers[m++] = buffer;
		}
		
		//update only Entries contained in order Vector, don't touch others
		
		Vector ligands = new Vector();//intended to keep track of position of ligand cells in sequence
		Vector ligandIndices = new Vector();
		
		int[] symbolCount = new int[order.size()];

		m = 0;
		for (int i = 0; i < sequences.size(); i++){
			
			Vector ligs = new Vector();
			Vector ind = new Vector();

			//check whether this sequence should be updated
			Sequence sequence = (Sequence)sequenceObjects.get(i);
			if (!order.contains(entries.get(sequence))){
				continue;
			}
			
//			System.out.println(sequence.getSequence().length() + ":" + sequence.getSequence());
			
			Cell[] cells = (Cell[])sequences.get(i);
			
			//count gaps first
			int counter = 0;
			for (int j = 0; j < cells.length; j++){
//				System.out.println("j = " + j + ", cells[j] = " + cells[j]);
				if (cells[j].symbol.equals("-"))counter++;
			}
			
			Cell[] cc = new Cell[cells.length - counter];
			int k = 0;
			StringBuffer buffer1 = new StringBuffer();
			for (int j = 0; j < cells.length; j++){
				if (cells[j].symbol.equals("-"))continue;

				if (cells[j].symbol.equals("#")){
					buffer1.append("X");
					ligs.add(cells[j]);
					ind.add(new Integer(k));
					continue;
				}
				else{
					buffer1.append(cells[j].symbol);
				}
				cc[k] = cells[j];
				k++;
			}
			
			sequences.remove(cells);
			sequences.add(i, cc);
			
			
			symbolCount[m++] = buffer1.length();
			
			ligands.add(ligs);
			ligandIndices.add(ind);
			
			((Sequence)sequenceObjects.get(i)).setSequence(buffer1.toString());
		}
		
		for (int i = 0; i < buffers.length; i++){
			StringBuffer b = buffers[i];
			Vector indices = (Vector)ligandIndices.get(i);
			if (indices.size() == 0) continue;//no ligands in this sequence
			int[] set = new int[indices.size()];
			for (int j = 0; j < indices.size(); j++){
				set[j] = ((Integer)indices.get(j)).intValue();
			}
			
			//now go through the buffer and insert X whenever the correct index is encountered
			//we can't number straight indices, because buffer contains also dashes, so only 
			//non-dash symbols have to be counted
			//we use regular expressions to insert X at every needed position, after each time
			//incrementing index by 1
			int factor = 0;
			String content = b.toString();
			for (int j = 0; j < set.length; j++){
				int index = set[j];
				index += factor;
				String pat = "^((\\-*?\\w\\-*?){" + index + "})(.*)";
				Pattern pattern = Pattern.compile(pat);
				Matcher matcher = pattern.matcher(content);
				if (matcher.matches()){
					String start = matcher.group(1);
					String end = matcher.group(3);
					//insert X between start and end and replace the buffer with the new value
					b = new StringBuffer();
					b.append(start);
					b.append("X");
					b.append(end);
					
					factor++;
				}
				
				content = b.toString();
				
			}
//			System.out.println(b);
			buffers[i] = new StringBuffer(content);
		}
		
		//now insert dashes at X positions in all sequences but those that have X's
		int n = 0;
		
		
		while (true){//stop when all buffers are out
			boolean x = false;
			boolean stop = true;
			for (int j = 0; j < buffers.length; j++){
//				System.out.println("j = " + j + ", length = " + buffers.length);
				if (n < buffers[j].length()) stop = false;//even one is sufficient to keep going
				if (n >= buffers[j].length()) continue;
				
				char ch = buffers[j].charAt(n);
				if (ch == 'X'){
					x = true;
					break;
				}
			}
			
			if (stop) break;
			
			if (x){
				//walk over the buffers again and insert dashes wherever there is no X at this position
				for (int j = 0; j < buffers.length; j++){
					
					if (n >= buffers[j].length()){
						buffers[j].append("-");
					}
					else{
						char ch = buffers[j].charAt(n);
						if (ch != 'X'){
							buffers[j].insert(n, "-");
						}
					}
				}
			}
			
			n++;
		}
		
		//update sequences
		Vector temp = new Vector();//temporary space for new Cell arrays, to avoid concurrent modification
		//exception
		
		//if the new sequence only contains dashes, leave the original intact (in case it's an organic molecule)
//		System.out.println("size of ligands = " + ligands.size());
		m = 0;
		for (int i = 0; i < sequences.size(); i++){
			
			if (!order.contains(entries.get(sequenceObjects.get(i)))){
				temp.add(sequences.get(i));
				continue;
			}
			
			Cell[] cells = (Cell[])sequences.get(i);
//			System.out.println("length of original cells = " + cells.length);
			Sequence sequence = (Sequence)sequenceObjects.get(i);
			
			Vector ligs = (Vector)ligands.get(m);
//			System.out.println("size of ligs = " + ligs.size());
			
			String name = sequence.getName();
			String line = buffers[m].toString();
			
			Pattern pattern = Pattern.compile("^\\-+$");
			Matcher matcher = pattern.matcher(line);
			if (matcher.matches()){
				//all dashes, don't replace sequence (there were no amino acids)
				temp.add(sequences.get(i));
				continue;
			}
			
			
			//compare the length of the new line with the previous character count
			int oldCount = symbolCount[m];
			int diff = oldCount - line.length();
			if (diff < 0) diff = 0;
			
//			System.out.println("diff = " + diff);
			
			char[] array = line.toCharArray();
			Cell[] cc = new Cell[array.length + diff];//new array
			
//			System.out.println("cc length = " + cc.length + ", array length = " + array.length);
			
			int counter = 0;//keeps position in the original array
			int j = 0;
			int ligCount = 0;
			StringBuffer sequenceBuffer = new StringBuffer();
			for (j = 0; j < array.length; j++){
				try{
					if (array[j] == '-'){
						cc[j] = new Cell(sequence, j, "-", null);
						sequenceBuffer.append("-");
					}
					else if (array[j] == 'X'){
						Cell cell = (Cell)ligs.get(ligCount++);
						cc[j] = cell;
						sequenceBuffer.append(cell.symbol);
					}
					else{
						cc[j] = cells[counter];
						cells[counter].index = j;
						sequenceBuffer.append(cc[j].symbol);
						counter++;
					}
				}
				catch (Exception ex){
//					ex.printStackTrace();
				}
			}
			
			//keep counting in case there were extra symbols in the original sequence
			if (diff > 0){
				try{
					counter++;
					for (int k = 0; k < diff; k++){
						cc[j] = cells[counter];
						cells[counter].index = j;
						sequenceBuffer.append(cc[j].symbol);
						counter++;
						j++;
					}
				}
				catch (Exception e){
					//do nothing, just get done with it
				}
			}
			
			m++;
			
			//check the cells
//			System.out.println("length of cc = " + cc.length);
			int r = 0;
			try{
				for (r = 0; r < cc.length; r++){
					Cell c = cells[r];
					if (c == null){
//						System.out.println("Null cell at position " + r);
					}
				}
			}
			catch (Exception s){
//				System.out.println("Exception at r = " + r);
			}

			
			sequence.setSequence(sequenceBuffer.toString());
			temp.add(cc);
		}
		
		sequences.clear();
		for (int i = 0; i < temp.size(); i++){
			sequences.add(temp.get(i));
		}
		
		adjustVScrollBar(false);
		computeResidueMax(false);
		revalidate();
		repaint();
		
	}
	
	/**
	 * This method takes the current data model, saves a reserve copy in case an undo will be needed
	 * or an error occurs, and runs remote clustalw on an EBI server
	 * 
	 *
	 */
	private void alignRemote(String input){
		
		AlignmentSettings.alignmentMode = AlignmentSettings.ALIGNMENT_SERVER;
		
		String result = AlignmentService.align(input);

		
		if (result == null || result.length() == 0){
			parent.displayErrorMessage("Alignment produced no data.");
			return;
		}
		
//		System.out.println("result = " + result);
		
		alignmentData = result;
		menuSequenceAlignmentDownload.setEnabled(true);
		
		processAlignmentOutput(alignmentData);
		
	}
	
	private void saveAlignmentOutput(){
		if (alignmentData == null || alignmentData.length() == 0) return;
		IOHandler.saveTextFile(alignmentData, parent, "aln");
	}
	
	public void setUndo(boolean flag){
		menuEditReset.setEnabled(flag);
		sequencePanel.setUndo(flag);
	}
	
	public boolean getEditMode(){
		return editMode;
	}
	
	public Vector getBackupSequences(){
		return backupSequences;
	}
	
	public void displayMessage(String text){
		structureDocument.parent.displayMessage(text);
	}
	
	public int getResidueMax(){
		return residueMax;
	}
	
	public void setMessage(String message, Color color){
		if (message == null){
			scrollBarFiller.remove(messageLabel);
		}
		else{
			messageLabel = new JLabel(message);
			messageLabel.setPreferredSize(new Dimension(buttonWidthCells*parameters.cellWidth-2, 10));
			messageLabel.setForeground(color);
			scrollBarFiller.add(messageLabel, BorderLayout.CENTER);
		}
		scrollBarFiller.revalidate();
		scrollBarFiller.repaint();
	}

	public final HashMap getCellResidueMapping() {
		return cellResidueMapping;
	}

	public final HashMap getResidueCellMapping() {
		return residueCellMapping;
	}
	
	public HashMap getStyleMap(){
		return styleMap;
	}
	
	/**
	 * This method is called internally by SequencePanel. It determines whether to fire a component event
	 * and handle internal change in response to that or replace residue in the viewer (if there is no 
	 * corresponding residue).
	 * @param sequence
	 * @param index
	 * @param replacement
	 */
	public void replaceResidue(Sequence sequence, int index, String replacement){
		
		Cell[] cells = (Cell[])sequences.get(sequenceObjects.indexOf(sequence));
		if (index > (cells.length-1)) return;

		if (styleMap.containsKey(sequence)){
			
			Residue residue = cells[index].residue;
			
			//check the target
			if (residue.getClassification() != Residue.COMPOUND_AMINO_ACID){
				parent.displayErrorMessage("Selected target residue is not amino acid.");
				return;
			}
			
			StructureStyles styles = (StructureStyles)styleMap.get(sequence);
			if (StylesPreferences.undoEnabled) structureDocument.parent.getStructureViewer().saveStateForUndo();
			styles.replaceStructureComponent(residue, AminoAcidInfo.getCodeFromLetter(replacement));
			
		}
		else{
			//do it all here
			sequence.getSequenceStyles().replaceResidue(cells[index], replacement);
		}
		
		structureDocument.parent.updateView();
	}
	
	public Vector getSequences(){
		return sequences;
	}
	
	public Vector getSequenceObjects(){
		return sequenceObjects;
	}
	
	public void setCellColors(float[][][] colors){
		
		try{
			for (int i = 0; i < sequences.size(); i++){
				Cell[] cells = (Cell[])sequences.get(i);
				for (int j = 0; j < cells.length; j++){
					if (cells[j] == null) continue;
					cells[j].color = styleManager.getColor(colors[i][j]);
				}
			}
		}
		catch (Exception e){
			System.out.println("Exception setting cell colors: " + e);
			return;
		}
		
		updateView();
		
	}
	
	public void setUndoMode(boolean enabled){
		undoMode = enabled;
	}
	
	public HashMap getEntries(){
		return entries;
	}
	
	public boolean isSelected(Sequence sequence){
		Vector selection = sequencePanel.getSelectedSequences();
		if (selection.contains(sequence)) return true;
		
		return false;
	}
	
	public void setSelected(Sequence sequence, boolean selected){
		
		if (selected){
			Vector selection = sequencePanel.getSelectedSequences();
			if (!selection.contains(sequence))selection.add(sequence);
		}
		else{
			Vector selection = sequencePanel.getSelectedSequences();
			if (selection.contains(sequence)) selection.remove(sequence);
		}
	}
	
	/**
	 * This method runs through the symbols in the sequence display of the given Sequence object
	 * and inserts dashes wherever necessary
	 * @param s
	 * @param sequence
	 */
	public void updateSequence(Sequence s, String sequence){
		
		Cell[] cells = (Cell[])sequences.get(sequenceObjects.indexOf(s));
		
		if (cells.length >= sequence.length()) return;
		
		int n = sequenceObjects.indexOf(s);
		
		//if cells is shorter, add the dashes from the sequence
		Cell[] newCells = new Cell[sequence.length()];
		StringBuffer buffer = new StringBuffer();
		int counter = 0;
		for (int i = 0; i < sequence.length(); i++){
			String symbol = sequence.substring(i, i+1);
			if (cells[counter].symbol.equals(symbol)){
				//copy the cell to the new array
				newCells[i] = cells[counter];
				buffer.append(symbol);
				counter++;
			}
			else if (symbol.equals("-")){
				//add a new Cell object
				newCells[i] = new Cell(s, i, "-", null);
				buffer.append("-");
			}
		}
		
		sequences.remove(cells);
		sequences.add(n, newCells);
		
		s.setSequence(buffer.toString());
		
	}
	
	/**
	 * this method is called when the loaded sequence needs to be updated to match the one
	 * previously loaded (for cases with homology modeling).
	 * From the original sequence, we take the fragments before and after structure coverage,
	 * then append them to the sequence already loaded, and update it with the original to
	 * get the correct dashes
	 * @param s
	 * @param sequence
	 * @param start
	 * @param end
	 */
	public void updateSequenceFromHomologyModel(Sequence s, String sequence, int start, int end){
		if (s.getSequence().length() == sequence.length()){
			
			//lengths are the same
			//apparently, full sequence was used, so we just need to update sequence
			updateSequence(s, sequence);
			
		}
		else{
			//sequences have different lengths - the structure covers only a part of the original sequence
/*			String pre = sequence.substring(0, start);
			String post = sequence.substring(end+1);
			
			String current = s.getSequence();
			StringBuffer result = new StringBuffer();
			if (pre != null && pre.length() > 0){
				result.append(pre);
			}
			
			result.append(current);
			
			if (post != null && post.length() > 0){
				result.append(post);
			}
*/			
			Cell[] cells = (Cell[])sequences.get(sequenceObjects.indexOf(s));
			
			if (cells.length >= sequence.length()) return;
			
			int n = sequenceObjects.indexOf(s);
			
			//if cells is shorter, add the dashes from the sequence
			Cell[] newCells = new Cell[sequence.length()];
			StringBuffer buffer = new StringBuffer();
			int counter = 0;
			for (int i = 0; i < sequence.length(); i++){
				String symbol = sequence.substring(i, i+1);
				if (i < start){
					//create a new Cell
					newCells[i] = new Cell(s, i, symbol, null);
					buffer.append(symbol);
					continue;
				}
				
				if (i > end){
					newCells[i] = new Cell(s, i, symbol, null);
					buffer.append(symbol);
					continue;
				}
				
				//this executes if we are in the structure coverage area
				if (cells[counter].symbol.equals(symbol)){
					//copy the cell to the new array
					newCells[i] = cells[counter];
					buffer.append(symbol);
					counter++;
				}
				else if (symbol.equals("-")){
					//add a new Cell object
					newCells[i] = new Cell(s, i, "-", null);
					buffer.append("-");
				}
			}
			
			sequences.remove(cells);
			sequences.add(n, newCells);
			
			s.setSequence(buffer.toString());
			
		
		}
	}
	
	private void translateDNA(){
		//use the selected sequence and check that the start residue is selected.
		//translation is from the selected residue to either the next selection or to the stop codon
		Sequence s = null;
		int count = sequences.size();
		if (count == 1){
			s = (Sequence)sequenceObjects.get(0);
		}
		else{
			Vector selectedSequences = sequencePanel.getSelectedSequences();
			if (selectedSequences.size() > 1){
				parent.displayErrorMessage("Only one sequence should be selected");
				return;
			}
			s = (Sequence)selectedSequences.get(0);
		}
		
		StringBuffer out = new StringBuffer();
		HashMap code = DNAProperties.getTranslationMap();
		
		StringBuffer triplet = new StringBuffer();
		
		//get the start residue and end residue if applicable
		boolean start = false;
		Cell[] cells = (Cell[])this.sequences.get(sequenceObjects.indexOf(s));
		
		int counter = 0;
		boolean selected = false;
		//check for content of non-DNA symbols
		for (int i = 0; i < cells.length; i++){
			String symbol = cells[i].symbol;
			if (!symbol.equals("A") && !symbol.equals("T") && !symbol.equals("C") && !symbol.equals("G")){
				if (!symbol.equals("#") && !symbol.equals("*")){
					counter++;
				}
			}
			if (cells[i].selected) selected = true;
		}
		
		double ratio = (double)counter/(double)cells.length;
		if (ratio > 0.5){
			parent.displayErrorMessage("More than half of the residues in the sequence are not DNA bases");
			return;
		}
		
		//to do a preliminary transcription into a complementary RNA strand, which gets translated into protein
		HashMap map = new HashMap();
		map.put("A", "T");
		map.put("T", "A");
		map.put("C","G");
		map.put("G", "C");
		
		for (int i = 0; i < cells.length; i++){
			Cell c = cells[i];
			
			if (!selected) start = true;//start from the beginning
			
			//check the symbol
			String symbol = c.symbol;
			if (!symbol.equals("A") && !symbol.equals("T") && !symbol.equals("C") && !symbol.equals("G")) continue;
			
			if (c.selected){
				if (start){
					//it's the last base to be interpreted
					triplet.append(map.get(c.symbol));
					
					
					start = false;
				}
				else{
					//the first base
					triplet.append(map.get(c.symbol));
					
					start = true;
				}
			}
			else{
				if (start){
					//continue building the sequence
					triplet.append(map.get(c.symbol));
				}
			}
			
			if (triplet.length() == 3){
				String aa = (String)code.get(triplet.toString());
				if (aa.equals("X")){
					//stop codon
					break;
				}
				out.append(aa);
				triplet = new StringBuffer();
			}
		}
	
//		System.out.println("out = " + out);
		
		//now simply add a new sequence
		SequenceEntry entry = new SequenceEntry();
		entry.setName("Name001");
		entry.setSequence(new Sequence("Name001", out.toString()));
		entry.setDescription("Translated sequence");
		parent.addEntry(entry, true, false);

	}
	
	public void printSetupHandler(){

		Thread t = new Thread(){
			public void run(){
				PrinterJob pj = PrinterJob.getPrinterJob();
				mPageFormat = pj.pageDialog(mPageFormat);
			}
		};
		t.start();
	}

	public void printHandler(boolean selection){
		
		if (sequences.size() == 0){
			structureDocument.parent.displayErrorMessage("No sequences loaded: nothing to print.");
			return;
		}
		
		try{
			
		final boolean useSelection = selection;
		
		Thread runner = new Thread(){
			public void run(){
				try{
					
				if (useSelection){
					if (getSelectedColumns().size() == 0){
						structureDocument.parent.displayErrorMessage("No columns are selected: nothing to print.");
						return;
					}
				}
					
				//use the current page format to get the printable area
				PrinterJob pj = PrinterJob.getPrinterJob();
		
				//set the cursor, 'cause the wait will be long
				setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
				book = new Book();
				
				//get the page format
				int realPageWidth = (int)mPageFormat.getImageableWidth();
				int realPageHeight = (int)(mPageFormat.getImageableHeight());
				
				int pageWidth = 2 * realPageWidth;
				int pageHeight = 2 * realPageHeight;
				
				//calculate how many cells and sequences will fit in a page's width and height
				int cells = (int)(pageWidth/parameters.cellWidth);
				int lines = (int)(pageHeight/parameters.cellHeight);
				
				//calculate the number f actual residues that can be on a single line
				int span = cells - (int)(buttonWidthCells*1.5) - 2;//to allow for longer terminating position numbers
				
//				System.out.println("##########\nOne page fits " + lines + " lines");
				
				int length = residueMax;
						
				//iterate through the pages each time creating a BufferedImage of the page size
				//and drawing a panel on top of it\
				int startResidue = 0;
				int stopResidue = startResidue + span;
				
				int firstResidue = startResidue;
				int lastResidue = length;

				if (useSelection){
					//print whatever is between the leftmost and rightmost selected columns
					Vector columns = getSelectedColumns();
					int min = 10000;
					int max = -1;
					for (int i = 0; i < columns.size(); i++){
						int c = ((Integer)columns.elementAt(i)).intValue();
						if (c < min){
							min = c;
						}
						if (c > max){
							max = c;
						}
					}
					
					if (min < 10000 && max > -1){
						if (max > min){
							startResidue = min;
							lastResidue = max + 1;//to correct zero-based indices
							stopResidue = Math.min(stopResidue, lastResidue);
						}
					}
				}
				
				
				int lineHeight = parameters.cellHeight;
				int offsetY = 0;
				boolean newPage = true;//indicates that the starting iteration will initialize an image
				BufferedImage image = null;
				Graphics2D g = null;
				
				int linesRendered = 0;//counter of lines drawn so far on the page
				int pagesRendered = 0;
				
				while (true){
					if (startResidue > lastResidue){
						break;
					}
					if (newPage){
						if (StylesPreferences.indexedColor){
							image = new BufferedImage(pageWidth,pageHeight, BufferedImage.TYPE_BYTE_INDEXED, ImageHandler.getIndexColorModel());
							g = image.createGraphics();
						}
						else{
							image = new BufferedImage(pageWidth,pageHeight, BufferedImage.TYPE_INT_ARGB);
							g = image.createGraphics();
							g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
						}
						newPage = false;
						
						offsetY = 0;
					}
					
					if (linesRendered >= lines-2){
						//skip to the next page first, because the scale will appear last on this page
						pagesRendered++;
						PagePrintable p = new PagePrintable(image, realPageWidth, realPageHeight, pageWidth, pageHeight);
						book.append(p, mPageFormat);
					
//						System.out.println("saving image before skipping to the next page");
//						ImageHandler.savePrint(image, (new Integer(pagesRendered)).toString());
							
						linesRendered = 0;
						if (StylesPreferences.indexedColor){
							image = new BufferedImage(pageWidth,pageHeight, BufferedImage.TYPE_BYTE_INDEXED, ImageHandler.getIndexColorModel());
							g = image.createGraphics();
						}
						else{
							image = new BufferedImage(pageWidth,pageHeight, BufferedImage.TYPE_INT_RGB);
							g = image.createGraphics();
							g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
						}
						newPage = false;
						
						offsetY = 0;
						
					}
					
					//increase offsetY by 10 pixels before each new block to improve readability
					offsetY += 10;
					
					Image line = scalePanel.createPrintableImage(startResidue, stopResidue, cells*parameters.cellWidth, lineHeight);
					g.drawImage(line, 0, offsetY, null);
					offsetY += lineHeight;
					linesRendered++;
					
					//start iterating over the sequences
					for (int j = 0; j < sequenceObjects.size(); j++){
						Sequence sequence = (Sequence)sequenceObjects.get(j);
						if (hiddenSequences.contains(sequence)){
							continue;
						}
						
						//get the image of the requested part of the sequence
						line = sequencePanel.createPrintableImage(j, startResidue, stopResidue, cells*parameters.cellWidth, lineHeight);
						g.drawImage(line, 0, offsetY, null);
						linesRendered++;
						offsetY += lineHeight;
						
						//check if this is the end of the page
						if (linesRendered >= lines){
							//we reached the end of the page
//							System.out.println("Rendered page at " + linesRendered + " + rendered lines");
							
							//now with this image, create a Printable instance and add it to Book
							pagesRendered++;
							PagePrintable p = new PagePrintable(image, realPageWidth, realPageHeight, pageWidth, pageHeight);
							book.append(p, mPageFormat);
					
//							System.out.println("Skipping to next page in the middle of block scan");
//							ImageHandler.savePrint(image, (new Integer(pagesRendered)).toString());
							
							linesRendered = 0;
							offsetY = 0;
							
//							System.out.println("start residue = " + startResidue);
							if (startResidue > lastResidue){
								break;
							}
							
							//create a new image to continue rendering
//							System.out.println("Creating a new page");
							if (StylesPreferences.indexedColor){
								image = new BufferedImage(pageWidth,pageHeight, BufferedImage.TYPE_BYTE_INDEXED, ImageHandler.getIndexColorModel());
								g = image.createGraphics();
							}
							else{
								image = new BufferedImage(pageWidth,pageHeight, BufferedImage.TYPE_INT_RGB);
								g = image.createGraphics();
								g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);			
							}
							g.setColor(Color.white);
							g.fillRect(0,0, pageWidth, pageHeight);
						}
						
					}
					
					//reset residue counter
					startResidue += span;
					stopResidue += span;
					
					if (linesRendered >= lines){
						//we reached the end of the page
						//now with this image, create a Printable instance and add it to Book
						pagesRendered++;
						PagePrintable p = new PagePrintable(image, realPageWidth, realPageHeight, pageWidth, pageHeight);
						book.append(p, mPageFormat);
					
//						System.out.println("Skipping to the next page");
//						ImageHandler.savePrint(image, (new Integer(pagesRendered)).toString());
							
						linesRendered = 0;
						offsetY = 0;
						
						newPage = true;
						
						if (startResidue > lastResidue){
							break;//the page is already saved
						}
					}
							
					stopResidue = Math.min(stopResidue, lastResidue);
					
					if (startResidue > lastResidue){
						//this is it. print what we have so far
		//				System.out.println("Hit the wall: start = " + startResidue);
						pagesRendered++;
						PagePrintable p = new PagePrintable(image, realPageWidth, realPageHeight, pageWidth, pageHeight);
						
						book.append(p, mPageFormat);
//						System.out.println("Done. saving the last one");
//						ImageHandler.savePrint(image, (new Integer(pagesRendered)).toString());
							
						break;
					}
				}
				
				g.dispose();
				pj.setPageable(book);
				if (pj.printDialog()){
					try{
						pj.print();
					}
					catch (PrinterException pe){
//						System.out.println("Exception " + pe);
						pe.printStackTrace();
						JOptionPane.showMessageDialog(parent.getApplicationFrame(), "Printing error: " + pe, "Error", JOptionPane.ERROR_MESSAGE);
					}
				}
				
				setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
				book = null;
				
				}
				catch (Error ee){
					structureDocument.parent.displayErrorMessage("Out of memory. Please restart the application and try again.");
				}
			}
		};
		runner.start();
		
		}
		catch (Exception ex){
			structureDocument.parent.displayExceptionMessage("Exception printing", ex);
		}
		catch (OutOfMemoryError error){
			structureDocument.parent.displayErrorMessage("Out of memory. Please restart the application and try again.");
		}
		
	}
	
	/**
	 * This method returns a Vector of Residues that correspond to sequence of the given Entry
	 * If no residue is present at the given position, null is used
	 * @param entry
	 * @return
	 */
	public Vector getAlignedResidues(Entry entry){
		
		Vector out = null;
		Set keys = entries.keySet();
		Iterator it = keys.iterator();
		while (it.hasNext()){
			Sequence sequence = (Sequence)it.next();
			Entry e = (Entry)entries.get(sequence);
			if (e == entry){
				Cell[] cells = (Cell[])sequences.get(sequenceObjects.indexOf(sequence));
				out = new Vector();
				for (int i = 0; i < cells.length; i++){
					Cell c = cells[i];
					if (c.symbol.equals("*")) continue;
					out.add(cells[i].residue);
				}
				
				break;
			}
		}
		
		return out;
	}
	
	public void performAlignment(){
		try{
			if (sequences.size() == 0){
				parent.displayErrorMessage("No sequences are loaded: nothing to align.");
				return;
			}
			
			if (sequences.size() == 1){
				parent.displayErrorMessage("Only one sequence is loaded. Nothing to align.");
				return;
			}
			
			if (sequences.size() > 200){
				parent.displayErrorMessage("Maximum size of alignment produced using Sirius interactively is limited to 200.");
				return;
			}
			
			//also check for identical names
			HashMap nameCheck = new HashMap();
			
			HashMap map = new HashMap();//name -> sequence for new sequences
			Vector names = new Vector();
			

			//concatenate the string representations of the loaded sequences
			StringBuffer buffer = new StringBuffer();
			for (int i = 0; i < sequenceObjects.size(); i++){
				Sequence s = (Sequence)sequenceObjects.get(i);
				
				buffer.append(">");
				buffer.append(s.getName());
				nameCheck.put(s.getName(), null);
				buffer.append("\n");
				String seq = s.getSequence();
				//compress it if necessary

				Pattern pattern = Pattern.compile("-");
				Matcher matcher = pattern.matcher(seq);
				String temp = matcher.replaceAll("");
				
				Pattern pattern2 = Pattern.compile("#");
				Matcher matcher2 = pattern2.matcher(temp);
				buffer.append(matcher2.replaceAll(""));
				buffer.append("\n");
				
				map.put(s.getName(), new StringBuffer());
				names.add(s.getName());
			}
			
			if (nameCheck.size() < sequenceObjects.size()){
				//there are duplicate names
				parent.displayErrorMessage("Two or more loaded sequences have identical names, which may affect parsing of the results. Please rename and rerun.");
				return;
			}


			String path = parent.getInstallationHome();
			File dir = new File(path + File.separator + "clustalw");
			sequencePanel.backupDataModel();
			if (dir.exists()){
				if (sequences.size() > 30){
					setMessage(" Aligning...", Color.RED);
				}
				
				alignLocal(buffer.toString());
				setMessage(null, null);
			}
			else{
				setMessage(" Aligning...", Color.RED);
				alignRemote(buffer.toString());
				setMessage(null, null);
			}
		}
		catch (Exception e){
			e.printStackTrace();
		}

	}
}