package edu.sdsc.sirius.viewers.TreeViewer;

import javax.swing.*;
import java.awt.*;
import edu.sdsc.mbt.*;
import edu.sdsc.sirius.dialogs.TableImpl.*;
import edu.sdsc.sirius.util.*;
import edu.sdsc.mbt.util.*;

import java.util.*;

/**
 * This class encapsulates a table that lists information about the currently active structure component
 * @author Sasha Buzko
 *
 */
public class InfoViewer extends JPanel {
	
	private StructureComponent structureComponent;
	private TableViewPanel contentPanel;
	private DataSet data = new DataSet();
	private Manager callable;
	
	private Vector searchTypes = new Vector();
	
	private String componentName = "None";//don't translate
	
	public InfoViewer(Manager callable){
		
		this.callable = callable;
		searchTypes.add("Component");//don't translate
		searchTypes.add(componentName);
		
		data.clearData();
		DataBlock block = new DataBlock("");
		block.setEntryId("");
		block.setName("");
		data.addData(block);
		
		contentPanel = new TableViewPanel(this, data, callable, searchTypes, TableViewPanel.FIRST_COLUMN_NARROW, false);
		this.setLayout(new BorderLayout());
		this.add(contentPanel, BorderLayout.CENTER);
		
		//wait for size setting before making this visible
		
	}
	
	public void setPanelSize(Dimension dim){
		contentPanel.resizePanel(dim);
		setVisible(true);
	}
	
	public void displayComponentInfo(StructureComponent sc){
		
		structureComponent = sc;
		
		if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
			Atom atom = (Atom)sc;
			data.clearData();
			
			searchTypes.set(1, "Atom");//don't translate
			
			//update the content of data set
			DataBlock line1 = new DataBlock("1");
			line1.setName("Atom name");
			line1.setDescription(atom.name);
			data.addData(line1);
			
			DataBlock line2 = new DataBlock("2");
			line2.setName("Atom number");
			line2.setDescription((new Integer(atom.number)).toString());
			data.addData(line2);
			
			if (atom.forceFieldType != null){
				DataBlock line3 = new DataBlock("3");
				line3.setName("Atom type");
				line3.setDescription(atom.forceFieldType);
				data.addData(line3);
			}

			DataBlock line4 = new DataBlock("4");
			line4.setName("Element");
			if (atom.element == 0){
				line4.setDescription("Unknown");
			}
			else{
				line4.setDescription(atom.getElement());
			}
			data.addData(line4);
			
			DataBlock line5 = new DataBlock("5");
			line5.setName("Partial charge");
			line5.setDescription(new Float(atom.partialCharge).toString());
			data.addData(line5);
			
			DataBlock line6 = new DataBlock("6");
			line6.setName("Formal charge");
			line6.setDescription((new Integer(atom.formalCharge)).toString());
			data.addData(line6);

			DataBlock line7 = new DataBlock("7");
			line7.setName("Atomic mass");
			if (atom.element == 0){
				line7.setDescription("None");
			}
			else{
				double mass = PeriodicTable.getElementWeight(atom.getElement());
				String mark = StringUtils.getFormattedNumber(mass, 4);
				line7.setDescription(mark);
			}
			data.addData(line7);
			
			DataBlock line8 = new DataBlock("8");
			line8.setName("Hybridization");
			if (atom.hybridization == Atom.SP){
				line8.setDescription("SP");
			}
			else if (atom.hybridization == Atom.SP2){
				line8.setDescription("SP2");
			}
			else{
				line8.setDescription("SP3");
			}
			data.addData(line8);
			
			DataBlock line9 = new DataBlock("9");
			line9.setName("Coordinates");
			String coords = StringUtils.getFormattedNumber(atom.coordinate[0], 3) + ", " 
					+ StringUtils.getFormattedNumber(atom.coordinate[1], 3) + ", " + StringUtils.getFormattedNumber(atom.coordinate[2],3);
			line9.setDescription(coords);
			data.addData(line9);
			
			this.remove(contentPanel);
			contentPanel = new TableViewPanel(this, data, callable, searchTypes, TableViewPanel.FIRST_COLUMN_NARROW, false);
			this.add(contentPanel, BorderLayout.CENTER);
			contentPanel.resizePanel(new Dimension(this.getSize().width, 250));
			this.revalidate();
			this.repaint();
		}
		else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
			Residue residue = (Residue)sc;
			
			searchTypes.set(1, "Residue");//don't translate
			
			//update the content of data set
			DataBlock line1 = new DataBlock("1");
			line1.setName("Residue name");
			line1.setDescription(residue.getCompoundCode());
			
			DataBlock line2 = new DataBlock("2");
			line2.setName("Number");
			line2.setDescription((new Integer(residue.getResidueId()).toString()));
			
			DataBlock line3 = new DataBlock("3");
			line3.setName("Atom count");
			line3.setDescription((new Integer(residue.getAtomCount())).toString());
			
			DataBlock line4 = new DataBlock("4");
			line4.setName("Classification");
			line4.setDescription(residue.getClassification());
			
			data.clearData();
			data.addData(line1);
			data.addData(line2);
			data.addData(line3);
			data.addData(line4);
			
			this.remove(contentPanel);
			contentPanel = new TableViewPanel(this, data, callable, searchTypes, TableViewPanel.FIRST_COLUMN_NARROW, false);
			this.add(contentPanel, BorderLayout.CENTER);
			contentPanel.resizePanel(new Dimension(this.getSize().width, 250));
			this.revalidate();
			this.repaint();
		}
		else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
			Chain chain = (Chain)sc;
			
			searchTypes.set(1, "Chain");//don't translate
			
			//update the content of data set
			DataBlock line1 = new DataBlock("1");
			line1.setName("Chain id");
			line1.setDescription(chain.getChainId());
			
			DataBlock line2 = new DataBlock("2");
			line2.setName("Residue count");
			line2.setDescription((new Integer(chain.getResidueCount()).toString()));
			
			DataBlock line3 = new DataBlock("3");
			line3.setName("Ribbon displayed");
			if (callable.getStructureViewer().hasRibbon(chain) > -1){
				line3.setDescription("Yes");
			}
			else{
				line3.setDescription("No");
			}
			
			DataBlock line4 = new DataBlock("4");
			line4.setName("Classification");
			line4.setDescription(chain.getClassification() + " chain");
			
			data.clearData();
			data.addData(line1);
			data.addData(line2);
			data.addData(line3);
			data.addData(line4);
			
			this.remove(contentPanel);
			contentPanel = new TableViewPanel(this, data, callable, searchTypes, TableViewPanel.FIRST_COLUMN_NARROW, false);
			this.add(contentPanel, BorderLayout.CENTER);
			contentPanel.resizePanel(new Dimension(this.getSize().width, 250));
			this.revalidate();
			this.repaint();
		}
		
		
	}
	
	public StructureComponent getActiveComponent(){
		return structureComponent;
	}
	
	public void clearDisplay(boolean update){

		this.remove(contentPanel);
		
		searchTypes.set(1, "None");//don't translate
		
		data.clearData();
		DataBlock block = new DataBlock("");
		block.setEntryId("");
		block.setName("");
		data.addData(block);
		
		contentPanel = new TableViewPanel(this, data, callable, searchTypes, TableViewPanel.FIRST_COLUMN_NARROW, false);
		this.add(contentPanel, BorderLayout.CENTER);
		if (update){
			contentPanel.resizePanel(new Dimension(this.getSize().width, 250));
			this.revalidate();
			this.repaint();
		}
		
	}
	
	public void printData(){
		Thread runner = new Thread(){
			public void run(){
				contentPanel.printData();
			}
		};
		runner.start();
	}
	
}