package edu.sdsc.sirius.viewers.TreeViewer;

import javax.swing.*;

import java.util.*;
import java.awt.*;
import java.awt.event.*;

import javax.swing.tree.*;
import javax.swing.event.*;
import javax.swing.border.*;

import edu.sdsc.mbt.*;
import edu.sdsc.mbt.viewables.*;

import edu.sdsc.sirius.io.*;
import edu.sdsc.sirius.util.*;
import edu.sdsc.sirius.viewers.*;
import edu.sdsc.sirius.viewers.SequenceViewerImpl.*;
import edu.sdsc.sirius.dialogs.*;

public class StructureBrowser extends JPanel implements Viewer, StructureStylesEventListener, SequenceStylesEventListener, StructureComponentEventListener {
	
	private Manager callable;
	private StructureViewer structureViewer;
	private SequenceViewer sequenceViewer;
	
	public static boolean expandSelection = false;
	
	private int width = 300;
	private int height = 450;

	private StructureDocument structureDocument;
	
	private boolean scrollToSelection = true;
	
	private JTree tree;
	private DefaultTreeModel model;
	
	private Vector selectedChainNodes = new Vector();
	private Vector selectedResidueNodes = new Vector();
	private Vector selectedAtomNodes = new Vector();
	private HashMap nodeMap = new HashMap();//StructureComponent -> StructureComponentNode
	
	private Object[] nodes;
	
	private Vector structureNodes = new Vector();
	
	private JScrollPane sp = new JScrollPane();
	
	private JPanel panel = new JPanel();
	private JButton buttonVisibility = new JButton("Display");
	private JButton buttonRendering = new JButton("Render");
	private JButton buttonColor = new JButton("Color");

	private final ImageIcon nodeVisible = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/tree_visible.gif"));//don't translate resource locations
	private final ImageIcon nodeInvisible = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/tree_invisible.gif"));
	
	private final ImageIcon PARENT_OPEN = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/parent_open.gif"));
	private final ImageIcon PARENT_CLOSED = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/parent_closed.gif"));
	
	private final ImageIcon STRUCTURE_OPEN = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/distance_open.gif"));
	private final ImageIcon STRUCTURE_CLOSED = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/distance_closed.gif"));
	
	private final ImageIcon CHAIN_OPEN = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/hb_open.gif"));
	private final ImageIcon CHAIN_CLOSED = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/hb_closed.gif"));
		
	private final ImageIcon RESIDUE_OPEN = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/bumps_open.gif"));
	private final ImageIcon RESIDUE_CLOSED = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/bumps_closed.gif"));

	private final ImageIcon ATOM_LABEL = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/label_visible.gif"));
	private final ImageIcon LABELS_INVISIBLE = new ImageIcon(getClass().getResource("/edu/sdsc/sirius/monitors/label_invisible.gif"));
	
	private JMenuBar menuBar;
	
	private DefaultMutableTreeNode rootNode;
	
	private JPopupMenu atomPopup;
	private JPopupMenu residuePopup;
	private JPopupMenu chainPopup;
	
	private boolean popupActive = false;
	private Atom activeAtom;
	private Residue activeResidue;
	private Chain activeChain;
	
	private JMenuItem chainPopupRibbon;
	private JCheckBoxMenuItem viewInfo;
	private JCheckBoxMenuItem optionsScroll;
	
	private JSplitPane splitPane;
	
	private InfoViewer infoViewer;
	
	private boolean internalSelection = false;

	
	public StructureBrowser(Manager c){
		
		this.callable = c;

		this.structureViewer = callable.getStructureViewer();
		this.sequenceViewer = callable.getSequenceViewer();
		
		setLayout(new BorderLayout());
		
		menuBar = createMenuBar();
		
		atomPopup = createAtomPopupMenu();
		residuePopup = createResiduePopupMenu();
		chainPopup = createChainPopupMenu();
		
		this.add(menuBar, BorderLayout.NORTH);
		
		
		
		createTree();
		
		sp.getViewport().add(tree);
		tree.setBorder(new EmptyBorder(5,5,5,5));
		
		add(sp, BorderLayout.CENTER);
		
		panel.setPreferredSize(new Dimension(250,45));
		panel.setLayout(null);
		
		buttonVisibility.setBounds(10, 10, 75, 25);
		buttonVisibility.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				showDisplay();
			}
		});
		
		buttonRendering.setBounds(90, 10, 75, 25);
		buttonRendering.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				showRender();
			}
		});

		buttonColor.setBounds(170, 10, 75, 25);
		buttonColor.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				showColor();
			}
		});
		
		panel.add(buttonVisibility);
		panel.add(buttonRendering);
		panel.add(buttonColor);
		
		add(panel, BorderLayout.SOUTH);
		
		setVisible(true);
	}
	
	public void processStructureDocumentEvent(StructureDocumentEvent structureDocumentEvent){
		
		try{
			
			int type = structureDocumentEvent.type;
			int change = structureDocumentEvent.change;
			
			if ( type == StructureDocumentEvent.TYPE_VIEWER ){
				if ( change == StructureDocumentEvent.CHANGE_ADDED ){
					structureDocument = structureDocumentEvent.structureDocument;
					viewerAdded(structureDocumentEvent.viewer);
				}
				else if ( change == StructureDocumentEvent.CHANGE_REMOVED ){
					structureDocument = null;
					viewerRemoved(structureDocumentEvent.viewer);
				}
			}
			else if ( type == StructureDocumentEvent.TYPE_DATA ){
			
				Entry e = structureDocumentEvent.entry;
				Structure structure = null;
				if (e instanceof StructureEntry){
					StructureEntry entry = (StructureEntry)e;
					structure = entry.getStructure();
					
					if (structure == null){
						return;
					}
					
					if (structure.getStructureMap() == null || structure.getStructureMap().getAtomCount() == 0){
						return;
					}
					
					if ( change == StructureDocumentEvent.CHANGE_ADDED ){
						if (structure.getStructureMap().getAtomCount() == 0) return;
						structureAdded( structure );
					}
					else if ( change == StructureDocumentEvent.CHANGE_REMOVED ){
						structureRemoved(structure, true);
					}
					else if (change == StructureDocumentEvent.CHANGE_RENAMED){
						//get new name and update the node representation
						String name = structureViewer.getStructureName(structure);
						
						for (int i = 0; i < structureNodes.size(); i++){
							StructureNode node = (StructureNode)structureNodes.get(i);
							if (node.getStructure() == structure){
								node.setName("Structure: " + name);
								break;
							}
						}
						
						tree.updateUI();
					}
					
				}
/*				else if (e instanceof SequenceEntry){
					SequenceEntry entry = (SequenceEntry)e;
					int count = entry.getSequenceCount();
					if (count > 1){
						for (int i = 0; i < count; i++){
							Sequence sequence = entry.getSequence(i);
							if ( change == StructureDocumentEvent.CHANGE_ADDED ){
								sequenceAdded( sequence );
							}
							else if ( change == StructureDocumentEvent.CHANGE_REMOVED ){
								sequenceRemoved( sequence, true );
							}
							else if (change == StructureDocumentEvent.CHANGE_RENAMED){
								//get the new name and apply to the relevant entry
								
							}
						}
					}
					else{
						Sequence sequence = entry.getSequence();
						if ( change == StructureDocumentEvent.CHANGE_ADDED ){
							sequenceAdded( sequence );
						}
						else if ( change == StructureDocumentEvent.CHANGE_REMOVED ){
							sequenceRemoved( sequence, true );
						}
						else if (change == StructureDocumentEvent.CHANGE_RENAMED){
							//get the new name and apply to the relevant entry
							for (int i = 0; i < structureNodes.size(); i++){
								StructureNode node = (StructureNode)structureNodes.get(i);
								if (node.getSequence() == sequence){
									node.setName(sequence.getName());
									break;
								}
							}
							
							tree.updateUI();
						}
					}
					
				}
*/				
				tree.updateUI();
			}
			else if (type == StructureDocumentEvent.TYPE_GLOBAL){
				if (change == StructureDocumentEvent.CHANGE_RELOAD){

					//clear everything and load back from StructureDocument
					Vector expandedChains = new Vector();//Chains to restore the initial state at least to some extent

					for (int i = 0; i < structureNodes.size(); i++){
						StructureNode node = (StructureNode)structureNodes.get(i);
						Structure structure = node.structure;
						Sequence sequence = node.sequence;
						if (structure != null){
							for (int j = 0; j < node.node.getChildCount(); j++){
								DefaultMutableTreeNode nnn = (DefaultMutableTreeNode)node.node.getChildAt(j);
								TreePath p = new TreePath(nnn.getPath());
								if (tree.isExpanded(p)){
									StructureComponentNode sn = (StructureComponentNode)nnn.getUserObject();
									expandedChains.add(sn.component);
								}
							}
							structureRemoved(structure, false);
						}
/*						else if (sequence != null){
							sequenceRemoved(sequence, false);
						}
*/					}
					
					//now load everything back
					ArrayList entries = structureDocument.getEntriesByType(StructureEntry.STRUCTURE_ENTRY);
					for ( int i = 0; i < entries.size(); i++ ){
						Entry entry = (Entry)entries.get( i );
						if (entry.TYPE == StructureEntry.STRUCTURE_ENTRY){
							structureAdded( ((StructureEntry)entry).getStructure() );
						}
					}
					
					entries = structureDocument.getEntriesByType(StructureEntry.FRAGMENT_ENTRY);
					for ( int i = 0; i < entries.size(); i++ ){
						Entry entry = (Entry)entries.get( i );
						if (entry.TYPE == StructureEntry.FRAGMENT_ENTRY){
							structureAdded( ((StructureEntry)entry).getStructure() );
						}
					}
/*					entries = structureDocument.getEntriesByType(StructureEntry.SEQUENCE_ENTRY);
					for ( int i = 0; i < entries.size(); i++ ){
						Entry entry = (Entry)entries.get( i );
						if (entry.TYPE == StructureEntry.SEQUENCE_ENTRY){
							int count = ((SequenceEntry)entry).getSequenceCount();
							if (count > 1){
								for (int j = 0; j < count; j++){
									Sequence sequence = ((SequenceEntry)entry).getSequence(j);
									sequenceAdded( sequence );
								}
							}
							else{
								Sequence sequence = ((SequenceEntry)entry).getSequence();
								sequenceAdded( sequence );
							}
						}
					}
*/					
					//restore expansion status
					if (expandedChains.size() > 0){
						for (int i = 0; i < expandedChains.size(); i++){
							try{
								StructureComponentNode sn = (StructureComponentNode)nodeMap.get(expandedChains.get(i));
								TreePath p = new TreePath(sn.node.getPath());
								tree.expandPath(p);
							}
							catch (Exception e){
								e.printStackTrace();
							}
						}
					}
					
					tree.updateUI();
					revalidate();
					repaint();
					
				}

			}
		}
		catch (Exception ex){
			structureDocument.parent.displayExceptionMessage("Exception processing StructureDocumentEvent", ex);
		}
	}
	
	private void viewerAdded( Viewer viewer )
	{
		if ( viewer != this ) return;
		
		ArrayList entries = structureDocument.getEntriesByType(StructureEntry.STRUCTURE_ENTRY);
		for ( int i = 0; i < entries.size(); i++ ){
			Entry entry = (Entry)entries.get( i );
			if (entry.TYPE == StructureEntry.STRUCTURE_ENTRY){
				structureAdded( ((StructureEntry)entry).getStructure() );
			}
		}
		
		entries = structureDocument.getEntriesByType(StructureEntry.FRAGMENT_ENTRY);
		for ( int i = 0; i < entries.size(); i++ ){
			Entry entry = (Entry)entries.get( i );
			if (entry.TYPE == StructureEntry.FRAGMENT_ENTRY){
				structureAdded( ((StructureEntry)entry).getStructure() );
			}
		}
		tree.updateUI();

	}

	/**
	 *  A Viewer was just removed from a StructureDocument.
	 */
	private void viewerRemoved( Viewer viewer )
	{
		if ( viewer != this ) return;
		
		ArrayList entries = structureDocument.getEntriesByType(StructureEntry.STRUCTURE_ENTRY);
		for ( int i = 0; i < entries.size(); i++ )
		{
			Entry entry = (Entry)entries.get( i );
			structureRemoved( ((StructureEntry)entry).getStructure(), true );
		}
		
/*		entries = structureDocument.getEntriesByType(StructureEntry.SEQUENCE_ENTRY);
		for ( int i = 0; i < entries.size(); i++ )
		{
			Entry entry = (Entry)entries.get( i );
			sequenceRemoved( ((SequenceEntry)entry).getSequence(), true );
		}
*/
	}
	
	private void structureAdded(Structure structure){
		
		if ( structure == null ) return;
		
		int counter = nodes.length;//total number of nodes in the tree at the beginning
		
		Vector newNodes = new Vector();
		for (int i = 0; i < nodes.length; i++){
			newNodes.add(nodes[i]);
		}
		
		//add structure node
		StructureNode snn = new StructureNode("Structure: " + structureViewer.getStructureName(structure), structure);
		DefaultMutableTreeNode sn = new DefaultMutableTreeNode(snn);
		newNodes.add(sn);
		
		snn.node = sn;
		
		rootNode.add(sn);
		

		
		structureNodes.add(snn);
		
		StructureMap map = structure.getStructureMap();
		for (int i = 0; i < map.getChainCount(); i++){
			Chain c = map.getChain(i);
			
			//create a chain node
			StructureComponentNode scn = new StructureComponentNode("Chain " + c.getChainId(), c, true);
			DefaultMutableTreeNode cn = new DefaultMutableTreeNode(scn);
			sn.add(cn);
			newNodes.add(cn);
			scn.node = cn;
//			scn.path = new TreePath(cn.getPath());
			scn.visible = c.visible;
			scn.selected = map.getStructureStyles().isSelected(c);
			

			
			nodeMap.put(c, scn);
			
			for (int j = 0; j < c.getResidueCount(); j++){
				Residue r = c.getResidue(j);
				StructureComponentNode scn2 = new StructureComponentNode("Residue " + r.getCompoundCode() + r.getResidueId(), r, true);
				DefaultMutableTreeNode rn = new DefaultMutableTreeNode(scn2);
				cn.add(rn);
				newNodes.add(rn);
				scn2.node = rn;
//				scn2.path = new TreePath(rn.getPath());
				nodeMap.put(r, scn2);
				
				scn2.visible = r.visible;
				scn2.selected = map.getStructureStyles().isSelected(r);
				
				for (int k = 0; k < r.getAtomCount(); k++){
					Atom a = r.getAtom(k);
					String label = null;
					if (a.forceFieldType == null){
						label = a.name;
					}
					else{
						label = a.name + " [" + a.forceFieldType + "]";
					}
					StructureComponentNode scn3 = new StructureComponentNode(label, a, false);
					DefaultMutableTreeNode an = new DefaultMutableTreeNode(scn3);
					rn.add(an);
					newNodes.add(an);
					scn3.node = an;
					scn3.visible = a.visible;
					scn3.selected = map.getStructureStyles().isSelected(a);
					nodeMap.put(a, scn3);
				}
			}
		}
	
		map.getStructureStyles().addStructureStylesEventListener(this);
		map.getStructureStyles().addStructureComponentEventListener(this);
		
		nodes = new DefaultMutableTreeNode[newNodes.size()];
		
		for (int i = 0; i < newNodes.size(); i++){
			nodes[i] = (DefaultMutableTreeNode)newNodes.get(i);
		}
		
		TreePath p = new TreePath(sn.getPath());
		tree.expandPath(p);
		
	}
	
	private void structureRemoved(Structure structure, boolean update){
		
		if ( structure == null ) return;
		
		
		//check how many structures are loaded
		//if only one, just do a complete clear and avoid traversing the whole tree
		if (structureNodes.size() == 1){
			nodeMap.clear();
			structureNodes.clear();
			
			for (int i = 0; i < nodes.length; i++){
				if (nodes[i] == rootNode) continue;
				((DefaultMutableTreeNode)nodes[i]).setUserObject(null);
				((DefaultMutableTreeNode)nodes[i]).removeFromParent();
			}
			
			nodes = null;
			rootNode.removeAllChildren();

			nodes = new DefaultMutableTreeNode[1];
			nodes[0] = rootNode;
		}
		else{
			int index = -1;
		
			Vector removedNodes = new Vector();
			
			DefaultMutableTreeNode node = null;
			//find the structure node
			for (int i = 0; i < structureNodes.size(); i++){
				StructureNode sn = (StructureNode)structureNodes.get(i);
				if (sn.getStructure() == structure){
					node = sn.node;
					index = i;
					break;
				}
			}
	
			if (node == null) return;
			
			//remove the tree
			for (int i = node.getChildCount()-1; i >= 0; i--){
				DefaultMutableTreeNode cn = (DefaultMutableTreeNode)model.getChild(node, i);
				StructureComponentNode scn = (StructureComponentNode)cn.getUserObject();
				nodeMap.remove(scn.component);
				cn.setUserObject(null);
	//			System.out.println("Chain node = " + cn);
				for (int j = model.getChildCount(cn)-1; j >= 0 ; j--){
					DefaultMutableTreeNode comp2 = (DefaultMutableTreeNode)model.getChild(cn, j);
					StructureComponentNode rn = (StructureComponentNode)comp2.getUserObject();
					nodeMap.remove(rn.component);
					comp2.setUserObject(null);
					for (int k = model.getChildCount(comp2)-1; k >= 0 ; k--){
						DefaultMutableTreeNode comp3 = (DefaultMutableTreeNode)model.getChild(comp2, k);
						StructureComponentNode an = (StructureComponentNode)comp3.getUserObject();
						comp2.remove(comp3);
						removedNodes.add(comp3);
						nodeMap.remove(an.component);
						comp3.setUserObject(null);
					}
					
					cn.remove(comp2);
					removedNodes.add(comp2);
					
				}
				
				node.remove(cn);
				removedNodes.add(cn);
				
			}
			
			node.removeAllChildren();
			
			rootNode.remove(index);
			removedNodes.add(node);
			
			Object[] newNodes = new Object[nodes.length - removedNodes.size()];
			int counter = 0;
			for (int i = 0; i < nodes.length; i++){
				if (removedNodes.contains(nodes[i])){
					continue;
				}
				newNodes[counter++] = nodes[i];
			}
			nodes = null;
			nodes = newNodes;
			
			structureNodes.remove(index);

		}
		
		if (update){
			tree.updateUI();
			
			TreePath path = new TreePath(rootNode.getPath());
			tree.expandPath(path);
			tree.revalidate();
			tree.repaint();
		}
		
		if (infoViewer != null && infoViewer.isVisible()){
			if (infoViewer.getActiveComponent() != null){
				if (infoViewer.getActiveComponent().structure == structure) infoViewer.clearDisplay(true);
			}
		}
		
		
		// Remove the StructureStylesEventListener so don't recieve updates.
		StructureMap structureMap = structure.getStructureMap( );
		StructureStyles structureStyles = structureMap.getStructureStyles( );
		structureStyles.removeStructureStylesEventListener( this );
		structureStyles.removeStructureComponentEventListener( this );
	}
	
	public void sequenceAdded(Sequence sequence){
		
		if ( sequence == null ) return;

		int counter = nodes.length;//total number of nodes in the tree at the beginning
		
		int idCounter = nodes.length;
		
		Vector newNodes = new Vector();
		for (int i = 0; i < nodes.length; i++){
			newNodes.add(nodes[i]);
		}
		
		//add structure node
		StructureNode snn = new StructureNode("Sequence: " + sequence.getName(), sequence);
		DefaultMutableTreeNode sn = new DefaultMutableTreeNode(snn);
		newNodes.add(sn);
		snn.node = sn;
		
		rootNode.add(sn);
		
		structureNodes.add(snn);
		
		String seq = sequence.getSequence();
		Cell[] cells = sequence.getCells();
		int count = 1;
		for (int i = 0; i < cells.length; i++){
			String r = cells[i].symbol;
			if (r.equals("-")) continue;
			//create a chain node
			StructureComponentNode scn = new StructureComponentNode("Residue " + r + count, cells[i]);
			DefaultMutableTreeNode cn = new DefaultMutableTreeNode(scn);
			sn.add(cn);
//			scn.path = new TreePath(cn.getPath());
			newNodes.add(cn);
			scn.node = cn;
			
			nodeMap.put(cells[i], scn);
			count++;
		}
	
		sequence.getSequenceStyles().addSequenceStylesEventListener(this);
		sequence.getSequenceStyles().addStructureComponentEventListener(this);
		
		nodes = new DefaultMutableTreeNode[newNodes.size()];
		
		for (int i = 0; i < newNodes.size(); i++){
			nodes[i] = (DefaultMutableTreeNode)newNodes.get(i);
		}
		
		TreePath path = new TreePath(rootNode.getPath());
		tree.expandPath(path);
		
	}
	
	public void sequenceRemoved(Sequence sequence, boolean update){
		
		if ( sequence == null ) return;
		
		Vector newNodes = new Vector();
		for (int i = 0; i < nodes.length; i++){
			newNodes.add(nodes[i]);
		}
		
		int counter = nodes.length;
		
		DefaultMutableTreeNode node = null;
		int index = -1;
		//find the sequence node
		for (int i = 0; i < structureNodes.size(); i++){
			StructureNode sn = (StructureNode)structureNodes.get(i);
			if (sn.getSequence() == sequence){
				node = sn.node;
				index = i;
				break;
			}
		}

		if (node == null) return;
		
		//remove the tree
		for (int i = node.getChildCount()-1; i >= 0; i--){
			DefaultMutableTreeNode cn = (DefaultMutableTreeNode)model.getChild(node, i);
			StructureComponentNode scn = (StructureComponentNode)cn.getUserObject();
						
			node.remove(cn);
			newNodes.remove(cn);
			nodeMap.remove(scn.cell);
			
		}
		
		rootNode.remove(index);
		newNodes.remove(node);
		
		nodes = new DefaultMutableTreeNode[newNodes.size()];
		for (int i = 0; i < newNodes.size(); i++){
			nodes[i] = (DefaultMutableTreeNode)newNodes.get(i);
		}
		
		if (update){
			tree.updateUI();
			TreePath path = new TreePath(rootNode.getPath());
			tree.expandPath(path);
			tree.revalidate();
			tree.repaint();
		}
		
		
		SequenceStyles sequenceStyles = sequence.getSequenceStyles( );
		sequenceStyles.removeSequenceStylesEventListener( this );
//		structureStyles.removeStructureComponentEventListener( this );
		structureNodes.remove(index);
		
		
	}
	
	private void clearSelection(){
		
		for (int i = 0; i < structureNodes.size(); i++){
			StructureNode n = (StructureNode)structureNodes.get(i);
			Structure structure = n.structure;
			if (structure != null){
				structure.getStructureMap().getStructureStyles().selectNone(true, false);
			}
			else{
				Sequence sequence = n.sequence;
				sequence.getSequenceStyles().selectNone(false, true);
			}
		}

		//clear selection
		for (int i = selectedChainNodes.size()-1; i >= 0 ; i--){
			StructureComponentNode cn = (StructureComponentNode)selectedChainNodes.get(i);									
			selectedChainNodes.remove(cn);
			cn.selected = false;
			Chain c = (Chain)cn.component;
			for (int j = 0; j < c.getResidueCount(); j++){
				Residue r = c.getResidue(j);
				StructureComponentNode rn = (StructureComponentNode)nodeMap.get(r);
				rn.selected = false;
				selectedResidueNodes.remove(rn);
				for (int k = 0; k < r.getAtomCount(); k++){
					Atom a = r.getAtom(k);
					StructureComponentNode an = (StructureComponentNode)nodeMap.get(a);
					an.selected = false;
					selectedAtomNodes.remove(an);
				}
			}
		}
		
		//remaining selected residues (individually, except those under the chain in question)
		for (int i = selectedResidueNodes.size()-1; i >= 0; i--){
			StructureComponentNode rn = (StructureComponentNode)selectedResidueNodes.get(i);									
			Residue r = (Residue)rn.component;
			for (int k = 0; k < r.getAtomCount(); k++){
				Atom a = r.getAtom(k);
				StructureComponentNode an = (StructureComponentNode)nodeMap.get(a);
				an.selected = false;
				selectedAtomNodes.remove(an);
			}
			selectedResidueNodes.remove(rn);
			rn.selected = false;
		}
		
		for (int i = selectedAtomNodes.size()-1; i >= 0; i--){
			StructureComponentNode an = (StructureComponentNode)selectedAtomNodes.get(i);							
			Atom a = (Atom)an.component;
			selectedAtomNodes.remove(an);
			an.selected = false;
		}
		
		tree.setSelectionPaths(null);
		
		callable.updateView();
		tree.updateUI();
		
	}
	
	private void setGlobalDeselect(){
		for (int i = 0; i < structureNodes.size(); i++){
			StructureNode n = (StructureNode)structureNodes.get(i);
			Structure structure = n.structure;
			if (structure != null){
				structure.getStructureMap().getStructureStyles().selectNone(true, false);
			}
			else{
				Sequence sequence = n.sequence;
				sequence.getSequenceStyles().selectNone(false, true);
			}
		}
	}
	
	private void createTree(){
		
		nodes = new DefaultMutableTreeNode[1];
		
		rootNode = new DefaultMutableTreeNode(new StructureComponentNode("Structures", null, true));
		nodes[0] = rootNode;
		
		model = new DefaultTreeModel(rootNode);
		tree = new JTree(model);
		tree.getSelectionModel().setSelectionMode( TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION );

		
		String os = System.getProperty("os.name");
		DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
		TreeCellRenderer customRenderer = new IconCellRenderer();
		
//		if (os.startsWith("Mac")){
//			tree.setCellRenderer(renderer);
//		}
//		else{
			tree.setCellRenderer(customRenderer);
//		}
		
		tree.setEditable(false);
		
		tree.setSelectionPath(null);
		tree.addMouseListener(new MouseInputAdapter(){
			public void mouseClicked(MouseEvent e){
				
				TreePath path = tree.getPathForLocation(e.getX(), e.getY());
				if (path == null) return;
				
				DefaultMutableTreeNode node = (DefaultMutableTreeNode)path.getLastPathComponent();
				
				if (e.isPopupTrigger()){
					processPopup(e, node);
					return;
				}
				
				if (popupActive){
					popupActive = false;
					return;
				}

				try{
					
					StructureComponentNode scn = (StructureComponentNode)node.getUserObject();
					StructureComponent sc = scn.component;
					
					
					if (sc == null){
						//it's a node from Sequence object
						//get the Sequence of this node
						DefaultMutableTreeNode seqnode = (DefaultMutableTreeNode)scn.node.getParent();
						StructureNode sequenceNode = (StructureNode)seqnode.getUserObject();
						Sequence sequence = sequenceNode.sequence;
						if (scn.selected){
							//deselect this residue
							sequence.getSequenceStyles().setSelected(scn.cell, false, true, false);
						}
						else{
							//select residue
							if (!e.isControlDown()){
								sequence.getSequenceStyles().selectNone(true, true);
							}
							sequence.getSequenceStyles().setSelected(scn.cell, true, true, false);
						}
						
						callable.updateView();
						
						return;
					}
					
					if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
						Chain chain = (Chain)sc;
						
						if (callable.getDisplayDialog() != null){
							callable.getDisplayDialog().processPick(DataService.componentToLabel(chain), chain);
							return;
						}
						
						
						internalSelection = true;
						StructureMap structureMap = chain.structure.getStructureMap();
						if (scn.selected){
							//this chain just got deselected
							chain.structure.getStructureMap().getStructureStyles().setSelected(chain, false, true, false);
						}
						else{
							//this chain just got selected
							if (!e.isControlDown()){
								chain.structure.getStructureMap().getStructureStyles().selectNone(true, true);
								setGlobalDeselect();
							}
							chain.structure.getStructureMap().getStructureStyles().setSelected(chain, true, true, false);
						}
						chain.structure.getStructureMap().getStructureStyles().updateChain(chain, true);
						internalSelection = false;
						callable.updateView();
					}
					else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
						Residue residue = (Residue)sc;
						
						if (callable.getDisplayDialog() != null){
							callable.getDisplayDialog().processPick(DataService.componentToLabel(residue), residue);
							return;
						}

						if (scn.selected){
							//deselect this residue
							residue.structure.getStructureMap().getStructureStyles().setSelected(residue, false, true, false);
						}
						else{
							if (!e.isControlDown()){
//								residue.structure.getStructureMap().getStructureStyles().selectNone(true, true);
								setGlobalDeselect();
							}
							residue.structure.getStructureMap().getStructureStyles().setSelected(residue, true, true, false);
						}
						
						//update the chain involved
						Chain chain = residue.structure.getStructureMap().getChain(residue.getAtom(0));
						residue.structure.getStructureMap().getStructureStyles().updateChain(chain, true);
//						residue.structure.getStructureMap().getStructureStyles().registerChainSelection(chain, scn.selected);
						
						callable.updateView();
					}
					else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
						Atom atom = (Atom)sc;

						if (callable.getDisplayDialog() != null){
							callable.getDisplayDialog().processPick(DataService.componentToLabel(atom), atom);
							return;
						}
						
						if (scn.selected){
							atom.structure.getStructureMap().getStructureStyles().setSelected(atom, false, true, false);
						}
						else{
							if (!e.isControlDown()){
								//deselect everything except the atom in question
//								atom.structure.getStructureMap().getStructureStyles().selectNone(true, false);
								setGlobalDeselect();
							}
							atom.structure.getStructureMap().getStructureStyles().setSelected(atom, true, true, false);
						}
						callable.updateView();
					}
					
					
				}
				catch (ClassCastException ex){
//					ex.printStackTrace();
					//it's either root or structure node
					try{
						StructureNode sn = (StructureNode)node.getUserObject();
						//check if it's a Sequence: if so, the sequence needs to be selected
						if (sn.getSequence() != null){
							Sequence sequence = sn.getSequence();
							
							/**
							 * TODO total selection of Sequence
							 */
							
/*							if (sn.selected){
								sequence.getSequenceStyles().selectNone(true, true);
								sn.selected = false;
							}
							else{
								if (!e.isControlDown()){
									clearSelection();
								}
								
								sequence.getSequenceStyles().selectAll();
								sn.selected = true;
							}
*/							
						}
					}
					catch (ClassCastException exx){
						
					}
				}
				
				callable.updateView();
			}
			
			public void mousePressed(MouseEvent e){
				
				TreePath path = tree.getPathForLocation(e.getX(), e.getY());
				if (path == null) return;
				
				DefaultMutableTreeNode node = (DefaultMutableTreeNode)path.getLastPathComponent();
				
				if (e.isPopupTrigger()){
					popupActive = true;
					processPopup(e, node);
					return;
				}
				
				
			}
			
			public void mouseReleased(MouseEvent e){
				
				TreePath path = tree.getPathForLocation(e.getX(), e.getY());
				if (path == null) return;
				
				DefaultMutableTreeNode node = (DefaultMutableTreeNode)path.getLastPathComponent();
				
				if (e.isPopupTrigger()){
					popupActive = true;
					processPopup(e, node);
					return;
				}
				
			}
		});
		
		tree.addTreeExpansionListener(new TreeExpansionListener(){
			public void treeExpanded(TreeExpansionEvent event){
				//check whether the newly shown nodes are selected and add them to the selection path if necessary
			}
			
			public void treeCollapsed(TreeExpansionEvent event){
				
			}
		});
		
		tree.setExpandsSelectedPaths(expandSelection);
		tree.setRootVisible(false);
	
	}
		
	class StructureComponentNode {
		
		protected String m_name;
		protected StructureComponent component;
		
		protected Icon nodeIconVisible = nodeVisible;
		protected Icon nodeIconInvisible = nodeInvisible;
		
		public DefaultMutableTreeNode node;
		
		private boolean parentNode = false;
		
		public boolean selected = false;
		public boolean visible = true;
		
//		public TreePath path;
		
		public Cell cell;//for Sequence components
		
		public StructureComponentNode(String name, StructureComponent component, boolean parent){
			m_name = name;
			this.component = component;
			this.parentNode = parent;
		}
		
		/**
		 * This constructor is for residue nodes in Sequence objects, with no real data structures attached
		 * @param id
		 * @param name
		 * @param index
		 */
		public StructureComponentNode(String name, Cell cell){
			m_name = name;
			this.cell = cell;
		}
				
		public String getName(){
			return m_name;
		}
		
		public void setName(String name){
			m_name = name;
		}
		
		public String toString(){
			return m_name;
		}
				
		public StructureComponent getStructureComponent(){
			return component;
		}
		
		public Icon getExpandedIcon(){
			if (this == rootNode.getUserObject()){
				return PARENT_OPEN;
			}
			
			if (component == null){
				return CHAIN_CLOSED;//it's a sequence residue
			}
			
			if (component.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
				return CHAIN_OPEN;
			}
			else if (component.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				return RESIDUE_OPEN;
			}
			else if (component.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
				return ATOM_LABEL;
			}

			return PARENT_OPEN;
		}
		
		public Icon getIcon(){
			if (this == rootNode.getUserObject()){
				//check whether it's collapsed or not
				TreePath p0 = new TreePath(rootNode.getPath());
				if (tree.isExpanded(p0)){
					return PARENT_OPEN;
				}
				else{
					return PARENT_CLOSED;
				}
			}
			
			if (component == null){
				return CHAIN_CLOSED;
			}
			
			if (component.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
				TreePath p0 = new TreePath(node.getPath());
				if (tree.isExpanded(p0)){
					return CHAIN_OPEN;
				}
				else{
					return CHAIN_CLOSED;
				}
			}
			else if (component.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				TreePath p0 = new TreePath(node.getPath());
				if (tree.isExpanded(p0)){
					return RESIDUE_OPEN;
				}
				else{
					return RESIDUE_CLOSED;
				}
			}
			else if (component.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
				return ATOM_LABEL;
			}

			
			return PARENT_OPEN;
		}
	}
	
	/**
	 * Top level structure/sequence node that can be a parent to the lower-level structure components
	 * @author Sasha Buzko
	 *
	 */
	class StructureNode {
		
		protected String m_name;
		protected Structure structure;
		
		protected Sequence sequence;
		
		public DefaultMutableTreeNode node;

		protected Icon nodeIconVisible = nodeVisible;
		protected Icon nodeIconInvisible = nodeInvisible;
		
		private boolean parentNode = false;
		
		public boolean selected = false;
		public boolean visible = true;
		
		public StructureNode(String name, Structure structure){
			m_name = name;
			this.structure = structure;
		}
		
		public StructureNode(String name, Sequence sequence){
			m_name = name;
			this.sequence = sequence;
		}

		public String getName(){
			return m_name;
		}
		
		public void setName(String name){
			m_name = name;
		}
		
		public String toString(){
			return m_name;
		}
		
		public Structure getStructure(){
			return structure;
		}
		
		public Icon getExpandedIcon(){
			return PARENT_OPEN;
		}
		
		public Icon getIcon(){
			TreePath p0 = new TreePath(node.getPath());
			if (tree.isExpanded(p0)){
				return PARENT_OPEN;
			}
			else{
				return PARENT_CLOSED;
			}
		}
		
		public Sequence getSequence(){
			return sequence;
		}
	}
	
	private boolean processSelectionEvent = true;

	
	class IconCellRenderer extends DefaultTreeCellRenderer {
		
		protected Color textSelectioColor;
		protected Color textNonSelectionColor;
		protected Color bkSelectionColor;
		protected Color bkNonSelectionColor;
		protected Color borderSelectionColor;
		
		protected Color textInvisibleColor = new Color(0.6f, 0.6f, 0.6f);
		
		protected boolean selected;
		protected boolean visible;

		/**
		 * The renderer constructor.
		 */
		public IconCellRenderer() {
			
			super();
			
			textSelectionColor = UIManager.getColor("Tree.selectionForeground");//don't translate Tree variables
//			textSelectionColor = Color.black;
			textNonSelectionColor = UIManager.getColor("Tree.textForeground");
//			textNonSelectionColor = Color.red;
			bkSelectionColor = UIManager.getColor("Tree.selectionBackground");
//			bkSelectionColor = Color.red;
			bkNonSelectionColor = UIManager.getColor("Tree.textBackground");
//	/		bkNonSelectionColor = TreeViewer.bgColor;
			borderSelectionColor = UIManager.getColor("Tree.selectionBorderColor");
			setOpaque(false);
		}

		/**
		 * Returns the renderer based on the current situation in the tree representation.
		 */
		public Component getTreeCellRendererComponent(JTree tree,
			Object value, boolean sel, boolean expanded, boolean leaf,
			int row, boolean hasFocus) {

			// Invoke default implementation
			Component result = null;
			DefaultMutableTreeNode node =
				(DefaultMutableTreeNode)value;
			Object obj = node.getUserObject();
			
			try{
				StructureComponentNode mn = (StructureComponentNode)obj;
				selected = mn.selected;
				result = super.getTreeCellRendererComponent(tree,value, selected, expanded, leaf, row, hasFocus);
				setText(mn.getName());
				visible = mn.visible;
			}
			catch (ClassCastException e){
				StructureNode sn = (StructureNode)obj;
				selected = sn.selected;
				visible = sn.visible;
				result = super.getTreeCellRendererComponent(tree, value, selected, expanded, leaf, row, hasFocus);
				setText(sn.getName());
			}
			catch (Exception eee){
				eee.printStackTrace();
			}
			
			
			if (obj instanceof StructureComponentNode) {
				StructureComponentNode idata = (StructureComponentNode)obj;
				setIcon(idata.getIcon());
			}
			else if (obj instanceof StructureNode){
				StructureNode idata = (StructureNode)obj;
				setIcon(idata.getIcon());
			}
			else{
				setIcon(null);
			}
			
			Font font = new Font("Helvetica", Font.PLAIN, 12);
			setFont(font);
			if (visible){
				setForeground(selected ? textSelectionColor : textNonSelectionColor);
				setBackground(selected ? bkSelectionColor : bkNonSelectionColor);
			}
			else{
				setForeground(textInvisibleColor);
				setBackground(selected ? bkSelectionColor : bkNonSelectionColor);

			}
	
			setBackgroundNonSelectionColor(new Color(0.0f,0.0f,0.0f,0.0f));
	
			return result;
		}
		
	}

	public void updateView(){
		tree.updateUI();
	}
	
	public void processStructureStylesEvent(StructureStylesEvent event){
		
		StructureComponent component = event.structureComponent;
		
		if (component == null){
			//it's a global event
			if (event.attribute == StructureStyles.ATTRIBUTE_SELECTION){
				
				//either everything is selected or everything deselected
				for (int i = 0; i < nodes.length; i++){
					try{
						StructureComponentNode n = (StructureComponentNode)((DefaultMutableTreeNode)nodes[i]).getUserObject();
						StructureComponent sc = n.component;
						if (sc == null){
							if (n.cell == null) continue;
							n.selected = n.cell.selected;
							if (n.selected){
								selectedResidueNodes.add(n);
							}
							else{
								selectedResidueNodes.remove(n);
							}
						}
						else{
							n.selected = sc.structure.getStructureMap().getStructureStyles().isSelected(sc);
							if (n.selected){
								if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
									selectedChainNodes.add(n);
								}
								else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
									selectedResidueNodes.add(n);
								}
								else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
									selectedAtomNodes.add(n);
								}
							}
							else{
								if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
									selectedChainNodes.remove(n);
								}
								else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
									selectedResidueNodes.remove(n);
								}
								else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
									selectedAtomNodes.remove(n);
								}
							}
						}
					}
					catch (Exception e){
//						e.printStackTrace();
					}
				}
				try{
					tree.setSelectionPaths(null);
				}
				catch (Exception e){}
			}
			else if (event.attribute == StructureStyles.ATTRIBUTE_VISIBILITY){
				//walk over all nodes
				for (int i = 0; i < nodes.length; i++){
					try{
						StructureComponentNode n = (StructureComponentNode)((DefaultMutableTreeNode)nodes[i]).getUserObject();
						StructureComponent sc = n.component;
						if (sc == null){
							n.visible = true;//sequence can't be hidden
						}
						else{
							n.visible = sc.isVisible();
						}
					}
					catch (Exception e){}
				}
			}
			
			tree.updateUI();
			
			return;
		}
		
		
		try{
			
		
		if (component.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
			Chain chain = (Chain)component;
			if (event.attribute == StructureStyles.ATTRIBUTE_SELECTION){
				boolean selected = component.structure.getStructureMap().getStructureStyles().isSelected(component);
				if (selected){
					StructureComponentNode scn = (StructureComponentNode)nodeMap.get(chain);
					if (scn == null) return;
					selectedChainNodes.add(scn);
					scn.selected = true;
					for (int i = 0; i < chain.getResidueCount(); i++){
						Residue r = chain.getResidue(i);
						StructureComponentNode rn = (StructureComponentNode)nodeMap.get(r);
						if (rn.selected) continue;
						rn.selected = true;
						selectedResidueNodes.add(rn);
						
						for (int j = 0; j < r.getAtomCount(); j++){
							Atom a = r.getAtom(j);
							StructureComponentNode an = (StructureComponentNode)nodeMap.get(a);
							if (an.selected) continue;
							an.selected = true;
							selectedAtomNodes.add(an);
						}
					}
					
//					if (!internalSelection && scrollToSelection){
//						tree.scrollPathToVisible(new TreePath(scn.node.getPath()));
//					}

				}
				else{
					StructureComponentNode scn = (StructureComponentNode)nodeMap.get(chain);
					if (scn == null) return;
					selectedChainNodes.remove(scn);
					scn.selected = false;
					
					//scan the children
					for (int i = 0; i < chain.getResidueCount(); i++){
						Residue r = chain.getResidue(i);
						StructureComponentNode rn = (StructureComponentNode)nodeMap.get(r);
						rn.selected = false;
						selectedResidueNodes.remove(rn);
						for (int k = 0; k < r.getAtomCount(); k++){
							Atom a = r.getAtom(k);
							StructureComponentNode an = (StructureComponentNode)nodeMap.get(a);
							an.selected = false;
							selectedAtomNodes.remove(an);
						}
					}

				}
				
				tree.setSelectionPaths(null);

			}
			else if (event.attribute == StructureStyles.ATTRIBUTE_VISIBILITY){
				StructureComponentNode scn = (StructureComponentNode)nodeMap.get(chain);
				if (scn == null) return;
				scn.visible = chain.visible;
				for (int j = 0; j < chain.getResidueCount(); j++){
					Residue r = chain.getResidue(j);
					StructureComponentNode scn2 = (StructureComponentNode)nodeMap.get(r);
					scn2.visible = r.visible;
					
					for (int i = 0; i < r.getAtomCount(); i++){
						Atom a = r.getAtom(i);
						StructureComponentNode scn3 = (StructureComponentNode)nodeMap.get(a);
						scn3.visible = a.visible;
					}
				}

			}
		}
		else if (component.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
			Residue residue = (Residue)component;
			if (event.attribute == StructureStyles.ATTRIBUTE_SELECTION){
				if (residue.structure.getStructureMap().getStructureStyles().isSelected(residue)){
					StructureComponentNode scn = (StructureComponentNode)nodeMap.get(residue);
					if (scn == null) return;
					if (scn.selected) return;//residue is already selected
					selectedResidueNodes.add(scn);
					scn.selected = true;
					for (int i = 0; i < residue.getAtomCount(); i++){
						Atom a = residue.getAtom(i);
						StructureComponentNode an = (StructureComponentNode)nodeMap.get(a);
						if (an.selected) continue;
						an.selected = true;
						selectedAtomNodes.add(an);
					}
					
					//check whether all residues in its parent chain are now selected
					//and mark the chain as selected
					Chain chain = residue.structure.getStructureMap().getChain(residue.getAtom(0));
					StructureMap map = chain.structure.getStructureMap();
					boolean selected = true;
					for (int i = 0; i < chain.getResidueCount(); i++){
						Residue rr = chain.getResidue(i);
						if (!map.getStructureStyles().isSelected(rr)){
							selected = false;
							break;
						}
					}
					
					if (selected){
						//select the chain
						StructureComponentNode scn3 = (StructureComponentNode)nodeMap.get(chain);
						if (scn3.selected) return;
						selectedChainNodes.add(scn3);
						scn3.selected = true;
					}
					
//					if (!internalSelection && scrollToSelection){
//						tree.scrollPathToVisible(new TreePath(scn.node.getPath()));
//					}

				}
				else{
					StructureComponentNode scn = (StructureComponentNode)nodeMap.get(residue);
					if (scn == null) return;
					if (!scn.selected) return;

					selectedResidueNodes.remove(scn);
					scn.selected = false;
					for (int i = 0; i < residue.getAtomCount(); i++){
						Atom a = residue.getAtom(i);
						StructureComponentNode an = (StructureComponentNode)nodeMap.get(a);
						if (!an.selected) continue;
						an.selected = false;
						selectedAtomNodes.remove(an);
					}
					
					//also deselect its parent chain, since with an incomplete residue selection it can't be considered selected
					Chain chain = residue.structure.getStructureMap().getChain(residue.getAtom(0));
					StructureComponentNode c = (StructureComponentNode)nodeMap.get(chain);
//					System.out.println("chain selection = " + c.selected);
					if (!c.selected) return;
					c.selected = false;
					selectedChainNodes.remove(c);

				}
				
				tree.setSelectionPaths(null);

			}
			else if (event.attribute == StructureStyles.ATTRIBUTE_VISIBILITY){
//				System.out.println("residue visibility = " + residue.visible);
				StructureComponentNode scn = (StructureComponentNode)nodeMap.get(residue);
				if (scn == null) return;
				scn.visible = residue.visible;
				
				for (int i = 0; i < residue.getAtomCount(); i++){
					Atom a = residue.getAtom(i);
					StructureComponentNode scn2 = (StructureComponentNode)nodeMap.get(a);
					scn2.visible = a.visible;
				}
				
				Chain chain = residue.structure.getStructureMap().getChain(residue.getAtom(0));
				StructureComponentNode scn3 = (StructureComponentNode)nodeMap.get(chain);
				if (scn.visible){
					//make the chain visible
					scn3.visible = true;
				}
				else{
					//check other residues in this chain: if nothing else is visible, hide the chain (if at least one is visible, show it)
					StructureMap map = chain.structure.getStructureMap();
					boolean visible = false;
					for (int i = 0; i < chain.getResidueCount(); i++){
						Residue rr = chain.getResidue(i);
						if (rr.visible){
							visible = true;
							break;
						}
					}
					
					scn3.visible = visible;
				}
				
			}
			else if (event.attribute == StructureStyles.ATTRIBUTE_STYLE){
				if (event.property == StructureStyles.PROPERTY_COLOR){
					
					//deselect the residue
					StructureComponentNode scn = (StructureComponentNode)nodeMap.get(residue);
					scn.selected = false;
					selectedResidueNodes.remove(scn);
					
					//check the atoms
					for (int i = 0; i < residue.getAtomCount(); i++){
						Atom a = residue.getAtom(i);
						StructureComponentNode an = (StructureComponentNode)nodeMap.get(a);
						an.selected = false;
						selectedAtomNodes.remove(an);
					}
					
					//also deselect its parent chain, since with an incomplete residue selection it can't be considered selected
					Chain chain = residue.structure.getStructureMap().getChain(residue.getAtom(0));
					StructureComponentNode c = (StructureComponentNode)nodeMap.get(chain);
					if (!c.selected) return;
					c.selected = false;
					selectedChainNodes.remove(c);

					
				}
				tree.setSelectionPaths(null);
			}
			
		}
		else if (component.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
			Atom atom = (Atom)component;
			if (event.attribute == StructureStyles.ATTRIBUTE_SELECTION){
				if (atom.structure.getStructureMap().getStructureStyles().isSelected(atom)){
					
					StructureComponentNode scn = (StructureComponentNode)nodeMap.get(atom);
					if (scn == null) return;
					if (scn.selected) return;
					selectedAtomNodes.add(scn);
					scn.selected = true;
					
					//check the parents (residue and chain)
					Residue r = atom.residue;
					StructureMap map = r.structure.getStructureMap();
					boolean selected = true;
					for (int i = 0; i < r.getAtomCount(); i++){
						Atom a = r.getAtom(i);
						if (!map.getStructureStyles().isSelected(a)){
							selected = false;
							break;
						}
					}
					
					if (selected){
						//select the residue and check the chain
						StructureComponentNode scn2 = (StructureComponentNode)nodeMap.get(r);
						selectedResidueNodes.add(scn2);
						scn2.selected = true;
						
						Chain chain = r.structure.getStructureMap().getChain(atom);
						selected = true;
						for (int i = 0; i < chain.getResidueCount(); i++){
							Residue rr = chain.getResidue(i);
							if (!map.getStructureStyles().isSelected(rr)){
								selected = false;
								break;
							}
						}
						
						if (selected){
							//select the chain
							StructureComponentNode scn3 = (StructureComponentNode)nodeMap.get(chain);
							if (scn3.selected) return;
							selectedChainNodes.add(scn3);
							scn3.selected = true;
						}
					}
					
//					if (!internalSelection && scrollToSelection){
//						tree.scrollPathToVisible(new TreePath(scn.node.getPath()));
//					}
				}
				else{
					StructureComponentNode scn = (StructureComponentNode)nodeMap.get(atom);
					if (scn == null) return;
					//now deselect this residue and its atoms
					selectedAtomNodes.remove(scn);
					scn.selected = false;

					//also deselect its parent residue, since with an incomplete residue selection it can't be considered selected
					StructureComponentNode r = (StructureComponentNode)nodeMap.get(atom.residue);
					if (!r.selected) return;//already deselected
					r.selected = false;
					selectedResidueNodes.remove(r);
					
					//deselect the chain as well
					StructureMap map = atom.structure.getStructureMap();
					StructureComponentNode c = (StructureComponentNode)nodeMap.get(map.getChain(atom));
					c.selected = false;
					selectedChainNodes.remove(c);
					
					repaint();
				}
				
				tree.setSelectionPaths(null);
			}
			else if (event.attribute == StructureStyles.ATTRIBUTE_VISIBILITY){
//				System.out.println("Atom visibility");
				StructureComponentNode scn = (StructureComponentNode)nodeMap.get(atom);
				if (scn == null) return;
				scn.visible = atom.visible;
				StructureMap map = atom.structure.getStructureMap();
				
				//check other atoms in this residue: if nothing else is visible, hide the residue (if at least one is visible, show it)
				Residue residue = atom.residue;
				boolean visible = false;
				for (int i = 0; i < residue.getAtomCount(); i++){
					Atom rr = residue.getAtom(i);
					if (rr.visible){
						visible = true;
						break;
					}
				}
				
				StructureComponentNode scn3 = (StructureComponentNode)nodeMap.get(residue);
				scn3.visible = visible;
				
				Chain chain = residue.structure.getStructureMap().getChain(atom);
				StructureComponentNode chainNode = (StructureComponentNode)nodeMap.get(chain);
				if (visible){
					//make sure the chain is shown
					chainNode.visible = true;
				}
				else{
					//if nothing else is visible, hide it
					visible = false;
					for (int i = 0; i < chain.getResidueCount(); i++){
						Residue rr = chain.getResidue(i);
						if (rr.visible){
							visible = true;
							break;
						}
					
						if (!visible){
							chainNode.visible = false;
						}
					}
				}
			}
			else if (event.attribute == StructureStyles.ATTRIBUTE_STYLE){
				if (event.property == StructureStyles.PROPERTY_COLOR){
					//deselect this atom
					
					StructureComponentNode scn = (StructureComponentNode)nodeMap.get(atom);
					if (scn == null) return;
					//now deselect this residue and its atoms
					selectedAtomNodes.remove(scn);
					scn.selected = false;

					//also deselect its parent residue, since with an incomplete residue selection it can't be considered selected
					StructureComponentNode r = (StructureComponentNode)nodeMap.get(atom.residue);
					if (!r.selected) return;//already deselected
					r.selected = false;
					selectedResidueNodes.remove(r);
					
					//deselect the chain as well
					StructureMap map = atom.structure.getStructureMap();
					StructureComponentNode c = (StructureComponentNode)nodeMap.get(map.getChain(atom));
					c.selected = false;
					selectedChainNodes.remove(c);

					
				}
				
			}

			
		}
		
///		tree.updateUI();
	
		}
		catch (Exception e){}

	}
	
	public void updateVisibility(Residue residue){

		StructureComponentNode scn = (StructureComponentNode)nodeMap.get(residue);
		if (scn == null) return;
		scn.visible = residue.visible;
		
		for (int i = 0; i < residue.getAtomCount(); i++){
			Atom a = residue.getAtom(i);
			StructureComponentNode scn2 = (StructureComponentNode)nodeMap.get(a);
			scn2.visible = a.visible;
		}
		
		Chain chain = residue.structure.getStructureMap().getChain(residue.getAtom(0));
		StructureComponentNode scn3 = (StructureComponentNode)nodeMap.get(chain);
		if (scn.visible){
			//make the chain visible
			scn3.visible = true;
		}
		else{
			//check other residues in this chain: if nothing else is visible, hide the chain (if at least one is visible, show it)
			StructureMap map = chain.structure.getStructureMap();
			boolean visible = false;
			for (int i = 0; i < chain.getResidueCount(); i++){
				Residue rr = chain.getResidue(i);
				if (rr.visible){
					visible = true;
					break;
				}
			}
			
			scn3.visible = visible;
		}
		
		tree.repaint();
	}
	
	public void processSequenceStylesEvent(SequenceStylesEvent event){
		
		Cell cell = event.cell;
		if (cell == null){
			//change applies to the entire sequence
			if (event.attribute == SequenceStyles.ATTRIBUTE_SELECTION){
				if (event.sequenceStyles.isSequenceSelected()){
					//select the entire sequence and its residues
					Sequence sequence = event.sequenceStyles.getSequence();
					for (int i = selectedResidueNodes.size()-1; i >= 0; i--){
						StructureComponentNode rn = (StructureComponentNode)selectedResidueNodes.get(i);									
						Cell r = (Cell)rn.cell;
						if (r.sequence != sequence) continue;
						selectedResidueNodes.add(rn);
						rn.selected = true;
					}
				}
				else{
					Sequence sequence = event.sequenceStyles.getSequence();
					for (int i = selectedResidueNodes.size()-1; i >= 0; i--){
						StructureComponentNode rn = (StructureComponentNode)selectedResidueNodes.get(i);									
						Cell r = (Cell)rn.cell;
						if (r.sequence != sequence) continue;
						selectedResidueNodes.remove(rn);
						rn.selected = false;
					}

				}
			}
			else if (event.attribute == SequenceStyles.ATTRIBUTE_NAME){
				Sequence s = event.sequenceStyles.getSequence();
				String name = s.getName();
				
				//locate the node and update its name
				for (int i = 0; i < structureNodes.size(); i++){
					StructureNode n = (StructureNode)structureNodes.get(i);
					if (s == n.getSequence()){
						n.setName("Sequence: " + name);
					}
				}
				
			}
			tree.updateUI();
			return;
		}
		else{
			//cell is not null
			if (event.attribute == SequenceStyles.ATTRIBUTE_SELECTION){
				if (cell.symbol.equals("-")) return;
				StructureComponentNode node = (StructureComponentNode)nodeMap.get(cell);
				if (event.sequenceStyles.isSelected(cell)){
					node.selected = true;
					selectedResidueNodes.add(node);
				}
				else{
					node.selected = false;
					selectedResidueNodes.remove(node);
				}
				
			}
		}
		
		
	}
	
	public void processStructureComponentEvent(StructureComponentEvent event){
		
		StructureComponent structureComponent = event.structureComponent;
		if (event.changeType == StructureComponentEvent.TYPE_REPLACE){
			
			if (structureComponent == null){
				//it must have come from SequenceStyles
				if (event.cell == null) return;
				Cell cell = event.cell;
				if (cell.symbol.equals("-")) return;
				String data = "Residue " + cell.symbol + cell.residueId;
				
				StructureComponentNode scn = (StructureComponentNode)nodeMap.get(cell);
				DefaultMutableTreeNode node = scn.node;
				scn.setName(data);
				
				return;
			}
			
			if (structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				
				
				Residue r = (Residue)structureComponent;
				String data = "Residue " + r.getCompoundCode() + r.getResidueId();
				StructureComponentNode scn = (StructureComponentNode)nodeMap.get(r);
				DefaultMutableTreeNode node = scn.node;
				int difference = node.getChildCount();
				
				Vector removedNodes = new Vector();
				
				//since the new residue contains different atoms, the child nodes need to be regenerated
				for (int i = difference-1; i >= 0; i--){
					DefaultMutableTreeNode aaa = (DefaultMutableTreeNode)node.getChildAt(i);
					StructureComponentNode an = (StructureComponentNode)aaa.getUserObject();
					node.remove(aaa);
					removedNodes.add(aaa);
					nodeMap.remove(an.component);
					selectedAtomNodes.remove(an);
				}
				
				//update the nodes array
				Object[] newNodes = new Object[nodes.length - difference];
				int counter = 0;
				for (int i = 0; i < nodes.length; i++){
					if (removedNodes.contains(nodes[i])) continue;
					newNodes[counter++] = nodes[i];
				}
				
				nodes = newNodes;
				
				scn.setName(data);
				
				StructureMap map = r.structure.getStructureMap();
				Vector addedNodes = new Vector();
				//add new nodes
				for (int i = 0; i < r.getAtomCount(); i++){
					Atom a = r.getAtom(i);
					StructureComponentNode scn3 = new StructureComponentNode(a.name, a, false);
					DefaultMutableTreeNode an = new DefaultMutableTreeNode(scn3);
					node.add(an);
					addedNodes.add(an);
					scn3.node = an;
					scn3.visible = a.visible;
					nodeMap.put(a, scn3);

					if (!scn.visible){
						scn3.visible = false;
					}
					
					if (scn.selected){
						scn3.selected = true;
						selectedAtomNodes.add(scn3);
					}
				}
				
				Object[] newNodes2 = new Object[nodes.length + addedNodes.size()];
				for (int i = 0; i < nodes.length; i++){
					newNodes2[i] = nodes[i];
				}
				
				//add the new atom nodes on top of that
				int j = nodes.length;
				for (int i = 0; i < addedNodes.size(); i++){
					newNodes2[j++] = addedNodes.get(i);
				}
				
				if (infoViewer != null && infoViewer.isVisible()){
					if (infoViewer.getActiveComponent() == r){
						infoViewer.clearDisplay(false);
						infoViewer.displayComponentInfo(r);
					}
				}
				
				nodes = newNodes2;
			}
			else if (structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
				Atom a = (Atom)structureComponent;
				String data = a.name;
				StructureComponentNode node = (StructureComponentNode)nodeMap.get(a);
				node.setName(data);
				if (infoViewer != null && infoViewer.isVisible()){
					if (infoViewer.getActiveComponent() == a){
						infoViewer.clearDisplay(false);
						infoViewer.displayComponentInfo(a);
					}
				}
				
			}
		}
		else if (event.changeType == StructureComponentEvent.TYPE_ADD){
			StructureComponent sc = event.structureComponent;
			
			if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
				Chain chain = (Chain)sc;
				//get the parent node
				Structure structure = chain.structure;
				StructureNode structureNode = null;
				for (int i = 0; i < structureNodes.size(); i++){
					StructureNode sn = (StructureNode)structureNodes.get(i);
					if (sn.structure == structure){
						structureNode = sn;
						break;
					}
				}
				
				if (structureNode == null) return;//no such structure
				
				StructureMap map = structure.getStructureMap();
				
				DefaultMutableTreeNode parent = structureNode.node;
				
				Vector newNodes = new Vector();
				
				//create a chain node
				StructureComponentNode scn = new StructureComponentNode("Chain " + chain.getChainId(), chain, true);
				DefaultMutableTreeNode cn = new DefaultMutableTreeNode(scn);
				parent.add(cn);
				newNodes.add(cn);
				scn.node = cn;
				scn.visible = chain.visible;
				
				nodeMap.put(chain, scn);
				
				for (int j = 0; j < chain.getResidueCount(); j++){
					Residue r = chain.getResidue(j);
					StructureComponentNode scn2 = new StructureComponentNode("Residue " + r.getCompoundCode() + r.getResidueId(), r, true);
					DefaultMutableTreeNode rn = new DefaultMutableTreeNode(scn2);
					cn.add(rn);
					newNodes.add(rn);
					scn2.node = rn;
					nodeMap.put(r, scn2);
					
					scn2.visible = r.visible;
					
					for (int k = 0; k < r.getAtomCount(); k++){
						Atom a = r.getAtom(k);
						StructureComponentNode scn3 = new StructureComponentNode(a.name, a, false);
						DefaultMutableTreeNode an = new DefaultMutableTreeNode(scn3);
						rn.add(an);
						newNodes.add(an);
						scn3.node = an;
						scn3.visible = a.visible;
						nodeMap.put(a, scn3);
					}
				}
				
				//update nodes
				Object[] nNodes = new Object[nodes.length + newNodes.size()];
				for (int i = 0; i < nodes.length; i++){
					nNodes[i] = nodes[i];
				}
				
				//now add the new ones
				int j = nodes.length;
				for (int i = 0; i < newNodes.size(); i++){
					nNodes[j++] = newNodes.get(i);
				}

				nodes = nNodes;
			}
			else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				Residue residue = (Residue)sc;
				
				//get the parent node
				Structure structure = residue.structure;
				StructureNode structureNode = null;
				for (int i = 0; i < structureNodes.size(); i++){
					StructureNode sn = (StructureNode)structureNodes.get(i);
					if (sn.structure == structure){
						structureNode = sn;
						break;
					}
				}
				
				if (structureNode == null) return;
				
				StructureMap map = structure.getStructureMap();
				
				Vector newNodes = new Vector();
				
				//check which chain it's going to
				Chain chain = map.getChain(residue.getAtom(0));
				StructureComponentNode chainNode = (StructureComponentNode)nodeMap.get(chain);
				DefaultMutableTreeNode cNode = chainNode.node;
				
				StructureComponentNode scn = new StructureComponentNode("Residue " + residue.getCompoundCode() + residue.getResidueId(), residue, true);
				DefaultMutableTreeNode rn = new DefaultMutableTreeNode(scn);
				cNode.add(rn);
				newNodes.add(rn);
				scn.node = rn;
				nodeMap.put(residue, scn);
				
				scn.visible = residue.visible;
				
				for (int k = 0; k < residue.getAtomCount(); k++){
					Atom a = residue.getAtom(k);
					StructureComponentNode scn3 = new StructureComponentNode(a.name, a, false);
					DefaultMutableTreeNode an = new DefaultMutableTreeNode(scn3);
					rn.add(an);
					newNodes.add(an);
					scn3.node = an;
					scn3.visible = a.visible;
					nodeMap.put(a, scn3);
				}
				
				//update nodes
				Object[] nNodes = new Object[nodes.length + newNodes.size()];
				for (int i = 0; i < nodes.length; i++){
					nNodes[i] = nodes[i];
				}
				
				//now add the new ones
				int j = nodes.length;
				for (int i = 0; i < newNodes.size(); i++){
					nNodes[j++] = newNodes.get(i);
				}

				nodes = nNodes;
				
			}
			else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
				
				Atom atom = (Atom)sc;
				
				//get the parent node
				Structure structure = atom.structure;
				StructureNode structureNode = null;
				for (int i = 0; i < structureNodes.size(); i++){
					StructureNode sn = (StructureNode)structureNodes.get(i);
					if (sn.structure == structure){
						structureNode = sn;
						break;
					}
				}
				
				if (structureNode == null) return;
				
				StructureMap map = structure.getStructureMap();
				Residue residue = atom.residue;
				StructureComponentNode residueNode = (StructureComponentNode)nodeMap.get(residue);
				DefaultMutableTreeNode rn = residueNode.node;
				
				StructureComponentNode scn3 = new StructureComponentNode(atom.name, atom, false);
				DefaultMutableTreeNode an = new DefaultMutableTreeNode(scn3);
				rn.add(an);
				scn3.node = an;
				scn3.visible = atom.visible;
				nodeMap.put(atom, scn3);
				
				//add the new node to the list
				
				Object[] nNodes = new Object[nodes.length + 1];
				for (int i = 0; i < nodes.length; i++){
					nNodes[i] = nodes[i];
				}
				
				//now add the new one
				nNodes[nodes.length] = an;

				nodes = nNodes;
				
				
			}
			
		}
		else if (event.changeType == StructureComponentEvent.TYPE_REMOVE){
			
			if (event.structureComponent == null){
				//it came from SequenceStyles
				Cell cell = event.cell;
				if (cell == null) return;
				if (cell.symbol.equals("-")) return;
				
				StructureComponentNode scn = (StructureComponentNode)nodeMap.get(cell);
				DefaultMutableTreeNode rn = scn.node;
				DefaultMutableTreeNode parent = (DefaultMutableTreeNode)scn.node.getParent();
			
				parent.remove(rn);
				selectedResidueNodes.remove(scn);
				
				nodeMap.remove(cell);
				
				//update nodes
				Object[] newNodes = new Object[nodes.length - 1];
				int counter = 0;
				for (int i = 0; i < nodes.length; i++){
					if (rn == nodes[i]) continue;
					newNodes[counter++] = nodes[i];
				}
				
				nodes = newNodes;
				
				//update numbering
				Sequence sequence = cell.sequence;
				for (int i = 0; i < parent.getChildCount(); i++){
					DefaultMutableTreeNode child = (DefaultMutableTreeNode)parent.getChildAt(i);
					StructureComponentNode sn = (StructureComponentNode)child.getUserObject();
					sn.setName("Residue " + sn.cell.symbol + sn.cell.residueId);
					
				}

				
				return;
			}
			
			if (event.structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				Residue residue = (Residue)structureComponent;
				
				StructureComponentNode scn = (StructureComponentNode)nodeMap.get(residue);
				DefaultMutableTreeNode rn = scn.node;
				DefaultMutableTreeNode parent = (DefaultMutableTreeNode)scn.node.getParent();
				
				Vector removedNodes = new Vector();
				
				parent.remove(rn);
				removedNodes.add(rn);
				selectedResidueNodes.remove(scn);
				
				nodeMap.remove(residue);
				
				//remove all atoms
				for (int i = rn.getChildCount()-1; i >=0; i--){
					DefaultMutableTreeNode child = (DefaultMutableTreeNode)rn.getChildAt(i);
					StructureComponentNode an = (StructureComponentNode)child.getUserObject();
					Atom a = (Atom)an.component;
					nodeMap.remove(a);
					selectedAtomNodes.remove(an);
					removedNodes.add(child);
					rn.remove(child);
				}
				
				//now update nodes array
				Object[] newNodes = new Object[nodes.length - removedNodes.size()];
				int counter = 0;
				for (int i = 0; i < nodes.length; i++){
					if (removedNodes.contains(nodes[i])) continue;
					newNodes[counter++] = nodes[i];
				}
				
				nodes = newNodes;
				
				if (infoViewer != null && infoViewer.isVisible()){
					if (infoViewer.getActiveComponent() == residue){
						infoViewer.clearDisplay(true);
					}
				}
			}
			else if (event.structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
				Atom atom = (Atom)event.structureComponent;
				
				StructureComponentNode an = (StructureComponentNode)nodeMap.get(atom);
				Residue residue = atom.residue;
				StructureComponentNode rn = (StructureComponentNode)nodeMap.get(residue);
				rn.node.remove(an.node);
				
				nodeMap.remove(atom);
				if (rn.selected){
					selectedAtomNodes.remove(an);
					
				}
				else{
					if (an.selected){
						selectedAtomNodes.remove(an);
					}
					else{
						//maybe now all other atoms become selected and residue should be marked selected
						/**
						 * TODO
						 */
					}
				}
				
				//update the list of nodes
				Object[] newNodes = new Object[nodes.length - 1];
				int counter = 0;
				for (int i = 0; i < nodes.length; i++){
					if (nodes[i] == an.node) continue;
					newNodes[counter++] = nodes[i];
				}
				
				nodes = newNodes;
				
				if (infoViewer != null && infoViewer.isVisible()){
					if (infoViewer.getActiveComponent() == atom){
						infoViewer.clearDisplay(true);
					}
				}

			}
		}

	}
	
	public void addAtom(Atom atom){
		
		//get the parent node
		Structure structure = atom.structure;
		StructureNode structureNode = null;
		for (int i = 0; i < structureNodes.size(); i++){
			StructureNode sn = (StructureNode)structureNodes.get(i);
			if (sn.structure == structure){
				structureNode = sn;
				break;
			}
		}
		
		if (structureNode == null) return;
		
		StructureMap map = structure.getStructureMap();
		Residue residue = atom.residue;
		StructureComponentNode residueNode = (StructureComponentNode)nodeMap.get(residue);
		DefaultMutableTreeNode rn = residueNode.node;
		
		StructureComponentNode scn3 = new StructureComponentNode(atom.name, atom, false);
		DefaultMutableTreeNode an = new DefaultMutableTreeNode(scn3);
		rn.add(an);
		scn3.node = an;
		scn3.visible = atom.visible;
		nodeMap.put(atom, scn3);
		scn3.selected = structure.getStructureMap().getStructureStyles().isSelected(atom);
		
		//add the new node to the list
		
		Object[] nNodes = new Object[nodes.length + 1];
		for (int i = 0; i < nodes.length; i++){
			nNodes[i] = nodes[i];
		}
		
		//now add the new one
		nNodes[nodes.length] = an;
		
		nodes = nNodes;

	}
	
	private void showDisplay(){
		callable.displayAction();
	}
	
	private void showRender(){
		callable.renderingAction();
	}
	
	private void showColor(){
		callable.colorAction();
	}
	
	private void openInfoPanel(){
		if (infoViewer == null) infoViewer = new InfoViewer(callable);
		infoViewer.setPreferredSize(new Dimension(150, 150));
		this.remove(sp);
		splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, sp, infoViewer);
		
		int h = this.getBounds().height;
		splitPane.setDividerLocation(h-250);
		this.add(splitPane);
		this.revalidate();
		
		infoViewer.setPanelSize(new Dimension(infoViewer.getSize().width, 250));
		viewInfo.setSelected(true);
		this.repaint();
	}
	
	private void closeInfoPanel(){
		
		if (splitPane == null) return;
		if (infoViewer != null) infoViewer.setVisible(false);
		this.remove(splitPane);
		this.add(sp);
		viewInfo.setSelected(false);
		this.revalidate();
		this.repaint();
	}
	
	private JMenuBar createMenuBar(){
		
		JMenuBar bar = new JMenuBar();
		
		JMenu action = new JMenu("Action");
		action.setMnemonic('a');
		JMenuItem actionDisplay = new JMenuItem("Structure visibility...");
		actionDisplay.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				callable.displayAction();
			}
		});
		
		JMenuItem actionRender = new JMenuItem("Structure rendering...");
		actionRender.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				callable.renderingAction();
			}
		});
		
		JMenuItem actionColor = new JMenuItem("Structure coloring...");
		actionColor.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				callable.colorAction();
			}
		});

		JMenuItem actionClear = new JMenuItem("Clear selection");
		actionClear.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, KeyEvent.CTRL_MASK, false));
		actionClear.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				clearSelection();
			}
		});
		
		JMenuItem actionReset = new JMenuItem("Reset tree display");
		actionReset.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, KeyEvent.CTRL_MASK, false));
		actionReset.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				//reset tree to default expansion state
				//walk through all nodes and make sure they are collapsed except for the Structure nodes
				Vector chains = new Vector();
				for (int i = 0; i < nodes.length; i++){
					DefaultMutableTreeNode n = (DefaultMutableTreeNode)nodes[i];
					try{
						StructureComponentNode scn = (StructureComponentNode)n.getUserObject();
						if (scn == null) continue;
						if (scn.component != null){
							if (scn.component.getStructureComponentType().equals(StructureComponentRegistry.TYPE_ATOM)) continue;
							if (scn.component.getStructureComponentType().equals(StructureComponentRegistry.TYPE_CHAIN)){
								chains.add(n);
								continue;
							}
						}
						tree.collapsePath(new TreePath(n.getPath()));
					}
					catch (ClassCastException ex){
						//it's a structure node, keep it open
						//check if it's a sequence node
						StructureNode sn = (StructureNode)n.getUserObject();
						if (sn.sequence != null){
							chains.add(n);
						}
					}
				}
				
				for (int i = 0; i < chains.size(); i++){
					DefaultMutableTreeNode n = (DefaultMutableTreeNode)chains.get(i);
					tree.collapsePath(new TreePath(n.getPath()));
				}
				
			}
		});


		action.add(actionDisplay);
		action.add(actionRender);
		action.add(actionColor);
		action.add(new JSeparator());
		action.add(actionClear);
		action.add(actionReset);
		
		bar.add(action);
		
		JMenu view = new JMenu("View");
		viewInfo = new JCheckBoxMenuItem("Open information panel");
		viewInfo.setSelected(false);
		viewInfo.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				if (viewInfo.isSelected()){
					openInfoPanel();
				}
				else{
					closeInfoPanel();
				}
			}
		});
		
		JMenuItem viewPrint = new JMenuItem("Print displayed data");
		viewPrint.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				if (infoViewer != null && infoViewer.isVisible()){
					infoViewer.printData();
				}
			}
		});
		
		JMenuItem viewInfoClear = new JMenuItem("Clear current info");
		viewInfoClear.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				if (infoViewer != null){
					infoViewer.clearDisplay(true);
				}
			}
		});
		

		view.add(viewInfo);
		view.add(new JSeparator());
		view.add(viewPrint);
		view.add(viewInfoClear);
		
		bar.add(view);
		
		JMenu options = new JMenu("Options");
		optionsScroll = new JCheckBoxMenuItem("Expand tree to selection");
		optionsScroll.setSelected(scrollToSelection);
		optionsScroll.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				scrollToSelection = optionsScroll.isSelected();
			}
		});
		
		options.add(optionsScroll);
		
		bar.add(options);
		
		return bar;
	}
	
	private JPopupMenu createAtomPopupMenu(){

		
		atomPopup = new JPopupMenu();
		JMenuItem blank = new JMenuItem("");
		blank.setEnabled(false);
		
		JMenuItem atomPopupInfo = new JMenuItem("Display atom info");
		atomPopupInfo.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				if (infoViewer == null || !infoViewer.isVisible()){
					openInfoPanel();
				}
				else if (!infoViewer.isVisible()){
					infoViewer.setVisible(true);
				}
				infoViewer.displayComponentInfo(activeAtom);
			}
		});
		JMenuItem atomPopupRename = new JMenuItem("Rename atom");
		atomPopupRename.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent a){
				Thread runner = new Thread(){
					public void run(){
						//get the clicked Atom node
						AtomRenameDialog.showDialog(callable.getApplicationFrame(), activeAtom);
						StructureComponentNode scn = (StructureComponentNode)nodeMap.get(activeAtom);
						if (activeAtom.forceFieldType == null){
							scn.setName(activeAtom.name);
						}
						else{
							scn.setName(activeAtom.name + " [" + activeAtom.forceFieldType + "]");
						}
						tree.repaint();

						if (infoViewer != null && infoViewer.isVisible()){
							if (infoViewer.getActiveComponent() == activeAtom){
								infoViewer.clearDisplay(false);
								infoViewer.displayComponentInfo(activeAtom);
							}
						}
						activeAtom = null;
					}
				};
				runner.start();
				
			}
		});
		
		JMenuItem atomPopupColor = new JMenuItem("Color atom");
		atomPopupColor.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (callable.getDisplayDialog() != null) return;
						callable.setDisplayDialogStatus(true, new ComponentColorDialog(callable.getApplicationFrame(), 
								callable, structureViewer, "Atom color: " + activeAtom.name, activeAtom, false));
					}
				};
				runner.start();
			}
		});
		JMenuItem atomPopupRender = new JMenuItem("Render atom");
		atomPopupRender.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (callable.getDisplayDialog() != null) return;
						callable.setDisplayDialogStatus(true, new ComponentRenderingDialog(callable.getApplicationFrame(), 
								callable, structureViewer, "Atom rendering: " + activeAtom.name, activeAtom));
					}
				};
				runner.start();
			}
		});
		JMenuItem atomPopupLabel = new JMenuItem("Label atom");
		atomPopupLabel.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (callable.getDisplayDialog() != null) return;
//						callable.setDisplayDialogStatus(true, new ComponentLabelDialog(callable.getApplicationFrame(), callable, structureViewer, "Label atom: " + activeAtom.name, activeAtom));
					}
				};
				runner.start();
			}
		});
		
		JMenuItem atomPopupFocus = new JMenuItem("Focus view on atom");
		atomPopupFocus.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				if (activeAtom == null) return;
				structureViewer.focusView(activeAtom);
				activeAtom = null;
			}
		});
		
		
		
		atomPopup.add(blank);
		atomPopup.add(atomPopupInfo);
		atomPopup.add(atomPopupRename);
		atomPopup.add(new JSeparator());
		atomPopup.add(atomPopupColor);
		atomPopup.add(atomPopupRender);
//		atomPopup.add(atomPopupLabel);
		atomPopup.add(new JSeparator());
		atomPopup.add(atomPopupFocus);
		
		return atomPopup;
		
	
	}
	
	private JPopupMenu createResiduePopupMenu(){

		
		residuePopup = new JPopupMenu();
		JMenuItem blank = new JMenuItem("");
		blank.setEnabled(false);
		
		JMenuItem residuePopupInfo = new JMenuItem("Display residue info");
		residuePopupInfo.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				if (infoViewer == null || !infoViewer.isVisible()){
					openInfoPanel();
				}
				else if (!infoViewer.isVisible()){
					infoViewer.setVisible(true);
				}
				infoViewer.displayComponentInfo(activeResidue);
			}
		});
		JMenuItem residuePopupRename = new JMenuItem("Rename residue");
		residuePopupRename.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent a){
				Thread runner = new Thread(){
					public void run(){
						//get the clicked Atom node
						ResidueRenameDialog.showDialog(callable.getApplicationFrame(), activeResidue);
						StructureComponentNode scn = (StructureComponentNode)nodeMap.get(activeResidue);
						scn.setName("Residue " + activeResidue.getCompoundCode() + activeResidue.getResidueId());
						for (int i = 0 ; i < activeResidue.getAtomCount(); i++){
							Atom a = activeResidue.getAtom(i);
							a.compound = activeResidue.getCompoundCode();
						}
						tree.repaint();
						
						if (infoViewer != null && infoViewer.isVisible()){
							if (infoViewer.getActiveComponent() == activeResidue){
								infoViewer.clearDisplay(false);
								infoViewer.displayComponentInfo(activeResidue);
							}
						}
						
						activeResidue = null;
					}
				};
				runner.start();
				
			}
		});
		
		JMenuItem residuePopupColor = new JMenuItem("Color residue");
		residuePopupColor.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (callable.getDisplayDialog() != null) return;
						callable.setDisplayDialogStatus(true, new ComponentColorDialog(callable.getApplicationFrame(), 
								callable, structureViewer, "Residue color: " + activeResidue.getCompoundCode() + activeResidue.getResidueId(), activeResidue, false));
					}
				};
				runner.start();
			}
		});
		JMenuItem residuePopupRender = new JMenuItem("Render residue");
		residuePopupRender.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (callable.getDisplayDialog() != null) return;
						callable.setDisplayDialogStatus(true, new ComponentRenderingDialog(callable.getApplicationFrame(), 
								callable, structureViewer, "Residue rendering: " + activeResidue.getCompoundCode() + activeResidue.getResidueId(), activeResidue));
					}
				};
				runner.start();
			}
		});
		
		
		
		residuePopup.add(blank);
		residuePopup.add(residuePopupInfo);
		residuePopup.add(residuePopupRename);
		residuePopup.add(new JSeparator());
		residuePopup.add(residuePopupColor);
		residuePopup.add(residuePopupRender);
		
		return residuePopup;
		
	
	}

	private JPopupMenu createChainPopupMenu(){

		
		chainPopup = new JPopupMenu();
		JMenuItem blank = new JMenuItem("");
		blank.setEnabled(false);
		
		JMenuItem chainPopupInfo = new JMenuItem("Display chain info");
		chainPopupInfo.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				if (infoViewer == null || !infoViewer.isVisible()){
					openInfoPanel();
				}
				else if (!infoViewer.isVisible()){
					infoViewer.setVisible(true);
				}
				infoViewer.displayComponentInfo(activeChain);
			}
		});
		

		JMenuItem chainPopupColor = new JMenuItem("Color chain residues");
		chainPopupColor.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (callable.getDisplayDialog() != null) return;
						callable.setDisplayDialogStatus(true, new ComponentColorDialog(callable.getApplicationFrame(), 
								callable, structureViewer, "Chain color: " + activeChain.getChainId(), activeChain, false));
					}
				};
				runner.start();
			}
		});
		
		JMenuItem chainPopupRender = new JMenuItem("Render chain residues");
		chainPopupRender.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (callable.getDisplayDialog() != null) return;
						callable.setDisplayDialogStatus(true, new ComponentRenderingDialog(callable.getApplicationFrame(), 
								callable, structureViewer, "Chain rendering: " + activeChain.getChainId(), activeChain));
					}
				};
				runner.start();
			}
		});
		
		
		chainPopupRibbon = new JMenuItem("Show ribbon");
		chainPopupRibbon.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (callable.getDisplayDialog() != null) return;
						String title = null;
						if (structureViewer.hasRibbon(activeChain) > -1){
							structureViewer.removeRibbon(activeChain, true);
							callable.updateView();
						}
						else{
							title = "Chain " + activeChain.getChainId() + ": add ribbon";
							callable.setDisplayDialogStatus(true, new ComponentRibbonDialog(callable.getApplicationFrame(), 
									callable, structureViewer, title, activeChain, true));
						}
					}
				};
				runner.start();
			}
		});
		
		JMenuItem chainPopupRibbonColor = new JMenuItem("Color ribbon");
		chainPopupRibbonColor.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						if (callable.getDisplayDialog() != null) return;
						callable.setDisplayDialogStatus(true, new ComponentColorDialog(callable.getApplicationFrame(), 
								callable, structureViewer, "Ribbon color: " + activeChain.getChainId(), activeChain, true));
					}
				};
				runner.start();
			}
		});

		chainPopup.add(blank);
		chainPopup.add(chainPopupInfo);
		chainPopup.add(new JSeparator());
		chainPopup.add(chainPopupRibbon);
		chainPopup.add(chainPopupRibbonColor);
		chainPopup.add(new JSeparator());
		chainPopup.add(chainPopupColor);
		chainPopup.add(chainPopupRender);
		
		return chainPopup;
		
	
	}
	
	private void processPopup(MouseEvent e, DefaultMutableTreeNode node){
		
		//check what node was clicked
		try{
			StructureComponentNode scn = (StructureComponentNode)node.getUserObject();
			StructureComponent sc = scn.component;
			if (sc == null){
				//it's a cell
				return;//for now
			}
			
			if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
				activeAtom = (Atom)sc;
				
				//correct the coordinates of the popup by taking into account tree's position within the viewport
				Point position = sp.getViewport().getViewPosition();
				int x = e.getX() - position.x;
				int y = e.getY() - position.y + 20;//correct for the position of the menu with regard to tree's rows
				atomPopup.show(this, x, y);
			}
			else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				activeResidue = (Residue)sc;
				
				//correct the coordinates of the popup by taking into account tree's position within the viewport
				Point position = sp.getViewport().getViewPosition();
				int x = e.getX() - position.x;
				int y = e.getY() - position.y + 20;//correct for the position of the menu with regard to tree's rows
				residuePopup.show(this, x, y);
			}
			else if (sc.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
				activeChain = (Chain)sc;
				
				if (structureViewer.hasRibbon(activeChain) > -1){
					chainPopupRibbon.setText("Remove ribbon");
				}
				else{
					chainPopupRibbon.setText("Show ribbon");
				}
				
				//correct the coordinates of the popup by taking into account tree's position within the viewport
				Point position = sp.getViewport().getViewPosition();
				int x = e.getX() - position.x;
				int y = e.getY() - position.y + 20;//correct for the position of the menu with regard to tree's rows
				chainPopup.show(this, x, y);
			}
			
			//and so on
		}
		catch (Exception ex){
//			ex.printStackTrace();
		}
	}
	
	public void setScrollToSelection(boolean scroll){
		scrollToSelection = scroll;
	}
	
	public void expandToNode(StructureComponent component){
		
		if (component == null) return;
		if (!scrollToSelection) return;
		
		StructureComponentNode scn = (StructureComponentNode)nodeMap.get(component);
		
		tree.scrollPathToVisible(new TreePath(scn.node.getPath()));
		tree.updateUI();
	}
	
	public void expandToNode(Cell cell){
		
		if (cell == null) return;
		if (!scrollToSelection) return;
		
		StructureComponentNode scn = (StructureComponentNode)nodeMap.get(cell);
		
		tree.scrollPathToVisible(new TreePath(scn.node.getPath()));
		tree.updateUI();
	}


	public void expandToNodes(Vector components){
		
		if (components == null || components.size() == 0) return;
		
		if (!scrollToSelection) return;
		
		StructureComponentNode scn = (StructureComponentNode)nodeMap.get(components.get(0));
		
		tree.scrollPathToVisible(new TreePath(scn.node.getPath()));
		tree.updateUI();
	}

}