//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs;

// Core
import java.util.*;

// GUI
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

import javax.swing.event.*;
import javax.swing.*;
import javax.swing.border.*;

// MBT
import edu.sdsc.mbt.*;
import edu.sdsc.sirius.util.Manager;
import edu.sdsc.sirius.viewers.*;

/**
 *  This class impements a dialog box used to color the currently displayed structures.
 *  <P>
 *  @see edu.sdsc.sirius.viewers.StructureViewer
 *  @see edu.sdsc.sirius.dialogs.DisplayDialog
 *  <P>
 *  @author Oleksandr V. Buzko
 */

public class ColorDialog extends JDialog implements DisplayDialog {

	private JFrame parentFrame;
	private Manager parent;
	private StructureViewer sv;
	
	private StructureComponent structureComponent;
	
	private DisplayDialog itself;

	private Container contentPane;
	private JPanel base = new JPanel();
	private JPanel setPanel = new JPanel();
	private JPanel colorPanel = new JPanel();
	
	private JPanel cPanel = new JPanel();//color selection
	
	private Border border1;
	private TitledBorder border2;
	private Border border;
	
	
	/**
	 * Static variable that determines what level of coloring is used. Selection color is the default value.
	 */	
	public static int setFlag = DialogConstants.APPLY_SELECTION;
	
	public static final int COLOR_BY_ELEMENT = 0;
	public static final int COLOR_BY_CHAIN = 1;
	public static final int COLOR_BY_CHARGE = 2;
	public static final int COLOR_CUSTOM = 3;
	
	/**
	 * Static variable that determines which type of coloring coloring is used.
	 */
	public static int colorType = COLOR_BY_ELEMENT;
	
	private ButtonGroup setGroup = new ButtonGroup();
	private JRadioButton radio1 = new JRadioButton("Currently selected", true);
	private JRadioButton radio2 = new JRadioButton("Structure", false);
	private JRadioButton radio3 = new JRadioButton("Chain", false);
	private JRadioButton radio4 = new JRadioButton("Residue", false);
	private JRadioButton radio5 = new JRadioButton("Atom", false);
	
	private ButtonGroup colorGroup = new ButtonGroup();
	private JRadioButton radio6 = new JRadioButton("By element", true);
	private JRadioButton radio65 = new JRadioButton("By chain", false);
	private JRadioButton radio67 = new JRadioButton("By partial charge", false);
	private JRadioButton radio7 = new JRadioButton("Custom color", false);
	
	private JTextField field = new JTextField(30);
	private JLabel header = new JLabel();
	private JLabel label = new JLabel();

	private JLabel rgbLabel = new JLabel();
	private JTextField rgbField = new JTextField(30);
	
	private JButton colorButton = null;
	private JButton okButton = new JButton("OK");
	private JButton cancelButton = new JButton("Cancel");
	
	private JLabel or = new JLabel("or");
	private JComboBox combo = new JComboBox();
	private Vector structures = new Vector();
	

	
	/**
	 * Default structure color when the dialog is open.
	 */
	public static Color structureColor = new Color(160,160,160);
	private BufferedImage m_image;
	
	private boolean colorFlag = false;
	
	/**
	 * Constructor that builds the dialog.
	 * @param f
	 * @param p
	 * @param s
	 */
	public ColorDialog(JFrame f, Manager p, StructureViewer s){
	
		super(f, " Structure color settings ", false);

		setFlag = DialogConstants.APPLY_SELECTION;
		colorType = COLOR_BY_ELEMENT;
	
		parentFrame = f;
		parent = p;
		sv = s;
		
		parent.setDisplayDialogStatus(true);
		
		itself = this;
		
		
		
		
		//create the panels
		contentPane = getContentPane();
		contentPane.setLayout(null);
		
		base.setBorder(new BevelBorder(BevelBorder.RAISED));
		base.setBounds(5, 5, 410, 290);
		base.setLayout(null);
		
		setPanel.setBounds(5,5,200,280);
		setPanel.setLayout(null);
		
		border1 = BorderFactory.createEmptyBorder(0,0,0,0);//left offset of the boxes in the box
		border2 = BorderFactory.createTitledBorder(new EtchedBorder(), " Apply to ");
		border2.setTitleColor(Color.black);
		border2.setTitleFont(new Font("Dialog", 0, 12));
		border = BorderFactory.createCompoundBorder(border1, border2);
		setPanel.setBorder(border);
		
		//text field
		header.setForeground(Color.black);
		header.setFont(new Font("Dialog", Font.PLAIN, 12));
		header.setText("Pick object to color");
		header.setPreferredSize(new Dimension(150,20));
		header.setBounds(20,60,150,20);
		header.setLocation(new Point(20,60));
		header.setVisible(true);
		
		label.setForeground(Color.black);
		label.setFont(new Font("Dialog", Font.PLAIN, 11));
		label.setText("Enter name or click structure");
		label.setPreferredSize(new Dimension(175,20));
		label.setBounds(20,175,170,20);
		label.setLocation(new Point(20,175));
		label.setVisible(true);
		
		field.setPreferredSize(new Dimension(160,20));
		field.setBounds(20,200,160,20);
		field.setLocation(new Point(20,200));
		field.setEditable(false);
		field.setEnabled(false);
		field.setVisible(true);
		field.addMouseListener(new MouseAdapter(){
			public void mouseClicked(MouseEvent m){
				if (radio1.isSelected()){
					setFlag = DialogConstants.APPLY_RESIDUE;
					radio1.setSelected(false);
					radio4.setSelected(true);
					field.setText("");
					field.setEditable(true);
					field.setEnabled(true);
					field.requestFocus();
				}
			}
		});
					
		field.addKeyListener(new KeyAdapter(){
			public void keyPressed(KeyEvent k){
				if (k.getKeyCode() == KeyEvent.VK_ENTER){
					//do the same thing as with OK button
					process();
				}
				else if(k.getKeyCode() == KeyEvent.VK_ESCAPE){
					//do the same thing as with Cancel button
					setVisible(false);
					parent.setDisplayDialogStatus(false);
					dispose();
				}
			}
		});
		
		or.setForeground(Color.black);
		or.setFont(new Font("Dialog", Font.PLAIN, 11));
		or.setPreferredSize(new Dimension(15,20));
		or.setBounds(20,230,15,20);
		or.setLocation(new Point(20,230));
		or.setVisible(true);
		
		//walk through the loaded structures
		structures.add("Select structure or set");
		for (int i = 0; i < sv.getLoadedStructures().size(); i++){
			Structure st = (Structure)sv.getLoadedStructures().get(i);
			structures.add(sv.getStructureName(st));
		}
		
		Vector setNames = sv.getAtomSetNames();
		for (int i = 0; i < setNames.size(); i++){
			structures.add(setNames.get(i));
		}
		

		combo = new JComboBox(structures);
		combo.setPreferredSize(new Dimension(145,20));
		combo.setBounds(35,230,145,20);
		combo.setLocation(new Point(35,230));
		combo.setVisible(true);
		if (structures.size() == 1) combo.setEnabled(false);
		combo.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				String choice = (String)combo.getSelectedItem();
				if (choice.startsWith("Select")) return;
				field.setText(choice);
				field.setEnabled(true);
				field.setEditable(true);
				setFlag = DialogConstants.APPLY_STRUCTURE;		
				radio2.setSelected(true);
			}
		});
		
		
		//define the radio buttons
		radio1.setLocation(20,25);
		radio1.setBounds(20,25,150,20);
		radio1.setVisible(true);
		radio1.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setFlag = DialogConstants.APPLY_SELECTION;
				field.setText("");
				field.setEditable(false);
				field.setEnabled(false);
			}
		});
		
		radio2.setLocation(20,85);
		radio2.setBounds(20,85,150,20);
		radio2.setVisible(true);
		radio2.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setFlag = DialogConstants.APPLY_STRUCTURE;
				field.setEditable(true);
				field.setEnabled(true);
				field.requestFocus();
			}
		});
		
		radio3.setLocation(20,105);
		radio3.setBounds(20,105,150,20);
		radio3.setVisible(true);
		radio3.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setFlag = DialogConstants.APPLY_CHAIN;
				field.setEditable(true);
				field.setEnabled(true);
				field.requestFocus();
			}
		});
		
		radio4.setLocation(20,125);
		radio4.setBounds(20,125,150,20);
		radio4.setVisible(true);
		radio4.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setFlag = DialogConstants.APPLY_RESIDUE;
				field.setEditable(true);
				field.setEnabled(true);
				field.requestFocus();
			}
		});
		
		radio5.setLocation(20,145);
		radio5.setBounds(20,145,150,20);
		radio5.setVisible(true);
		radio5.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setFlag = DialogConstants.APPLY_ATOM;
				field.setEditable(true);
				field.setEnabled(true);
				field.requestFocus();
			}
		});
		
		//draw a line between the radio buttons
		JSeparator line = new JSeparator();
		line.setLocation(10,55);
		line.setBounds(10,55,180,5);
		line.setVisible(true);
		
		setPanel.add(line);
		setPanel.add(header);
		
		setGroup.add(radio1);
		setGroup.add(radio2);
		setGroup.add(radio3);
		setGroup.add(radio4);
		setGroup.add(radio5);
		
		setPanel.add(radio1);
		setPanel.add(radio2);
		setPanel.add(radio3);
		setPanel.add(radio4);
		setPanel.add(radio5);
		
		setPanel.add(label);
		setPanel.add(field);
		setPanel.add(or);
		setPanel.add(combo);
		
		
		//color panel
		colorPanel.setBounds(205,5,200,280);
		colorPanel.setLayout(null);
		
		border1 = BorderFactory.createEmptyBorder(0,0,0,0);//left offset of the boxes in the box
		border2 = BorderFactory.createTitledBorder(new EtchedBorder(), "  Color ");
		border2.setTitleColor(Color.black);
		border2.setTitleFont(new Font("Dialog", 0, 12));
		border = BorderFactory.createCompoundBorder(border1, border2);
		colorPanel.setBorder(border);
		
		//add radio buttons
		radio6.setBounds(20,25,150,15);
		radio6.setVisible(true);
		radio6.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				colorType = COLOR_BY_ELEMENT;
				rgbField.setText("");
				rgbField.setEditable(false);
				rgbField.setEnabled(false);
				colorButton.setEnabled(false);
			}
		});
		
		radio65.setBounds(20,45,150,15);
		radio65.setVisible(true);
		radio65.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				colorType = COLOR_BY_CHAIN;
				rgbField.setText("");
				rgbField.setEditable(false);
				rgbField.setEnabled(false);
				colorButton.setEnabled(false);
			}
		});

		radio67.setBounds(20,65,150,15);
		radio67.setVisible(true);
		radio67.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				colorType = COLOR_BY_CHARGE;
				rgbField.setText("");
				rgbField.setEditable(false);
				rgbField.setEnabled(false);
				colorButton.setEnabled(false);
			}
		});

		radio7.setBounds(20,85,150,15);
		radio7.setVisible(true);
		radio7.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				colorType = COLOR_CUSTOM;
				rgbField.setEditable(true);
				rgbField.setEnabled(true);
				colorButton.setEnabled(true);
			}
		});

		colorGroup.add(radio6);
		colorGroup.add(radio65);
		colorGroup.add(radio67);
		colorGroup.add(radio7);
		
		colorPanel.add(radio6);
		colorPanel.add(radio65);
		colorPanel.add(radio67);
		colorPanel.add(radio7);


		//add color squares
		
		m_image = new BufferedImage(162,122, BufferedImage.TYPE_INT_ARGB);

		Graphics2D g2 = m_image.createGraphics();
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		g2.setFont(new Font("Arial", Font.PLAIN, 10));

		//first row
		g2.setColor(Color.white);
		g2.fillRect(0,0,40,30);
		
		g2.setColor(new Color(160,160,160));
		g2.fillRect(40,0,40,30);
		
		g2.setColor(Color.black);
		g2.fillRect(80,0,40,30);
		
		g2.setColor(Color.red);
		g2.fillRect(120,0,40,30);
		
		//second row
		g2.setColor(Color.yellow);
		g2.fillRect(0,30,40,30);
		
		g2.setColor(new Color(210,210,0));
		g2.fillRect(40,30,40,30);
		
		g2.setColor(new Color(150,150,0));
		g2.fillRect(80,30,40,30);
		
		g2.setColor(new Color(255,100,200));
		g2.fillRect(120,30,40,30);
		
		//third row
		g2.setColor(new Color(100,200,255));
		g2.fillRect(0,60,40,30);
		
		g2.setColor(new Color(100,100,255));
		g2.fillRect(40,60,40,30);
		
		g2.setColor(Color.blue);
		g2.fillRect(80,60,40,30);
		
		g2.setColor(new Color(200,0,200));
		g2.fillRect(120,60,40,30);
		
		//fourth row
		g2.setColor(Color.green);
		g2.fillRect(0,90,40,30);

		g2.setColor(new Color(0,200,0));
		g2.fillRect(40,90,40,30);
		
		g2.setColor(new Color(0,150,0));
		g2.fillRect(80,90,40,30);
		
		g2.setColor(new Color(150,0,0));
		g2.fillRect(120,90,40,30);
		
		//lines
		g2.setColor(Color.black);
		g2.drawLine(0,0,0,120);
		g2.drawLine(0,1,160,1);
		g2.drawLine(0,120,160,120);
		g2.drawLine(160,0,160,120);
		
		for (int i = 0; i <= 3; i++){
			//horizontal line
			g2.drawLine(0, i*30, 160, i*30);
			
			//vertical line
			g2.drawLine(i*40, 0, i*40, 120);
		}
		
		
		
		
		ImageIcon image = new ImageIcon(m_image);
		JLabel imageLabel = new JLabel();
		imageLabel.setIcon(image);
		
		JPanel cPanel = new JPanel();
		cPanel.setBounds(15,105,165,125);
		cPanel.setLayout(null);
		
		imageLabel.setBounds(1, 1, 161, 121);
		cPanel.add(imageLabel);
		
		//set up the rgb text field
		
		rgbLabel.setForeground(Color.black);
		rgbLabel.setFont(new Font("Dialog", Font.PLAIN, 12));
		rgbLabel.setText("Enter RGB color components");
		rgbLabel.setBounds(15,230,170,20);
		rgbLabel.setVisible(true);
		
		rgbField.setBounds(15,250,100,20);
		rgbField.setEditable(false);
		rgbField.setEnabled(false);
		rgbField.setVisible(true);
		rgbField.addMouseListener(new MouseAdapter(){
			public void mouseClicked(MouseEvent m){
				colorType = COLOR_CUSTOM;
				radio6.setSelected(false);
				radio7.setSelected(true);
				rgbField.setEditable(true);
				rgbField.setEnabled(true);
				colorButton.setEnabled(true);
			}
		});
					
		
		
		//add the color chooser button
		colorButton = new JButton("Choose");
		colorButton.setFont(new Font("Helvetica", Font.PLAIN, 10));
		colorButton.setBounds(120, 250, 70, 20);
		colorButton.setVisible(true);
		colorButton.setEnabled(false);

		colorButton.addMouseListener(new MouseAdapter(){
			public void mouseClicked(MouseEvent m){
				colorType = COLOR_CUSTOM;
				radio6.setSelected(false);
				radio7.setSelected(true);
				rgbField.setEditable(true);
				rgbField.setEnabled(true);
				colorButton.setEnabled(true);
				
				colorFlag = true;
				Color c = JColorChooser.showDialog(null, "Choose selection color", structureColor);
				colorFlag = false;
				if (c == null){
					return;
				}
				
				structureColor = c;
				
				String text = structureColor.getRed() + "," + structureColor.getGreen() + "," + structureColor.getBlue();
				rgbField.setText(text);
				
			}
		});


		colorButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				colorFlag = true;
				Color c = JColorChooser.showDialog(null, "Choose selection color", structureColor);
				colorFlag = false;
				if (c == null){
					return;
				}
				
				structureColor = c;
				
				String text = structureColor.getRed() + "," + structureColor.getGreen() + "," + structureColor.getBlue();
				rgbField.setText(text);
			}
		});
		
		
		//add mouse listener to detect which color has been clicked
		MouseInputAdapter mia = new MouseInputAdapter(){

			public void mouseClicked(MouseEvent e){
			
				if (colorType == COLOR_BY_ELEMENT){
					radio7.setSelected(true);
					radio6.setSelected(false);
					rgbField.setEnabled(true);
					colorButton.setEnabled(true);
					colorType = COLOR_CUSTOM;
				}
			
				//get the pixel
				int X = e.getX();//with the margins
				int Y = e.getY();
				
				if ((X > m_image.getWidth())||(Y > m_image.getHeight())){
					return;
				}
				
				long pixel = m_image.getRGB(X,Y);

				if (pixel == -1){
					structureColor = Color.white;
					rgbField.setText("255,255,255");
				}
				else if (pixel == -16777216){
					structureColor = Color.black;
					rgbField.setText("0,0,0");
				}
				else if (pixel == -256){
					structureColor = Color.yellow;
					rgbField.setText("255,255,0");
				}
				else if (pixel == -65536){
					structureColor = Color.red;
					rgbField.setText("255,0,0");
				}
				else if (pixel == -6250336){
					structureColor = new Color(160,160,160);
					rgbField.setText("160,160,160");
				}
				else if (pixel == -2960896){
					structureColor = new Color(210,210,0);
					rgbField.setText("210,210,0");
				}
				else if (pixel == -6908416){
					structureColor = new Color(150,150,0);
					rgbField.setText("150,150,0");
				}
				else if (pixel == -39736){
					structureColor = new Color(255,100,200);
					rgbField.setText("255,100,200");
				}
				else if (pixel == -10172161){
					structureColor = new Color(100,200,255);
					rgbField.setText("100,200,255");
				}
				else if (pixel == -10197761){
					structureColor = new Color(100,100,255);
					rgbField.setText("100,100,255");
				}
				else if (pixel == -16776961){
					structureColor = Color.blue;
					rgbField.setText("0,0,255");
				}
				else if (pixel == -3669816){
					structureColor = new Color(200,0,200);
					rgbField.setText("200,0,200");
				}
				else if (pixel == -16711936){
					structureColor = Color.green;
					rgbField.setText("0,255,0");
				}
				else if (pixel == -16726016){
					structureColor = new Color(0,200,0);
					rgbField.setText("0,200,0");
				}
				else if (pixel == -16738816){
					structureColor = new Color(0,150,0);
					rgbField.setText("0,150,0");
				}
				else if (pixel == -6946816){
					structureColor = new Color(150,0,0);
					rgbField.setText("150,0,0");
				}
				
			}
		};
		
		cPanel.addMouseListener(mia);
		
		colorPanel.add(cPanel);
		colorPanel.add(rgbLabel);
		colorPanel.add(rgbField);
		colorPanel.add(colorButton);
		
		//add panels to the base
		base.add(setPanel);
		base.add(colorPanel);
		
		
		//buttons
		okButton.setBounds(115, 305, 80, 25);
		okButton.setVisible(true);
		okButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				process();
			}
		});
		
		
		cancelButton.setBounds(235, 305, 80, 25);
		cancelButton.setVisible(true);
		cancelButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
		});
		
		contentPane.add(base);
		contentPane.add(okButton);
		contentPane.add(cancelButton);
		
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
			
			public void windowDeactivated(WindowEvent e){
				if (colorFlag){
					return;
				}
				toFront();
			}
		});
				
		setSize(430, 370);
		
		//add this to accommodate both applet and application cases
		if (parentFrame == null){
//			setLocationRelativeTo(null);
			setBounds(0, 0, getSize().width, getSize().height);
		}
		else{
		
			//set location
			Dimension d1 = getSize();
			Dimension d2 = parentFrame.getSize();
		
			int x = 5 + DialogConstants.toolbarWidth;
			int y = DialogConstants.level + DialogConstants.toolbarHeight;
			
			setBounds(x + parentFrame.getX(), y + parentFrame.getY(), d1.width, d1.height);
		}
		
		setResizable(false);
		setVisible(true);
	}
	
	/**
	 * This method is called by StructureViewer in response to a mouse click and sets the currently
	 * active text field to the String representation of the clicked StructureComponent.
	 * @see edu.sdsc.sirius.viewers.StructureViewer
	 * @see edu.sdsc.mbt.StructureComponent
	 */
	public void processPick(String data, StructureComponent structureComponent){
		this.structureComponent = structureComponent;
		if (field.isEditable()){
			field.setText(data);
		}
	}
	
	private void process(){
		
		//add error checking for selection bounds
		//eg., selection level is residue, but entered only structure name
		try{
			//first, check the number of tags
			int count = 0;
			StringTokenizer tok = new StringTokenizer(field.getText(), ":", false);
			while (tok.hasMoreTokens()){
				Object o = tok.nextToken();
				count++;
			}
	
			if (setFlag != 0){
				if (count < setFlag){
					setFlag = count;
				}
			}
		
		
			setVisible(false);
			parent.setDisplayDialogStatus(false);
			
			ColorEvent event = new ColorEvent(itself);
			event.setFlag = setFlag;
			event.colorType = colorType;
			event.structureColor = structureColor;
			event.structureComponent = structureComponent;
			event.componentId = field.getText();
			
			sv.processDisplayDialogEvent(event);
			
		}
		catch (Exception e){
			setVisible(false);
			parent.setDisplayDialogStatus(false);
			parent.displayExceptionMessage("Exception coloring structure", e);

		}

		dispose();

	}
}
