//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

import javax.swing.event.*;
import javax.swing.*;
import javax.swing.border.*;

// MBT
import edu.sdsc.mbt.*;
import edu.sdsc.mbt.viewables.*;
import edu.sdsc.sirius.util.Manager;
import edu.sdsc.sirius.viewers.*;

/**
 *  This class impements a dialog box used to color the currently displayed structures.
 *  <P>
 *  @see edu.sdsc.sirius.viewers.StructureViewer
 *  @see edu.sdsc.sirius.dialogs.DisplayDialog
 *  <P>
 *  @author Oleksandr V. Buzko
 */

public class ComponentColorDialog extends JDialog implements DisplayDialog {

	private JFrame parentFrame;
	private Manager parent;
	private StructureViewer sv;
	
	private DisplayDialog itself;

	private Container contentPane;
	private JPanel base = new JPanel();
	private JPanel colorPanel = new JPanel();
	
	private JPanel cPanel = new JPanel();//color selection
	
	private Border border1;
	private TitledBorder border2;
	private Border border;
	
	
	public static final int COLOR_BY_ELEMENT = 0;
	public static final int COLOR_BY_CHAIN = 1;
	public static final int COLOR_BY_CHARGE = 2;
	public static final int COLOR_CUSTOM = 3;
	
	/**
	 * Static variable that determines which type of coloring coloring is used.
	 */
	public static int colorType = COLOR_BY_ELEMENT;
	
	private ButtonGroup colorGroup = new ButtonGroup();
	private JRadioButton radio6 = new JRadioButton("Coloring by element", true);
	private JRadioButton radio5 = new JRadioButton("Coloring by charge", false);
	private JRadioButton radio7 = new JRadioButton("Custom color", false);
	
	private JLabel rgbLabel = new JLabel();
	private JTextField rgbField = new JTextField(30);
	
	private JButton colorButton = null;
	private JButton okButton = new JButton("OK");
	private JButton cancelButton = new JButton("Cancel");
	

	
	/**
	 * Default structure color when the dialog is open.
	 */
	public static Color structureColor = new Color(160,160,160);
	private BufferedImage m_image;
	
	private boolean colorFlag = false;
	
	private int width = 230;
	private int height = 350;
	
	private boolean ribbon = false;
	
	private StructureComponent structureComponent;
	
	/**
	 * Constructor that builds the dialog.
	 * @param f
	 * @param p
	 * @param s
	 */
	public ComponentColorDialog(JFrame f, Manager p, StructureViewer s, String title, StructureComponent sc, boolean ribbon){
	
		super(f, title, false);

		colorType = COLOR_BY_ELEMENT;
	
		parentFrame = f;
		parent = p;
		sv = s;
		structureComponent = sc;
		this.ribbon = ribbon;
		
		parent.setDisplayDialogStatus(true);
		
		itself = this;
		structureComponent = sc;
		
		
		
		//create the panels
		contentPane = getContentPane();
		contentPane.setLayout(null);
		
		base.setPreferredSize(new Dimension(210, 270));
		base.setBorder(new BevelBorder(BevelBorder.RAISED));
		base.setLocation(new Point(5,5));
		base.setBounds(5, 5, 210, 270);
		base.setLayout(null);
		
		//color panel
		colorPanel.setPreferredSize(new Dimension(200, 260));
		colorPanel.setLocation(new Point(205,0));
		colorPanel.setBounds(5,5,200,260);
		colorPanel.setLayout(null);
		
		border1 = BorderFactory.createEmptyBorder(0,0,0,0);//left offset of the boxes in the box
		border2 = BorderFactory.createTitledBorder(new EtchedBorder(), "  Color ");
		border2.setTitleColor(Color.black);
		border2.setTitleFont(new Font("Dialog", 0, 12));
		border = BorderFactory.createCompoundBorder(border1, border2);
		colorPanel.setBorder(border);
		
		//add radio buttons
		radio6.setLocation(20,20);
		radio6.setBounds(20,20,150,15);
		radio6.setVisible(true);
		radio6.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				colorType = COLOR_BY_ELEMENT;
				rgbField.setText("");
				rgbField.setEditable(false);
				rgbField.setEnabled(false);
				colorButton.setEnabled(false);
			}
		});
		
		radio5.setLocation(20,40);
		radio5.setBounds(20,40,150,15);
		radio5.setVisible(true);
		radio5.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				colorType = COLOR_BY_CHARGE;
				rgbField.setText("");
				rgbField.setEditable(false);
				rgbField.setEnabled(false);
				colorButton.setEnabled(false);
			}
		});

		radio7.setLocation(20,60);
		radio7.setBounds(20,60,150,15);
		radio7.setVisible(true);
		radio7.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				colorType = COLOR_CUSTOM;
				rgbField.setEditable(true);
				rgbField.setEnabled(true);
				colorButton.setEnabled(true);
			}
		});

		colorGroup.add(radio6);
		colorGroup.add(radio5);
		colorGroup.add(radio7);
		
		colorPanel.add(radio6);
		colorPanel.add(radio5);
		colorPanel.add(radio7);


		//add color squares
		
		m_image = new BufferedImage(162,122, BufferedImage.TYPE_INT_ARGB);

		Graphics2D g2 = m_image.createGraphics();
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		g2.setFont(new Font("Arial", Font.PLAIN, 10));

		//first row
		g2.setColor(Color.white);
		g2.fillRect(0,0,40,30);
		
		g2.setColor(new Color(160,160,160));
		g2.fillRect(40,0,40,30);
		
		g2.setColor(Color.black);
		g2.fillRect(80,0,40,30);
		
		g2.setColor(Color.red);
		g2.fillRect(120,0,40,30);
		
		//second row
		g2.setColor(Color.yellow);
		g2.fillRect(0,30,40,30);
		
		g2.setColor(new Color(210,210,0));
		g2.fillRect(40,30,40,30);
		
		g2.setColor(new Color(150,150,0));
		g2.fillRect(80,30,40,30);
		
		g2.setColor(new Color(255,100,200));
		g2.fillRect(120,30,40,30);
		
		//third row
		g2.setColor(new Color(100,200,255));
		g2.fillRect(0,60,40,30);
		
		g2.setColor(new Color(100,100,255));
		g2.fillRect(40,60,40,30);
		
		g2.setColor(Color.blue);
		g2.fillRect(80,60,40,30);
		
		g2.setColor(new Color(200,0,200));
		g2.fillRect(120,60,40,30);
		
		//fourth row
		g2.setColor(Color.green);
		g2.fillRect(0,90,40,30);

		g2.setColor(new Color(0,200,0));
		g2.fillRect(40,90,40,30);
		
		g2.setColor(new Color(0,150,0));
		g2.fillRect(80,90,40,30);
		
		g2.setColor(new Color(150,0,0));
		g2.fillRect(120,90,40,30);
		
		//lines
		g2.setColor(Color.black);
		g2.drawLine(0,0,0,120);
		g2.drawLine(0,1,160,1);
		g2.drawLine(0,120,160,120);
		g2.drawLine(160,0,160,120);
		
		for (int i = 0; i <= 3; i++){
			//horizontal line
			g2.drawLine(0, i*30, 160, i*30);
			
			//vertical line
			g2.drawLine(i*40, 0, i*40, 120);
		}
		
		
		
		
		ImageIcon image = new ImageIcon(m_image);
		JLabel imageLabel = new JLabel();
		imageLabel.setIcon(image);
		
		JPanel cPanel = new JPanel();
		cPanel.setPreferredSize(new Dimension(165, 125));
		cPanel.setLocation(new Point(15,80));
		cPanel.setBounds(15,80,165,125);
		cPanel.setLayout(null);
		
		imageLabel.setBounds(1, 1, 161, 121);
		cPanel.add(imageLabel);
		
		//set up the rgb text field
		
		rgbLabel.setForeground(Color.black);
		rgbLabel.setFont(new Font("Dialog", Font.PLAIN, 12));
		rgbLabel.setText("Enter RGB color components");
		rgbLabel.setPreferredSize(new Dimension(170,20));
		rgbLabel.setBounds(15,210,170,20);
		rgbLabel.setLocation(new Point(15,210));
		rgbLabel.setVisible(true);
		
		rgbField.setPreferredSize(new Dimension(100,20));
		rgbField.setBounds(15,230,100,20);
		rgbField.setLocation(new Point(15,230));
		rgbField.setEditable(false);
		rgbField.setEnabled(false);
		rgbField.setVisible(true);
		rgbField.addMouseListener(new MouseAdapter(){
			public void mouseClicked(MouseEvent m){
				colorType = COLOR_CUSTOM;
				radio6.setSelected(false);
				radio5.setSelected(false);
				radio7.setSelected(true);
				rgbField.setEditable(true);
				rgbField.setEnabled(true);
				colorButton.setEnabled(true);
			}
		});
					
		
		
		//add the color chooser button
		colorButton = new JButton("Choose");
		colorButton.setFont(new Font("Helvetica", Font.PLAIN, 10));
		colorButton.setPreferredSize(new Dimension(70,20));
		colorButton.setLocation(new Point(120, 230));
		colorButton.setBounds(120, 230, 70, 20);
		colorButton.setVisible(true);
		colorButton.setEnabled(false);

		colorButton.addMouseListener(new MouseAdapter(){
			public void mouseClicked(MouseEvent m){
				colorType = COLOR_CUSTOM;
				radio6.setSelected(false);
				radio5.setSelected(false);
				radio7.setSelected(true);
				rgbField.setEditable(true);
				rgbField.setEnabled(true);
				colorButton.setEnabled(true);
				
				colorFlag = true;
				Color c = JColorChooser.showDialog(null, "Choose selection color", structureColor);
				colorFlag = false;
				if (c == null){
					return;
				}
				
				structureColor = c;
				
				String text = structureColor.getRed() + "," + structureColor.getGreen() + "," + structureColor.getBlue();
				rgbField.setText(text);
				
			}
		});


		colorButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				colorFlag = true;
				Color c = JColorChooser.showDialog(null, "Choose selection color", structureColor);
				colorFlag = false;
				if (c == null){
					return;
				}
				
				structureColor = c;
				
				String text = structureColor.getRed() + "," + structureColor.getGreen() + "," + structureColor.getBlue();
				rgbField.setText(text);
			}
		});
		
		
		//add mouse listener to detect which color has been clicked
		MouseInputAdapter mia = new MouseInputAdapter(){

			public void mouseClicked(MouseEvent e){
			
				if (colorType == COLOR_BY_ELEMENT){
					radio7.setSelected(true);
					radio6.setSelected(false);
					rgbField.setEnabled(true);
					colorButton.setEnabled(true);
					colorType = COLOR_CUSTOM;
				}
				else if (colorType == COLOR_BY_CHARGE){
					radio7.setSelected(true);
					radio5.setSelected(false);
					rgbField.setEnabled(true);
					colorButton.setEnabled(true);
					colorType = COLOR_CUSTOM;
				}
			
				//get the pixel
				int X = e.getX();//with the margins
				int Y = e.getY();
				
				if ((X > m_image.getWidth())||(Y > m_image.getHeight())){
					return;
				}
				
				long pixel = m_image.getRGB(X,Y);

				if (pixel == -1){
					structureColor = Color.white;
					rgbField.setText("255,255,255");
				}
				else if (pixel == -16777216){
					structureColor = Color.black;
					rgbField.setText("0,0,0");
				}
				else if (pixel == -256){
					structureColor = Color.yellow;
					rgbField.setText("255,255,0");
				}
				else if (pixel == -65536){
					structureColor = Color.red;
					rgbField.setText("255,0,0");
				}
				else if (pixel == -6250336){
					structureColor = new Color(160,160,160);
					rgbField.setText("160,160,160");
				}
				else if (pixel == -2960896){
					structureColor = new Color(210,210,0);
					rgbField.setText("210,210,0");
				}
				else if (pixel == -6908416){
					structureColor = new Color(150,150,0);
					rgbField.setText("150,150,0");
				}
				else if (pixel == -39736){
					structureColor = new Color(255,100,200);
					rgbField.setText("255,100,200");
				}
				else if (pixel == -10172161){
					structureColor = new Color(100,200,255);
					rgbField.setText("100,200,255");
				}
				else if (pixel == -10197761){
					structureColor = new Color(100,100,255);
					rgbField.setText("100,100,255");
				}
				else if (pixel == -16776961){
					structureColor = Color.blue;
					rgbField.setText("0,0,255");
				}
				else if (pixel == -3669816){
					structureColor = new Color(200,0,200);
					rgbField.setText("200,0,200");
				}
				else if (pixel == -16711936){
					structureColor = Color.green;
					rgbField.setText("0,255,0");
				}
				else if (pixel == -16726016){
					structureColor = new Color(0,200,0);
					rgbField.setText("0,200,0");
				}
				else if (pixel == -16738816){
					structureColor = new Color(0,150,0);
					rgbField.setText("0,150,0");
				}
				else if (pixel == -6946816){
					structureColor = new Color(150,0,0);
					rgbField.setText("150,0,0");
				}
				
			}
		};
		
		cPanel.addMouseListener(mia);
		
		colorPanel.add(cPanel);
		colorPanel.add(rgbLabel);
		colorPanel.add(rgbField);
		colorPanel.add(colorButton);
		
		//add panels to the base
		base.add(colorPanel);
		
		
		//buttons
		okButton.setPreferredSize(new Dimension(80,25));
		okButton.setLocation(new Point(10, 285));
		okButton.setBounds(10, 285, 80, 25);
		okButton.setVisible(true);
		okButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				process();
			}
		});
		
		
		cancelButton.setPreferredSize(new Dimension(80,25));
		cancelButton.setLocation(new Point(110, 285));
		cancelButton.setBounds(110, 285, 80, 25);
		cancelButton.setVisible(true);
		cancelButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
		});
		
		contentPane.add(base);
		contentPane.add(okButton);
		contentPane.add(cancelButton);
		
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
			
			public void windowDeactivated(WindowEvent e){
				if (colorFlag){
					return;
				}
				toFront();
			}
		});
				
		
		Dimension dim = getToolkit().getScreenSize();
		if ((parentFrame.getX() + parentFrame.getWidth() + width) > dim.getWidth()){
			setBounds(dim.width - width, parentFrame.getY(), width, height);
		}
		else{
			setBounds(parentFrame.getX() + parentFrame.getWidth(), parentFrame.getY(), width, height);
		}
		
		setSize(width, height);
		setResizable(false);
		setVisible(true);
	}
	
	/**
	 * This method is called by StructureViewer in response to a mouse click and sets the currently
	 * active text field to the String representation of the clicked StructureComponent.
	 * @see edu.sdsc.sirius.viewers.StructureViewer
	 * @see edu.sdsc.mbt.StructureComponent
	 */
	public void processPick(String data, StructureComponent structureComponent){}
	
	private void process(){
		
		//add error checking for selection bounds
		//eg., selection level is residue, but entered only structure name
		try{
		
			setVisible(false);
			parent.setDisplayDialogStatus(false);
			if (structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_ATOM){
				AtomColor ac = null;
				if (colorType == COLOR_BY_ELEMENT){
					ac = AtomColorByElement.create();
				}
				else if (colorType == COLOR_BY_CHARGE){
					ac = AtomColorByCharge.create();
				}
				else{
					ac = AtomColorFactory.getAtomColor(structureColor);
				}
				structureComponent.structure.getStructureMap().getStructureStyles().setAtomColor(structureComponent,ac, true, true);
				structureComponent.structure.getStructureMap().getStructureStyles().setSelected(structureComponent, false, true, false);
				parent.updateView();
			}
			else if (structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_RESIDUE){
				RibbonColor ribbonColor = null;
				ResidueColor rc = null;
				if (colorType == COLOR_BY_ELEMENT){
					rc = ResidueColorDefault.create();
					ribbonColor = RibbonColorDefault.create();
					structureComponent.structure.getStructureMap().getStructureStyles().setResidueColor(structureComponent,rc, true, true);
					structureComponent.structure.getStructureMap().getStructureStyles().setSelected(structureComponent, false, true, false);
				}
				else if (colorType == COLOR_BY_CHARGE){
					rc = ResidueColorByCharge.create();
					ribbonColor = RibbonColorDefault.create();
					structureComponent.structure.getStructureMap().getStructureStyles().setResidueColor(structureComponent,rc, true, true);
					structureComponent.structure.getStructureMap().getStructureStyles().setSelected(structureComponent, false, true, false);
				}
				else{
					rc = ResidueColorFactory.getResidueColor(structureColor);
					ribbonColor = RibbonColorFactory.getRibbonColor(structureColor);
					
					structureComponent.structure.getStructureMap().getStructureStyles().setResidueColor(structureComponent,rc, true, true);
					structureComponent.structure.getStructureMap().getStructureStyles().setSelected(structureComponent, false, true, false);
				}
				
				if (StylesPreferences.commonRibbonColoring){
					StructureMap map = ((Residue)structureComponent).structure.getStructureMap();
					structureComponent.structure.getStructureMap().getStructureStyles().setRibbonColor((Residue)structureComponent, ribbonColor, true, false);
					structureComponent.structure.getStructureMap().getStructureStyles().updateChain(map.getChain(((Residue)structureComponent).getAtom(0)), true);
				}

				
				parent.updateView();
			}
			else if (structureComponent.getStructureComponentType() == StructureComponentRegistry.TYPE_CHAIN){
				
				Chain chain = (Chain)structureComponent;
				StructureMap map = chain.structure.getStructureMap();
				if (ribbon){
					sv.colorRibbon(chain, colorType, structureColor);
					return;
				}

				ResidueColor rc = null;
				RibbonColor ribbonColor = null;
				if (colorType == COLOR_BY_ELEMENT){
					rc = ResidueColorDefault.create();
					ribbonColor = RibbonColorDefault.create();
				}
				else if (colorType == COLOR_BY_CHARGE){
					rc = ResidueColorByCharge.create();
					ribbonColor = RibbonColorDefault.create();
				}
				else{
					rc = ResidueColorFactory.getResidueColor(structureColor);
					ribbonColor = RibbonColorFactory.getRibbonColor(structureColor);
				}

				for (int i = 0; i < chain.getResidueCount(); i++){
					Residue r = chain.getResidue(i);
					r.structure.getStructureMap().getStructureStyles().setResidueColor(r, rc, true, true);
					r.structure.getStructureMap().getStructureStyles().setSelected(r, false, true, false);
					
					if (StylesPreferences.commonRibbonColoring){
						map.getStructureStyles().setRibbonColor(r, ribbonColor, true, false);
					}
				}

				if (StylesPreferences.commonRibbonColoring){
					structureComponent.structure.getStructureMap().getStructureStyles().updateChain(chain, true);
				}
				
				parent.updateView();
			}
		}
		catch (Exception e){
			setVisible(false);
			parent.setDisplayDialogStatus(false);
			parent.displayExceptionMessage("Exception coloring structure", e);

		}

		dispose();

	}
}
