//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs;

import java.util.*;

import javax.swing.*;

import java.awt.*;
import java.awt.event.*;

import edu.sdsc.sirius.dialogs.TableImpl.*;
import edu.sdsc.sirius.io.*;
import edu.sdsc.sirius.util.Manager;
import edu.sdsc.mbt.Structure;
import edu.sdsc.mbt.viewables.*;

import java.io.*;
import java.net.URL;

/**
 * This class implements a dialog that displays the currently loaded entries.
 * It retrieves data from structureDocument, packages it as DataSet and displays it in ResultPanel.
 * @see edu.sdsc.mbt.util.DataSet
 * @see edu.sdsc.sirius.search.ResultPanel
 * 
 * @author Oleksandr V. Buzko
 */
public class DataListerDialog extends JDialog implements ResultPanelCallable, EntryLister {
	
	private Container contentPane;
	private JDialog self;
	
	private int width = 300;
	private int height = 350;
	
	private TableViewPanel contentPanel;
	private String workingDir = ".";
	
	private Vector searchTypes;
	
	private Manager parent;
	private JFrame parentFrame;
	
	private boolean express = true;
	
	private StructureDocument structureDocument;
	
	private JPanel base = new JPanel();
	private JPanel buttonPanel = new JPanel();
	private JButton loadButton = new JButton("Load");
	private JButton removeButton = new JButton("Remove");
	private JButton closeButton = new JButton("Close");
	
	private String initialTitle;
	
	private HashMap replacement;
	private String parentId;
	
	private DataSet set;
	
	private String filename;
	
	private File file;
	private URL url;
	private Vector data;
	private FileEntryLoader loader;
	
	private boolean scores = false;//whether we need another data column
	
	private Entry[] entries;//array that stores entry object for each loaded item and null for those not loaded
	
	private JCheckBoxMenuItem menuScanMode = new JCheckBoxMenuItem("Enable last/next mode");
	private JCheckBoxMenuItem menuHB = new JCheckBoxMenuItem("Display hydrogen bonds");

		
	/**
	 * Constructor that creates the dialog.
	 * @param f parent frame
	 * @param p Manager object that will process the RequestSet
	 * @param title dialog title
	 * @param modal whether the dialog is modal (other frames in the application will be inaccessible)
	 */
	public DataListerDialog (JFrame f, Manager p, StructureDocument d, FileEntryLoader loader, Vector data, File file){
	
		super(f, "Record browser", false);
		initialTitle = "Record browser - available records";
		parent = p;
		this.loader = loader;
		this.file = file;
		structureDocument = d;
		this.data = data;
		
		express = parent.getListScanMode();
		initialize();
		
	}
	
	public DataListerDialog (JFrame f, Manager p, StructureDocument d, FileEntryLoader loader, Vector data, URL url){
	
		super(f, "Record browser", false);
		initialTitle = "Record browser - available records";
		parent = p;
		this.loader = loader;
		this.url = url;
		structureDocument = d;
		this.data = data;
		
		initialize();
		
	}
	
	private void initialize(){
		//get file name
		if (file != null){
		String fname = file.toString();
			int index = fname.lastIndexOf(File.separator);
			filename = fname.substring(index+1);
			long length = file.length();
	
			if (length > 5000000){
				parent.displayWaitScreen();
			}
			
		}
		else if (url != null){
			filename = "PDB entry";
		}
		else{
			parent.displayErrorMessage("No data found in the PDB record");
			dispose();
			return;
		}
		

		
		set = new DataSet();
		//get the dataset from the structureDocument
		for (int j = 0; j < data.size(); j++){
			//create a DataBlock out of this entry
			if (data.get(j) instanceof String){
				scores = false;
				DataBlock block = new DataBlock(new Integer(j).toString());
				block.setName((String)data.get(j));
				set.addData(block);
			}
			else if (data.get(j) instanceof SDFHeader){
				scores = true;
				express = true;
				SDFHeader h = (SDFHeader)data.get(j);
				String name = h.name;
				String value = h.value;
				DataBlock block = new DataBlock(new Integer(j).toString());
				block.setName(name);
				block.setDescription(value);
				set.addData(block);
			}
		}
		
		//get the menubar
		JMenuBar menuBar = new JMenuBar();
		JMenu menuOptions = new JMenu("Options");

		menuScanMode.setSelected(scores);//if these are SDFs with scores, enable the scan mode
		menuScanMode.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				parent.setListScanMode(menuScanMode.isSelected());
				express = menuScanMode.isSelected();
				if (express){
					loadButton.setText("Last");
					removeButton.setText("Next");
				}
				else{
					loadButton.setText("Load");
					removeButton.setText("Remove");
				}
			}
		});
		
		express = menuScanMode.isSelected();
		
		menuHB.setSelected(scores);//same for HB (makes no sense to calculate HBs between NMR models)
		menuOptions.add(menuScanMode);
		menuOptions.add(menuHB);
		
		menuBar.add(menuOptions);
		
		JMenu menuTools = new JMenu("Tools");
		JMenuItem menuToolsFind = new JMenuItem("Find entry...");
		menuToolsFind.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				Thread runner = new Thread(){
					public void run(){
						
						String input = (String)JOptionPane.showInputDialog(base, "Enter name of the entry", "Find entry", 
								JOptionPane.INFORMATION_MESSAGE, null, null, "");
						
						
						if (input == null){
							return;
						}

						//now look through the data model and determine index of the desired record
						if (set == null) return;
						
						contentPanel.deselectAll();
						
						boolean found = false;
						int index = -1;//first found index
						for (int i = 0; i < set.size(); i++){
							DataBlock block = set.getData(i);
							if (block.getName().startsWith(input)){
								//this is the index
								//select the row and scroll to it
								contentPanel.setSelection(true, i);
								if (index == -1) index = i;
								found = true;
							}
						}
						
						if (!found){
							parent.displayMessage("No matching records found");
						}
						else{
							//scroll to the first index
							contentPanel.scrollToRow(index);
						}
					}
				};
				runner.start();
			}
		});
		
		JMenuItem menuToolsDeselect = new JMenuItem("Clear entry selection");
		menuToolsDeselect.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				Thread runner = new Thread(){
					public void run(){
						
						contentPanel.deselectAll();
						
					}
				};
				runner.start();
			}
		});

		
		
		parent.setListScanMode(scores);
		
		
		
		//go through the available data options and determine the future table headers
		searchTypes = new Vector();
		searchTypes.add("Name");
		if (scores){
			searchTypes.add("Score");
		}
		parentFrame = parent.getApplicationFrame();
		
		contentPane = getContentPane();
		contentPane.setLayout(new BorderLayout());
		
		
		self = this;
		
		contentPane.add(menuBar, BorderLayout.NORTH);
		

//		base.setBorder(new BevelBorder(BevelBorder.LOWERED));
		
		
		//add window listener
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				parent.removeEntryLister((EntryLister)self);
				setVisible(false);
				dispose();
			}
		});
		
		addComponentListener(new ComponentAdapter(){
			public void componentResized(ComponentEvent e){
				Dimension size = getSize();
				if (contentPanel == null) return;
				contentPanel.resizePanel(size);
			}	
		});
		
		base.setLayout(new BorderLayout());
		contentPane.add(base, BorderLayout.CENTER);
		
		
		//panel that holds buttons
		JPanel lowerPanel = new JPanel();
		lowerPanel.setLayout(new BorderLayout());
		JPanel borderPanel = new JPanel();
		borderPanel.setLayout(new BorderLayout());
		borderPanel.add(new JSeparator(), BorderLayout.NORTH);
		
		lowerPanel.add(borderPanel, BorderLayout.NORTH);
		
		buttonPanel.setLayout(new FlowLayout(FlowLayout.CENTER, 10, 5));
		
		loadButton.setPreferredSize(new Dimension(80,25));
		if (express) loadButton.setText("Last");
		loadButton.setVisible(true);
		loadButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				if (contentPanel != null){
					if (express){
						lastEntry();
						return;
					}
					if (contentPanel.getSelectedRowCount() > 0){
						if (express){
							lastEntry();
						}
						else{
							loadSelection();
						}
					}
				}
			}
		});
		
		removeButton.setPreferredSize(new Dimension(80,25));
		if (express) removeButton.setText("Next");
		removeButton.setVisible(true);
		removeButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				if (contentPanel != null){
					if (express){
						nextEntry();
						return;
					}
					
					if (contentPanel.getSelectedRowCount() > 0){
						if (express){
							nextEntry();
						}
						else{
							removeSelection();
						}
					}
				}
			}
		});
		
		closeButton.setPreferredSize(new Dimension(80,25));
		closeButton.setVisible(true);
		closeButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				parent.removeEntryLister((EntryLister)self);
				setVisible(false);
				dispose();
			}
		});
		
		buttonPanel.add(loadButton);
		buttonPanel.add(removeButton);
		buttonPanel.add(closeButton);
		
		
		lowerPanel.add(buttonPanel, BorderLayout.CENTER);
		contentPane.add(lowerPanel, BorderLayout.SOUTH);
		
				
		if (set != null && set.size() > 0){
			setData(set);
			setTitle(filename + ": " + set.size() + " records");
		}
		else{
			parent.displayErrorMessage("Nothing to display: no records found");
			dispose();
			return;
		}
		
		entries = new Entry[set.size()];
				
		parent.removeWaitScreen();
		
		Dimension dim = getToolkit().getScreenSize();
		if ((parentFrame.getX() + parentFrame.getWidth() + width) > dim.getWidth()){
			setBounds(dim.width - width, parentFrame.getY(), width, height);
		}
		else{
			setBounds(parentFrame.getX() + parentFrame.getWidth(), parentFrame.getY(), width, height);
		}
		setSize(width, height);
		
		setVisible(true);
	}
	
	/**
	 * Sets the display to the given DataSet.
	 */
	public void setData(DataSet set){
	
		if (contentPanel != null){
			return;
		}
		
		//used to show a table in a previously open frame
		//(after a new search initiated from this frame)
//		contentPanel = new ResultPanel(this, set, parent, ResultPanel.NAVIGATION_TABLE, searchTypes, replacement);
//		contentPane.remove(base);
		contentPanel = new TableViewPanel(base, set, parent, searchTypes, TableViewPanel.SECOND_COLUMN_NARROW, true);
		base.add(contentPanel, BorderLayout.CENTER);
		base.revalidate();
		base.repaint();
	}
	
	public void loadSelection(){
		
		//get the selected indices
		int[] selection = contentPanel.getSelectedRows();
		if (selection.length == 0) return;
		
		for (int i = 0; i < selection.length; i++){
			int row = selection[i];
			if (row >= set.size()) continue;
			if (entries[row] != null) continue;//already loaded
			Entry e = null;
			if (file != null){
				e = loader.load(file, row);
			}
			else if (url != null){
				e = loader.load(url, row);
			}
			parent.addEntry(e, true, true);
			entries[row] = e;
			contentPanel.setSelection(true, row);
		}
		contentPanel.clearRowSelection();
	}
	
	public void removeSelection(){
		
		//get the selected indices
		int[] selection = contentPanel.getSelectedRows();
		if (selection.length == 0) return;
		
		for (int i = 0; i < selection.length; i++){
			int row = selection[i];
			if (row >= set.size()) continue;
			if (entries[row] == null) continue;//not loaded
//			System.out.println(entries[row]);
//			System.out.println("structureDocument = " + structureDocument);
			structureDocument.parent.removeCloseListener(entries[row]);
			structureDocument.removeEntry(entries[row]);
			entries[row] = null;
			contentPanel.setSelection(false, row);
		}
		contentPanel.clearRowSelection();
				
	}
	
	public void removeEntry(Entry entry){
		for (int i = 0; i < entries.length; i++){
			if (entries[i] == entry){
				entries[i] = null;
				contentPanel.setSelection(false, i);
			}
		}
		contentPanel.clearRowSelection();
	}
	
	public void clearEntries(){
		for (int i = 0; i < entries.length; i++){
			entries[i] = null;
			contentPanel.setSelection(false, i);
		}
		contentPanel.clearRowSelection();
	}

	public void setNumberOfHits(int n){}
	
	public void nextEntry(){
		
		//check what's currently loaded
		int next = -1;
		int target = -1;
		for (int i = 0; i < entries.length; i++){
			Entry e = entries[i];
			if (e != null){
				target = i;
				next = i + 1;
				break;
			}
		}
		
		if (next >= entries.length){
			parent.displayMessage("No more entries.");
			return;
		}
		
		if (target == -1){
			//nothing is loaded. Look at the selection
			int[] selection = contentPanel.getSelectedRows();
			if (selection.length != 1){
				parent.displayMessage("Only one item must be selected");
				return;
			}
			
			next = selection[0];
		}
		else{
			structureDocument.parent.removeCloseListener(entries[target]);
			structureDocument.removeEntry(entries[target]);
			entries[target] = null;
			contentPanel.setSelection(false, target);
		}
		
		//load next
		if (entries[next] != null) return;//already loaded
		Entry e = null;
		if (file != null){
			e = loader.load(file, next);
		}
		else if (url != null){
			e = loader.load(url, next);
		}
		
		parent.addEntry(e, true, true);
		entries[next] = e;
		contentPanel.setSelection(true, next);
		
		contentPanel.clearRowSelection();
		
		//show HBs if needed
		//calculate HBs between the newly loaded entry and the first of loaded
		//entries that did not come from the SDF file
		if (menuHB.isSelected()){
			Vector entryList = parent.getStructureViewer().getLoadedEntries();
			
			Entry protein = null;
			
			for (int i = 0; i < entryList.size(); i++){
				Entry ee = (Entry)entryList.get(i);
				//check whether this entry is in the entries array
				boolean present = false;//whether the loaded entry is in the SDF list
				for (int j = 0; j < entries.length; j++){
					if (entries[j] == ee){
						present = true;
						break;
					}
				}
				
				if (!present){
					protein = ee;
					break;
				}
			}
			
			if (protein != null){
				showHydrogenBonds(protein, e);
			}
		
		}
	}
	
	public void lastEntry(){
		
		//check what's currently loaded
		int last = -1;
		int target = -1;
		for (int i = 0; i < entries.length; i++){
			Entry e = entries[i];
			if (e != null){
				target = i;
				last = i - 1;
				break;
			}
		}
		
		if (last < 0) return;
		
		if (target == -1){
			//nothing is loaded. Look at the selection
			int[] selection = contentPanel.getSelectedRows();
			if (selection.length != 1){
				parent.displayMessage("Only one item must be selected");
				return;
			}
			
			last = selection[0];
		}
		else{
			structureDocument.parent.removeCloseListener(entries[target]);
			structureDocument.removeEntry(entries[target]);
			entries[target] = null;
			contentPanel.setSelection(false, target);
		}
		
		//load last
		if (entries[last] != null) return;//already loaded
		Entry e = null;
		if (file != null){
			e = loader.load(file, last);
		}
		else if (url != null){
			e = loader.load(url, last);
		}
		parent.addEntry(e, true, true);
		entries[last] = e;
		contentPanel.setSelection(true, last);
		
		contentPanel.clearRowSelection();
			
		//show HBs if needed
		//calculate HBs between the newly loaded entry and the first of loaded
		//entries that did not come from the SDF file
		if (menuHB.isSelected()){
			Vector entryList = parent.getStructureViewer().getLoadedEntries();
			
			Entry protein = null;
			
			for (int i = 0; i < entryList.size(); i++){
				Entry ee = (Entry)entryList.get(i);
				//check whether this entry is in the entries array
				boolean present = false;//whether the loaded entry is in the SDF list
				for (int j = 0; j < entries.length; j++){
					if (entries[j] == ee){
						present = true;
						break;
					}
				}
				
				if (!present){
					protein = ee;
					break;
				}
			}
			
			if (protein != null){
				showHydrogenBonds(protein, e);
			}
		
		}

	}
	
	public void setSelectedIndices(Vector v){}
	
	private void showHydrogenBonds(Entry entry1, Entry entry2){
		
		if (entry1 == null || entry2 == null) return;
		
		try{
			Vector ligand = new Vector();
			Vector protein = new Vector();
			
			Structure pStructure = ((StructureEntry)entry1).getStructure();
			Structure lStructure = ((StructureEntry)entry2).getStructure();
			
			for (int i = 0; i < lStructure.getStructureMap().getAtomCount(); i++){
				ligand.add(lStructure.getStructureMap().getAtom(i));
			}
			
			for (int i = 0; i < pStructure.getStructureMap().getAtomCount(); i++){
				protein.add(pStructure.getStructureMap().getAtom(i));
			}
			
			parent.getStructureViewer().detectHydrogenBonds(ligand, protein, 3.3, false);
		}
		catch (Exception ex){
			parent.displayExceptionMessage("Exception displaying hydrogen bonds in the complex", ex);
		}
		
	}
	
 
}