package edu.sdsc.sirius.dialogs;

// MBT
import edu.sdsc.mbt.image.*;

import edu.sdsc.mbt.viewables.*;
import edu.sdsc.sirius.util.Manager;
import edu.sdsc.sirius.util.FileNameFilter;
import edu.sdsc.sirius.util.PagePrintable;

// Core
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.border.*;
import java.awt.image.*;
import java.io.*;
import java.awt.print.*;

import javax.imageio.*;
import javax.imageio.spi.*;
import javax.imageio.stream.*;

public class DistributionViewer extends JDialog {

	private HashMap data;
	private JTabbedPane pane = new JTabbedPane();
	private JPanel imagePanel = new JPanel();
	private JPanel tablePanel = new JPanel();
	
	private JDialog distributionFrame;
	private JFrame parent;
	
//	private HashMap aa;
//	private HashMap tt;
	
	private String[] aminoacids = { "A", "C", "D", "E", "F", "G", "H", "I", "K", "L","M", "N", "P", "Q", "R","S", "T", "V", "W", "Y", "-" };
	private int[] scales = { 1, 5, 10, 50, 100, 500, 1000, 5000, 10000 };//allowed scale intervals
	private static String[] organismList = { "Homo sapiens", "Mus musculus", "Drosophila melanogaster", "Caenorhabditis elegans", 
		"Saccharomyces cerevisiae", "Arabidopsis thaliana", "Gallus gallus", "Rattus norvegicus", "Xenopus laevis", "Oryza sativa", "Other" };
	
	private static Vector organisms = new Vector();
	
	static{
		organisms.add("Homo sapiens");
		organisms.add("Mus musculus");
		organisms.add("Drosophila melanogaster");
		organisms.add("Caenorhabditis elegans");
		organisms.add("Saccharomyces cerevisiae");
		organisms.add("Arabidopsis thaliana");
		organisms.add("Gallus gallus");
		organisms.add("Rattus norvegicus");
		organisms.add("Xenopus laevis");
		organisms.add("Oryza sativa");
		
		
		
	};
	
	private static HashMap organismLookup = new HashMap();
	static{
		for (int i = 0 ; i < organismList.length; i++){
			organismLookup.put(organismList[i], "");
		}
	};
	
	
	private HashMap organismIntColors = new HashMap();//int RGB color -> type
	private int total = 0;//total number of aminoacid occurrences across all organisms in the set
	
	private HashMap colors = new HashMap();
	private BufferedImage m_image;
	private Graphics2D g2;
	
	private JTable m_table;
	private DistributionData m_data;
	private JScrollPane sp = new JScrollPane();
	private JViewport v = new JViewport();
	
	private int imageWidth = 920;
	private int imageHeight = 660;
	private int leftMargin = 50;
	private int bottomMargin = 60;
	private int topMargin = 50;
	
	private int rowHeight = 18;
	
	private Object value = new Object();//as a placeholder value for hashes
	
	private MouseInputAdapter mia1;
	private MouseInputAdapter mia2;
	protected static int counter1 = 0;
	protected static int counter2 = 0;
	
	private JPopupMenu imagePopup;
	private JPopupMenu tablePopup;

	private Book book = null;
	public PageFormat mPageFormat;
	protected int m_maxNumPage = 1;
	
	
	private Manager callable;

	
	public DistributionViewer(JFrame f, StructureDocument sd, HashMap d){
	
		super(f, "Residue distribution", true);
		parent = f;
		data = d;
		
		callable = sd.parent;
		
		//initialize the color maps
		colors.put("Homo sapiens", Color.red);
		colors.put("Mus musculus", new Color(100,200,255));
		colors.put("Drosophila melanogaster", new Color(255, 0, 255));
		colors.put("Caenorhabditis elegans", Color.blue);
		colors.put("Saccharomyces cerevisiae", Color.yellow);
		colors.put("Arabidopsis thaliana", new Color(0,170,0));
		colors.put("Gallus gallus", new Color(100,100,255));
		colors.put("Rattus norvegicus", new Color(150,0,0));
		colors.put("Xenopus laevis", new Color(150,150,0));
		colors.put("Oryza sativa", new Color(0,200,0));
		colors.put("Other", new Color(160,160,160));
		
		organismIntColors.put(new Integer(-65536), "Homo sapiens");
		organismIntColors.put(new Integer(-16733696), "Arabidopsis thaliana");
		organismIntColors.put(new Integer(-65281), "Drosophila melanogaster");
		organismIntColors.put(new Integer(-16776961), "Caenorhabditis elegans");
		organismIntColors.put(new Integer(-256), "Saccharomyces cerevisiae");
		organismIntColors.put(new Integer(-10172161), "Mus musculus");
		organismIntColors.put(new Integer(-6946816), "Rattus norvegicus");
		organismIntColors.put(new Integer(-10197761), "Gallus gallus");
		organismIntColors.put(new Integer(-6908416), "Xenopus laevis");
		organismIntColors.put(new Integer(-16726016), "Oryza sativa");
		organismIntColors.put(new Integer(-6250336), "Other");
		
		
		
		//create the image and place it on the image panel
		imagePanel.setPreferredSize(new Dimension(880,695));
		tablePanel.setPreferredSize(new Dimension(880,695));
		tablePanel.setLayout(new BorderLayout());
		
		Image graph = paintImage();
		ImageIcon icon = new ImageIcon(graph);
		JLabel image = new JLabel(icon);
		imagePanel.add(image);
		
		imagePopup = createImagePopup();
		
		mia1 = new MouseInputAdapter(){
		
			boolean popup = false;
//			private int counter = 0;
			
			public void mousePressed(MouseEvent e){
				setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
				checkPopup(e);
			}

			public void mouseReleased(MouseEvent e){
				checkPopup(e);
			}
			
			public void mouseClicked(MouseEvent e){
				checkPopup(e);
			}

			public void checkPopup(MouseEvent e){
				if (e.isPopupTrigger()){
					popup = true;
				}
				start(e, popup);
			}
			
			public void start(MouseEvent e, boolean p){
				counter1++;
				if (counter1 == 3){
					counter1 = 0;
					if (popup){
						imagePopup.show(imagePanel, e.getX(), e.getY());
//						System.out.println("popup");
					}
					else{
//						System.out.println("Processing click");
					}
					popup = false;
				}
				else{
//					System.out.println("counter = " + counter);
				}
			}
			
			public void mouseMoved(MouseEvent e){
				int lastX = e.getX();
				int lastY = e.getY();
				
				//calculate which part of the X axis this point corresponds to
				int spanX = lastX - leftMargin - 15;
								
				int N = (int)(spanX/30);
				
				if ((N > 20)||(N < 0)){
					eraseTooltip();
					return;
				}
				String aaa = aminoacids[N];
				
				int left = leftMargin + 15 + N*30;
				int right = left + 15;
				
				if ((lastX > left)&&(lastX < right)){
					//now get the color of the pixel and find out which type to look up
					int pixel = m_image.getRGB(lastX,lastY);
					if (pixel == -1){
						eraseTooltip();
						return;//it's the white background
					}
					else{
						if (organismIntColors.containsKey(new Integer(pixel))){
							String organism = (String)organismIntColors.get(new Integer(pixel));
							
							//get the count
							HashMap temp = (HashMap)data.get(aaa);
							
							int resCount = 0;
							if (organism.equals("Other")){
								//manually count
								Set kt = temp.keySet();
								Iterator ii = kt.iterator();
								while (ii.hasNext()){
									String o = (String)ii.next();
									//check whether this organism is in the standard list
									if (!organisms.contains(o)){
										resCount += ((Integer)temp.get(o)).intValue();
									}
								}
							}
							else{
								resCount = ((Integer)temp.get(organism)).intValue();
							}

							
							int percent = (int)((resCount*100)/total);
							String tooltip = "";
							if (resCount == 1){
								tooltip = organism + ": " + resCount + " occurrence of " + aaa + " (" + percent + "%)";
							}
							else{
								tooltip = organism + ": " + resCount + " occurrences of " + aaa + " (" + percent + "%)";
							}
							
							//fill the area of tooltip display
							eraseTooltip();
							
							g2.setColor(Color.black);
							g2.setFont(new Font("Helvetica", Font.PLAIN, 16));
							g2.drawString(tooltip, 530, imageHeight - 15);
							repaint();

						}
						else{
							//fill the area of tooltip display
							eraseTooltip();
						}
					}
					
					
				}
				else{
					eraseTooltip();
				}

			}
			
		};
		image.addMouseListener( mia1 );
		image.addMouseMotionListener( mia1 );
		
		
		//now handle the tabular representation of the data
		v = sp.getViewport();
//		v.setBackground(new Color(180,180,180));
		v.setBackground(Color.white);
		tablePanel.add(sp, BorderLayout.CENTER);
		tablePanel.setBorder(new EmptyBorder(10,5,5,5));
		
		distributionFrame = this;
		
		m_data = new DistributionData(data, total);
		m_table = new JTable();
		
		m_table.setAutoCreateColumnsFromModel(false);
		m_table.setModel(m_data);
		m_table.setRowSelectionAllowed(true);
		m_table.setCellSelectionEnabled(true);
		m_table.setRowHeight(rowHeight);

		//determine whether blank rows are necessary
		int rows = m_data.getDataRowCount();
		int totalHeight = rows*rowHeight;
		
		if (totalHeight < tablePanel.getPreferredSize().height){
			int blanks = (tablePanel.getPreferredSize().height - totalHeight)/rowHeight + 2;//panel size difference
			m_data.addBlanks(blanks);
			sp.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER);
		}
		
		tablePopup = createTablePopup();
		
		mia2 = new MouseInputAdapter(){
		
			boolean popup = false;
			
			public void mousePressed(MouseEvent e){
				setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
				checkPopup(e);
			}

			public void mouseReleased(MouseEvent e){
				checkPopup(e);
			}
			
			public void mouseClicked(MouseEvent e){
				checkPopup(e);
			}

			public void checkPopup(MouseEvent e){
				if (e.isPopupTrigger()){
					popup = true;
				}
				start(e, popup);
			}
			
			public void start(MouseEvent e, boolean p){
				counter2++;
				if (counter2 == 3){
					counter2 = 0;
					if (popup){
						tablePopup.show(m_table, e.getX(), e.getY());
//						System.out.println("popup");
					}
					else{
//						System.out.println("Processing click");
					}
					popup = false;
				}
				else{
//					System.out.println("counter = " + counter);
				}
			}
		};
			
		m_table.addMouseListener( mia2 );
		m_table.addMouseMotionListener( mia2 );
		

		ColorRenderer renderer = null;

		//take care of the renderers
		for (int k = 0; k <= (aminoacids.length + 1); k++){
			renderer = new ColorRenderer(data);
			renderer.setHorizontalAlignment(m_data.m_columns[k].m_alignment);
			
			TableColumn column = new TableColumn(k, m_data.m_columns[k].m_width, renderer, null);
			m_table.addColumn(column);
		}

		JTableHeader header = m_table.getTableHeader();
		header.setUpdateTableInRealTime(false);
		
		//close the previous view (if any);
		v.setView(m_table);
		
		pane.addTab("   Bar graph   ", imagePanel);
		pane.addTab("   Distribution table   ", tablePanel);
		pane.setPreferredSize(new Dimension(930, 700));
		
		getContentPane().add(pane);
		
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				setVisible(false);
				dispose();
			}
		});

//		System.out.println("Total = " + total);

		//printing stuff
		PrinterJob pj = PrinterJob.getPrinterJob();
		mPageFormat = pj.defaultPage();
		mPageFormat.setOrientation(PageFormat.LANDSCAPE);
		pj = null;

		

		setTitle("Residue distribution at the selected position in the alignment");

		setSize(new Dimension(950, 725));
		setLocationRelativeTo(null);
		setResizable(false);
		setVisible(true);

	}
	
	public void eraseTooltip(){
		g2.setColor(Color.white);
		g2.fillRect(525, imageHeight-35,380,30);
		repaint();
	}
		
	
	public Image paintImage(){
	
		//use the data HashMap to create the image
		m_image = new BufferedImage(imageWidth, imageHeight, BufferedImage.TYPE_INT_RGB);
		g2 = m_image.createGraphics();
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		
		//draw the background
		g2.setColor(Color.white);
		g2.fillRect(0,0,imageWidth, imageHeight);
		g2.setColor(Color.black);
		g2.drawRect(0,0,imageWidth - 1, imageHeight - 1);
		
		//draw the axes
		g2.drawLine(leftMargin,30,leftMargin, 600);
		g2.drawLine(leftMargin, 600, leftMargin + 640, 600);
		
		//draw aminoacid labels under the X axis
		g2.setFont(new Font("Helvetica", Font.BOLD, 12));
		
		for (int i = 0; i < aminoacids.length; i++){
			g2.drawString(aminoacids[i], leftMargin + 15 + i*30 + 5, 615);//+15 for the space before the first bar starts
		}
		
		
		//get the data from the hash and figure out the scale of the drawing
		//split the data into hashkeys per each aminoacid and create an organism->type lookup table
		
		//process the data
		
/*		HashMap aa = new HashMap();
		
		Set keys = data.keySet();
		Iterator it = keys.iterator();
		while (it.hasNext()){
			String a = (String)it.next();
			
			System.out.println("a = " + a);
			
			//update count by organism
			HashMap temp = (HashMap)data.get(a);
			System.out.println("temp = " + temp);
			Set k = temp.keySet();
			Iterator i = k.iterator();
			while (i.hasNext()){
				String organism = (String)i.next();
				Integer cc = (Integer)temp.get(organism);
				System.out.println("cc = " + cc);
				if (cc == null) continue;
				int count = cc.intValue();
				
				//get the current count for this aminoacid from this organism
				HashMap map = (HashMap)tt.get(a);
				System.out.println("a = " + a + ", organism = " + organism);
				int current = ((Integer)map.get(organism)).intValue();
				current += count;
				
				total += count;
				map.put(organism, new Integer(current));
				tt.put(a, map);
				
				
				//same thing for the image
				HashMap tempo = (HashMap)aa.get(a);
				try{
					current = ((Integer)tempo.get(organism)).intValue();
					current += count;
					tempo.put(organism, new Integer(current));
				}
				catch (NullPointerException ex){
					current = ((Integer)tempo.get("Other")).intValue();
					current += count;
					tempo.put("Other", new Integer(current));
				}
				aa.put(a, tempo);
			}
			
		}
*/		
		//use the aa hash to get maxCount
		Set set = data.keySet();
		Iterator it = set.iterator();
		
		int maxCount = 0;
		
		while (it.hasNext()){
			String aaa = (String)it.next();
			HashMap temp = (HashMap)data.get(aaa);
			Set ss = temp.keySet();
			Iterator iii = ss.iterator();
			int aaTotal = 0;
			while (iii.hasNext()){
				String org = (String)iii.next();
				int counter = ((Integer)temp.get(org)).intValue();
				aaTotal += counter;
				total += counter;
			}
			
			if (aaTotal > maxCount){
				maxCount = aaTotal;
			}
		}
		
		//at this point, aa HashMap contains residue counts grouped by residue, followed by organism 
//		System.out.println("maxCount = " + maxCount);
		
		//consider that display should be about 25% higher than the highest bar
		int verticalLimit = (int)(maxCount*1.25);
		
///		System.out.println("verticalLimit = " + verticalLimit);
		
		//find out which scale to use
		int interval = 0;
		for (int i = 0; i < scales.length; i++){
			int ratio = verticalLimit/scales[i];
			if (ratio < 5){
				interval = scales[i];
				break;
			}
		}
		
		//calculate the entire height of the scale in interval units
		int span = (verticalLimit/interval) + 1;//to have some extra space
		
		int units = span*interval;
		int graphHeight = imageHeight - topMargin - bottomMargin;

		double ratio = ((double)units)/((double)graphHeight);//units per pixel
		
//		System.out.println("ratio = " + ratio);
		
		//now draw the scale
		g2.setColor(Color.black);
		g2.setFont(new Font("Helvetica", Font.PLAIN, 10));
		int counter = 0;//in units
		int startY = imageHeight - bottomMargin;
		int Y = 0;
		while (true){
			int value = counter*interval;
			Y = (int)(value/ratio);
//			System.out.println("Y = " + Y + " at value = " + value);
			
			g2.drawLine(leftMargin-10,startY - Y, leftMargin, startY - Y);
			g2.drawString((new Integer(value)).toString(), leftMargin-38, startY - Y+3);
			counter++;
			if (counter > span){
				break;
			}
		}

		//draw the color scheme on the right side of the graph
		int rectX = leftMargin+650;
		int rectY = 80;
		g2.setFont(new Font("Helvetica", Font.PLAIN, 12));
		for (int i = 0; i < organismList.length; i++){
			String org = organismList[i];
			g2.setColor((Color)colors.get(org));
			g2.fillRect(rectX, rectY, 40, 15);
			g2.setColor(Color.black);
			g2.drawRect(rectX, rectY, 40, 15);
			g2.drawString(org, rectX+50, rectY+12);
			rectY += 25;
		}
		
		//now use the derived ratio and draw the residue count bars, splitting them by organism type
		//go through the aa hash
		int startX = leftMargin + 15;
		for (int j = 0; j < aminoacids.length; j++){
		
			int baseY = imageHeight - bottomMargin;
		
			String aminoacid = aminoacids[j];
			
			HashMap temp = (HashMap)data.get(aminoacid);
//			System.out.println(aminoacid + ": organisms = " + temp.size());
			
			if (temp == null || temp.size() == 0){
				startX += 30;
				continue;
			}
			
			int baseHeight = 0;//height of the bar so far
			for (int k = 0; k < organismList.length; k++){
				try{
					
					int count = 0;
					//check for rare organisms while we are in the Other category
					if (organismList[k].equals("Other")){
						//iterate over all organisms in the data
						Set kt = temp.keySet();
						Iterator ii = kt.iterator();
						while (ii.hasNext()){
							String o = (String)ii.next();
							//check whether this organism is in the standard list
							if (!organisms.contains(o)){
								count += ((Integer)temp.get(o)).intValue();
							}
						}
					}
					else{
						count = ((Integer)temp.get(organismList[k])).intValue();
					}
					
					int pixels = (int)((double)count/ratio);
					
					g2.setColor((Color)colors.get(organismList[k]));
					g2.fillRect(startX,baseY - baseHeight - pixels, 15, pixels);
				
					g2.setColor(Color.black);
					g2.drawRect(startX,baseY - baseHeight - pixels, 15, pixels);
					
				
					baseHeight += pixels;
				}
				catch (Exception e){}
			}
			
			
			startX += 30;
		}
		
		
		return m_image;
	}
	
	public JPopupMenu createImagePopup(){
	
		JPopupMenu popup = new JPopupMenu();
		
		JMenuItem blank = new JMenuItem("");
		blank.setEnabled(false);
		JMenuItem save = new JMenuItem("Save image...");
		save.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				
				saveImage();
				repaint();
			}
				
		});
		
		
		JMenuItem printSetup = new JMenuItem("Print setup...");
		printSetup.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				printSetupHandler();
			}
		});
		JMenuItem print = new JMenuItem("Print image...");
		print.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				printImage();
			}
		});
		
		JMenuItem close = new JMenuItem("Close");
		close.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setVisible(false);
				dispose();
			}
		});
		
		popup.add(blank);
		popup.add(save);
		popup.add(new JSeparator());
		popup.add(printSetup);
		popup.add(print);
		popup.add(new JSeparator());
		popup.add(close);
		 
		return popup;
	}
		
	public void saveImage(){

		
		eraseTooltip();

		try{
			ImageHandler.saveImage(callable, m_image);
		}
		catch (IOException e){
			JOptionPane.showMessageDialog(this, "File writing error: " + e, "Error", JOptionPane.ERROR_MESSAGE);
		}

	}
	
	public void printImage(){
		
		//print the currently displayed image
		
		Thread t = new Thread(){
			public void run(){

				PrinterJob pj = PrinterJob.getPrinterJob();

				//set the cursor, 'cause the wait will be long
				setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

				book = new Book();
				
				//get image dimensions
				int iWidth = m_image.getWidth();
				int iHeight = m_image.getHeight();
				double aspectRatio = imageWidth/imageHeight;
				
				//get the page format
				int pageWidth = (int)mPageFormat.getImageableWidth();
				int pageHeight = (int)(mPageFormat.getImageableHeight()*0.9);//make it a bit more legible by compression


				//now with this image, create a Printable instance and add it to Book
				PagePrintable p = new PagePrintable(m_image, pageWidth, pageHeight, iWidth, iHeight);

				book.append(p, mPageFormat);




				pj.setPageable(book);
				if (pj.printDialog()){
					try{
						pj.print();
					}
					catch (PrinterException pe){
						System.out.println("Exception " + pe);
						JOptionPane.showMessageDialog(distributionFrame, "Printing error: " + pe, "Error", JOptionPane.ERROR_MESSAGE);
					}
				}

				setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
				book = null;
			}
		};
		
		t.start();
		
		
	}			
	
	
	public JPopupMenu createTablePopup(){
	
		JPopupMenu popup = new JPopupMenu();
		
		JMenuItem blank = new JMenuItem("");
		blank.setEnabled(false);
		JMenuItem save = new JMenuItem("Export table...");
		save.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				
				//display the dialog
				JFileChooser chooser = new JFileChooser(".");
				chooser.setDialogTitle("Save as a CSV file...");

				chooser.setSelectedFile(new File("*.csv"));
				chooser.setDialogType(JFileChooser.SAVE_DIALOG);

				FileNameFilter filter = new FileNameFilter("csv", "CSV files");
				chooser.setFileFilter(filter);

				chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
				int result = chooser.showSaveDialog(null);
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				String fname = filename.toString();

				int dot = fname.lastIndexOf(".");
//				System.out.println("fname is " + fname);
//				System.out.println("Index of dot is " + dot);
				if (dot <= 0){
					//no extension supplied
					fname += ".cvs";
				}
				

//				System.out.println("Filename chosen: " + fname);
				
				saveTable(fname);
				repaint();

				chooser = null;
			}
				
		});
		
		
		JMenuItem printSetup = new JMenuItem("Print setup...");
		printSetup.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				printSetupHandler();
			}
		});
		
		JMenuItem print = new JMenuItem("Print table...");
		print.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				Thread runner = new Thread(){
					public void run(){
						printData();
					}
				};
				runner.start();
			}
		});
		
		JMenuItem close = new JMenuItem("Close");
		close.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setVisible(false);
				dispose();
			}
		});
		
		
		popup.add(blank);
		popup.add(save);
		popup.add(new JSeparator());
		popup.add(printSetup);
		popup.add(print);
		popup.add(new JSeparator());
		popup.add(close);
		 
		return popup;
	}
	
	public void saveTable(String fname){
	
		//save the table as a comma-separated values file
		//go line by line separating columns by commas and lines by newlines
		
		String filename = fname;
		fname = null;
		
		eraseTooltip();

		try{
			FileOutputStream out = new FileOutputStream(filename);
			PrintWriter writer = new PrintWriter(out);
			String line = "";
			for (int i = 0; i < m_table.getColumnCount(); i++){
				line +=  m_data.m_columns[i].getTitle();
				if (i != (m_table.getColumnCount()-1)){
					line += ",";
				}
			}
			writer.println(line);
			
			//now go through the lines
			for (int j = 0; j < m_table.getRowCount(); j++){
				line = "";
				for (int k = 0; k < m_table.getColumnCount(); k++){
					String value = (m_data.getValueAt(j, k)).toString();
					line += value;
					if (k != (m_table.getColumnCount()-1)){
						line += ",";
					}
				}
				writer.println(line);
			}
			
			writer.flush();
			writer = null;
			
		
		}
		catch (IOException e){
			JOptionPane.showMessageDialog(this, "File writing error: " + e, "Error", JOptionPane.ERROR_MESSAGE);
		}
		
		JOptionPane.showMessageDialog(null,"Image saved to " + filename);

		filename = null;

	}
	
	public void printSetupHandler(){

		//this method is called from Viewer when Print Setup is selected
		//currently active frame is found and this method is called

		Thread t = new Thread(){
			public void run(){
				PrinterJob pj = PrinterJob.getPrinterJob();
				mPageFormat = pj.pageDialog(mPageFormat);
			}
		};
		t.start();
	}
	
	
	public void printData() {

		//use the current page format to get the printable area
		PrinterJob pj = PrinterJob.getPrinterJob();
		
		//set the cursor, 'cause the wait will be long
		setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		book = new Book();
		
		int pageWidth = 0;
		int pageHeight = 0;
		if (mPageFormat.getOrientation() == PageFormat.PORTRAIT) {
			pageWidth = (int)mPageFormat.getImageableWidth();
			pageHeight = (int)mPageFormat.getImageableHeight();
		}
		else {
			pageWidth = (int)mPageFormat.getImageableWidth();
			pageHeight = (int)mPageFormat.getImageableHeight();
		}
		
		boolean newPage = true;
		BufferedImage m_image = null;
		Graphics2D g = null;
		FontMetrics fm = null;
		int y = 20;//start a bit lower
		
		int imageWidth = (int)(pageWidth*1.7);
		int imageHeight = (int)(pageHeight*1.7);
		int nRow = 0;
		while (true){
			
			if (newPage){
				m_image = new BufferedImage(imageWidth, imageHeight, BufferedImage.TYPE_INT_RGB);
				g = m_image.createGraphics();
				g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				newPage = false;
				g.setColor(Color.white);
				g.fillRect(0,0, imageWidth, imageHeight);
				
				//draw the header
				g.setColor(Color.black);
				Font headerFont = m_table.getFont().deriveFont(Font.BOLD);
//				int size = headerFont.getSize();
//				System.out.println("header font = " + size);
				g.setFont(headerFont);
				fm = g.getFontMetrics();

				TableColumnModel colModel = m_table.getColumnModel();
				int nColumns = colModel.getColumnCount();
				int x[] = new int[nColumns];
				x[0] = 5;
        
				int h = fm.getAscent();
				y += h;//add ascent of header font
 
				for (int nCol=0; nCol < nColumns; nCol++){
					TableColumn tk = colModel.getColumn(nCol);
					int width = tk.getWidth();
					if (x[nCol] + width > imageWidth) {
						nColumns = nCol;
						break;
					}
					if (nCol+1<nColumns){
						x[nCol+1] = x[nCol] + width;
					}
					String title = (String)tk.getIdentifier();
					g.drawString(title, x[nCol], y);
				}
				
				y += h*2;
			}
			
			//start iterating over the table and drawing the data on one line
			g.setFont(m_table.getFont());
			fm = g.getFontMetrics();
			int h = fm.getHeight();
			int lineX = 5;
			for (int nCol=0; nCol < m_table.getColumnCount(); nCol++) {
				TableColumn tk = m_table.getColumnModel().getColumn(nCol);
				int width = tk.getWidth();
				int col = m_table.getColumnModel().getColumn(nCol).getModelIndex();
				Object obj = m_data.getValueAt(nRow, col);
				if (obj == null){
					continue;
				}
				String str = obj.toString();
				if (nCol == 0){
					//check for long organism names
					int w = fm.stringWidth(str);
					if (w > (width - 2)){
						g.drawString(str, lineX, y);
						//and cover the rest of the string with a white rectangle
						g.setColor(Color.white);
						g.fillRect(lineX+width-5, y-h, w, h+2);
						g.setColor(Color.black);
					}
					else{
						g.drawString(str, lineX, y);
					}
				}
				else{
					g.drawString(str, lineX, y);
				}
				lineX += width;
			}
			y += h;
			
			if (y > imageHeight){
				PagePrintable p = new PagePrintable(m_image, pageWidth, pageHeight, imageWidth, imageHeight);
				book.append(p, mPageFormat);
//				savePrint(m_image, "1");
					
				//create a new image
				m_image = new BufferedImage(imageWidth, imageHeight, BufferedImage.TYPE_INT_RGB);
				g = m_image.createGraphics();
				g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				newPage = false;
				g.setColor(Color.white);
				g.fillRect(0,0, imageWidth, imageHeight);
				
				y = 20;
				//draw the header
				g.setColor(Color.black);
				Font headerFont = m_table.getFont().deriveFont(Font.BOLD);
				g.setFont(headerFont);
				fm = g.getFontMetrics();

				TableColumnModel colModel = m_table.getColumnModel();
				int nColumns = colModel.getColumnCount();
				int x[] = new int[nColumns];
				x[0] = 5;
        
				y += fm.getAscent();//add ascent of header font
 
				for (int nCol=0; nCol < nColumns; nCol++){
					TableColumn tk = colModel.getColumn(nCol);
					int width = tk.getWidth();
					if (x[nCol] + width > imageWidth) {
						nColumns = nCol;
						break;
					}
					if (nCol+1<nColumns){
						x[nCol+1] = x[nCol] + width;
					}
					String title = (String)tk.getIdentifier();
					g.drawString(title, x[nCol], y);
				}
				
				y += h;
			}
			
			nRow++;
			if (nRow > m_table.getRowCount()-1){
				PagePrintable p = new PagePrintable(m_image, pageWidth, pageHeight, imageWidth, imageHeight);
				book.append(p, mPageFormat);
//				savePrint(m_image, "2");
				break;
			}
			
			
		}
		
		pj.setPageable(book);
		if (pj.printDialog()){
			try{
				pj.print();
			}
			catch (PrinterException pe){
				System.out.println("Exception " + pe);
				JOptionPane.showMessageDialog(this, "Printing error: " + pe, "Error", JOptionPane.ERROR_MESSAGE);
			}
		}
			
		setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		book = null;
	}
	
	private void savePrint( BufferedImage image, String filename ){

		try
		{
			File file = new File("D:\\print" + filename + ".png");
			// Get the currently-selected item in the combo box
			FormatDescriptor fd =
				(FormatDescriptor) (getWriterFormats().elementAt(2));

			// Get the ImageWriter stored in this FormatDescriptor
			ImageWriter imageWriter = fd.imageWriter;

			// Get an output stream for the file
			FileOutputStream fout = new FileOutputStream( file );

			// Turn it into an ImageOutputStreamn
			ImageOutputStream ios = ImageIO.createImageOutputStream( fout );

			// Plug this stream into the ImageWriter
			imageWriter.setOutput( ios );

			// Write the image
			imageWriter.write( image );

			// Dont forget to close the file!
			fout.close();
		}
		catch( IOException ie )
		{
			System.out.println("Unable to save image");
			ie.printStackTrace( );
		}
	}
	
	static private Vector getWriterFormats( )
	{
		Vector formats = new Vector( );
		Hashtable seen = new Hashtable( );

		String names[] = ImageIO.getWriterFormatNames( );
		for ( int i=0; i<names.length; ++i )
		{
			String name = names[i];
			Iterator writers = ImageIO.getImageWritersByFormatName( name );
			while ( writers.hasNext() )
			{
				ImageWriter iw = (ImageWriter) writers.next( );
				ImageWriterSpi iws = iw.getOriginatingProvider( );
				String suffixes[] = iws.getFileSuffixes( );
				boolean allseen = true;
				for ( int j=0; j<suffixes.length; ++j )
				{
					String suffix = suffixes[j];
					suffix = suffix.toLowerCase( );
					if ( ! seen.containsKey( suffix ) )
					{
						seen.put( suffix, suffix );
						String description = name + " (*." + suffix + ")";
						FormatDescriptor fd =
							new FormatDescriptor( suffix, iw, description );
						formats.addElement( fd );
					}
				}
			}
		}
	

		return formats;
	}

		
	static private class FormatDescriptor
	{
		public String suffix;
		public ImageWriter imageWriter;
		public String description;

		public FormatDescriptor( String suffix, ImageWriter imageWriter,
			String description )
		{
			this.suffix = suffix;
			this.imageWriter = imageWriter;
			this.description = description;
		}

		public String toString( )
		{
			return description;
		}
	}
	
		
/*	public static void main(String[] args){
	
		HashMap distribution = new HashMap();
/*		HashMap temp = new HashMap();
		temp.put("A", new Integer(2));
		temp.put("C", new Integer(1));
		temp.put("D", new Integer(1));
		temp.put("E", new Integer(1));
		temp.put("F", new Integer(1));
		temp.put("G", new Integer(1));
		temp.put("H", new Integer(0));
		temp.put("I", new Integer(0));
		temp.put("K", new Integer(0));
		temp.put("L", new Integer(0));
		temp.put("M", new Integer(0));
		temp.put("N", new Integer(1));
		temp.put("P", new Integer(0));
		temp.put("Q", new Integer(0));
		temp.put("R", new Integer(0));
		temp.put("S", new Integer(1));
		temp.put("T", new Integer(3));
		temp.put("V", new Integer(0));
		temp.put("W", new Integer(3));
		temp.put("Y", new Integer(1));
		
		distribution.put("Homo sapiens", temp);
		
		HashMap second = new HashMap();
		second.put("A", new Integer(2));
		second.put("C", new Integer(1));
		second.put("D", new Integer(1));
		second.put("F", new Integer(1));
		second.put("K", new Integer(2));
		second.put("L", new Integer(1));
		second.put("M", new Integer(0));
		second.put("N", new Integer(0));
		second.put("P", new Integer(1));
		second.put("W", new Integer(1));
		second.put("Y", new Integer(2));
		second.put("-", new Integer(2));
		
		distribution.put("Zea mays", second);
		
		HashMap third = new HashMap();
		third.put("A", new Integer(2));
		third.put("C", new Integer(1));
		third.put("D", new Integer(1));
		third.put("F", new Integer(1));
		third.put("K", new Integer(0));
		third.put("L", new Integer(1));
		third.put("M", new Integer(0));
		third.put("N", new Integer(0));
		third.put("P", new Integer(1));
		third.put("W", new Integer(1));
		third.put("Y", new Integer(2));
		
		
		distribution.put("Saccharomyces cerevisiae", third);
		try{
			ObjectInputStream objIn = new ObjectInputStream(new GZIPInputStream(new FileInputStream("distribution.dat")));
			distribution =  (HashMap)objIn.readObject();
	
		}
		catch (Exception ex){
			System.out.println("Exception reading distribution: " + ex);
		}
		DistributionViewer v = new DistributionViewer(parent, distribution);
	}

*/
}


class DistributionColumnData {

	private String m_title = "";
	public int m_width;
	public int m_alignment;

	public DistributionColumnData(String title, int width, int alignment){
		
		m_title = title;
		m_width = width;
		m_alignment = alignment;

	}
	
	public String getTitle(){
		return m_title;
	}
}



class DistributionData extends AbstractTableModel {

	public DistributionColumnData m_columns[] = null;
	protected HashMap m_hash;//stores the actual data
	protected int m_total = 0;
	
	protected int totalRows = 0;
	protected int dataRows = 0;

	private static final int WIDTH = 65;
	private static final int ALIGNMENT = JLabel.CENTER;
	
	private String[] aminoacids = { "A", "C", "D", "E", "F", "G", "H", "I", "K", "L","M", "N", "P", "Q", "R","S", "T", "V", "W", "Y", "-" };
	private static String[] organismList = { "Homo sapiens", "Mus musculus", "Drosophila melanogaster", "Caenorhabditis elegans", "Saccharomyces serevisiae", "Arabidopsis thaliana", "Gallus gallus", "Rattus norvegicus", "Xenopus laevis", "Oryza sativa", "Other" };
	private Vector organisms = new Vector();//the order in which organisms appear
	
	public DistributionData(HashMap data, int total){

		//data is a HashMap of HashMaps: aminoacid -> organism -> count
		//total stores the total number of occurrences of everything everywhere (for the bottom right cell)
		m_total = total;
		
		//fill out m_columns[] with aminoacid names
		//first find out how many there are
		int k = aminoacids.length;

		m_columns = new DistributionColumnData[k+2];//to add the organism and total columns

		//enter data for the compounds column
		m_columns[0] = new DistributionColumnData("Organism", 500, JLabel.LEFT);

		//enter the actual column parameters
		for (int i = 0; i < k; i++){
			m_columns[i+1] = new DistributionColumnData(aminoacids[i], WIDTH, ALIGNMENT);
		}
		m_columns[k+1] = new DistributionColumnData("Total", WIDTH, ALIGNMENT);

		//assign the actual data
		m_hash = data;
		
		HashMap tempOrg = new HashMap();//all organisms found in the data
		for (int j = 0; j < aminoacids.length; j++){
			HashMap temp = (HashMap)m_hash.get(aminoacids[j]);
			//the keys of this hash are organism names
			Set keys = temp.keySet();
			Iterator it = keys.iterator();
			while (it.hasNext()){
				tempOrg.put(it.next(), "");
			}
		}
		
		//get a sorted list
		Vector temp = new Vector();
		Set keys = tempOrg.keySet();
		Iterator it = keys.iterator();
		while (it.hasNext()){
			temp.add(it.next());
		}
		
		Collections.sort(temp);
		
		//scan the data to sort all organisms
		for (int i = 0; i < organismList.length; i++){
			if (temp.contains(organismList[i])){
				organisms.add(organismList[i]);
			}
		}
		
		//now remove all duplicate organisms from temp
		for (int i = temp.size()-1; i >= 0; i--){
			if (organisms.contains(temp.elementAt(i))){
				temp.removeElementAt(i);
			}
		}
		
		//add the organisms to the initial set
		organisms.addAll(temp);
//		System.out.println("organisms = " + organisms);
		
		dataRows = organisms.size() + 1;//the totals line
		totalRows = dataRows;
//		System.out.println("Size of m_data = " + m_hash);

	}
	
	public void addBlanks(int blanks){
		
		//add specified number of blank rows to the table
		totalRows += blanks;
		
		
	}

	public int getRowCount(){
		return m_hash==null ? 0 : totalRows;//types currently defined + the total line
	}
	
	public int getDataRowCount(){
		return dataRows;
	}

	public int getColumnCount(){
		return m_columns.length;
	}

	public String getColumnName(int column){
		return m_columns[column].getTitle();
	}

	public boolean isCellEditable(int nRow, int nColumn){
		return false;
	}

	public Object getValueAt(int nRow, int nColumn){
		if (nRow < 0 || nRow > getRowCount()){
			return "";
		}
		
		if (nRow > getDataRowCount()-1){
			return " ";
		}
		
		if ((nColumn < 0)||(nColumn > 22)){
			return "";
		}
		
//		System.out.println("Column = " + nColumn);
//		System.out.println("Row = " + nRow);

		//first, get the right line HashMap
		if (nRow == (getDataRowCount() - 1)){
			//it's the last row with total counts per each aminoacid over all organism types
			if (nColumn == 0){
				return "Total";
			}
			if (nColumn == (getColumnCount() - 1)){
				//return the total number of hits seen in the graph
				return new Integer(m_total);
			}
			//otherwise, return count of all types at the corresponding aminoacid
			int totalCount = 0;
			for (int m = 0; m < (getDataRowCount() - 1); m++){
				Integer vv = (Integer)getValueAt(m, nColumn);
				if (vv != null) totalCount += vv.intValue();
			}
			return new Integer(totalCount);
		}
		
			
		String organism = (String)organisms.elementAt(nRow);
//		System.out.println(type + " in row "+ nRow + " and column " + nColumn);
		if (nColumn == 0){
			return organism;
		}
		if (nColumn == (getColumnCount()-1)){
			//return the total count for this type
			int totalCount = 0;
			for (int m = 1; m < getColumnCount()-1; m++){
				Integer vv = (Integer)getValueAt(nRow, m);
				if (vv != null) totalCount += vv.intValue();
			}
			return new Integer(totalCount);
		}
		
		String aa = aminoacids[nColumn - 1];
			
//		System.out.println("Aminoacid = " + aa);
		HashMap temp = (HashMap)m_hash.get(aa);
		Object v = temp.get(organism);

		return v;
	}
	
	public String getTitle(){
		return "Residue distribution profile at selected position in the alignment";
	}

}


class ColorRenderer extends JLabel implements TableCellRenderer {

	private HashMap data = null;//contains lines of scores

	public ColorRenderer(HashMap d){
		super();
		data = d;
		setOpaque(true);
		
	}

	public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column){
		
//		System.out.println("value = " + value + " at row = " + row + " and column = " + column);
		
		//find out whether it's the 0-column and paint the cell according to the type it shows
		setForeground(Color.black);
//		if (column == 0){
			setFont(new Font("Helvetica", Font.PLAIN, 12));
			if (row == (table.getRowCount()-1)){
				setBackground(Color.white);
			}
			else{
//				System.out.println("Row = " + row);
				if ((row%2) == 0){
					setBackground(new Color(210,220,255));
				}
				else{
					setBackground(Color.white);
				}
			}	
//		}
//		else{
///			setBackground(Color.white);
//			setFont(new Font("Helvetica", Font.PLAIN, 14));
//		}

		if (value != null){
			setText(" " + value.toString());
		}
		else{
			setText("-");
		}

		return this;
	}
}
