//  $Id: ElementSelectionPanel.java,v 1.1 2006/10/21 17:52:13 Sasha Buzko Exp $
//
//  Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//  a facility operated jointly by the University of California,
//  San Diego (UCSD) and General Atomics, San Diego, California, USA.
//
//  Users and possessors of this source code are hereby granted a
//  nonexclusive, royalty-free copyright and design patent license to
//  use this code in individual software.  License is not granted for
//  commercial resale, in whole or in part, without prior written
//  permission from SDSC.  This source is provided "AS IS" without express
//  or implied warranty of any kind.
//
//  For further information, please see:  http://mbt.sdsc.edu
//
//  History:
//  $Log: ElementSelectionPanel.java,v $
//  Revision 1.1  2006/10/21 17:52:13  Sasha Buzko
//  Refactored the project to move all new code to edu.sdsc.sirius package.
//
//  Revision 1.1  2006/05/20 17:02:01  Sasha Buzko
//  Updated version
//
//  Revision 1.1  2006/04/30 20:13:55  Sasha Buzko
//  New version of the app
//
//  Revision 1.1  2006/04/15 19:42:19  Sasha Buzko
//  Initial commit
//
//  Revision 1.2  2006/02/24 22:26:54  Administrator
//  *** empty log message ***
//
//  Revision 1.1  2005/12/26 03:53:47  Administrator
//  *** empty log message ***
//
//  Revision 1.1  2005/11/13 04:35:23  Administrator
//  *** empty log message ***
//
//  Revision 1.1  2003/11/25 17:23:35  moreland
//  First version.
//
//  Revision 1.0  2003/10/21 20:45:18  moreland
//  First revision.
//


package edu.sdsc.sirius.dialogs;


// Core
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

// MBT
import edu.sdsc.mbt.util.*;


/**
 *  An example program which exersises the PeriodicTable class.
 *  <P>
 *  @see edu.sdsc.mbt.viewables.ColorMap
 *  <P>
 *  @author John L. Moreland
 */
public class ElementSelectionPanel
	extends JPanel
{
	
	private ElementSelectionDialog dialog = null;
	//
	// Define a 2D table of element numbers than will be used to build the GUI.
	// Each row has the same number of items to enable automatic GUI layout.
	// A "0" value means there should be an empty/invisible tile.
	//
	private static final int elementNumbers[][] =
	{
		{  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2 },
		{  3,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  6,  7,  8,  9, 10 },
		{ 11, 12,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 13, 14, 15, 16, 17, 18 },
		{ 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36 },
		{ 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54 },
		{ 55, 56, 57, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86 },
		{ 87, 88, 89,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118 },
		{  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
		{  0,  0, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71,  0,  0 },
		{  0,  0, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99,100,101,102,103,  0,  0 }
	};


	/**
	 *  Construct an instance of this class using the content described by the
	 *  PeriodicTable and ElementStyles classes.
	 */
	public ElementSelectionPanel(ElementSelectionDialog d)
	{
		
		dialog = d;
		
		// Use a grid of same-size tiles to form the table.
		setLayout( new GridLayout( elementNumbers.length, elementNumbers[0].length ) );
		// Leave some space around the edges of the panel.
		setBorder( BorderFactory.createEmptyBorder( 10, 10, 10, 10 ) );

		// Walk through the 2D table of element numbers to build the GUI.
		for ( int i=0; i<elementNumbers.length; i++ )
		{
			for ( int j=0; j<elementNumbers[i].length; j++ )
			{
				int elementNumber = elementNumbers[i][j];
				JComponent component = null;
				if ( elementNumber == 0 )
				{
					// An empty cell.
					component = (JComponent) new JLabel( "" );
				}
				else
				{
					// A non-empty cell.
					Element element = PeriodicTable.getElement( elementNumber );
					JButton button = null;
					// If there is an element object use the symbol as the name.
					if ( element == null )
						button = new JButton( String.valueOf( elementNumber ) );
					else
						button = new JButton( element.symbol );
					// button = (JButton) new PtButton( element );
					component = (JComponent) button;

					// Enable the user to click the element tile.
					PtListener ptListener = new PtListener( element, this, button );
					button.addActionListener( ptListener );

					// Color the element tile acording to the ElementStyles class.
					float[] rgb =
						ElementProperties.getElementColor( elementNumber );
					if ( rgb != null )
					{
						Color color = new Color( rgb[0], rgb[1], rgb[2] );
						if ( color != null ) component.setBackground( color );
					}

					// Replace the border to get rid of the text insets.
					component.setBorder( BorderFactory.createEtchedBorder() );
				}

				// Add the element tile to the panel layout.
				add( component );
			}
		}
	}


	/**
	 * Called in response to a click on an element tile.
	 */
	public void tileClicked( Element element, JButton button )
	{
//		System.out.println( "tileClicked = " + element );
//		System.out.println("symbol = " + element.symbol);
		dialog.setElement(element.symbol);
	}

	//
	// Inner classes.
	//
	////////////////////////////////////////////////////////////////////////


	/**
	 * An inner class which knows how to respond to a click on an element tile.
	 */
	public class PtListener
		implements ActionListener
	{
		Element element = null;
		ElementSelectionPanel elementSelectionPanel = null;
		JButton button = null;
		public PtListener( Element el, ElementSelectionPanel pte, JButton bt )
		{
			this.element = el;
			this.elementSelectionPanel = pte;
			this.button = bt;
		}
		public void actionPerformed( ActionEvent actionEvent )
		{
//			System.out.println("click on " + element.symbol);
			elementSelectionPanel.tileClicked( element, button );
		}
	};


	/**
	 * An inner class which knows how to draw an element tile.
	 */
	public class PtButton
		extends JButton
	{
		Element element = null;
		String atomicNumber = null;
		public PtButton( Element el )
		{
			this.element = el;
			float[] rgb =
				ElementProperties.getElementColor( element.atomic_number );
			if ( rgb != null )
			{
				Color color = new Color( rgb[0], rgb[1], rgb[2] );
				if ( color != null ) setBackground( color );
			}
			atomicNumber = String.valueOf( element.atomic_number );
		}
		public void paint( Graphics graphics )
		{
			int width = getWidth( );
			int height = getHeight( );
			int halfWidth = width / 2;
			int halfHeight = height / 2;

			// Clear the tile.
			graphics.setColor( getBackground() );
			graphics.fillRect( 0, 0, width, height );

			// Draw the element symbol.
			graphics.drawString( element.symbol, halfWidth, height );
			FontMetrics fontMetrics = graphics.getFontMetrics( );
			int stringWidth = fontMetrics.stringWidth( element.symbol );
			int halfStringWidth = stringWidth / 2;
			graphics.setColor( Color.black );
			graphics.drawString( element.symbol, halfWidth-halfStringWidth, halfHeight );

			// Draw the element atomic number.
			stringWidth = fontMetrics.stringWidth( atomicNumber );
			halfStringWidth = stringWidth / 2;
			graphics.drawString( atomicNumber, halfWidth-halfStringWidth, halfHeight+fontMetrics.getAscent() );

			// Frame the tile.
			graphics.setColor( Color.black );
			graphics.drawRect( 0, 0, width-1, height-1 );
		}
	};
}

