//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs;

import java.util.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.border.*;

import edu.sdsc.mbt.viewables.*;
import edu.sdsc.sirius.dialogs.TableImpl.*;
import edu.sdsc.sirius.io.*;
import edu.sdsc.sirius.util.*;
import edu.sdsc.mbt.*;

/**
 * This class implements a dialog that displays the currently loaded entries.
 * It retrieves data from structureDocument, packages it as DataSet and displays it in ResultPanel.
 * @see edu.sdsc.mbt.util.DataSet
 * 
 * @author Oleksandr V. Buzko
 */
public class EntryListerDialog extends JDialog implements ResultPanelCallable {
	
	public static final int SAVE_ENTRY = 0;
	public static final int CLOSE_ENTRY = 1;
	public static final int LOAD_STRUCTURE = 2;
	
	private int dialogType = SAVE_ENTRY;

	private Container contentPane;
	private JDialog self;
	
	private TableViewPanel contentPanel;
	private String workingDir = ".";
	
	private Vector searchTypes;
	
	private Manager parent;
	private JFrame parentFrame;
	private StructureDocument structureDocument;
	
	private JPanel base = new JPanel();
	private JPanel buttonPanel = new JPanel();
	private JButton okButton;
	private JButton cancelButton = new JButton("Cancel");
	
	private String initialTitle;
	private String parentId;
	
	private DataSet set;
		
	/**
	 * Constructor that creates the dialog.
	 * @param f parent frame
	 * @param p Manager object that will process the RequestSet
	 * @param title dialog title
	 * @param modal whether the dialog is modal (other frames in the application will be inaccessible)
	 */
	public EntryListerDialog (JFrame f, Manager p, boolean modal, StructureDocument sd, int type){
	
		super(f, "Currently loaded", modal);
		initialTitle = "Currently loaded";
		structureDocument = sd;
		parent = p;
		dialogType = type;
		
		set = new DataSet();
		//get the dataset from the structureDocument

		for (int j = 0; j < structureDocument.getEntryCount(); j++){
			Entry entry = (Entry)structureDocument.getEntry(j);
			
			if (entry == null){
				continue;
			}
			
			//create a DataBlock out of this entry
			DataBlock block = new DataBlock(new Integer(j).toString());
			block.setName(entry.getName());
			if (entry.getDescription() == null && entry.getSourceUrl() != null){
				block.setDescription(entry.getSourceUrl().toString());
			}
			else{
				block.setDescription(entry.getDescription());
			}
			set.addData(block);
		}
		

		//get the menubar
		//go through the available data options and determine the future table headers
		searchTypes = new Vector();
		searchTypes.add("Name");
		searchTypes.add("Description");
		parentFrame = parent.getApplicationFrame();
		
		contentPane = getContentPane();
		contentPane.setLayout(new BorderLayout());
		
		
		self = this;
		
		base.setBorder(new BevelBorder(BevelBorder.LOWERED));
		base.setLayout(new BorderLayout());
		
		//add window listener
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				setVisible(false);
				dispose();
			}
		});
		
		addComponentListener(new ComponentAdapter(){
			public void componentResized(ComponentEvent e){
				Dimension size = getSize();
				if (contentPanel == null) return;
				contentPanel.resizePanel(size);
			}	
		});
		
		contentPane.add(base, BorderLayout.CENTER);
		
		
		//panel that holds buttons
		JPanel lowerPanel = new JPanel();
		lowerPanel.setLayout(new BorderLayout());
		JPanel borderPanel = new JPanel();
		borderPanel.setLayout(new BorderLayout());
		borderPanel.add(new JSeparator(), BorderLayout.NORTH);
		
		lowerPanel.add(borderPanel, BorderLayout.NORTH);
		
		buttonPanel.setLayout(new FlowLayout(FlowLayout.CENTER, 40, 5));
		
		if (dialogType == SAVE_ENTRY){
			okButton = new JButton("Save");
		}
		else if (dialogType == CLOSE_ENTRY){
			okButton = new JButton("Close");
		}
		else{
			okButton = new JButton("OK");
		}
		okButton.setPreferredSize(new Dimension(80,25));
		okButton.setVisible(true);
		okButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				if (contentPanel != null){
					if (contentPanel.getSelectedRowCount() > 0){
						setVisible(false);
						processSelection();
					}
				}
			}
		});
		
		
		cancelButton.setPreferredSize(new Dimension(80,25));
		cancelButton.setVisible(true);
		cancelButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setVisible(false);
				dispose();
			}
		});
		
		buttonPanel.add(okButton);
		buttonPanel.add(cancelButton);
		
		
		lowerPanel.add(buttonPanel, BorderLayout.CENTER);
		contentPane.add(lowerPanel, BorderLayout.SOUTH);
		
				
		if (set != null && set.size() > 0){
			
			//used to show a table in a previously open frame
			//(after a new search initiated from this frame)
			contentPanel = new TableViewPanel(base, set, parent, searchTypes, TableViewPanel.FIRST_COLUMN_NARROW, false);
			base.add(contentPanel);
			contentPane.add(base, BorderLayout.CENTER);
			
			if (dialogType != LOAD_STRUCTURE){
				setTitle(initialTitle + ": " + set.size() + " entries");
			}
			else{
				setTitle(initialTitle + ": " + set.size() + " structures available");
			}
		}
		else{
			parent.displayErrorMessage("Nothing to display: no entries loaded.");
			dispose();
			return;
		}
				
		
		setSize(370,200);
		
		if (parentFrame == null){
//			setLocationRelativeTo(null);
			setBounds(0, 0, getSize().width, getSize().height);
		}
		else{
			//set location
			Dimension d1 = getSize();
			Dimension d2 = parentFrame.getSize();
		
			int x = Math.max((d2.width - d1.width)/2, 0);
			int y = Math.max((d2.height - d1.height)/2, 0);
		
			setBounds(x + parentFrame.getX(), y + parentFrame.getY(), d1.width, d1.height);
		}
		
		setVisible(true);
	}
	
	public void processSelection(){
		
		//get the selected indices
		int[] selection = contentPanel.getSelectedRows();
		if (selection.length == 0) return;
		
		//depending on the type of the dialog, process the entries
		if (dialogType == SAVE_ENTRY){
			if (selection.length > 1){
				parent.displayErrorMessage("Only the first selected entry will be saved.");
			}
			
			Entry e = structureDocument.getEntry(selection[0]);
			try{
				StructureEntry se = (StructureEntry)e;
				IOHandler.saveStructureFile(se, parent);
			}
			catch (Exception ex){
				IOHandler.selection = false;
				SequenceEntry sse = (SequenceEntry)e;
				Vector sequences = new Vector();
				//save the sequence entry
				for (int i = 0; i < sse.getSequenceCount(); i++){
					Sequence s = sse.getSequence(i);
					sequences.add(s.getCells());
				}
				
				IOHandler.saveAlignment(sequences, parent);
			}
		}
		else if (dialogType == CLOSE_ENTRY){
			for (int i = 0; i < selection.length; i++){
				DataBlock line = set.getData(selection[i]);
				if (line == null) continue;
				String id = line.getId();
				structureDocument.removeEntry(id);
			}
		}
		dispose();
		
	}
	
	public void refreshList(){
		
		base.remove(contentPanel);
		set.clearData();
		for (int j = 0; j < structureDocument.getEntryCount(); j++){
			Entry entry = (Entry)structureDocument.getEntry(j);
			
			if (entry == null){
				continue;
			}
			
			//create a DataBlock out of this entry
			DataBlock block = new DataBlock(entry.getName());
			block.setEntryId(entry.getName());
			if (entry.getDescription() == null && entry.getSourceUrl() != null){
				block.setName(entry.getSourceUrl().toString());
			}
			else{
				block.setName(entry.getDescription());
			}
			set.addData(block);
		}
		
		contentPanel = new TableViewPanel(base, set, parent, searchTypes, TableViewPanel.FIRST_COLUMN_NARROW, false);
		base.add(contentPanel);
		contentPane.add(base, BorderLayout.CENTER);
		
		base.revalidate();
		base.repaint();
	}
	
	public void setNumberOfHits(int n){}
	
	public void setSelectedIndices(Vector v){}
}