package edu.sdsc.sirius.dialogs;

// Core
import java.io.*;
import java.util.*;

// GUI
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.border.*;

// MBT
import edu.sdsc.mbt.*;
import edu.sdsc.mbt.util.PeriodicTable;
import edu.sdsc.mbt.viewables.*;
import edu.sdsc.sirius.util.*;
import edu.sdsc.sirius.viewers.*;
import edu.sdsc.sirius.viewers.SequenceViewerImpl.*;


public class HomologyModelDialog extends JDialog implements DisplayDialog {

	private JFrame parentFrame;
	private Manager parent;
	private StructureViewer sv;
	private StructureComponent structureComponent;
	
	private DisplayDialog itself;

	private Container contentPane;
	private JPanel base = new JPanel();
	
	private JPanel topPanel = new JPanel();
	private JPanel choicePanel = new JPanel();
	private JPanel pathPanel = new JPanel();
	
	private Border border1;
	private TitledBorder border2;
	private Border border;
	
	private final int USE_ALL_ENTRIES = 0;
	private final int USE_SELECTED_ENTRIES = 1;
	
	private final int FULL_LENGTH_SPAN = 0;//entire sequence span is used for the model
	private final int COLUMN_SELECTED_SPAN = 1;
	private final int USER_DEFINED_SPAN = 2;//manually entered column numbers
	
	
	private int set = USE_ALL_ENTRIES;
	private int span = COLUMN_SELECTED_SPAN;
	
	private JTextField pathField = new JTextField(30);
	private JButton pathButton = new JButton("Browse...");
	
	private ButtonGroup group = new ButtonGroup();
	private JRadioButton allSet = new JRadioButton("Use all loaded entries", true);
	private JRadioButton selectedSet = new JRadioButton("Use selected entries", false);
	
	private ButtonGroup group2 = new ButtonGroup();
	private JRadioButton fullButton = new JRadioButton("Full length sequences", false);
	private JRadioButton columnButton = new JRadioButton("Fragments bounded by selected columns", true);
	private JRadioButton manualButton = new JRadioButton("Manually defined bounds", false);
	
	private JTextField startField = new JTextField(20);
	private JTextField endField = new JTextField(20);

	private JLabel startLabel = new JLabel("From:");
	private JLabel endLabel = new JLabel(" to ");

	
	private JButton okButton = new JButton("OK");
	private JButton cancelButton = new JButton("Cancel");
	
	private boolean onTop = true;
	
	private String tempDirectory;
	
	
	public HomologyModelDialog(JFrame f, Manager p, StructureViewer s){

		super(f, "Build homology model", false);

		parentFrame = f;
		parent = p;
		sv = s;
		
		parent.setDisplayDialogStatus(true);
		
		itself = this;
		
		//create the panels
		contentPane = getContentPane();
		contentPane.setLayout(null);
		
		
		base.setBorder(new BevelBorder(BevelBorder.RAISED));
		base.setBounds(5, 5, 300, 250);
		base.setLayout(null);
		
		topPanel.setBounds(5,5,290,70);
		topPanel.setLayout(null);
		
		border1 = BorderFactory.createEmptyBorder(0,0,0,0);//left offset of the boxes in the box
		border2 = BorderFactory.createTitledBorder(new EtchedBorder(), " Entry selection ");
		border2.setTitleColor(Color.black);
		border2.setTitleFont(new Font("Dialog", 0, 12));
		border = BorderFactory.createCompoundBorder(border1, border2);
		topPanel.setBorder(border);
		
	
		//define the radio buttons
		allSet.setBounds(20,20,150,20);
		allSet.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				set = USE_ALL_ENTRIES;
			}
		});

		selectedSet.setBounds(20,40,150,20);
		selectedSet.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				set = USE_SELECTED_ENTRIES;
			}
		});
		
		group.add(allSet);
		group.add(selectedSet);
		
		topPanel.add(allSet);
		topPanel.add(selectedSet);
		
		base.add(topPanel);
		
		
		choicePanel.setBounds(5,75,290,110);
		choicePanel.setLayout(null);
		
		border1 = BorderFactory.createEmptyBorder(0,0,0,0);//left offset of the boxes in the box
		border2 = BorderFactory.createTitledBorder(new EtchedBorder(), " Sequence span to be used ");
		border2.setTitleColor(Color.black);
		border2.setTitleFont(new Font("Dialog", 0, 12));
		border = BorderFactory.createCompoundBorder(border1, border2);
		choicePanel.setBorder(border);
		
		
		//define the radio buttons
		fullButton.setBounds(20,20,200,20);
		fullButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				span = FULL_LENGTH_SPAN;
				startField.setText("");
				startField.setEnabled(false);
				endField.setText("");
				endField.setEnabled(false);
			}
		});

		columnButton.setBounds(20,40,240,20);
		columnButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				span = COLUMN_SELECTED_SPAN;
				startField.setText("");
				startField.setEnabled(false);
				endField.setText("");
				endField.setEnabled(false);
			}
		});
		
		manualButton.setBounds(20,60,200,20);
		manualButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				span = USER_DEFINED_SPAN;
				startField.setEnabled(true);
				endField.setEnabled(true);
				startField.requestFocus();
			}
		});
		
		startLabel.setBounds(60, 80, 50, 20);
		startField.setBounds(110, 80, 50, 20);
		startField.setEnabled(span == USER_DEFINED_SPAN);
		endLabel.setBounds(180, 80, 30, 20);
		endField.setBounds(220, 80, 50, 20);

		
		group2.add(fullButton);
		group2.add(columnButton);
		group2.add(manualButton);
		
		choicePanel.add(fullButton);
		choicePanel.add(columnButton);
		choicePanel.add(manualButton);
		
		choicePanel.add(startLabel);
		choicePanel.add(startField);
		choicePanel.add(endLabel);
		choicePanel.add(endField);

		
		base.add(choicePanel);
		
		
		
		pathPanel.setBounds(5,185,290,60);
		pathPanel.setLayout(null);
		
		border1 = BorderFactory.createEmptyBorder(0,0,0,0);//left offset of the boxes in the box
		border2 = BorderFactory.createTitledBorder(new EtchedBorder(), " Modeller directory location ");
		border2.setTitleColor(Color.black);
		border2.setTitleFont(new Font("Dialog", 0, 12));
		border = BorderFactory.createCompoundBorder(border1, border2);
		pathPanel.setBorder(border);
		
		pathField = new JTextField(100);
		pathField.setBounds(20,25,170,20);
		pathField.setText(StylesPreferences.defaultModellerDirectory);
		
		pathButton.setBounds(200,25,80,20);
		pathButton.setFont(new Font("Dialog", Font.PLAIN, 11));
		pathButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				//get the new directory from the user
				//browse the filesystem. select directories
				onTop = false;
				JFileChooser chooser = new JFileChooser(StylesPreferences.workingDirectory);
				chooser.setApproveButtonText("Select");
				chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
				int result = chooser.showOpenDialog(parentFrame);
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				String newDirectory = filename.toString();
				pathField.setText(newDirectory);
				onTop = true;
			}
		});

		
		pathPanel.add(pathField);
		pathPanel.add(pathButton);
		
		base.add(pathPanel);
		
		//buttons
		okButton.setBounds(65, 260, 80, 25);
		okButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				process();
			}
		});
		
		cancelButton.setBounds(160, 260, 80, 25);
		cancelButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
		});
		
		pathField.requestFocus();

		
		contentPane.add(base);
		contentPane.add(okButton);
		contentPane.add(cancelButton);
		
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
			
			public void windowDeactivated(WindowEvent e){
				if (onTop) toFront();
			}
		});
				
		setSize(315, 320);
		
		//add this to accommodate both applet and application cases
		if (parentFrame == null){
			setBounds(0, 0, getSize().width, getSize().height);
		}
		else{
		
			//set location
			Dimension d1 = getSize();
			Dimension d2 = parentFrame.getSize();
		
			int x = 5 + DialogConstants.toolbarWidth;
			int y = DialogConstants.level + DialogConstants.toolbarHeight;
		
			setBounds(x + parentFrame.getX(), y + parentFrame.getY(), d1.width, d1.height);
		}

		
		setResizable(false);
		
		setVisible(true);
		


	}
	
	private void process(){
		
		int start = 0;//zero-based column indices - inclusive
		int end = 0;
		try{
			if (span == USER_DEFINED_SPAN){
				start = Integer.parseInt(startField.getText().trim());
				end = Integer.parseInt(endField.getText().trim());
			}
			else if (span == COLUMN_SELECTED_SPAN){
				//start and end are derived from the current column selection
				Vector cols = parent.getSequenceViewer().getSelectedColumns();
				if (cols.size() != 2){
					onTop = false;
					parent.displayErrorMessage("Only two columns must be selected for start and end of the alignment area.");
					onTop = true;
					return;
				}
				
				start = (Integer)cols.get(0);
				end = (Integer)cols.get(1);
				
				if (start > end){
					//swap them, since it could happen if the end column is selected before the start column
					int temp = start;
					start = end;
					end = temp;
				}
				
			}
			else{
				end = parent.getSequenceViewer().getResidueMax() - 1;//to account for the 0-based index
			}
		}
		catch (NumberFormatException ex){
			onTop = false;
			parent.displayErrorMessage("Start and end columns for sequence span must be integers.");
			onTop = true;
			return;
		}
		
		if (end <= start){
			onTop = false;
			parent.displayErrorMessage("Selected sequence span is less than one residue.");
			onTop = true;
			return;
		}
		
		final int startIndex = start;//for passing as arguments to the manager
		final int endIndex = end;
		
//		System.out.println("start = " + start + ", end = " + end);
		
		if (set == USE_SELECTED_ENTRIES){
			//check whether anything is selected
			if (parent.getSequenceViewer().getSelectedSequences().size() < 2){
				onTop = false;
				parent.displayErrorMessage("At least two entries must be selected in Sequence Viewer for this option.");
				onTop = true;
				return;
			}
		}
		
		setVisible(false);
		onTop = false;
		
		//now get the entries in question
		Vector structures = new Vector();//vector of templates (there can be more than one): Structure objects
		Vector sequences = new Vector();//Sequence objects that correspond to the loaded structures and the target
		Sequence target = null;
		Vector seqs = parent.getSequenceViewer().getSequenceObjects();

		for (int i = 0; i < seqs.size(); i++){
			Sequence sequence = (Sequence)seqs.get(i);
			if (set == USE_SELECTED_ENTRIES){
				if (!parent.getSequenceViewer().isSelected(sequence)){
					continue;//ignore anything that is not selected
				}
			}
			
			if (parent.getSequenceViewer().getStyleMap().containsKey(sequence)){
				//there is a structure associated with this entry
				StructureStyles styles = (StructureStyles)parent.getSequenceViewer().getStyleMap().get(sequence);
				structures.add(styles.getStructureMap().getStructure());
				sequences.add(sequence);
			}
			else{
				if (target == null){
					target = sequence;
					sequences.add(target);
				}
				else{
					//it's a second sequence that could possibly be the target
					parent.displayErrorMessage("Two or more target sequences have been selected. Make sure there is only one possible target sequence.");
					parent.setDisplayDialogStatus(false);
					return;
				}
			}
		}
		
		final Sequence ts = target;
		
		StylesPreferences.modellerDirectory = pathField.getText().trim();
		
		//create temporary directory
		String home = System.getProperty("user.home");
		try{
			tempDirectory = "D:\\.model";
//			tempDirectory = home + File.separator + ".model";
			File tempdir = new File(tempDirectory);
			boolean exists = tempdir.exists();
			boolean result = tempdir.mkdir();
			if (!exists && !result){
				parent.displayErrorMessage("Unable to create temporary directory: " + tempdir);
				parent.setDisplayDialogStatus(false);
				return;
			}
			
			File alignmentFile = new File(tempDirectory + File.separator + "model.ali");
		
			//now get the actual sequence/structure data
			
			//save the alignment file
			//int the process, check whether the current Sequence has an associated Structure
			//if so, check the start and end Residues and save the appropriate part of the structure
			PrintWriter printer = new PrintWriter(new FileWriter(alignmentFile));
			
			Vector knowns = new Vector();//names of structures
			
			//scan over all loaded sequences
			
			HashMap styles = parent.getSequenceViewer().getStyleMap();
			for (int i = 0; i < sequences.size(); i++){
				Sequence sequence = (Sequence)sequences.get(i);
				Cell[] cells = sequence.getCells();
				
				
				Structure structure = null;
				Residue startResidue = null;
				Residue endResidue = null;
				
				//check for structure first: if the structure can't be saved, sequence shouldn't be there either
				if (styles.containsKey(sequence)){
					//it means there is an associated structure
					structure = ((StructureStyles)styles.get(sequence)).getStructureMap().getStructure();
					
					if (structure == null) continue;
					
//					System.out.println("Start = " + start + ", end = " + end);
//					System.out.println("Sequence length = " + cells.length);
					
					Cell startCell = cells[start];
					Cell endCell = cells[end];
					startResidue = startCell.residue;
					endResidue = endCell.residue;
					
//					System.out.println("startResidue = " + startResidue.getCompoundCode());
//					System.out.println("endResidue = " + endResidue.getCompoundCode());
					
					if (startResidue == null){
						//walk forward until residue is non-null
						int index = start;
						while (true){
							startResidue = cells[++index].residue;
							if (startResidue != null) break;
							if (index >= cells.length) break;
						}
					}
					
					if (endResidue == null){
						//walk back until residue is non-null
						int index = end;
						while (true){
							endResidue = cells[--index].residue;
							if (endResidue != null) break;
							if (index == 0) break;
						}
					}
					
					if (startResidue == null || endResidue == null){
//						System.out.println("Ignoring structure");
						continue;//ignore this structure/sequence
					}
					
					//here, both residues are non-null
					//save the structure between them, inclusive
					String name = structure.getUrlString();
					name = name.replace(':', '#');
					if (name.lastIndexOf("/") > -1) name = name.substring(name.lastIndexOf("/")+1);
					if (name.lastIndexOf("\\") > -1) name = name.substring(name.lastIndexOf("\\")+1);//just in case
					
					String fname = tempDirectory + File.separator + name + ".pdb";
					try{
						savePdb(structure, startResidue, endResidue, fname);
					}
					catch (Exception ex){
						ex.printStackTrace();
						continue;
					}
					
					knowns.add(name);

				}
				
				
				
				//************IMPORTANT: to comply with strange rules defined by modeller, sequence name is simplified to target ********
				
				
				String n = sequence.getName();
				
				//now print the information
				if (structure == null){
					printer.println(">P1;model");
					printer.print("sequence:model:");
				}
				else{
					printer.println(">P1;" + n);
					printer.print("structureX:" + n + ":");
				}
				
				//now print residue/chain numbers for the fragment to be modeled
				//count non-dash symbols
				int count = 0;
//				System.out.println("cells length = " + cells.length);
				for (int j = start; j <= end; j++){
					
					if (cells[j].symbol == "#" || cells[j].symbol == "*" || cells[j].symbol == "-") continue;
					count++;
				}
				
//				System.out.println("startResidue = " + startResidue + ", endResidue = " + endResidue);
				
				if (startResidue != null && endResidue != null){
					printer.print(StringUtils.rightPadded(new Integer(startResidue.getResidueId()).toString(), 5, " "));
					if (startResidue.getChainId().equals("_")){
						printer.print(": :");
					}
					else{
						printer.print(":" + startResidue.getChainId() + ":");
					}
					printer.print(StringUtils.rightPadded(new Integer(endResidue.getResidueId()).toString(), 5, " "));
					if (endResidue.getChainId().equals("_")){
						printer.println(": : : : :");
					}
					else{
						printer.println(":" + endResidue.getChainId() + ": : : :");
					}
				}
				else{
					printer.print("1    : :");
					printer.print(StringUtils.rightPadded(new Integer(count).toString(), 5, ' '));
					printer.println(": : : : : ");
				}
				
				//then record the sequence 60 characters per line
				StringBuffer buffer = new StringBuffer();
				int counter = 0;
				boolean init = true;
				for (int j = start; j <= end; j++){
					if (cells[j].symbol == "#" || cells[j].symbol == "*") continue;
					buffer.append(cells[j].symbol);
					counter++;
					if (counter == 60){
						if (init){
							printer.print(buffer.toString());
							init = false;
						}
						else{
							printer.print("\n");
							printer.print(buffer.toString());
						}
						counter = 0;
						buffer = new StringBuffer();
					}
				}
				
				if (buffer.length() > 0){
					if (!init) printer.print("\n");
					printer.print(buffer.toString());
				}
				
				printer.println("*");//finish printing the current sequence

				
				
			}
			
			printer.flush();
			printer.close();
			
			//now prepare the default.py script to launch the modeling job
			final String script = tempDirectory + File.separator + "default.py";
			PrintWriter writer = new PrintWriter(new FileWriter(script));
			
			writer.println("from modeller import *              # Load standard Modeller classes");
			writer.println("from modeller.automodel import *    # Load the automodel class");
			writer.println("");
			writer.println("log.verbose()    # request verbose output");
			writer.println("env = environ()  # create a new MODELLER environment to build this model in");
			writer.println("\n# directories for input atom files");
//			writer.println("env.io.atom_files_directory = \'" + tempDirectory + "\'");
			writer.println("env.io.atom_files_directory = \'.\'");
			writer.println("\na = automodel(env,");
			writer.println("              alnfile  = 'model.ali',     # alignment filename");
			writer.print("              knowns   = ");
			
			for (int j = 0; j < knowns.size(); j++){
				String known = (String)knowns.get(j);
				writer.print("\'" + known + "\',");
			}
			
			writer.println("              # codes of the templates");
			
			writer.println("              sequence = \'model\')              # code of the target");
			writer.println("a.starting_model= 1                 # index of the first model ");
			writer.println("a.ending_model  = 1                 # index of the last model");
			writer.println("                                    # (determines how many models to calculate)");
			writer.println("\na.make()                            # do the actual homology modeling");
			writer.flush();
			writer.close();
			
			
			
			//now start the actual modeling process with created input files
			final String modeller = pathField.getText().trim() + File.separator + "bin" + File.separator + "mod9v2";
			
			//check the OS: for Windows, create a bat file
			String os = System.getProperty("os.name");
			if (os.startsWith("Win")){
				PrintWriter p = new PrintWriter(new FileWriter(tempDirectory + File.separator + "run.bat"));
				
				//get the drive letter from the temp directory
				String drive = tempDirectory.substring(0, tempDirectory.indexOf(":"));
				p.println(drive + ":");
				p.println("cd " + tempDirectory);
				p.println(modeller + " " + tempDirectory + File.separator + "default.py");
				p.println("exit");
				p.flush();
				p.close();
				
				
				Thread runner = new Thread(){
					public void run(){
						final Manager manager = parent;
						final String original = ts.getName();
						try{
							String[] cmd = { "cmd", "/c", tempDirectory + File.separator + "run.bat" };

							manager.displayWaitScreen();
							
							Process process = Runtime.getRuntime().exec(cmd);
							process.waitFor();
							
							int exit = process.exitValue();
							
							//this thread will now talk to the parent with the results
							manager.removeWaitScreen();
							
							manager.processHomologyModel(tempDirectory, original, startIndex, endIndex);
						}
						catch (Exception ex){
							ex.printStackTrace();
							manager.removeWaitScreen();
						}
					}
				};
				runner.start();

			}
			else if (os.startsWith("Linux")){
				parent.displayErrorMessage("No Linux support yet. Coming up.");
				parent.setDisplayDialogStatus(false);
				return;
			}
			
			parent.setDisplayDialogStatus(false);
			dispose();
		
		}
		catch (Exception e){
			parent.removeWaitScreen();
			parent.displayExceptionMessage("Unable to save input data.", e);
			parent.setDisplayDialogStatus(false);
			return;
		}

	
	}
	
	/**
	 * THis method saves the specified range of residues to a PDB file. Non-amino acid residues are skipped
	 * @param structure
	 * @param start
	 * @param end
	 * @param fname
	 */
	private void savePdb(Structure structure, Residue start, Residue end, String fname) throws IOException {
		
		FileWriter fr = new FileWriter(fname);
		PrintWriter printer = new PrintWriter(fr);
		
		int chainIdCounter = 0;

		//iterate over all atoms in the structure
		int counter = 1;//atom number counter: will use this, in case reshuffling will be needed (e.g., with added hydrogens in proteins)
		HashMap reference = new HashMap();//old atom number -> new counter
		
		HashMap atomNames = new HashMap();//element -> current index
		
		boolean save = false;
		for (int i = 0 ; i < structure.getStructureMap().getResidueCount(); i++){
			
			Residue residue = structure.getStructureMap().getResidue(i);
			if (!save && residue == start){
				save = true;
			}
			
			
			if (!save) continue;
			
			//walk through the atoms of this residue and save them in order
			for (int j = 0; j < residue.getAtomCount(); j++){
				Atom a = residue.getAtom(j);
				
				reference.put(new Integer(a.number), new Integer(counter));
				
				if (!ProteinProperties.isAminoAcid(a.compound) && !DNAProperties.isNucleicAcid(a.compound)){
					continue;
				}
				
				//the ATOM keyword
				printer.print(StringUtils.rightPadded("ATOM",6," "));

				
				//atom number
				if (PeriodicTable.getElementSymbol(a.element).length() > 1){
					//start the name immediately
					printer.print(StringUtils.leftPadded((new Integer(counter)).toString(),5, " ") + " ");
				
					//atom type
					printer.print(StringUtils.rightPadded(a.name, 4, " ") + " ");
				}
				else{
					if (a.name.length() > 3){
						printer.print(StringUtils.leftPadded((new Integer(counter)).toString(),5, " ") + " ");
						
						//atom type
						printer.print(StringUtils.rightPadded(a.name, 4, " ") + " ");
					}
					else{
						printer.print(StringUtils.leftPadded((new Integer(counter)).toString(),5, " ") + "  ");
						
						//atom type
						printer.print(StringUtils.rightPadded(a.name, 3, " ") + " ");
					}
					
				}
				
				//residue
				printer.print(StringUtils.rightPadded(a.compound, 3, " ") + " ");
				
				//chain
				if (a.chain_id.equals("_")){
					printer.print(" ");
				}
				else{
					printer.print(StringUtils.rightPadded(a.chain_id, 1, " "));
				}
				
				//residue number
				printer.print(StringUtils.leftPadded((new Integer(a.residue_id)).toString(), 4, " ") + "     ");
//				System.out.println("residue id = " + a.residue_id);
				
				//coordinates
				String x = StringUtils.getFormattedNumber(a.coordinate[0], 3);
				String y = StringUtils.getFormattedNumber(a.coordinate[1], 3);
				String z = StringUtils.getFormattedNumber(a.coordinate[2], 3);
				
				//make sure the part of the number after the point has three digits
				int index = x.indexOf(".");
				String subX = x.substring(0, index+1);
				String ext = x.substring(index+1);
				x = subX + StringUtils.rightPadded(ext, 3, "0");
				
				index = y.indexOf(".");
				String subY = y.substring(0, index+1);
				ext = y.substring(index+1);
				y = subY + StringUtils.rightPadded(ext, 3, "0");
	
				index = z.indexOf(".");
				String subZ = z.substring(0, index+1);
				ext = z.substring(index+1);
				z = subZ + StringUtils.rightPadded(ext, 3, "0");
	
				printer.print(StringUtils.leftPadded(x, 7, " ") + " ");
				printer.print(StringUtils.leftPadded(y, 7, " ") + " ");
				printer.print(StringUtils.leftPadded(z, 7, " ") + "  ");
				
				//occupancy
				printer.print(StringUtils.rightPadded("1.00", 4, " "));
				
				//bfactor
				printer.print(StringUtils.leftPadded("0.00", 6, " "));
				
				//the spacing
				printer.print(StringUtils.leftPadded(" ", 11, " "));
				
				printer.print(StringUtils.rightPadded(a.getElement(), 3, " "));
				
				printer.print("\n");
				counter++;
			}
			
			if (save && residue == end){
				save = false;
				break;
			}


		}
		
		printer.println("END");
		printer.close();

	}
	
	public void processPick(String data, StructureComponent structureComponent){}
	
}
