//  $Id: ImageFileChooser.java,v 1.1 2006/10/21 17:52:14 Sasha Buzko Exp $
//
//  Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//  a facility operated jointly by the University of California,
//  San Diego (UCSD) and General Atomics, San Diego, California, USA.
//
//  Users and possessors of this source code are hereby granted a
//  nonexclusive, royalty-free copyright and design patent license to
//  use this code in individual software.  License is not granted for
//  commercial resale, in whole or in part, without prior written
//  permission from SDSC.  This source is provided "AS IS" without express
//  or implied warranty of any kind.
//
//  For further information, please see:  http://mbt.sdsc.edu
//
//  History:
//  $Log: ImageFileChooser.java,v $
//  Revision 1.1  2006/10/21 17:52:14  Sasha Buzko
//  Refactored the project to move all new code to edu.sdsc.sirius package.
//
//  Revision 1.1  2006/05/20 17:02:00  Sasha Buzko
//  Updated version
//
//  Revision 1.1  2006/04/30 20:13:56  Sasha Buzko
//  New version of the app
//
//  Revision 1.1  2006/04/15 19:42:19  Sasha Buzko
//  Initial commit
//
//  Revision 1.1  2005/11/13 04:35:23  Administrator
//  *** empty log message ***
//
//  Revision 1.1  2003/07/14 21:26:36  moreland
//  Added "gui" package.
//
//  Revision 1.0  2003/05/05 16:06:13  moreland
//  First version.
//


package edu.sdsc.sirius.dialogs;


import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.*;
import javax.imageio.*;
import javax.imageio.spi.*;
import javax.imageio.stream.*;
import javax.swing.*;


public class ImageFileChooser
	extends JFileChooser
{
	// Enables selection of any supported Image Format plug-in.
	private JComboBox imageFormatComboBox;

	// Which AWT Component to use as the parent for the chooser window.
	private Component parentComponent;

	/**
	 *  Construct a new ImageFileChooser object.
	 */
	public ImageFileChooser( Component parentComponent )
	{
		super( new File( System.getProperty( "user.dir" ) ) );
		this.parentComponent = parentComponent;
	}

	/**
	 *  Prompt the user to load an image file using any of the supported
	 *  Image Format plug-ins.
	 */
	public BufferedImage load( )
	{
		int ret = showOpenDialog( parentComponent );
		if ( ret == JFileChooser.APPROVE_OPTION )
		{
			try
			{
				File file = getSelectedFile();
				String filename = file.getName();

				// Get the file suffix
				String suffix = filename.substring( filename.lastIndexOf( '.' ) + 1 );

				// Get a set of ImageReaders that
				// support that suffix
				Iterator readers = ImageIO.getImageReadersBySuffix( suffix );

				// Pick the first one to use
				ImageReader imageReader = (ImageReader) readers.next( );

				// Get an input stream for the file
				FileInputStream fin = new FileInputStream( file );

				// Turn it into an ImageInputStream
				ImageInputStream iis = ImageIO.createImageInputStream( fin );

				// Plug this stream into the ImageReader
				imageReader.setInput( iis, false );

				// Read the image
				BufferedImage image = imageReader.read( 0 );

				// Dont forget to close the file!
				fin.close( );

				return image;
			}
			catch( IOException ie )
			{
				ie.printStackTrace( );
				return null;
			}
		}

		return null;
	}

	/**
	 *  Describes a Format plug-in for use as a JFileChooser filter.
	 */
	static private class FormatDescriptor
	{
		public String suffix;
		public ImageWriter imageWriter;
		public String description;

		public FormatDescriptor( String suffix, ImageWriter imageWriter,
			String description )
		{
			this.suffix = suffix;
			this.imageWriter = imageWriter;
			this.description = description;
		}

		public String toString( )
		{
			return description;
		}
	}

	/**
	 *  Return a vector of FormatDescriptor objects, one for each
	 *  supported Image Format plug-ins.
	 */
	static private Vector getWriterFormats( )
	{
		Vector formats = new Vector( );
		Hashtable seen = new Hashtable( );

		String names[] = ImageIO.getWriterFormatNames( );
		for ( int i=0; i<names.length; ++i )
		{
			String name = names[i];
			Iterator writers = ImageIO.getImageWritersByFormatName( name );
			while ( writers.hasNext() )
			{
				ImageWriter iw = (ImageWriter) writers.next( );
				ImageWriterSpi iws = iw.getOriginatingProvider( );
				String suffixes[] = iws.getFileSuffixes( );
				boolean allseen = true;
				for ( int j=0; j<suffixes.length; ++j )
				{
					String suffix = suffixes[j];
					suffix = suffix.toLowerCase( );
					if ( ! seen.containsKey( suffix ) )
					{
						seen.put( suffix, suffix );
						String description = name + " (*." + suffix + ")";
						FormatDescriptor fd =
							new FormatDescriptor( suffix, iw, description );
						formats.addElement( fd );
					}
				}
			}
		}

		return formats;
	}

	/**
	 *  Show the custom Image Save dialog box.
	 */
	private void augmentSaveDialog( BufferedImage image )
	{
		Vector formats = getWriterFormats( );

		imageFormatComboBox = new JComboBox( formats );
		final JComboBox cb = imageFormatComboBox;
		imageFormatComboBox.addActionListener( new ActionListener( )
		{
			public void actionPerformed( ActionEvent e )
			{
				FormatDescriptor fd =
					(FormatDescriptor) imageFormatComboBox.getSelectedItem( );
				String suffix = fd.suffix;
				updateFileSuffix( suffix );
			}
		} );

		Component north = getComponent( 0 );
		Component center = getComponent( 1 );
		Component south = getComponent( 2 );

		remove( north );
		remove( center );
		remove( south );
		
		JPanel jp = new JPanel( );
		jp.setLayout( new BorderLayout( 10, 10 ) );
		jp.add( north, BorderLayout.NORTH );
//		jp.add( center, BorderLayout.CENTER );
		jp.add( south, BorderLayout.SOUTH );

		JLabel formatLabel = new JLabel( "Select Format:" );

		JPanel formatRow = new JPanel( );
		formatRow.setLayout( new BorderLayout( 10, 10 ) );
		formatRow.add( formatLabel, BorderLayout.WEST );
		formatRow.add( imageFormatComboBox, BorderLayout.CENTER );

		JPanel augmentation = new JPanel( );
		augmentation.setLayout( new BorderLayout( 10, 10 ) );
		augmentation.add( formatRow, BorderLayout.SOUTH );

		setLayout( new BoxLayout( this, BoxLayout.Y_AXIS ) );
		add( augmentation );
		Dimension fillerDimension = new Dimension( 100, 10 );
		add( new Box.Filler( fillerDimension, fillerDimension, fillerDimension ) );
		add( jp );

		setSize( 450, 550 );
//		setResizeable(false);
		setLocation( 30, 30 );
	}

	/**
	 *  Prompt the user to save the given BufferedImage object, and,
	 *  if the user approves, save the file to the user-specified file.
	 */
	public void save( BufferedImage image )
	{
		augmentSaveDialog( image );

		int ret = showSaveDialog( parentComponent );
		if ( ret == JFileChooser.APPROVE_OPTION )
		{
			try
			{
				File file = getSelectedFile( );

				// Get the currently-selected item in the combo box
				FormatDescriptor fd =
					(FormatDescriptor) imageFormatComboBox.getSelectedItem( );

				// Get the ImageWriter stored in this FormatDescriptor
				ImageWriter imageWriter = fd.imageWriter;

				// Get an output stream for the file
				FileOutputStream fout = new FileOutputStream( file );

				// Turn it into an ImageOutputStreamn
				ImageOutputStream ios = ImageIO.createImageOutputStream( fout );

				// Plug this stream into the ImageWriter
				imageWriter.setOutput( ios );

				// Write the image
				imageWriter.write( image );

				// Dont forget to close the file!
				fout.close();
			}
			catch( IOException ie )
			{
				ie.printStackTrace( );
			}
		}
	}

	/**
	 *  Recursively walk and print an arbitrary AWT/Swing Component tree.
	 *  NOTE: THIS IS ONLY HERE FOR DEBUGGING PURPOSES. IT MAY BE
	 *  REMOVED IN THE FUTURE.
	 */
	private void printComponents( Component comp, int level, int item )
	{
		for ( int i=0; i<level; i++ )
			System.out.print( "   " );
		System.out.print( "level=" + level + ", item=" + item + ", name=" );

		if ( comp instanceof javax.swing.text.JTextComponent )
		{
			javax.swing.text.JTextComponent textComponent =
				(javax.swing.text.JTextComponent) comp;
			System.out.println( textComponent.getText() );
		}
		else
		{
			System.out.println( comp.getName() );
		}

		if ( comp instanceof Container )
		{
			Container cont = (Container) comp;
			for ( int c=0; c<cont.getComponentCount(); c++ )
			{
				printComponents( cont.getComponent(c), level+1, c );
			}
		}
	}

	/**
	 *  Whenever the user selects a different image file format plug-in,
	 *  automatically update the file name extension of the user-supplied
	 *  file name.
	 */
	private void updateFileSuffix( String suffix )
	{
		File file = getSelectedFile( );

		if ( file == null )
		{
			// JDK 1.4.0 and earlier has a bug which prevents the
			// user-typed file name from being get/set correctly.
			// The in this if-block is a workaround.
			Container containerA = (Container) getComponent( 2 );
			Container containerB = (Container) containerA.getComponent( 1 );
			Container containerC = (Container) containerB.getComponent( 2 );
			Container containerD = (Container) containerC.getComponent( 2 );
			JTextField jTextField = (JTextField) containerD.getComponent(1);
			
/*			System.out.println("START");

			for (int i = 0; i < getComponentCount(); i++){
				Container a = (Container)getComponent(i);
				Component aa = (Component)a;
				if (aa instanceof JTextField){
					System.out.println("FOUND");
				}
				else{
					for (int j = 0; j < a.getComponentCount(); j++){
						Container b = (Container)a.getComponent(j);
						Component bb = (Component)b;
						if (bb instanceof JTextField){
							System.out.println("FOUND");
						}
						else{
							for (int k = 0; k < b.getComponentCount(); k++){
								Container c = (Container)b.getComponent(k);
								Component cc = (Component)c;
								if (cc instanceof JTextField){
									System.out.println("FOUND");
								}
								else{
									for (int f = 0; f < c.getComponentCount(); f++){
										Container d = (Container)c.getComponent(f);
										Component dd = (Component)d;
										if (dd instanceof JTextField){
											System.out.println("FOUND");
										}
										else{
											for (int n = 0; n < d.getComponentCount(); n++){
												Container m = (Container)d.getComponent(n);
												Component mm = (Component)m;
												if (mm instanceof JTextField){
													System.out.println("FOUND HERE");
													System.out.println(i + "," + j + "," + k + "," + f + "," + n);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			
			System.out.println("DONE");
*/				
			String typedName = jTextField.getText();
			
//			if ( typedName == null ) return;
			int lp = typedName.lastIndexOf( "." );
			if ( lp == -1 ){
				//just add the dot and the suffix
				typedName = typedName + "." + suffix;
			}
			else{
				typedName = typedName.substring( 0, lp ) + "." + suffix;
			}
			jTextField.setText( typedName );
		}
		else
		{
			if ( file == null ) return;
			String filename = file.getPath( );
			int lp = filename.lastIndexOf( "." );
			if ( lp == -1 ) return;
			filename = filename.substring( 0, lp ) + "." + suffix;
			setSelectedFile( new File( filename ) );
		}
	}
}

