package edu.sdsc.sirius.dialogs;

// Core
import java.io.*;

// GUI
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.border.*;

// MBT
import edu.sdsc.mbt.*;
import edu.sdsc.sirius.util.*;

import edu.sdsc.sirius.io.*;
import edu.sdsc.sirius.md.*;

public class MDSetupDialog extends JDialog implements DisplayDialog {

	private JFrame parentFrame;
	private Manager parent;
	
	private DisplayDialog itself;

	private Container contentPane;
	private JPanel base = new JPanel();
	
	private JPanel inputPanel = new JPanel();
	private JPanel waterPanel = new JPanel();
	private JPanel optionsPanel = new JPanel();
	private JPanel outputPanel = new JPanel();
	private JPanel destinationPanel = new JPanel();
	
	private JCheckBox destination = new JCheckBox("Create SGE script");
	
	private Border border1;
	private TitledBorder border2;
	private Border border;
	
	private JTextField inputField1 = new JTextField(50);
	private JTextField inputField2 = new JTextField(50);
	
	private ButtonGroup group = new ButtonGroup();
	private JRadioButton implicitButton = new JRadioButton("Implicit solvent", false);
	private JRadioButton waterButton = new JRadioButton("Explicit water", true);
	
	private JTextField dirField = new JTextField(50);
	private JTextField destField = new JTextField(50);
	private JButton destButton;
	
	private JTextField minSteps = new JTextField(30);
	private JTextField eqSteps = new JTextField(30);
	private JTextField nodeCount = new JTextField(30);
	
	private JTextField totalRuns = new JTextField(30);
	private JTextField mdSteps = new JTextField(40);
	private JTextField saveFrequency = new JTextField(30);
	
	private JCheckBox ibCluster = new JCheckBox("InfiniBand cluster");

	
	private JButton okButton = new JButton("OK");
	private JButton cancelButton = new JButton("Cancel");
	
	
	public MDSetupDialog(JFrame f, Manager p){

		super(f, "Prepare molecular dynamics simulation", false);

		parentFrame = f;
		parent = p;
		
		parent.setDisplayDialogStatus(true);
		
		itself = this;
		
		//create the panels
		contentPane = getContentPane();
		contentPane.setLayout(null);
		
		
		base.setBorder(new BevelBorder(BevelBorder.RAISED));
		base.setBounds(5, 5, 495, 370);
		base.setLayout(null);
		
		//first panel - input files
		inputPanel.setBounds(5,5,485,85);
		inputPanel.setLayout(null);
		
		inputPanel.setBorder(createTitledBorder("Input files"));
		
		JLabel input1 = new JLabel("prmtop:");
		JLabel input2 = new JLabel("inpcrd:");
		
		input1.setBounds(20, 25, 60, 20);
		input2.setBounds(20, 50, 60, 20);
		
		inputField1.setBounds(90,25,275,20);
				
		JButton inputButton1 = new JButton("Browse...");
		inputButton1.setFont(new Font("Dialog", Font.PLAIN, 12));
		inputButton1.setBounds(370,25,90,20);
		inputButton1.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
				chooser.setApproveButtonText("Select");
				int result = chooser.showOpenDialog(parent.getApplicationFrame());
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				inputField1.setText(filename.toString());
				parent.setLastUsedDirectory(filename.toString());
			}
		});
		
		
		inputPanel.add(input1);
		inputPanel.add(inputField1);
		inputPanel.add(inputButton1);
		
		inputField2.setBounds(90,50,275,20);
		
		JButton inputButton2 = new JButton("Browse...");
		inputButton2.setFont(new Font("Dialog", Font.PLAIN, 12));
		inputButton2.setBounds(370,50,90,20);
		inputButton2.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
				chooser.setApproveButtonText("Select");
				int result = chooser.showOpenDialog(parent.getApplicationFrame());
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				inputField2.setText(filename.toString());
				parent.setLastUsedDirectory(filename.toString());
			}
		});

		inputPanel.add(input2);
		inputPanel.add(inputField2);
		inputPanel.add(inputButton2);
		
		
		base.add(inputPanel);
		
		//water
		waterPanel.setBounds(5,90,485,50);
		waterPanel.setLayout(null);
		
		waterPanel.setBorder(createTitledBorder("Solvent treatment"));
		implicitButton.setBounds(60, 20, 120, 20);
		waterButton.setBounds(250, 20, 120, 20);
		
		group.add(implicitButton);
		group.add(waterButton);
		
		waterPanel.add(implicitButton);
		waterPanel.add(waterButton);
		
		base.add(waterPanel);
		
		//parameters
		optionsPanel.setBounds(5,140,485,125);
		optionsPanel.setLayout(null);
		optionsPanel.setBorder(createTitledBorder("Simulation options"));
		
		JLabel minStepsLabel = new JLabel("Minimization steps:");
		minStepsLabel.setBounds(20,20,120,20);
		minSteps.setBounds(150, 20, 50, 20);
		minSteps.setText("10000");
		
		JLabel eqStepsLabel = new JLabel("Equilibration steps:");
		eqStepsLabel.setBounds(20, 45, 120, 20);
		eqSteps.setBounds(150, 45, 50, 20);
		eqSteps.setText("250000");
		
		JLabel threadCountLabel = new JLabel("Number of nodes:");
		threadCountLabel.setBounds(20, 70, 120, 20);
		nodeCount.setBounds(150, 70, 50, 20);
		nodeCount.setText("1");
		
		JLabel totalRunsLabel = new JLabel("Total MD runs:");
		totalRunsLabel.setBounds(20, 95, 120, 20);
		totalRuns.setBounds(150, 95, 50, 20);
		totalRuns.setText("10");
		
		JLabel mdStepsLabel = new JLabel("Steps in each run:");
		mdStepsLabel.setBounds(240, 20, 120, 20);
		mdSteps.setBounds(370, 20, 50, 20);
		mdSteps.setText("250000");
		
		JLabel saveFrequencyLabel = new JLabel("Save frequency:");
		saveFrequencyLabel.setBounds(240, 45, 120, 20);
		saveFrequency.setBounds(370, 45, 50, 20);
		saveFrequency.setText("500");
		
		ibCluster.setSelected(false);
		ibCluster.setBounds(240, 70, 120, 20);
		
		destination.setSelected(true);
		destination.setBounds(240, 95, 140, 20);
		destination.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				destField.setEnabled(destination.isSelected());
				destField.setEditable(destination.isSelected());
				destButton.setEnabled(destination.isSelected());
			}
		});
		
		optionsPanel.add(minStepsLabel);
		optionsPanel.add(minSteps);
		optionsPanel.add(eqStepsLabel);
		optionsPanel.add(eqSteps);
		optionsPanel.add(threadCountLabel);
		optionsPanel.add(nodeCount);
		optionsPanel.add(totalRunsLabel);
		optionsPanel.add(totalRuns);
		optionsPanel.add(mdStepsLabel);
		optionsPanel.add(mdSteps);
		optionsPanel.add(saveFrequencyLabel);
		optionsPanel.add(saveFrequency);
		optionsPanel.add(ibCluster);
		optionsPanel.add(destination);
		
		
		base.add(optionsPanel);
		
		//output directory
		outputPanel.setBounds(5,265,485,50);
		outputPanel.setLayout(null);
		outputPanel.setBorder(createTitledBorder("Output directory for simulation files"));

		dirField.setBounds(25,20,335,20);
		
		JButton dirButton = new JButton("Browse...");
		dirButton.setFont(new Font("Dialog", Font.PLAIN, 12));
		dirButton.setBounds(370,20,90,20);
		dirButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
				chooser.setApproveButtonText("Select");
				chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
				int result = chooser.showOpenDialog(parent.getApplicationFrame());
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				dirField.setText(filename.toString());
			}
		});
		
		
		outputPanel.add(dirField);
		outputPanel.add(dirButton);	
		
		base.add(outputPanel);
		
		
		//output directory
		destinationPanel.setBounds(5,315,485,50);
		destinationPanel.setLayout(null);
		destinationPanel.setBorder(createTitledBorder("Destination directory for MD results"));
		
		destField.setBounds(25,20,335,20);
		
		destButton = new JButton("Browse...");
		destButton.setFont(new Font("Dialog", Font.PLAIN, 12));
		destButton.setBounds(370,20,90,20);
		destButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
				chooser.setApproveButtonText("Select");
				chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
				int result = chooser.showOpenDialog(parent.getApplicationFrame());
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				destField.setText(filename.toString());
			}
		});
		
		destinationPanel.add(destField);
		destinationPanel.add(destButton);	

		
		
		base.add(destinationPanel);
		
		contentPane.add(base);
		
		//buttons
		okButton.setBounds(150, 380, 80, 25);
		okButton.setVisible(true);
		okButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				process();
			}
		});
		
		cancelButton.setBounds(270, 380, 80, 25);
		cancelButton.setVisible(true);
		cancelButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
		});
		
		contentPane.add(base);
		contentPane.add(okButton);
		contentPane.add(cancelButton);
		
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
			
		});
				
		setSize(510, 435);
		
		if (parent == null){
			setLocationRelativeTo(null);
		}
		else{
			//set location
			Dimension d1 = getSize();
			Dimension d2 = parent.getApplicationFrame().getSize();
		
			int x = Math.max((d2.width - d1.width)/2, 0);
			int y = Math.max((d2.height - d1.height)/2, 0);
		
			setBounds(x + parent.getApplicationFrame().getX(), y + parent.getApplicationFrame().getY(), d1.width, d1.height);
		}
		
		setResizable(false);
		
		setVisible(true);
		


	}
	
	private void process(){

		//process the entered data
		try{
			String prmtop = inputField1.getText().trim();
			String inpcrd = inputField2.getText().trim();
			String outdir = dirField.getText().trim();
			
			int minStepsCount = Integer.parseInt(minSteps.getText().trim());
			int eqStepsCount = Integer.parseInt(eqSteps.getText().trim());
			int nodes = Integer.parseInt(nodeCount.getText().trim());
			int totalRunsCount = Integer.parseInt(totalRuns.getText().trim());
			int steps = Integer.parseInt(mdSteps.getText().trim());
			int frequency = Integer.parseInt(saveFrequency.getText().trim());
			
			
			if (totalRunsCount < 2){
				totalRunsCount = 2;
				parent.displayMessage("Total MD run count is set to a minimum of two");
			}
			
			boolean explicitWater = waterButton.isSelected();
			
			//now prepare the input files in the specified directory
			//first, copy over the prmtop and inpcrd files
			if (!prmtop.endsWith("prmtop")){
				parent.displayErrorMessage("Selected prmtop file has an incorrect extension");
				return;
			}
			if (!inpcrd.endsWith("inpcrd")){
				parent.displayErrorMessage("Selected inpcrd file has an incorrect extension");
				return;
			}
			
			File prmtopFile = new File(prmtop);
			File inpcrdFile = new File(inpcrd);
			
			File prmtopOutFile = new File(outdir + File.separator + prmtopFile.getName());
			File inpcrdOutFile = new File(outdir + File.separator + inpcrdFile.getName());
			
			FileTransfer transfer = new FileTransfer();
			//check whether source and destination directories are different
			
			if (!prmtopFile.toString().equals(prmtopOutFile.toString())){
				boolean a = transfer.copy(prmtopFile, prmtopOutFile);
				if (!a){
					parent.displayErrorMessage("Unable to copy prmtop file to the selected directory");
					return;
				}
			}
			if (!inpcrdFile.toString().equals(inpcrdOutFile.toString())){
				boolean b = transfer.copy(inpcrdFile, inpcrdOutFile);
				if (!b){
					parent.displayErrorMessage("Unable to copy prmtop file to the selected directory");
					return;
				}
			}
			
			if (explicitWater){
				//in this case, we need to know the number of non-solvent residues
				//get it from the prmtop file
				ParameterSet parameters = MDUtils.readParameterFile(new File(prmtop));
				
				if (parameters.solventStartResidue < 0){
					parent.displayErrorMessage("No solvent found in the input. Check the input files and try again.");
					return;
				}
				
				int residues = parameters.solventStartResidue - 1;
				
				//two minimization files are needed
				//as well as reference lines in the input files
				PrintWriter printer = new PrintWriter(new FileWriter(outdir + File.separator + "wat_min1.in"));
				printer.println("initial minimisation solvent");
				printer.println(" &cntrl");
				printer.println("	imin   = 1,");
				printer.println("	maxcyc = " + minStepsCount + ",");
				printer.println("	ncyc   = " + (int)(minStepsCount/2) + ",");
				printer.println("	ntb    = 1,");
				printer.println("	ntr    = 1,");
				printer.println("	cut    = 15");
				printer.println(" /");
				printer.println("Hold the protein fixed");
				printer.println("500.0");
				printer.println("RES 1 " + residues);
				printer.println("END");
				printer.println("END");
				printer.flush();
				printer.close();

				//second min file
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "wat_min2.in"));
				printer.println("initial minimisation whole system");
				printer.println(" &cntrl");
				printer.println("	imin   = 1,");
				printer.println("	maxcyc = " + minStepsCount + ",");
				printer.println("	ncyc   = " + (int)(minStepsCount/2) + ",");
				printer.println("	ntb    = 1,");
				printer.println("	ntr    = 0,");
				printer.println("	cut    = 15");
				printer.println(" /");
				printer.flush();
				printer.close();
				 
				//heating file
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "wat_heat.in"));
				printer.println("protein: heating");
				printer.println(" &cntrl");
				printer.println("	imin   = 0,");
				printer.println("	irest  = 0,");
				printer.println("	ntx    = 1,");
				printer.println("	ntb    = 1,");
				printer.println("	cut    = 10,");
				printer.println("	ntr    = 1,");
				printer.println("	ntc    = 2,");
				printer.println("	ntf    = 2,");
				printer.println("	tempi  = 0.0,");
				printer.println("	temp0  = 300.0,");
				printer.println("	ntt    = 3,");
				printer.println("	gamma_ln = 2.0,");
				printer.println("	nstlim = 25000, dt = 0.002,");
				printer.println("	ntpr = 250, ntwx = 250, ntwr = 1000");
				printer.println(" /");
				printer.println("Keep protein fixed with weak restraints");
				printer.println("10.0");
				printer.println("RES 1 " + residues);
				printer.println("END");
				printer.println("END");
				printer.flush();
				printer.close();
				
				//density equilibration file
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "wat_density.in"));
				printer.println("protein: density");
				printer.println(" &cntrl");
				printer.println("	imin   = 0,");
				printer.println("	irest  = 1,");
				printer.println("	ntx    = 5,");
				printer.println("	ntb    = 2,");
				printer.println("	cut    = 10,");
				printer.println("	ntr    = 1,");
				printer.println("	ntp    = 1,");
				printer.println("	ntc    = 2,");
				printer.println("	ntf    = 2,");
				printer.println("	temp0  = 300.0,");
				printer.println("	ntt    = 3,");
				printer.println("	gamma_ln = 2.0,");
				printer.println("	nstlim = 25000, dt = 0.002,");
				printer.println("	ntpr = 250, ntwx = 250, ntwr = 1000");
				printer.println(" /");
				printer.println("Keep protein fixed with weak restraints");
				printer.println("5.0");
				printer.println("RES 1 " + residues);
				printer.println("END");
				printer.println("END");
				printer.flush();
				printer.close();
				
				//full equilibration file
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "wat_eq.in"));
				printer.println("protein: equilibration");
				printer.println(" &cntrl");
				printer.println("	imin   = 0,");
				printer.println("	irest  = 1,");
				printer.println("	ntx    = 5,");
				printer.println("	ntb    = 2,");
				printer.println("	cut    = 10,");
				printer.println("	ntp    = 1,");
				printer.println("	ntc    = 2,");
				printer.println("	ntf    = 2,");
				printer.println("	taup    = 2.0,");
				printer.println("	temp0  = 300.0,");
				printer.println("	ntt    = 3,");
				printer.println("	gamma_ln = 2.0,");
				printer.println("	nstlim = 250000, dt = 0.002,");
				printer.println("	ntpr = 500, ntwx = 500");
				printer.println(" /");
				printer.flush();
				printer.close();
				
				//md input file
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "wat_md.in"));
				printer.println("Production simulation");
				printer.println(" &cntrl");
				printer.println("  imin = 0, irest = 1, ntx = 7,");
				printer.println("  ntb = 2, pres0 = 1.0, ntp = 1,");
				printer.println("  taup = 2.0,");
				printer.println("  cut = 10, ntr = 0,");
				printer.println("  ntc = 2, ntf = 2,");
				printer.println("  temp0 = 300.0,");
				printer.println("  ntt = 3, gamma_ln = 2.0,");
				printer.println("  nstlim = " + steps + ", dt = 0.002,");
				printer.println("  ntpr = " + frequency + ", ntwx = " + frequency + ", ntwr = 1000");
				printer.println(" /");
				printer.flush();
				printer.close();
				
				//get the input file base name
				String n = (new File(prmtop)).getName();
				String basename = n.substring(0, n.indexOf("."));
				
				if (!ibCluster.isSelected()){
					PrintWriter p = new PrintWriter(new FileWriter(outdir + File.separator + "mpd_check.sh"));
					p.println("#!/bin/bash");
					p.println("a=$(/data/mpich2/bin/mpdtrace | grep MPICH2)");
					p.println("length=${#a}");
					p.println("if [ $length != 0 ];");
					p.println("then");
					p.println("/data/mpich2/bin/mpd&");
					p.println("fi\n");
					p.println("sleep 10");
					p.flush();
					p.close();
				}
				

				
				//write out the job script file
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "run.sh"));
				printer.println("#!/bin/csh");
				printer.println("set MDSTARTJOB=2");
				printer.println("set MDENDJOB=" + totalRunsCount);
				printer.println("set MDCURRENTJOB=$MDSTARTJOB");
				printer.println("set MDINPUT=0");
				
				if (!ibCluster.isSelected()){
					printer.println("chmod +x ./mpd_check.sh");
					printer.println("./mpd_check.sh\n");
				}
				
				printer.println("echo -n \"Starting Script at: \"");
				printer.println("date");
				printer.println("echo \"\"");
				printer.println("echo \"Running minimization...\"");
				int nThreads = nodes*4;
				
				
				
				if (nodes > 1 || ibCluster.isSelected()){
					printer.println("/opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i wat_min1.in -o " + basename + "_min1.out \\");
				}
				else{
					printer.println("/data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i wat_min1.in -o " + basename + "_min1.out \\");
				}
				printer.println("	-p " + basename + ".prmtop -c " + basename + ".inpcrd \\");
				printer.println("	-ref " + basename + ".inpcrd -r " + basename + "_min1.rst\n"); 

				if (nodes > 1 || ibCluster.isSelected()){
					printer.println("/opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i wat_min2.in -o " + basename + "_min2.out \\");
				}
				else{
					printer.println("/data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i wat_min2.in -o " + basename + "_min2.out \\");
				}
				printer.println("	-p " + basename + ".prmtop -c " + basename + "_min1.rst \\");
				printer.println("	-r " + basename + "_min2.rst\n"); 

				
				//heating
				printer.println("echo \"Running heating...\"");
				if (nodes > 1 || ibCluster.isSelected()){
					printer.println("/opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i wat_heat.in -o " + basename + "_heat.out \\");
				}
				else{
					printer.println("/data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i wat_heat.in -o " + basename + "_heat.out \\");
				}
				printer.println("	-p " + basename + ".prmtop -c " + basename + "_min2.rst -ref " + basename + "_min2.rst  \\");
				printer.println("	-r " + basename + "_heat.rst -x " + basename + "_heat.mdcrd\n");

				
				//density equilibration
				printer.println("echo \"Running density equilibration...\"");
				if (nodes > 1 || ibCluster.isSelected()){
					printer.println("/opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i wat_density.in -o " + basename + "_density.out \\");
				}
				else{
					printer.println("/data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i wat_density.in -o " + basename + "_density.out \\");
				}
				printer.println("	-p " + basename + ".prmtop -c " + basename + "_heat.rst -ref " + basename + "_heat.rst  \\");
				printer.println("	-r " + basename + "_density.rst -x " + basename + "_density.mdcrd\n");
				
				//equilibration
				printer.println("echo \"Running equilibration...\"");
				if (nodes > 1 || ibCluster.isSelected()){
					printer.println("/opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i wat_eq.in -o " + basename + "_eq.out \\");
				}
				else{
					printer.println("/data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i wat_eq.in -o " + basename + "_eq.out \\");
				}
				printer.println("	-p " + basename + ".prmtop -c " + basename + "_density.rst -ref " + basename + "_density.rst  \\");
				printer.println("	-r " + basename + "_eq.rst -x " + basename + "_eq.mdcrd\n");
				
				
				printer.println("   echo -n \"Job 1 started at: \"");
				printer.println("   date");
				
				if (nodes > 1 || ibCluster.isSelected()){
					printer.println("/opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i wat_md.in -o " + basename + "_run1.out \\");
				}
				else{
					printer.println("/data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i wat_md.in -o " + basename + "_run1.out \\");
				}
				printer.println("	-p " + basename + ".prmtop -c " + basename + "_eq.rst \\");
				printer.println("	-r " + basename + "_run1.rst -x " + basename + "_run1.mdcrd\n");

				printer.println("	gzip " + basename + "_run1.mdcrd\n");
				printer.println("   echo -n \"Job 1 finished at: \"");
				printer.println("   date");

				
				printer.println("while ( $MDCURRENTJOB <= $MDENDJOB )");
				printer.println("   echo -n \"Job $MDCURRENTJOB started at: \"");
				printer.println("   date");
				printer.println("   @ MDINPUT = $MDCURRENTJOB - 1");
				
				if (ibCluster.isSelected()){
					printer.println("   /opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i wat_md.in \\");
				}
				else{
					printer.println("   /data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i wat_md.in \\");
				}
				printer.println("                            -o " + basename + "_run$MDCURRENTJOB.out \\");
				printer.println("                            -p " + basename + ".prmtop \\");
				printer.println("                            -c " + basename + "_run$MDINPUT.rst \\");
				printer.println("                            -r " + basename + "_run$MDCURRENTJOB.rst \\");
				printer.println("                            -x " + basename + "_run$MDCURRENTJOB.mdcrd");
				printer.println("	gzip " + basename + "_run$MDCURRENTJOB.mdcrd");
				printer.println("   echo -n \"Job $MDCURRENTJOB finished at: \"");
				printer.println("   date");
				printer.println("	@ MDCURRENTJOB = $MDCURRENTJOB + 1");
				printer.println("end");
				printer.println("echo \"ALL DONE\"");
				printer.flush();
				printer.close();
				

			}
			else{
				//now, let's create the minimization input file
				PrintWriter printer = new PrintWriter(new FileWriter(outdir + File.separator + "min.in"));
				printer.println("Initial minimization");
				printer.println(" &cntrl");
				printer.println("	imin=1, maxcyc=" + minStepsCount + ", ncyc=" + (int)(minStepsCount/2) + ",");
				printer.println("	cut=15, ntb=0, igb=1");
				printer.println(" /");
				printer.flush();
				printer.close();
	
				
				//next, the equilibration input file
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "eq.in"));
				printer.println("Initial MD equilibration");
				printer.println(" &cntrl");
				printer.println("	imin=0, irest=0,");
				printer.println("	nstlim=" + eqStepsCount + ", dt=0.001, ntc=1,");
				printer.println("	ntpr=100, ntwx=100,");
				printer.println("	cut=15, ntb=0, igb=1,");
				printer.println("	ntt=3, gamma_ln=1.0,");
				printer.println("	tempi=0.0, temp0=300.0,");
				printer.println(" /");
				printer.flush();
				printer.close();
				
				//md simulation input file
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "md.in"));
				printer.println("Production MD");
				printer.println(" &cntrl");
				printer.println("	imin=0, irest=1, ntx=5,");
				printer.println("	nstlim=" + steps + ", dt=0.001, ntc=1,");
				printer.println("	ntpr=" + frequency + ", ntwx=" + frequency + ",");
				printer.println("	cut=15, ntb=0, igb=1,");
				printer.println("	ntt=3, gamma_ln=1.0,");
				printer.println("	tempi=300.0, temp0=300.0,");
				printer.println(" /");
				printer.flush();
				printer.close();
				
				//get the input file base name
				String n = (new File(prmtop)).getName();
				String basename = n.substring(0, n.indexOf("."));
				
				int nThreads = nodes*4;

				if (!ibCluster.isSelected()){
					PrintWriter p = new PrintWriter(new FileWriter(outdir + File.separator + "mpd_check.sh"));
					p.println("#!/bin/bash");
					p.println("a=$(/data/mpich2/bin/mpdtrace | grep MPICH2)");
					p.println("length=${#a}");
					p.println("if [ $length != 0 ];");
					p.println("then");
					p.println("/data/mpich2/bin/mpd&");
					p.println("fi\n");
					p.println("sleep 10");
					p.flush();
					p.close();
				}
				

				//now write out the automation script
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "run.sh"));
				printer.println("#!/bin/csh");
				printer.println("set MDSTARTJOB=2");
				printer.println("set MDENDJOB=" + totalRunsCount);
				printer.println("set MDCURRENTJOB=$MDSTARTJOB");
				printer.println("set MDINPUT=0");

				if (!ibCluster.isSelected()){
					printer.println("chmod +x ./mpd_check.sh");
					printer.println("./mpd_check.sh\n");
				}
				
				printer.println("echo -n \"Starting Script at: \"");
				printer.println("date");
				printer.println("echo \"\"");

				if (ibCluster.isSelected()){
					printer.println("/opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i min.in -o " + basename + "_min.out \\");
				}
				else{
					printer.println("/data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i min.in -o " + basename + "_min.out \\");
				}
				printer.println("	-p " + basename + ".prmtop -c " + basename + ".inpcrd \\");
				printer.println("	-r " + basename + "_min.rst\n"); 

				if (ibCluster.isSelected()){
					printer.println("/opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i eq.in -o " + basename + "_eq.out \\");
				}
				else{
					printer.println("/data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i eq.in -o " + basename + "_eq.out \\");
				}
				printer.println("	-p " + basename + ".prmtop -c " + basename + "_min.rst \\");
				printer.println("	-r " + basename + "_eq.rst -x " + basename + "_eq.mdcrd\n");

				if (ibCluster.isSelected()){
					printer.println("/opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i md.in -o " + basename + "_run1.out \\");
				}
				else{
					printer.println("/data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i md.in -o " + basename + "_run1.out \\");
				}
				printer.println("	-p " + basename + ".prmtop -c " + basename + "_eq.rst \\");
				printer.println("	-r " + basename + "_run1.rst -x " + basename + "_run1.mdcrd\n");
				printer.println("	gzip " + basename + "_run1.mdcrd");

				printer.println("while ( $MDCURRENTJOB <= $MDENDJOB )");
				printer.println("   echo -n \"Job $MDCURRENTJOB started at: \"");
				printer.println("   date");
				printer.println("   @ MDINPUT = $MDCURRENTJOB - 1");
				if (ibCluster.isSelected()){
					printer.println("   /opt/openmpi/bin/mpiexec -n " + nThreads + " -machinefile /home/sasha/hosts_" + nodes + " " +
							"/data/amber9_ib/exe/pmemd -O -i md.in \\");
				}
				else{
					printer.println("   /data/mpich2/bin/mpirun -n " + nThreads + " /data/amber9/exe/pmemd -O -i md.in \\");
				}
				printer.println("                            -o " + basename + "_run$MDCURRENTJOB.out \\");
				printer.println("                            -p " + basename + ".prmtop \\");
				printer.println("                            -c " + basename + "_run$MDINPUT.rst \\");
				printer.println("                            -r " + basename + "_run$MDCURRENTJOB.rst \\");
				printer.println("                            -x " + basename + "_run$MDCURRENTJOB.mdcrd");
				printer.println("	gzip " + basename + "_run$MDCURRENTJOB.mdcrd");
				printer.println("   echo -n \"Job $MDCURRENTJOB finished at: \"");
				printer.println("   date");
				printer.println("	@ MDCURRENTJOB = $MDCURRENTJOB + 1");
				printer.println("end");
				printer.println("echo \"ALL DONE\"");
				printer.flush();
				printer.close();

			}
			
			//create SGE script if requested
			if (destination.isSelected()){
				File out = new File(outdir);
				PrintWriter printer = new PrintWriter(new FileWriter(outdir + File.separator + "sge.sh"));
				printer.println("cd " + out.getParent());
				printer.println("tar cvf " + out.getName() + ".tar " + out.getName());
				printer.println("cd /amber");
				printer.println("cp " + out.getParent() + File.separator + out.getName() + ".tar " + out.getName() + ".tar");
				printer.println("tar xvf " + out.getName() + ".tar");
				printer.println("rm " + out.getName() + ".tar");
				printer.println("cd " + out.getName());

				printer.println("chmod +x run.sh");
				printer.println("./run.sh");

				printer.println("tar cvf " + out.getName() + ".tar *");
				printer.println("cp " + out.getName() + ".tar " + destField.getText().trim() + File.separator + out.getName() + ".tar");
				printer.println("cd " + destField.getText().trim());
				printer.println("tar xvf " + destField.getText().trim() + File.separator + out.getName() + ".tar");
				printer.println("rm " + destField.getText().trim() + File.separator + out.getName() + ".tar");
				printer.println("cd /amber");
				printer.println("rm -rf " + out.getName());
				printer.flush();
				printer.close();
				
			}
			

			
			setVisible(false);
			parent.displayMessage("Molecular Dynamics setup completed");
		}
		catch (Exception ex){
			parent.displayExceptionMessage("Exception preparing MD simulation", ex);
		}
		
		
		
		parent.setDisplayDialogStatus(false);
		
		dispose();
	}
	
	private Border createTitledBorder(String title){
		
		Border border1 = BorderFactory.createEmptyBorder(0,0,0,0);//left offset of the boxes in the box
		TitledBorder border2 = BorderFactory.createTitledBorder(new EtchedBorder(), " " + title + " ");
		border2.setTitleColor(Color.black);
		border2.setTitleFont(new Font("Dialog", 0, 12));
		Border border = BorderFactory.createCompoundBorder(border1, border2);

		return border;
	}

	
	public void processPick(String data, StructureComponent structureComponent){}
	
}
