package edu.sdsc.sirius.dialogs;

// Core
import java.io.*;

// GUI
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.border.*;

// MBT
import edu.sdsc.mbt.*;
import edu.sdsc.sirius.util.*;

import edu.sdsc.sirius.io.*;

public class NAMDSetupDialog extends JDialog implements DisplayDialog {

	private JFrame parentFrame;
	private Manager parent;
	
	private DisplayDialog itself;

	private Container contentPane;
	private JPanel base = new JPanel();
	
	private JPanel inputPanel = new JPanel();
	private JPanel waterPanel = new JPanel();
	private JPanel optionsPanel = new JPanel();
	private JPanel outputPanel = new JPanel();
	
	private Border border1;
	private TitledBorder border2;
	private Border border;
	
	private JTextField inputField1 = new JTextField(50);
	private JTextField inputField2 = new JTextField(50);
	private JTextField inputField3 = new JTextField(50);
	
	private JTextField dirField = new JTextField(50);
	
	private JTextField minSteps = new JTextField(30);
	private JTextField eqSteps = new JTextField(30);
	private JTextField timestep = new JTextField(30);
	
	private JTextField totalRuns = new JTextField(30);
	private JTextField mdSteps = new JTextField(40);
	private JTextField saveFrequency = new JTextField(30);
	
	private JTextField baseNameField = new JTextField(30);
	private JCheckBox aceBox = new JCheckBox("ACEMD");
	
	private JButton okButton = new JButton("OK");
	private JButton cancelButton = new JButton("Cancel");
	
	
	public NAMDSetupDialog(JFrame f, Manager p){

		super(f, "Prepare NAMD simulation", false);

		parentFrame = f;
		parent = p;
		
		parent.setDisplayDialogStatus(true);
		
		itself = this;
		
		//create the panels
		contentPane = getContentPane();
		contentPane.setLayout(null);
		
		
		base.setBorder(new BevelBorder(BevelBorder.RAISED));
		base.setBounds(5, 5, 495, 320);
		base.setLayout(null);
		
		//first panel - input files
		inputPanel.setBounds(5,5,485,135);
		inputPanel.setLayout(null);
		
		inputPanel.setBorder(createTitledBorder("Input files"));
		
		JLabel input1 = new JLabel("psf:");
		JLabel input2 = new JLabel("pdb:");
		JLabel input3 = new JLabel("params:");
		
		input1.setBounds(20, 30, 60, 20);
		input2.setBounds(20, 60, 60, 20);
		input3.setBounds(20, 90, 60, 20);
		
		inputField1.setBounds(90,30,275,20);
				
		JButton inputButton1 = new JButton("Browse...");
		inputButton1.setFont(new Font("Dialog", Font.PLAIN, 12));
		inputButton1.setBounds(370,30,90,20);
		inputButton1.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
				chooser.setApproveButtonText("Select");
				int result = chooser.showOpenDialog(parent.getApplicationFrame());
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				inputField1.setText(filename.toString());
				parent.setLastUsedDirectory(filename.toString());
			}
		});
		
		
		inputPanel.add(input1);
		inputPanel.add(inputField1);
		inputPanel.add(inputButton1);
		
		inputField2.setBounds(90,60,275,20);
		
		JButton inputButton2 = new JButton("Browse...");
		inputButton2.setFont(new Font("Dialog", Font.PLAIN, 12));
		inputButton2.setBounds(370,60,90,20);
		inputButton2.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
				chooser.setApproveButtonText("Select");
				int result = chooser.showOpenDialog(parent.getApplicationFrame());
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				inputField2.setText(filename.toString());
				parent.setLastUsedDirectory(filename.toString());
			}
		});

		inputPanel.add(input2);
		inputPanel.add(inputField2);
		inputPanel.add(inputButton2);
		
		
		inputField3.setBounds(90,90,275,20);
		JButton inputButton3 = new JButton("Browse...");
		inputButton3.setFont(new Font("Dialog", Font.PLAIN, 12));
		inputButton3.setBounds(370,90,90,20);
		inputButton3.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
				chooser.setApproveButtonText("Select");
				int result = chooser.showOpenDialog(parent.getApplicationFrame());
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				inputField3.setText(filename.toString());
				parent.setLastUsedDirectory(filename.toString());
			}
		});
		
		inputPanel.add(input3);
		inputPanel.add(inputField3);
		inputPanel.add(inputButton3);

		
		base.add(inputPanel);
		
		
		//parameters
		optionsPanel.setBounds(5,140,485,125);
		optionsPanel.setLayout(null);
		optionsPanel.setBorder(createTitledBorder("Simulation options"));
		
		JLabel minStepsLabel = new JLabel("Minimization steps:");
		minStepsLabel.setBounds(20,20,120,20);
		minSteps.setBounds(150, 20, 50, 20);
		minSteps.setText("10000");
		
		JLabel eqStepsLabel = new JLabel("Equilibration steps:");
		eqStepsLabel.setBounds(20, 45, 120, 20);
		eqSteps.setBounds(150, 45, 50, 20);
		eqSteps.setText("250000");
		
		JLabel timestepLabel = new JLabel("Timestep, fs:");
		timestepLabel.setBounds(20, 70, 120, 20);
		timestep.setBounds(150, 70, 50, 20);
		timestep.setText("2");
		
		JLabel totalRunsLabel = new JLabel("Total MD runs:");
		totalRunsLabel.setBounds(20, 95, 120, 20);
		totalRuns.setBounds(150, 95, 50, 20);
		totalRuns.setText("10");
		
		JLabel mdStepsLabel = new JLabel("Steps in each run:");
		mdStepsLabel.setBounds(240, 20, 120, 20);
		mdSteps.setBounds(370, 20, 50, 20);
		mdSteps.setText("500000");
		
		JLabel saveFrequencyLabel = new JLabel("Save frequency:");
		saveFrequencyLabel.setBounds(240, 45, 120, 20);
		saveFrequency.setBounds(370, 45, 50, 20);
		saveFrequency.setText("5000");
		
		JLabel basenameLabel = new JLabel("Basename:");
		basenameLabel.setBounds(240, 70, 120, 20);
		baseNameField.setBounds(370, 70, 90, 20);
		baseNameField.setText("");
		
		JLabel aceLabel = new JLabel("Production format:");
		aceLabel.setBounds(240, 95, 120, 20);
		aceBox.setBounds(370, 95, 60, 20);
		
		optionsPanel.add(minStepsLabel);
		optionsPanel.add(minSteps);
		optionsPanel.add(eqStepsLabel);
		optionsPanel.add(eqSteps);
		optionsPanel.add(timestepLabel);
		optionsPanel.add(timestep);
		optionsPanel.add(totalRunsLabel);
		optionsPanel.add(totalRuns);
		optionsPanel.add(mdStepsLabel);
		optionsPanel.add(mdSteps);
		optionsPanel.add(saveFrequencyLabel);
		optionsPanel.add(saveFrequency);
		optionsPanel.add(basenameLabel);
		optionsPanel.add(baseNameField);
		optionsPanel.add(aceBox);
		optionsPanel.add(aceLabel);
		
		base.add(optionsPanel);
		
		//output directory
		outputPanel.setBounds(5,265,485,50);
		outputPanel.setLayout(null);
		outputPanel.setBorder(createTitledBorder("Output directory"));

		dirField.setBounds(25,20,335,20);
		
		JButton dirButton = new JButton("Browse...");
		dirButton.setFont(new Font("Dialog", Font.PLAIN, 12));
		dirButton.setBounds(370,20,90,20);
		dirButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
				chooser.setApproveButtonText("Select");
				chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
				int result = chooser.showOpenDialog(parent.getApplicationFrame());
				if (result == JFileChooser.CANCEL_OPTION){
					return;
				}
				File filename = chooser.getSelectedFile();
				dirField.setText(filename.toString());
			}
		});
		
		
		outputPanel.add(dirField);
		outputPanel.add(dirButton);		
		
		
		base.add(outputPanel);
		
		contentPane.add(base);
		
		//buttons
		okButton.setBounds(150, 330, 80, 25);
		okButton.setVisible(true);
		okButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				process();
			}
		});
		
		cancelButton.setBounds(270, 330, 80, 25);
		cancelButton.setVisible(true);
		cancelButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
		});
		
		contentPane.add(base);
		contentPane.add(okButton);
		contentPane.add(cancelButton);
		
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				setVisible(false);
				parent.setDisplayDialogStatus(false);
				dispose();
			}
			
		});
				
		setSize(510, 385);
		
		if (parent == null){
			setLocationRelativeTo(null);
		}
		else{
			//set location
			Dimension d1 = getSize();
			Dimension d2 = parent.getApplicationFrame().getSize();
		
			int x = Math.max((d2.width - d1.width)/2, 0);
			int y = Math.max((d2.height - d1.height)/2, 0);
		
			setBounds(x + parent.getApplicationFrame().getX(), y + parent.getApplicationFrame().getY(), d1.width, d1.height);
		}
		
		setResizable(false);
		
		setVisible(true);
		


	}
	
	private void process(){

		//process the entered data
		try{
			String psfTemp = inputField1.getText().trim();
			String pdbTemp = inputField2.getText().trim();
			String paramsTemp = inputField3.getText().trim();
			String outdir = dirField.getText().trim();
			String basename = baseNameField.getText().trim();
			
			String psf = psfTemp.substring(psfTemp.lastIndexOf(File.separator)+1);
			String pdb = pdbTemp.substring(pdbTemp.lastIndexOf(File.separator)+1);
			String params = paramsTemp.substring(paramsTemp.lastIndexOf(File.separator)+1);
			
			int minStepsCount = Integer.parseInt(minSteps.getText().trim());
			int eqStepsCount = Integer.parseInt(eqSteps.getText().trim());
			int step = Integer.parseInt(timestep.getText().trim());
			int totalRunCount = Integer.parseInt(totalRuns.getText().trim());
			int steps = Integer.parseInt(mdSteps.getText().trim());
			int frequency = Integer.parseInt(saveFrequency.getText().trim());
			boolean acemd = aceBox.isSelected();
			
			if (totalRunCount < 2){
				totalRunCount = 2;
				parent.displayMessage("Total MD run count is set to a minimum of two");
			}
			
			//now prepare the input files in the specified directory
			//first, copy over the prmtop and inpcrd files
			if (!psf.endsWith("psf")){
				parent.displayErrorMessage("Selected PSF file has an incorrect extension");
				return;
			}
			if (!pdb.endsWith("pdb")){
				parent.displayErrorMessage("Selected PDB file has an incorrect extension");
				return;
			}
			
			File psfFile = new File(psfTemp);
			File pdbFile = new File(pdbTemp);
			
			File psfOutFile = new File(outdir + File.separator + psfFile.getName());
			File pdbOutFile = new File(outdir + File.separator + pdbFile.getName());
			
			FileTransfer transfer = new FileTransfer();
			//check whether source and destination directories are different
			
			if (!psfFile.toString().equals(psfOutFile.toString())){
				boolean a = transfer.copy(psfFile, psfOutFile);
				if (!a){
					parent.displayErrorMessage("Unable to copy psf file to the selected directory");
					return;
				}
			}
			if (!pdbFile.toString().equals(pdbOutFile.toString())){
				boolean b = transfer.copy(pdbFile, pdbOutFile);
				if (!b){
					parent.displayErrorMessage("Unable to copy pdb file to the selected directory");
					return;
				}
			}
			
			
			//two minimization files are needed
			//as well as reference file
			PrintWriter printer = new PrintWriter(new FileWriter(outdir + File.separator + "min1.namd"));
			PrintWriter reference = new PrintWriter(new FileWriter(outdir + File.separator + basename + "_reference.pdb"));
			
			printer.println("# molecular system");
			printer.println("structure\t" + psf);
			printer.println("coordinates\t" + pdb + "\n");
			printer.println("temperature        0       ;# initial temperature");

			printer.println("set outputname\t" + basename + "_min1");
			printer.println("set em_restart\t" + basename + "_min1_restart");

			printer.println("\n# force field");
			printer.println("paratypecharmm on");
			printer.println("parameters " + params);
			printer.println("exclude scaled1-4");
			printer.println("1-4scaling	1.0\n");
			
			//get the dimensions of the solvated box
			//go through all atoms of the pdb file and get the coordinates of all corners of the water
			//box. then calculate the dimensions and location of the center
			
			//NOTE: at the same time, record the reference pdb file with froze protein and flexible water and ions
			BufferedReader reader = new BufferedReader(new FileReader(pdbTemp));
			String line = null;
			double minX = 10000;
			double minY = 10000;
			double minZ = 10000;
			double maxX = -10000;
			double maxY = -10000;
			double maxZ = -10000;
			while ((line = reader.readLine()) != null){
				if (line.indexOf("TIP3") > 0){
					//it's water: get coordinates of the atom
					double x = Double.parseDouble(line.substring(30, 38));
					double y = Double.parseDouble(line.substring(38, 46));
					double z = Double.parseDouble(line.substring(46, 54));
					
					if (x < minX) minX = x;
					if (y < minY) minY = y;
					if (z < minZ) minZ = z;
					if (x > maxX) maxX = x;
					if (y > maxY) maxY = y;
					if (z > maxZ) maxZ = z;
					
					//now update the line to contain 0.0 in the occupancy field
					String temp1 = line.substring(0, 56);
					String temp2 = line.substring(60);
					String result = temp1 + "0.00" + temp2;
					reference.println(result);
				}
				else if (line.indexOf("ION") > 0){
					//update the line to contain 0.0 in the occupancy field
					String temp1 = line.substring(0, 56);
					String temp2 = line.substring(60);
					String result = temp1 + "0.00" + temp2;
					reference.println(result);
				}
				else{
					//print as is
					reference.println(line);
				}
			}
			reader.close();
			reference.close();
			
			double basisVector1 = maxX - minX;
			double basisVector2 = maxY - minY;
			double basisVector3 = maxZ - minZ;
			
			double centerX = (maxX + minX)/2;
			double centerY = (maxY + minY)/2;
			double centerZ = (maxZ + minZ)/2;
			
/*			System.out.println("maxX = " + maxX);
			System.out.println("minX = " + minX);
			System.out.println("maxY = " + maxY);
			System.out.println("minY = " + minY);
			System.out.println("maxZ = " + maxZ);
			System.out.println("minZ = " + minZ);
			
*/			
			String bv1 = StringUtils.getFormattedNumber(basisVector1, 1);
			String bv2 = StringUtils.getFormattedNumber(basisVector2, 1);
			String bv3 = StringUtils.getFormattedNumber(basisVector3, 1);
			
//			System.out.println("bv1 = " + bv1 + ", bv2 = " + bv2 + ", bv3 = " + bv3);
			
			String cx = StringUtils.getFormattedNumber(centerX, 1);
			String cy = StringUtils.getFormattedNumber(centerY, 1);
			String cz = StringUtils.getFormattedNumber(centerZ, 1);
			
				
			printer.println("cellBasisVector1\t" + bv1 + " 0.0 0.0");
			printer.println("cellBasisVector2\t0.0 " + bv2 + " 0.0");
			printer.println("cellBasisVector3\t0.0 0.0 " + bv3);
			printer.println("cellOrigin " + cx + " " + cy + " " + cz);
			printer.println("wrapAll on\n");

			printer.println("PME on");
			printer.println("PMETolerance 0.000001");
			printer.println("PMEGridSpacing 1.0\n");

			printer.println("# approximations");
			printer.println("switching on");
			printer.println("switchdist 8");
			printer.println("cutoff 11");
			printer.println("pairlistdist 12.5\n");

			printer.println("reassignFreq	1000");
			printer.println("reassignTemp	25");
			printer.println("reassignIncr	25");
			printer.println("reassignHold	300\n");

			printer.println("# integrator");
			printer.println("timestep 1.0 # in fs.");//timestep is 1 fs in pre-production runs
			printer.println("stepspercycle 20");
			printer.println("nonbondedFreq 2");
			printer.println("rigidBonds water");
			printer.println("rigidIterations 500\n");

			printer.println("# Output");
			printer.println("outputName          $outputname\n");

			printer.println("dcdfreq             100     ;# how often we output trajectories");
			printer.println("xstFreq             100");
			printer.println("outputEnergies      100");
			printer.println("outputPressure      100");
			printer.println("outputtiming        100");
			printer.println("binaryoutput        yes\n");

			printer.println("# for restarting:");
			printer.println("restartname         $em_restart");
			printer.println("restartfreq         " + minStepsCount);
			printer.println("restartsave         yes");
			printer.println("binaryrestart       yes\n"); 

			printer.println("fixedAtoms on");
			printer.println("fixedAtomsFile\t" + basename + "_reference.pdb");

			printer.println("# First minimize with fixedAtoms on");
			printer.println("minimize " + minStepsCount);

			printer.flush();
			printer.close();

			
			
			//second min file
			printer = new PrintWriter(new FileWriter(outdir + File.separator + "min2.namd"));
			printer.println("# molecular system"); 
			printer.println("structure\t" + psf);
			printer.println("coordinates\t" + pdb);
			printer.println("bincoordinates\t " + basename + "_min1_restart." + minStepsCount + ".coor\n");

			printer.println("temperature        0       ;# initial temperature");
			printer.println("set outputname\t" + basename + "_min2");  
			printer.println("set em_restart\t" + basename + "_min2_restart");

			printer.println("\n# force field");
			printer.println("paratypecharmm on");
			printer.println("parameters " + params);
			printer.println("exclude scaled1-4");
			printer.println("1-4scaling	1.0\n");
			
			printer.println("cellBasisVector1\t" + bv1 + " 0.0 0.0");
			printer.println("cellBasisVector2\t0.0 " + bv2 + " 0.0");
			printer.println("cellBasisVector3\t0.0 0.0 " + bv3);
			printer.println("cellOrigin " + cx + " " + cy + " " + cz);
			printer.println("wrapAll on\n");

			printer.println("PME on");
			printer.println("PMETolerance 0.000001");
			printer.println("PMEGridSpacing 1.0\n");

			printer.println("# approximations");
			printer.println("switching on");
			printer.println("switchdist 8");
			printer.println("cutoff 11");
			printer.println("pairlistdist 12.5\n");

			printer.println("reassignFreq	1000");
			printer.println("reassignTemp	25");
			printer.println("reassignIncr	25");
			printer.println("reassignHold	300\n");

			printer.println("# integrator");
			printer.println("timestep 1.0 # in fs.");
			printer.println("stepspercycle 20");
			printer.println("nonbondedFreq 2");
			printer.println("rigidBonds water");
			printer.println("rigidIterations 500\n");

			printer.println("# Output");
			printer.println("outputName          $outputname\n");

			printer.println("dcdfreq             100     ;# how often we output trajectories");
			printer.println("xstFreq             100");
			printer.println("outputEnergies      100");
			printer.println("outputPressure      100");
			printer.println("outputtiming        100");
			printer.println("binaryoutput        yes\n");

			printer.println("# for restarting:");
			printer.println("restartname         $em_restart");
			printer.println("restartfreq         " + minStepsCount);
			printer.println("restartsave         yes");
			printer.println("binaryrestart       yes\n"); 

			printer.println("fixedAtoms off");
			printer.println("minimize " + minStepsCount);
			printer.flush();
			printer.close();
			 
			
			//heating file
			printer = new PrintWriter(new FileWriter(outdir + File.separator + "heating.namd"));
			printer.println("# molecular system");
			printer.println("set MOL " + basename);
			printer.println("structure\t" + psf);
			printer.println("coordinates\t" + pdb);

			printer.println("\n# after minimization");
			printer.println("set inprot       ${MOL}_min2_restart." + minStepsCount);
			printer.println("bincoordinates   ${inprot}.coor");
			printer.println("binvelocities    ${inprot}.vel");
			printer.println("extendedSystem   ${inprot}.xsc\n");

			printer.println("set outputname     ${MOL}_heat");
			printer.println("set heat_restart   ${MOL}_heat_restart\n");

			printer.println("set desiredtemp    300       ;# initial temperature\n");
			printer.println("\n# force field");
			printer.println("paratypecharmm on");
			printer.println("parameters " + params);
			printer.println("exclude scaled1-4");
			printer.println("1-4scaling	1.0\n");
			
			printer.println("wrapAll on");
			
			printer.println("PME on");
			printer.println("PMETolerance 0.000001");
			printer.println("PMEGridSpacing 1.0\n");

			printer.println("# approximations");
			printer.println("switching off");
			printer.println("cutoff 11");
			printer.println("pairlistdist 12.5\n");

			printer.println("rigidBonds water");
			printer.println("rigidIterations 100\n");
			
			printer.println("# integrator");
			printer.println("timestep 1.0 # in fs.");
			printer.println("stepspercycle 20");
			printer.println("nonbondedFreq 2\n");
			
			printer.println("# Output");
			printer.println("outputName          $outputname\n");

			printer.println("dcdfreq             100     ;# how often we output trajectories");
			printer.println("xstFreq             100");
			printer.println("outputEnergies      100");
			printer.println("outputPressure      100");
			printer.println("outputtiming        100");
			printer.println("binaryoutput        yes\n");
			
			printer.println("# for restarting:");
			printer.println("restartname         $heat_restart");
			printer.println("restartfreq         6000");
			printer.println("restartsave         yes");
			printer.println("binaryrestart       yes\n"); 
			
			printer.println("reassignFreq	20");
			printer.println("reassignTemp	0");
			printer.println("reassignIncr	1");
			printer.println("reassignHold	300\n");
			
			printer.println("run 6000");
			
			printer.flush();
			printer.close();
			
			
			
			//density equilibration file
			printer = new PrintWriter(new FileWriter(outdir + File.separator + "equilibration.namd"));

			printer.println("# molecular system");
			printer.println("set MOL\t" + basename);
			printer.println("structure\t" + psf);
			printer.println("coordinates\t" + pdb);

			printer.println("\nset inprot       ${MOL}_heat_restart.6000");
			printer.println("bincoordinates   ${inprot}.coor");
			printer.println("binvelocities    ${inprot}.vel");
			printer.println("extendedSystem   ${inprot}.xsc\n");

			printer.println("set temperature 300	;# the temperature");

			printer.println("set outputname     ${MOL}_eq");
			printer.println("set eq_restart     ${MOL}_eq_restart\n");

			printer.println("firsttimestep       0 ;# reset to 0 for equilibration run.");

			printer.println("fixedAtoms off\n");

			printer.println("# force field");
			printer.println("paratypecharmm on");
			printer.println("parameters\t" + params);
			printer.println("exclude scaled1-4");
			printer.println("1-4scaling 1.0");

			printer.println("wrapAll on\n");

			printer.println("PME on");
			printer.println("PMETolerance	0.000001");
			printer.println("PMEGridSpacing	1.0\n");

			printer.println("# approximations");
			printer.println("switching on");
			printer.println("switchdist 8");
			printer.println("cutoff 10");
			printer.println("pairlistdist 11.5\n");

			printer.println("# integrator");
			printer.println("timestep      1.0  ;# in fs");
			printer.println("stepspercycle 20   ;# default");
			printer.println("nonbondedFreq 1    ;# same as BerendsenPressureFreq");
			printer.println("fullElectFrequency 2\n");

			printer.println("# output");
			printer.println("outputName          $outputname");
			printer.println("dcdfreq             1000     ;# every ps");
			printer.println("xstFreq             1000");
			printer.println("outputEnergies      1000");
			printer.println("outputPressure      1000");
			printer.println("outputtiming        1000");
			printer.println("binaryoutput        yes\n");

			printer.println("# for restarting:");
			printer.println("restartname         $eq_restart");
			printer.println("restartfreq         " + eqStepsCount);
			printer.println("restartsave         yes");
			printer.println("binaryrestart       yes   ;# yes preserves more accuracy\n");

			printer.println("# Temperature coupling via Langevin dynamics.");
			printer.println("langevin            on");
			printer.println("langevinTemp        $temperature");
			printer.println("langevinDamping     1.0");
			printer.println("langevinHydrogen    on\n");

			printer.println("# Pressure coupling by Nose-Hoover Langevin piston pressure control");
			printer.println("LangevinPiston       on");
			printer.println("LangevinPistonTarget 1.01325 ;# in bar.");
			printer.println("LangevinPistonPeriod 100     ;# in fs.");
			printer.println("LangevinPistonDecay  50");
			printer.println("LangevinPistonTemp   $temperature     ;# set same as langevinTemp.\n");

			printer.println("useGroupPressure yes\n");

			printer.println("# to use SHAKE to constraint all bonds during free MD only");
			printer.println("rigidBonds water");
			printer.println("#rigidTolerance	0.000001");
			printer.println("rigidIterations	100");
			printer.println("run\t" + eqStepsCount);
			
			printer.flush();
			printer.close();
			

			
			if (acemd){
				//this option is for ACEMD production input

				for (int i = 1; i <= totalRunCount; i++){
					int j = i - 1;
				
					printer = new PrintWriter(new FileWriter(outdir + File.separator + basename + "_run" + i + ".namd"));
					
					
					printer.println("# initial config");
					printer.println("coordinates\t" + pdb);
					printer.println("structure\t" + psf);

					if (i == 1){
						printer.println("bincoordinates\t" + basename + "_eq_restart." + eqStepsCount + ".coor");
						printer.println("binvelocities\t" + basename + "_eq_restart." + eqStepsCount + ".vel\n");
					}
					else{
						printer.println("bincoordinates\t" + basename + "_run" + j + ".coor");
						printer.println("binvelocities\t" + basename + "_run" + j + ".vel");
						printer.println("binindex\t" + basename + "_run" + j + ".idx\n");
					}
					printer.println("temperature     300\n");

					printer.println("# integrator params");
					printer.println("timestep\t" + step);
					printer.println("hydrogenscale   1");
					printer.println("rigidbonds      all\n"); 

					printer.println("# force field params");
					printer.println("parameters\t" + params);
					printer.println("exclude         scaled1-4");
					printer.println("1-4scaling      1.0");
					printer.println("switching       on");
					printer.println("switchdist      7.5");
					printer.println("cutoff          9\n");

					printer.println("# output params");
					printer.println("outputname\t" + basename + "_run" + i);
					printer.println("dcdfile\t" + basename + "_run" + i + ".dcd");
					printer.println("dcdfreq\t" + frequency + "\n");

					printer.println("# periodic cell");
					printer.println("celldimension\t" + bv1 + " " + bv2 + " " + bv3);

					printer.println("# full electrostatics");
					printer.println("pme             on");
					
					//figure out the dimensions of the PME grid
					//for each dimension, take the next number divisible by 2, 3 or 5
					long ci = Math.round(basisVector1);
					while (true){
						if (ci%2 == 0) break;
						if (ci%3 == 0) break;
						if (ci%5 == 0) break;
						ci++;
					}
					long cj = Math.round(basisVector2);
					while (true){
						if (cj%2 == 0) break;
						if (cj%3 == 0) break;
						if (cj%5 == 0) break;
						cj++;
					}
					long ck = Math.round(basisVector3);
					while (true){
						if (ck%2 == 0) break;
						if (ck%3 == 0) break;
						if (ck%5 == 0) break;
						ck++;
					}
					
					if (ci%2 != 0) ci++;
					if (cj%2 != 0) cj++;
					if (ck%2 != 0) ck++;
					
					printer.println("pmegridsizex\t" + ci);
					printer.println("pmegridsizey\t" + cj);
					printer.println("pmegridsizez\t" + ck);

					printer.println("\nlangevin on");
					printer.println("langevintemp	300");
					printer.println("langevindamping 0.1");
					printer.println("langevinhydrogen	on");
					
					printer.println("energyfreq\t" + frequency);

					printer.println("run\t" + steps);

					printer.flush();
					printer.close();
				}
				
				//write out the corrector.pl Perl script that edits the first run input file,
				//based on the output from the equilibration - this information is only known
				//once the equilibration is completed and has to be edited at runtime
				
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "corrector.pl"));
				
				printer.println("#!/usr/bin/perl");

				printer.println("$basename = $ARGV[0];");
				printer.println("$count = $ARGV[1];");

				printer.println("open (IN, $ARGV[2])|| die \"Unable to open $ARGV[2]:$!\\n\";");
				printer.println("@xsc = <IN>;");
				printer.println("close IN||die \"Unable to close $ARGV[2]:$!\\n\";");

				printer.println("@list = split(\" \", $xsc[2]);");
				printer.println("@data = ();");
				printer.println("foreach $item (@list){");
				printer.println("\tunless ($item eq \"0\"){");
				printer.println("\t\tpush(@data, $item);");
				printer.println("\t}");
				printer.println("}\n");
				printer.println("#form the line");
				printer.println("$update = \"celldimension\t\".$data[1].\" \".$data[2].\" \".$data[3].\"\\n\";");

				printer.println("#now walk over all run input files and correct them");
				printer.println("for $i (1 .. $count){");
				printer.println("\t$name = $basename.\"_run\".$i.\".namd\";");
				printer.println("\topen (F, $name)||die \"Unable to open $name for update:$!\\n\";");
				printer.println("\t@content = <F>;");
				printer.println("\tclose F || die \"Unable to close $name:$!\\n\";");

				printer.println("\topen (FF, \">$name\")||die \"Unable to open $name:$!\\n\";");
					
				printer.println("\tforeach $line (@content){");
				printer.println("\t\tif ($line=~/celldimension/){");
				printer.println("\t\t\tprint FF $update;");
				printer.println("\t\t}");
				printer.println("\t\telse{");
				printer.println("\t\t\tprint FF $line;");
				printer.println("\t\t}");
				printer.println("\t}");
				printer.println("\tclose FF||die \"Unable to close $name after update:$!\\n\";");
				printer.println("}");

				printer.flush();
				printer.close();
				
				
				//write out the job script file
				//start with preparation, and complete with the production run
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "run.sh"));
				printer.println("#!/bin/bash");
				printer.println("source /home/sasha/.ace");
				printer.println("echo -n \"Starting NAMD script at: \"");
				printer.println("date");
				printer.println("echo \"\"");
				printer.println("echo \"Running minimization...\"");
				printer.println("charmrun /data/namd/NAMD_2.6_Linux-amd64/namd2 ++local +p4 min1.namd > " + basename + "_min1.out");
				printer.println("charmrun /data/namd/NAMD_2.6_Linux-amd64/namd2 ++local +p4 min2.namd > " + basename + "_min2.out");
				printer.println("echo \"Running heating...\"");
				printer.println("charmrun /data/namd/NAMD_2.6_Linux-amd64/namd2 ++local +p4 heating.namd > " + basename + "_heat.out");
				printer.println("echo \"Running equilibration...\"");
				printer.println("charmrun /data/namd/NAMD_2.6_Linux-amd64/namd2 ++local +p4 equilibration.namd > " + basename + "_eq.out\n");
				
				//here, add a call to a Perl script that updates the cell dimensions in the first run input file,
				//according to the output of the equilibration xcs file
				printer.println("chmod +x corrector.pl");
				printer.println("./corrector.pl " + basename + " " + totalRunCount + " " + basename + "_eq_restart." + eqStepsCount + ".xsc");
				
				
				printer.println("for i in {1.." + totalRunCount + "}");
				printer.println("do");
				printer.println("\techo \"Running production simulation $i...\"");
				printer.println("\tmpirun -n 3 --mca btl ^openib,udapl acemd " + basename + "_run$i.namd --device 3 > " + basename + "_run$i.out");
				printer.println("done");
				
				
				printer.flush();
				printer.close();
			}
			else{
				//this option is for purely NAMD simulation
				for (int i = 1; i <= totalRunCount; i++){
					int j = i - 1;
					printer = new PrintWriter(new FileWriter(outdir + File.separator + basename + "_run" + i + ".namd"));
					printer.println("# molecular system");
					printer.println("set MOL\t" + basename);
					printer.println("structure\t" + psf);
					printer.println("coordinates\t" + pdb);

					if (i == 1){
						printer.println("set inprot       ${MOL}_eq_restart." + eqStepsCount);
					}
					else{
						printer.println("set inprot       ${MOL}_run" + j + "_restart." + steps);
					}
					printer.println("bincoordinates   ${inprot}.coor");
					printer.println("binvelocities    ${inprot}.vel");
					printer.println("extendedSystem   ${inprot}.xsc\n");

					printer.println("set outputname     ${MOL}_run" + i);
					printer.println("set prod_restart     ${MOL}_run" + i +"_restart\n");

					printer.println("set temperature	300");

					printer.println("firsttimestep       0 ;# reset to 0 for equilibration run.");

					printer.println("fixedAtoms off\n");

					printer.println("# force field");
					printer.println("paratypecharmm on");
					printer.println("parameters\t" + params);
					printer.println("exclude scaled1-4");
					printer.println("1-4scaling 1.0");

					printer.println("wrapAll on\n");

					printer.println("PME on");
					printer.println("PMETolerance	0.000001");
					printer.println("PMEGridSpacing	1.0\n");

					printer.println("# approximations");
					printer.println("switching on");
					printer.println("switchdist 8");
					printer.println("cutoff 10");
					printer.println("pairlistdist 11.5\n");

					printer.println("# integrator");
					printer.println("timestep\t" + step + ";# in fs");
					printer.println("stepspercycle 20   ;# default");
					printer.println("nonbondedFreq 2    ;# same as BerendsenPressureFreq");
					printer.println("fullElectFrequency 4\n");

					printer.println("# output");
					printer.println("outputName          $outputname");
					printer.println("dcdfreq         	 " + frequency);
					printer.println("xstFreq             " + frequency);
					printer.println("outputEnergies      " + frequency);
					printer.println("outputPressure      " + frequency);
					printer.println("outputtiming        " + frequency);
					printer.println("binaryoutput        yes\n");

					printer.println("# for restarting:");
					printer.println("restartname         $eq_restart");
					printer.println("restartfreq         " + steps);//one restart file at the end
					printer.println("restartsave         yes");
					printer.println("binaryrestart       yes   ;# yes preserves more accuracy\n");

					printer.println("# Temperature coupling via Langevin dynamics.");
					printer.println("langevin            on");
					printer.println("langevinTemp        $temperature");
					printer.println("langevinDamping     1.0");
					printer.println("langevinHydrogen    on\n");

					printer.println("# Pressure coupling by Nose-Hoover Langevin piston pressure control");
					printer.println("LangevinPiston       on");
					printer.println("LangevinPistonTarget 1.01325 ;# in bar.");
					printer.println("LangevinPistonPeriod 100     ;# in fs.");
					printer.println("LangevinPistonDecay  50");
					printer.println("LangevinPistonTemp   $temperature     ;# set same as langevinTemp.\n");

					printer.println("margin\t3.0");
					printer.println("useGroupPressure yes\n");

					printer.println("# to use SHAKE to constraint all bonds during free MD only");
					printer.println("rigidBonds water");
					printer.println("#rigidTolerance	0.000001");
					printer.println("rigidIterations	100");
					printer.println("run\t" + steps);
					
					printer.flush();
					printer.close();
				}

				
				//write out the job script file
				printer = new PrintWriter(new FileWriter(outdir + File.separator + "run.sh"));
				printer.println("#!/bin/bash");

				printer.println("echo -n \"Starting NAMD script at: \"");
				printer.println("date");
				printer.println("echo \"\"");
				printer.println("echo \"Running minimization...\"");
				printer.println("charmrun /data/namd/NAMD_2.6_Linux-amd64/namd2 ++local +p4 min1.namd");
				printer.println("charmrun /data/namd/NAMD_2.6_Linux-amd64/namd2 ++local +p4 min2.namd");
				printer.println("echo \"Running heating...\"");
				printer.println("charmrun /data/namd/NAMD_2.6_Linux-amd64/namd2 ++local +p4 heating.namd");
				printer.println("echo \"Running equilibration...\"");
				printer.println("charmrun /data/namd/NAMD_2.6_Linux-amd64/namd2 ++local +p4 equilibration.namd\n");
				printer.println("for i in {1.." + totalRunCount + "}");
				printer.println("do");
				printer.println("echo \"Running production simulation $i...\"");
				printer.println("charmrun /data/namd/NAMD_2.6_Linux-amd64/namd2 ++local +p4 " + outdir + File.separator + basename + "_run$i.namd");
				printer.println("done");
				
				printer.flush();
				printer.close();
					
				
			}
			
			setVisible(false);
			parent.displayMessage("Molecular Dynamics setup completed");
		}
		catch (Exception ex){
			ex.printStackTrace();
			parent.displayExceptionMessage("Exception preparing MD simulation", ex);
		}
		
		
		
		parent.setDisplayDialogStatus(false);
		
		dispose();
	}
	
	private Border createTitledBorder(String title){
		
		Border border1 = BorderFactory.createEmptyBorder(0,0,0,0);//left offset of the boxes in the box
		TitledBorder border2 = BorderFactory.createTitledBorder(new EtchedBorder(), " " + title + " ");
		border2.setTitleColor(Color.black);
		border2.setTitleFont(new Font("Dialog", 0, 12));
		Border border = BorderFactory.createCompoundBorder(border1, border2);

		return border;
	}

	
	public void processPick(String data, StructureComponent structureComponent){}
	
}
