//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs;

import java.util.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.border.*;

import edu.sdsc.mbt.viewables.*;
import edu.sdsc.sirius.viewers.*;
import edu.sdsc.sirius.surface.*;
import edu.sdsc.sirius.dialogs.TableImpl.*;
import edu.sdsc.sirius.util.*;
import edu.sdsc.mbt.*;

/**
 * This class implements a dialog that displays the currently loaded entries.
 * It retrieves data from structureDocument, packages it as DataSet and displays it in ResultPanel.
 * @see edu.sdsc.mbt.util.DataSet
 * 
 * @author Oleksandr V. Buzko
 */
public class SurfaceEditDialog extends JDialog implements ResultPanelCallable, DisplayDialog {
	
	private Container contentPane;
	private JDialog self;
	
	private TableViewPanel contentPanel;
	
	private Vector searchTypes;
	
	private Manager parent;
	private JFrame parentFrame;
	private StructureViewer viewer;
	
	private JPanel base = new JPanel();
	private JPanel buttonPanel = new JPanel();
	private JButton deleteButton = new JButton("Delete");
	private JButton hideButton = new JButton("Hide");
	private JButton showButton = new JButton("Show");
	private JButton closeButton = new JButton("Close");
	
	private Object value = new Object();
	
	private HashMap selection = new HashMap();//SurfaceComponent -> Object
	
	private String parentId;
	
	private DataSet set;
		
	/**
	 * Constructor that creates the dialog.
	 * @param f parent frame
	 * @param p Manager object that will process the RequestSet
	 * @param title dialog title
	 * @param modal whether the dialog is modal (other frames in the application will be inaccessible)
	 */
	public SurfaceEditDialog (JFrame f, Manager p, StructureViewer sv){
	
		super(f, "Manage surfaces", false);
		parent = p;
		this.viewer = sv;
		
		set = new DataSet();
		//get the dataset from the structureDocument

		Vector surfaces = sv.getSurfaceObjects();
		for (int j = 0; j < surfaces.size(); j++){
			SurfaceComponent sc = (SurfaceComponent)surfaces.get(j);
			
			if (sc == null){
				continue;
			}
			
			String name = "Surface" + new Integer(j+1).toString();
			String description = new Integer(sc.getAtoms().size()).toString() + " atoms";
			//create a DataBlock out of this entry
			DataBlock block = new DataBlock(new Integer(j).toString());
			block.setName(name);
			block.setDescription(description);
			set.addData(block);
		}
		

		//get the menubar
		//go through the available data options and determine the future table headers
		searchTypes = new Vector();
		searchTypes.add("Name");
		searchTypes.add("Description");
		parentFrame = parent.getApplicationFrame();
		
		contentPane = getContentPane();
		contentPane.setLayout(new BorderLayout());
		
		
		self = this;
		
		base.setBorder(new BevelBorder(BevelBorder.LOWERED));
		base.setLayout(new BorderLayout());
		
		//add window listener
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				setVisible(false);
				dispose();
			}
		});
		
		addComponentListener(new ComponentAdapter(){
			public void componentResized(ComponentEvent e){
				Dimension size = getSize();
				if (contentPanel == null) return;
				contentPanel.resizePanel(size);
			}	
		});
		
		contentPane.add(base, BorderLayout.CENTER);
		
		
		//panel that holds buttons
		JPanel lowerPanel = new JPanel();
		lowerPanel.setLayout(new BorderLayout());
		JPanel borderPanel = new JPanel();
		borderPanel.setLayout(new BorderLayout());
		borderPanel.add(new JSeparator(), BorderLayout.NORTH);
		
		lowerPanel.add(borderPanel, BorderLayout.NORTH);
		
		buttonPanel.setLayout(new FlowLayout(FlowLayout.CENTER, 40, 5));
		
		deleteButton.setPreferredSize(new Dimension(80,25));
		deleteButton.setVisible(true);
		deleteButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				if (contentPanel != null){
					if (contentPanel.getSelectedRowCount() > 0){
						//delete selected surface
						Set keys = selection.keySet();
						Iterator it = keys.iterator();
						while (it.hasNext()){
							SurfaceComponent sc = (SurfaceComponent)it.next();
							selection.remove(sc);
							viewer.deleteSurface(sc, false);
						}
						refreshList();
						viewer.getGeometryViewer().updateView();
					}
				}
			}
		});
		
		
		closeButton.setPreferredSize(new Dimension(80,25));
		closeButton.setVisible(true);
		closeButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ae){
				setVisible(false);
				dispose();
			}
		});
		
		buttonPanel.add(deleteButton);
		buttonPanel.add(closeButton);
		
		
		lowerPanel.add(buttonPanel, BorderLayout.CENTER);
		contentPane.add(lowerPanel, BorderLayout.SOUTH);
		
				
		if (set != null && set.size() > 0){
			
			//used to show a table in a previously open frame
			//(after a new search initiated from this frame)
			contentPanel = new TableViewPanel(base, set, parent, searchTypes, TableViewPanel.FIRST_COLUMN_NARROW, false);
			contentPanel.setCallable(this);
			base.add(contentPanel);
			contentPane.add(base, BorderLayout.CENTER);
			
		}
		else{
			parent.displayErrorMessage("Nothing to display: no surfaces loaded.");
			dispose();
			return;
		}
				
		
		setSize(370,200);
		
		if (parentFrame == null){
//			setLocationRelativeTo(null);
			setBounds(0, 0, getSize().width, getSize().height);
		}
		else{
		
			//set location
			Dimension d1 = getSize();
			Dimension d2 = parentFrame.getSize();
		
			int x = 5 + DialogConstants.toolbarWidth;
			int y = DialogConstants.level + DialogConstants.toolbarHeight;
		
			setBounds(x + parentFrame.getX(), y + parentFrame.getY(), d1.width, d1.height);
		}
		
		setVisible(true);
	}
	
	public void refreshList(){
		
		base.remove(contentPanel);
		set.clearData();
		Vector surfaces = viewer.getSurfaceObjects();
		for (int j = 0; j < surfaces.size(); j++){
			SurfaceComponent sc = (SurfaceComponent)surfaces.get(j);
			
			if (sc == null){
				continue;
			}
			
			String name = "Surface" + new Integer(j+1).toString();
			String description = new Integer(sc.getAtoms().size()).toString() + " atoms";
			//create a DataBlock out of this entry
			DataBlock block = new DataBlock(name);
			block.setEntryId(name);
			block.setName(description);
			set.addData(block);
		}
		
		contentPanel = new TableViewPanel(base, set, parent, searchTypes, TableViewPanel.FIRST_COLUMN_NARROW, false);
		contentPanel.setCallable(this);
		base.add(contentPanel);
		contentPane.add(base, BorderLayout.CENTER);
		
		base.revalidate();
		base.repaint();
	}
	
	public void setSelectedIndices(Vector indices){
		Vector surfaces = viewer.getSurfaceObjects();
		for (int i = 0; i < surfaces.size(); i++){
			SurfaceComponent sc = (SurfaceComponent)surfaces.get(i);
			StructureStyles styles = sc.structure.getStructureMap().getStructureStyles();
			Vector atoms = sc.getAtoms();
			
			boolean selected = false;
			for (int j = 0; j < indices.size(); j++){
				int index = ((Integer)indices.get(j)).intValue();
				if (index == i){
					selected = true;
					break;
				}
			}
			
			if (selection.containsKey(sc)){
				//this surface is already selected
				if (!selected){
					//deselect it
					for (int j = 0; j < atoms.size(); j++){
						styles.setSelected((Atom)atoms.get(j), false, true, true);
					}
					selection.remove(sc);
				}
			}
			else{
				if (selected){
					//select it
					for (int j = 0; j < atoms.size(); j++){
						styles.setSelected((Atom)atoms.get(j), true, true, true);
					}
					selection.put(sc, value);
				}
			}
		}
				
		viewer.getGeometryViewer().updateView();
	}
	
	public void setNumberOfHits(int n){}
	
	public void processPick(String a, StructureComponent c){}
	
}