//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs.TableImpl;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import javax.swing.table.*;


/**
 * This class extends AbstractTableModel to allow for custom behavior of the table elements in
 * ResultPanel, such as sorting and filtering.
 * @see edu.sdsc.sirius.search.ResultPanel
 * 
 * @author Oleksandr V. Buzko
 */
class PubmedResultData extends AbstractTableModel {

	public ColumnData m_columns[];
	protected Vector m_vector;//stores the actual data
	protected Vector m_current;
	
	protected int sortColumn = 1;//column used to sort content
	protected boolean sortAsc = true;
	protected Vector currentIndices = null;
	protected Vector currentColumns = null;
	
	protected int blankRowCount = 0;
	protected Vector blankRows = new Vector();
	
	protected HashMap checkedRows = new HashMap();
	
	private HashMap referenceHash = new HashMap();//PubmedLineData -> DataBlock

	public static final int COL_AUTHOR = 0;
	public static final int COL_YEAR = 1;
	public static final int COL_JOURNAL = 2;
	public static final int COL_TITLE = 3;
	
	private Vector hiddenRows = new Vector();//row number -> LineData object
	
	public PubmedResultData(DataSet data, Vector searchTypes, HashMap columnWidths, HashMap columnAlign, Vector col, Vector ind){

		int k = searchTypes.size();
		
		m_columns = new ColumnData[k];
		m_vector = new Vector();
		m_current = new Vector();//same as m_vector, except it stores only the currently displayed rows
		
		currentColumns = col;
		currentIndices = ind;
		
		//enter the actual column parameters
		for (int i = 0; i < k; i++){
			String headerName = null;
			String name = (String)searchTypes.elementAt(i);
			headerName = name;
			m_columns[i] = new ColumnData(name, headerName, ((Integer)columnWidths.get(name)).intValue(), ((Integer)columnAlign.get(name)).intValue());
		}
		
		//assign the actual data - convert DataSet into a Vector of Vectors
		for (int i = 0; i < data.size(); i++){
			PubmedLineData temp = new PubmedLineData();
			DataBlock block = data.getData(i);
			temp.author = block.getAuthor();
			temp.year = block.getYear();
			temp.journal = block.getJournal();
			temp.title = block.getTitle();
			
			referenceHash.put(temp, block);
			m_vector.add(temp);
			m_current.add(temp);
		}
		
		Collections.sort(m_vector, new ResultComparator(sortColumn, sortAsc, currentIndices));
	}

	public int getRowCount(){
		return m_current==null ? 0 : m_current.size();
	}
	
	public int getHiddenRowCount(){
		return hiddenRows.size();
	}
	
	public int getSubsetRowCount(){
		//the currently displayed subset - blank lines
		int count = m_current==null ? 0 : m_current.size();
		count -= blankRowCount;
		return count;
	}
	
	public int getFullRowCount(){
		int count = m_vector==null ? 0 : m_vector.size();
		count -= blankRowCount;
		return count;
	}

	public int getColumnCount(){
		return m_columns.length;
	}

	public int getSortColumn(){
		return sortColumn;
	}
	
	public HashMap getCheckedRows(){
		return checkedRows;
	}

	public boolean isCellEditable(int nRow, int nColumn){
		return false;
	}
	
	public void insertBlankRow(){
		PubmedLineData blank = new PubmedLineData();
		m_vector.add(blank);
		m_current.add(blank);

		blankRows.add(blank);
		blankRowCount++;
	}
	
	public void removeBlankRow(){
		if (blankRowCount == 0){
//			System.out.println("No blanks to remove");
			return;
		}
		PubmedLineData line = (PubmedLineData)blankRows.lastElement();
		if (blankRows.contains(line)){
			m_vector.remove(line);
			m_current.remove(line);
			blankRows.remove(line);
			blankRowCount--;
		}
	}
	
	public int getBlankRowCount(){
		return blankRowCount;
	}
	
	public void displayLine(PubmedLineData data, int index){
		if (hiddenRows.contains(data)){
			//show it
			hiddenRows.remove(data);
			m_current.insertElementAt(data, index);
		}
	}
	
	public void hideLine(PubmedLineData data){
		if (!hiddenRows.contains(data)){
			hiddenRows.add(data);
			m_current.remove(data);
		}
	}
	
	public void resort(){
		Collections.sort(m_current, new ResultComparator(sortColumn, sortAsc, currentIndices));
		
	}
	
	public Object getValueAt(int nRow, int nColumn){

		if (nRow < 0 || nRow > getRowCount()){
			return "";
		}
		
		PubmedLineData data = (PubmedLineData)m_current.elementAt(nRow);
		if (blankRows.contains(data)){
			return " ";
		}
		switch (nColumn) {
			case COL_AUTHOR:
				return data.author;
			case COL_YEAR:
				return data.year;
			case COL_JOURNAL:
				return data.journal;
			case COL_TITLE:
				return data.title;
		}
		return "";
	}
	
	public Object getFullValueAt(int nRow, int nColumn){

		if (nRow < 0 || nRow > getFullRowCount()){
			return "";
		}
		
		PubmedLineData data = (PubmedLineData)m_vector.elementAt(nRow);
		if (blankRows.contains(data)){
			return " ";
		}
		switch (nColumn) {
			case COL_AUTHOR:
				return data.author;
			case COL_YEAR:
				return data.year;
			case COL_JOURNAL:
				return data.journal;
			case COL_TITLE:
				return data.title;
		}
		return "";
	}
	
	
	public void setValueAt(Object value, int nRow, int nCol) {
	
		if (nRow < 0 || nRow>=getRowCount() || value == null){
			return;
		}
		
		PubmedLineData data = (PubmedLineData)m_current.elementAt(nRow);
		if (blankRows.contains(data)) return;
		
		switch (nCol) {
			case COL_AUTHOR:
				data.author = (String)value;
			case COL_YEAR:
				data.year = (String)value;
			case COL_JOURNAL:
				data.journal = (String)value;
			case COL_TITLE:
				data.title = (String)value;
		}	
	}
	
	public void insert(PubmedLineData d, int nRow) {
	
		if (nRow < 0){
			nRow = 0;
		}
		if (nRow > m_vector.size()){
			nRow = m_vector.size();
		}
		m_vector.insertElementAt(d, nRow);

	}
	
	public boolean delete(int nRow){
	
		if (nRow < 0 || nRow >= m_vector.size()) return false;
		m_vector.remove(nRow);
		return true;
	}
			
	public Vector getData(){
		return m_vector;
	}
	
	public PubmedLineData getRow(int nRow){
		return (PubmedLineData)m_vector.elementAt(nRow);
	}
	
	public DataBlock getDataBlock(int nRow){
		return (DataBlock)referenceHash.get(m_current.elementAt(nRow));
	}
	
	
	public String getTitle(){
		return "Pubmed search: result set";
	}
	
	public String getColumnName(int column){
		String str = m_columns[column].getHeaderTitle();
		if (column == sortColumn){
			str += sortAsc ? " >>" : " <<";
		}
		return str;
	}
	
	class ColumnListener extends MouseAdapter {

		protected JTable m_table = null;
		
		public ColumnListener(JTable table){
			m_table = table;
		}
		
		public void mouseClicked(MouseEvent e){
		
			TableColumnModel model = m_table.getColumnModel();
			int columnModelIndex = model.getColumnIndexAtX(e.getX());
			int modelIndex = model.getColumn(columnModelIndex).getModelIndex();
			
			if (modelIndex < 0) return;			
			if (sortColumn == modelIndex){
				sortAsc = !sortAsc;
			}
			else{
				sortColumn = modelIndex;
				sortAsc = true;//to make sure the new column is first sorted in ascending order
			}
	
			for (int i = 0; i < currentColumns.size(); i++){
				TableColumn column = (TableColumn)currentColumns.elementAt(i);
				column.setHeaderValue(getColumnName(column.getModelIndex()));
			}
			
			m_table.getTableHeader().repaint();
			
			Collections.sort(m_current, new ResultComparator(modelIndex, sortAsc, currentIndices));
			m_table.tableChanged(new TableModelEvent(PubmedResultData.this));
			m_table.repaint();
		}
	}
	
	class ResultComparator implements Comparator {
	
		protected int m_sortCol;
		protected boolean m_sortAsc;
		protected Vector currentIndices;
		
		public ResultComparator(int sortCol, boolean sortAsc, Vector curInd){
			m_sortCol = sortCol;
			m_sortAsc = sortAsc;
			currentIndices = curInd;
		}
		
		public int compare(Object o1, Object o2){
		
			if (!(o1 instanceof PubmedLineData) || !(o2 instanceof PubmedLineData)){
				return 0;
			}
			
			//check whether the objects are blanks that should not be compared
			if ((blankRows.contains(o1))&&(blankRows.contains(o2))){
				return 0;
			}
			else if (blankRows.contains(o1)){
				return 1;
			}
			else if (blankRows.contains(o2)){
				return -1;
			}
						
			PubmedLineData v1 = (PubmedLineData)o1;
			PubmedLineData v2 = (PubmedLineData)o2;
			
			int result = 0;
			try{
				switch (m_sortCol){
				case COL_AUTHOR:	//description
					if (currentIndices.contains(new Integer(COL_AUTHOR))){
						String d1 = v1.author.toUpperCase();
						String d2 = v2.author.toUpperCase();
					
						result = d1.compareTo(d2);
						break;
					}
					else{
						return 0;
					}
					
					case COL_YEAR:	//name
						if (currentIndices.contains(new Integer(COL_YEAR))){
							if (v1.year == null && v2.year == null){
								return 0;
							}
							else if (v1.year == null){
								return 1;
							}
							else if (v2.year == null){
								return -1;
							}
			
							String year1 = v1.year.trim().toUpperCase();
							String year2 = v2.year.trim().toUpperCase();
							if (year1.equals("NULL")){
								return 1;
							}
							else if (year2.equals("NULL")){
								return -1;
							}
							else if (year1.equals("NULL") && year2.equals("NULL")){
								return 0;
							}
							result = year1.compareTo(year2);
							break;
						}
						else{
							return 0;
						}
					case COL_JOURNAL:	//description
						if (currentIndices.contains(new Integer(COL_JOURNAL))){
							String d1 = v1.journal.toUpperCase();
							String d2 = v2.journal.toUpperCase();
						
							result = d1.compareTo(d2);
							break;
						}
						else{
							return 0;
						}
						
					case COL_TITLE:	//description
						if (currentIndices.contains(new Integer(COL_TITLE))){
							String d1 = v1.title.toUpperCase();
							String d2 = v2.title.toUpperCase();
						
							result = d1.compareTo(d2);
							break;
						}
						else{
							return 0;
						}
				}
				
				if (!sortAsc){
					result = -result;
				}
				
			}
			catch (NumberFormatException ex){
				return 0;
			}
			catch (NullPointerException e){
				return 0;
			}
			
			return result;
		}
		
		public boolean equals(Object obj){
			if (obj instanceof ResultComparator){
				ResultComparator compObj = (ResultComparator)obj;
				return (compObj.m_sortCol == m_sortCol)&&(compObj.m_sortAsc == m_sortAsc);
			}
			return false;
		}
	
	}
}
