//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs.TableImpl;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.table.*;
import java.awt.image.*;


/**
 * This class implements a panel used to display sequence data retrieved from a database. Each entry
 * is contained within a DataBlock as a part of a DataSet.
 * This class permits extended functionality including sorting, dynamic adding and removal of columns,
 * filtering of entries by content of any of the fields, custom buttons to display entry-specific information, 
 * as well as checkboxes in each line to trigger loading of the corresponding entry.
 * 
 * @see edu.sdsc.mbt.util.DataBlock
 * @see edu.sdsc.mbt.DataSet
 * @see edu.sdsc.mbt.util.Entry
 * 
 * @author Oleksandr V. Buzko
 */
public class PubmedResultPanel extends JPanel {
	
	private DataSet dataSet;
	private JPanel parentPanel;
	
	private JScrollPane scrollPane;
	private JScrollBar scrollBar;
	private JViewport vp;
	protected int viewOffset = 0;//offset in pixels when part of the table is scrolled out of the view
//	private JPanel filterPanel = new JPanel();
	
	private PubmedResultData m_data;
	private ResultTable m_table;
	private int columnCount = 0;
	private HashMap columns = new HashMap();
	private Vector columnOrder = new Vector();
	private Vector currentColumns = new Vector();
	public Vector currentIndices = new Vector();//stores indices of columns that are displayed
	
	private Vector searchTypes = new Vector();
	
	public HashMap columnWidths = new HashMap();
	public HashMap columnAlign = new HashMap();
	
	public static Vector defaultColumns = new Vector();
		
	protected int extraHeight = 100;//everything in the frame other than the table itself
	protected int tableHeight = 0;
	
	protected int[] selectedRows = { -1 };
	
//	protected Manager parent = null;
	
	//filter fields
//	private JLabel filterLabel = new JLabel("Filter:");
	private JTextField authorField = new JTextField(10);
	private JTextField journalField = new JTextField(20);
	private JTextField titleField = new JTextField(50);
	private JTextField yearField = new JTextField(10);
	
	private boolean tooltips = false;
	private String target;
	
	private HashMap columnRef = new HashMap();//contains references from fields to the corresponding table titles
	
	static{
	
		defaultColumns.add("Author");
		defaultColumns.add("Year");
		defaultColumns.add("Journal");
		defaultColumns.add("Title");
	}
	
	/**
	 * Constructs the <code>ResultPanel</code> when created as in a frame typically storing search results.
	 * @param p parent frame
	 * @param data <code>DataSet</code> to initialize the table
	 * @param callable Manager object that will handle entry loading
	 * @param pp 
	 */
	public PubmedResultPanel(JPanel parent, DataSet data){
		dataSet = data;
		parentPanel = parent;

		
		if (dataSet == null){
			System.out.println("No data to display");
			return;
		}
		
		//set layout
		setLayout(new BorderLayout());
		
		//get the list of search terms
//		searchTypes = SearchRegistry.getTypeNames();
		searchTypes.add("Author");
		searchTypes.add("Year");
		searchTypes.add("Journal");
		searchTypes.add("Title");
		
		columnWidths.put("Author", new Integer(80));
		columnWidths.put("Year", new Integer(60));
		columnWidths.put("Journal", new Integer(100));
		columnWidths.put("Title", new Integer(420));
		
		columnAlign.put("Author", new Integer(JLabel.LEFT));
		columnAlign.put("Year", new Integer(JLabel.CENTER));
		columnAlign.put("Journal", new Integer(JLabel.LEFT));
		columnAlign.put("Title", new Integer(JLabel.LEFT));
		
		columnRef.put("Author", authorField);
		columnRef.put("Year", yearField);
		columnRef.put("Journal", journalField);
		columnRef.put("Title", titleField);
		
		//create the table
		m_data = new PubmedResultData(dataSet, searchTypes, columnWidths, columnAlign, currentColumns, currentIndices);
		m_table = new ResultTable();

		m_table.setAutoCreateColumnsFromModel(false);
		m_table.setModel(m_data);
		
		DefaultListSelectionModel selectionModel = new DefaultListSelectionModel();
		selectionModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		m_table.setSelectionModel(selectionModel);
		
		m_table.setRowSelectionAllowed(true);
//		m_table.setCellSelectionEnabled(true);
		
		m_table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
//		m_table.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
//		m_table.setAutoResizeMode(JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);

		TableCellRenderer renderer = null;
		TableCellEditor editor = null;
		columnCount = m_data.m_columns.length;
		
		//take care of the renderers
		for (int k = 0; k < columnCount; k++){
			TableColumn column = null;
			renderer = new ColorRenderer();
			((ColorRenderer)renderer).setHorizontalAlignment(m_data.m_columns[k].m_alignment);
			column = new TableColumn(k, m_data.m_columns[k].m_width, renderer, null);
			
			//set minimum size for each column to be equal the original size
			String tag = m_data.m_columns[k].m_title;
			column.setMinWidth(((Integer)columnWidths.get(tag)).intValue() - 10);//initial width - 10 pixels
			
			final String label = m_data.m_columns[k].m_title;
			columns.put(label, column);
			columnOrder.add(label);
			
			if (defaultColumns.contains(label)){
				m_table.addColumn(column);
				currentColumns.add(column);
				currentIndices.add(new Integer(k));
			}
		}
		
		JTableHeader header = m_table.getTableHeader();
		header.setUpdateTableInRealTime(true);
		header.setReorderingAllowed(false);
		
		header.addMouseListener(m_data.new ColumnListener(m_table));
		
		m_table.getColumnModel().addColumnModelListener(new ColumnMovementListener());
		m_table.setMinimumWidth(m_table.getSize().width);
		
		scrollPane = new JScrollPane(m_table);
		
		scrollBar = scrollPane.getHorizontalScrollBar();
		scrollBar.addAdjustmentListener(new AdjustmentListener( ){
			public void adjustmentValueChanged( AdjustmentEvent ae ) {
				viewOffset = ae.getValue();
			}
		});
		
		//add key listener to the table to respond to Enter key events
		m_table.addKeyListener(new KeyAdapter(){
			public void keyPressed(KeyEvent k){
/*				if (k.getKeyCode() == KeyEvent.VK_ENTER){
					//do the same thing as with OK button
					loadSelection();
					m_table.clearSelection();
				}
*/				if(k.getKeyCode() == KeyEvent.VK_ESCAPE){
					//do the same thing as with Cancel button
					m_table.clearSelection();
				}
			}
		});
		
		
		setVisible(true);

		vp = scrollPane.getViewport();
		
		add(scrollPane, BorderLayout.CENTER);
		
		//go over currently loaded columns and add their corresponding fields
		int offset = 43;//from the left where the label is located (added to all measurements)
		for (int i = 0; i < currentIndices.size(); i++){
			int currentIndex = ((Integer)currentIndices.elementAt(i)).intValue();
			String title = (String)m_data.m_columns[currentIndex].m_title;

			TableColumn current = m_table.getColumnModel().getColumn(i);
			
			if (columnRef.containsKey(title)){
				final JTextField pointer = (JTextField)columnRef.get(title);//reference to the associated field
				
				
				//get the column size
				int w = current.getWidth();
				pointer.setLocation(new Point(offset, 10));
				pointer.setPreferredSize(new Dimension(w-6, 20));
				pointer.setSize(w-6, 20);
				pointer.setBounds(offset, 10, w - 6, 20);
				pointer.setVisible(true);
				
				offset += w;//5 pixel margins
				
			}
		}
		
		//make sure the blank lines are painted if necessary
		if (parentPanel != null){
			resizePanel(parentPanel.getSize());
		}

		setVisible(true);
	}
	
	/**
	 * Removes selection of the rows.
	 */
	public void clearRowSelection(){
		m_table.clearSelection();
	}
	
	public int getSelectedRowCount(){
		return m_table.getSelectedRows().length;
	}
	
	public int[] getSelectedRows(){
		return m_table.getSelectedRows();
	}
	
	public DataBlock getDataBlock(int row){
		return m_data.getDataBlock(row);
	}
	
	public void displayColumn(String title, boolean show){
		
		//show or hide column identified by its title
		TableColumnModel model = m_table.getColumnModel();
		
		//first, remove the columns to reinsert them later in the correct order
		for (int k = 0; k < currentColumns.size(); k++){
			TableColumn c = (TableColumn)currentColumns.elementAt(k);
			model.removeColumn(c);
		}
		
		currentIndices.clear();
		
		//now go over the columns and add the right ones
		for (int k = 0; k < columnOrder.size(); k++){
			String cTitle = (String)columnOrder.elementAt(k);
			TableColumn c = (TableColumn)columns.get(cTitle);
			if ((cTitle).equals(title)){
				if (show){
					currentColumns.add(c);
					currentIndices.add(new Integer(k));
					model.addColumn(c);
					if (m_data.getSortColumn() != k){
						//reset this column's header
						c.setHeaderValue(m_data.getColumnName(c.getModelIndex()));
					}
				}
				else{
					currentColumns.remove(c);
				}
			}
			else{
				if (currentColumns.contains(c)){
					model.addColumn(c);
					currentIndices.add(new Integer(k));
				}
			}
			int[] rows = m_table.getSelectedRows();
			
			m_table.tableChanged(new TableModelEvent(m_data));
			m_table.getTableHeader().repaint();
			m_table.repaint();
			
			for (int i = 0; i < rows.length; i++){
//				System.out.println("Selecting row " + rows[i]);
				m_table.addRowSelectionInterval(rows[i],rows[i]);
			}
		}
	}
	
	public int getColumnCount(){
		return columnCount;
	}
	
	class ColumnMovementListener implements TableColumnModelListener {

		public void columnAdded(TableColumnModelEvent e){
			columnCount++;
		}
		
		public void columnRemoved(TableColumnModelEvent e){
			columnCount--;
		}
		
		public void columnMarginChanged(ChangeEvent e){}
		public void columnMoved(TableColumnModelEvent e){}
		public void columnSelectionChanged(ListSelectionEvent e){}
		
	}
	
	
	/**
	 * Handles repainting of the table panel in response to resizing of the container frame.
	 * Adds filler lines if table height is less than the height of the view.
	 * @param size new size of the frame as a <code>Dimension</code>
	 */
	public void resizePanel(Dimension size){
		
		tableHeight = m_table.getPreferredSize().height;
		int factor = extraHeight;//may need to be increased if the scrollbar is present
        
		//get the dimensions of the viewport
		Dimension extent = vp.getExtentSize();
		if (extent.width < m_table.getPreferredSize().width){
			factor += 15;
		}
		int height = size.height - factor;
		
    	if (height > tableHeight){
			//need to add more blank rows
			//calculate how many
			int rows = ((height - tableHeight)/(m_table.getRowHeight())) + 2;
			for (int i = 0; i < rows; i++){
				m_data.insertBlankRow();
			}
			
			m_table.revalidate();
			tableHeight = m_table.getSize().height;
			m_table.tableChanged(new TableModelEvent(m_data));
			scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);
			m_table.repaint();
		}
		else{
			//there are some blanks
			int blanks = m_data.getBlankRowCount();
			int rows = (tableHeight - height)/(m_table.getRowHeight());

			for (int i = 0; i < rows; i++){
				if (i >= blanks){
					break;
				}
				m_data.removeBlankRow();
			}
			
			if (m_data.getBlankRowCount() == 0){
				scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);	
			}
			
			tableHeight = m_table.getSize().height;
			
			m_table.tableChanged(new TableModelEvent(m_data));
			m_table.repaint();
		}
    	
    	
    	
	}
	
	/**
	 * Method that handles mouse clicks depending on the row where they occurred.
	 */
	public void processClick(int nRow){
		m_table.clearSelection();
//		System.out.println("Received click at row " + nRow);
		//identify id by row number
//		parent.displayInfo((m_data.getValueAt(nRow, 1)).toString());
	}
	
	public void enableTooltips(boolean enable){
		tooltips = enable;
	}
	
	class ColorRenderer extends JLabel implements TableCellRenderer {
	
		private int type = ResultPanel.SEARCH_TABLE;
		protected String tooltip = null;

		protected Graphics2D g = null;
		protected FontMetrics fm = null;
	
		public ColorRenderer(){
			super();
			setOpaque(true);
		
			BufferedImage m_image = new BufferedImage(1, 1, BufferedImage.TYPE_INT_RGB);
			g = m_image.createGraphics();
		}

		public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column){
		
			if (isSelected){
				setBackground(new Color(120,120,255));
			}
			else{
				if ((row%2) == 0){
					if (type == ResultPanel.PATTERN_TABLE){
						setBackground(new Color(210,220,255));
					}
					else if (type == ResultPanel.SEARCH_TABLE){
						setBackground(new Color(225,225,180));
					}
					else if (type == ResultPanel.NAVIGATION_TABLE){
						setBackground(new Color(210,230,230));
					}
					else{
						//default if nothing else matches
						setBackground(new Color(220,220,220));
					}
				}
				else{
					setBackground(Color.white);
				}
			}
			setFont(new Font("Dialog", Font.PLAIN, 12));
			if (value == null){
				value = "null";
			}
			setText(" " + value.toString() + " ");
		
			if (tooltips){
				//get width of the text and the column and see whether it fits
				int columnWidth = table.getColumnModel().getColumn(column).getWidth();
				g.setFont(table.getFont());
				fm = g.getFontMetrics();
				int stringWidth = (int)(fm.stringWidth(value.toString())*1.1);
			
				if (stringWidth > columnWidth){
//					tooltip = "<html><body bgcolor=\"cdd7ed\">&nbsp " + value.toString() + "&nbsp</html>";
					tooltip = "<html><body>&nbsp " + value.toString() + "&nbsp</html>";
				}
				else{
					tooltip = null;
				}
			}
			else{
				tooltip = null;
			}
			return this;
		}
	
		public String getToolTipText(){
				return tooltip;
		}
		

	}


}



