//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs.TableImpl;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import javax.swing.table.*;


/**
 * This class extends AbstractTableModel to allow for custom behavior of the table elements in
 * ResultPanel, such as sorting and filtering.
 * @see edu.sdsc.sirius.search.ResultPanel
 * 
 * @author Oleksandr V. Buzko
 */
class ResultData extends AbstractTableModel {

	public ColumnData m_columns[];
	protected Vector m_vector;//stores the actual data
	protected Vector m_current;
	
	protected int sortColumn = 1;//column used to sort content
	protected boolean sortAsc = true;
	protected Vector currentIndices = null;
	protected Vector currentColumns = null;
	
	protected int blankRowCount = 0;
	protected Vector blankRows = new Vector();
	
	protected HashMap checkedRows = new HashMap();

//	public static final int COL_ID = 0;
	public static final int COL_NAME = 0;
	public static final int COL_DESCRIPTION = 1;
	public static final int COL_EVALUE = 2;
	
	private HashMap replacement = null;
	
	private boolean sort = true;
	
	private Vector hiddenRows = new Vector();//row number -> LineData object
	
	public ResultData(DataSet data, Vector fieldNames, HashMap columnWidths, HashMap columnAlign, Vector col, Vector ind, HashMap replacement){

		int k = fieldNames.size();
		
		this.replacement = replacement;

		m_columns = new ColumnData[k];
		m_vector = new Vector();
		m_current = new Vector();//same as m_vector, except it stores only the currently displayed rows
		
		currentColumns = col;
		currentIndices = ind;
		
		//enter the actual column parameters
		for (int i = 0; i < k; i++){
			String headerName = null;
			String name = (String)fieldNames.elementAt(i);
			if (replacement != null && replacement.get(name) != null){
				headerName = (String)replacement.get(name);
			}
			else{
				headerName = name;
			}
			m_columns[i] = new ColumnData(name, headerName, ((Integer)columnWidths.get(name)).intValue(), ((Integer)columnAlign.get(name)).intValue());
		}
		
///		System.out.println("data = " + data.size());

		//assign the actual data - convert DataSet into a Vector of LineData
		for (int i = 0; i < data.size(); i++){
			DataBlock block = data.getData(i);
			try{
				LineData temp = null;
				
				//check for generic data representation
				if (block.fields.size() == 0){
					temp = new LineData();
					temp.data = block;
					temp.id = block.getId();
					temp.name = block.getName();
					temp.bold = block.getBold();
					temp.pending = block.getPending();
					temp.description = block.getDescription();
					if (temp.description == null || temp.description.equals("null")){
						temp.description = " - ";
					}
					temp.evalue = block.getEvalue();
					temp.marked = block.marked;
				}
				else{
					temp = new LineData(block.fields.size());
					for (int j = 0; j < fieldNames.size(); j++){
						String value = (String)block.fields.get(fieldNames.get(j));
						if (value == null || value.equals("null")){
							value = "";
						}
						temp.values[j] = value;
					}
					temp.marked = block.marked;
				}
				
				m_vector.add(temp);
				m_current.add(temp);
			}
			catch (Exception ee){
				ee.printStackTrace();
			}

		}
		
		if (sort) Collections.sort(m_vector, new ResultComparator(sortColumn, sortAsc, currentIndices));
	}
	
	public void setSortingMode(boolean sort){
		this.sort = sort;
	}

	public int getRowCount(){
		return m_current==null ? 0 : m_current.size();
	}
	
	public int getHiddenRowCount(){
		return hiddenRows.size();
	}
	
	public int getSubsetRowCount(){
		//the currently displayed subset - blank lines
		int count = m_current==null ? 0 : m_current.size();
		count -= blankRowCount;
		return count;
	}
	
	public int getFullRowCount(){
		int count = m_vector==null ? 0 : m_vector.size();
		count -= blankRowCount;
		return count;
	}

	public int getColumnCount(){
		return m_columns.length;
	}

	public int getSortColumn(){
		return sortColumn;
	}
	
	public HashMap getCheckedRows(){
		return checkedRows;
	}

	public boolean isCellEditable(int nRow, int nColumn){
		return false;
	}
	
	public void insertBlankRow(){
		LineData blank = new LineData();
		m_vector.add(blank);
		m_current.add(blank);

		blankRows.add(blank);
		blankRowCount++;
	}
	
	public void removeBlankRow(){
		if (blankRowCount == 0){
//			System.out.println("No blanks to remove");
			return;
		}
		LineData line = (LineData)blankRows.lastElement();
		if (blankRows.contains(line)){
			m_vector.remove(line);
			m_current.remove(line);
			blankRows.remove(line);
			blankRowCount--;
		}
	}
	
	public int getBlankRowCount(){
		return blankRowCount;
	}
	
	public void displayLine(LineData data, int index){
		if (hiddenRows.contains(data)){
			//show it
			hiddenRows.remove(data);
			m_current.insertElementAt(data, index);
		}
	}
	
	public void hideLine(LineData data){
		if (!hiddenRows.contains(data)){
			hiddenRows.add(data);
			m_current.remove(data);
		}
	}
	
	public void resort(){
		Collections.sort(m_current, new ResultComparator(sortColumn, sortAsc, currentIndices));
		
	}
	
	public Object getValueAt(int nRow, int nColumn){

		if (nRow < 0 || nRow > getRowCount()){
			return "";
		}
		
		LineData data = (LineData)m_current.elementAt(nRow);
		if (blankRows.contains(data)){
			return " ";
		}
		
		if (data.generic){
			return data.values[nColumn];
		}
		
		switch (nColumn) {
//			case COL_ID:
//				if (data.id == null) return "";
//				return data.id;
			case COL_NAME:
				if (data.name == null) return "";
				return data.name;
			case COL_DESCRIPTION:
				if (data.description == null) return "";
				return data.description;
			case COL_EVALUE:
				if (data.evalue == null) return "";
				return data.evalue;
		}
		return "";
	}
	
	public Object getFullValueAt(int nRow, int nColumn){

		if (nRow < 0 || nRow > getFullRowCount()){
			return "";
		}
		
		LineData data = (LineData)m_vector.elementAt(nRow);
		if (blankRows.contains(data)){
			return " ";
		}
		
		if (data.generic){
			return data.values[nColumn];
		}
		
		switch (nColumn) {
//			case COL_ID:
//				if (data.id == null) return "";
//				return data.id;
			case COL_NAME:
				if (data.name == null) return "";
				return data.name;
			case COL_DESCRIPTION:
				if (data.description == null) return "";
				return data.description;
			case COL_EVALUE:
				if (data.evalue == null) return "";
				return data.evalue;
		}
		return "";
	}
	
	
	public void setValueAt(Object value, int nRow, int nCol) {
	
		if (nRow < 0 || nRow>=getRowCount() || value == null){
			return;
		}
		
		LineData data = (LineData)m_current.elementAt(nRow);
		if (blankRows.contains(data)) return;
		
		if (data.generic){
			data.values[nCol] = (String)value;
			return;
		}
		
		switch (nCol) {
//			case COL_ID:
//				data.id = (String)value;
			case COL_NAME:
				data.name = (String)value;
			case COL_DESCRIPTION:
				data.description = (String)value;
			case COL_EVALUE:
				data.evalue = (String)value;
		}	
	}
	
	public void insert(LineData d, int nRow) {
	
		if (nRow < 0){
			nRow = 0;
		}
		if (nRow > m_vector.size()){
			nRow = m_vector.size();
		}
		m_vector.insertElementAt(d, nRow);

	}
	
	public boolean delete(int nRow){
	
		if (nRow < 0 || nRow >= m_vector.size()) return false;
		m_vector.remove(nRow);
		return true;
	}
			
	public Vector getData(){
		return m_vector;
	}
	
	public LineData getRow(int nRow){
		return (LineData)m_vector.elementAt(nRow);
	}
	
	
	public String getTitle(){
		return "PKR search: result set";
	}
	
	public String getColumnName(int column){
		String str = m_columns[column].getHeaderTitle();
		if (sort){
			if (column == sortColumn){
				str += sortAsc ? " >>" : " <<";
			}
		}
		return str;
	}
	
	class ColumnListener extends MouseAdapter {

		protected JTable m_table = null;
		
		public ColumnListener(JTable table){
			m_table = table;
		}
		
		public void mouseClicked(MouseEvent e){
		
			TableColumnModel model = m_table.getColumnModel();
			int columnModelIndex = model.getColumnIndexAtX(e.getX());
			int modelIndex = model.getColumn(columnModelIndex).getModelIndex();
			
			if (modelIndex < 0) return;			
			if (sortColumn == modelIndex){
				sortAsc = !sortAsc;
			}
			else{
				sortColumn = modelIndex;
				sortAsc = true;//to make sure the new column is first sorted in ascending order
			}
	
			for (int i = 0; i < currentColumns.size(); i++){
				TableColumn column = (TableColumn)currentColumns.elementAt(i);
				column.setHeaderValue(getColumnName(column.getModelIndex()));
			}
			
			m_table.getTableHeader().repaint();
			
			Collections.sort(m_current, new ResultComparator(modelIndex, sortAsc, currentIndices));
			m_table.tableChanged(new TableModelEvent(ResultData.this));
			m_table.repaint();
		}
	}
	
	class ResultComparator implements Comparator {
	
		protected int m_sortCol;
		protected boolean m_sortAsc;
		protected Vector currentIndices;
		
		public ResultComparator(int sortCol, boolean sortAsc, Vector curInd){
			m_sortCol = sortCol;
			m_sortAsc = sortAsc;
			currentIndices = curInd;
		}
		
		public int compare(Object o1, Object o2){
		
			if (!(o1 instanceof LineData) || !(o2 instanceof LineData)){
				return 0;
			}
			
			//check whether the objects are blanks that should not be compared
			if ((blankRows.contains(o1))&&(blankRows.contains(o2))){
				return 0;
			}
			else if (blankRows.contains(o1)){
				return 1;
			}
			else if (blankRows.contains(o2)){
				return -1;
			}
						
			LineData v1 = (LineData)o1;
			LineData v2 = (LineData)o2;
			
			int result = 0;
			try{
				switch (m_sortCol){
					case COL_NAME:	//name
						if (currentIndices.contains(new Integer(COL_NAME))){
							if (v1.name == null && v2.name == null){
								return 0;
							}
							else if (v1.name == null){
								return 1;
							}
							else if (v2.name == null){
								return -1;
							}
			
							String name1 = v1.name.trim().toUpperCase();
							String name2 = v2.name.trim().toUpperCase();
							if (name1.equals("NULL")){
								return 1;
							}
							else if (name2.equals("NULL")){
								return -1;
							}
							else if (name1.equals("NULL") && name2.equals("NULL")){
								return 0;
							}
							result = name1.compareTo(name2);
							break;
						}
						else{
							return 0;
						}
					case COL_DESCRIPTION:	//description
						if (currentIndices.contains(new Integer(COL_DESCRIPTION))){
							String d1 = v1.description.toUpperCase();
							String d2 = v2.description.toUpperCase();
						
							result = d1.compareTo(d2);
							break;
						}
						else{
							return 0;
						}
					case COL_EVALUE:
						if (currentIndices.contains(new Integer(COL_EVALUE))){
							String id1 = v1.id;
							String id2 = v2.id;
					
							double n1 = Double.parseDouble(id1);
							double n2 = Double.parseDouble(id2);
							result = n1<n2 ? -1 : (n1>n2 ? 1 : 0);
							break;
						}
						else{
							return 0;
						}
				}
				
				if (!sortAsc){
					result = -result;
				}
				
			}
			catch (NumberFormatException ex){
				return 0;
			}
			catch (NullPointerException e){
				return 0;
			}
			
			return result;
		}
		
		public boolean equals(Object obj){
			if (obj instanceof ResultComparator){
				ResultComparator compObj = (ResultComparator)obj;
				return (compObj.m_sortCol == m_sortCol)&&(compObj.m_sortAsc == m_sortAsc);
			}
			return false;
		}
	
	}
}
