//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs.TableImpl;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;

import javax.swing.table.*;
import java.awt.print.*;
import java.awt.image.*;

import edu.sdsc.sirius.util.Manager;
import edu.sdsc.sirius.util.PagePrintable;

/**
 * This class implements a panel used to display sequence data retrieved from a database. Each entry
 * is contained within a DataBlock as a part of a DataSet.
 * This class permits extended functionality including sorting, dynamic adding and removal of columns,
 * filtering of entries by content of any of the fields, custom buttons to display entry-specific information, 
 * as well as checkboxes in each line to trigger loading of the corresponding entry.
 * 
 * @see edu.sdsc.mbt.util.DataBlock
 * @see edu.sdsc.mbt.DataSet
 * @see edu.sdsc.mbt.util.Entry
 * 
 * @author Oleksandr V. Buzko
 */
public class ResultPanel extends JPanel {
	
	//different types of tables that differ in their background colors, etc.
	public static final int SEARCH_TABLE = 0;
	public static final int PATTERN_TABLE = 1;
	public static final int NAVIGATION_TABLE = 2;
	
	private int tableType = SEARCH_TABLE;

	private DataSet dataSet;
	private JFrame parentFrame;
	private JDialog parentDialog;
	
	private JScrollPane scrollPane;
	private JScrollBar scrollBar;
	private JViewport vp;
	protected int viewOffset = 0;//offset in pixels when part of the table is scrolled out of the view
	private JPanel filterPanel = new JPanel();
	
	private ResultData m_data;
	private ResultTable m_table;
	private int columnCount = 0;
	private HashMap columns = new HashMap();
	private Vector columnOrder = new Vector();
	private Vector currentColumns = new Vector();
	public Vector currentIndices = new Vector();//stores indices of columns that are displayed
	
	private Vector searchTypes = new Vector();
	
	public HashMap columnWidths = new HashMap();
	public HashMap columnAlign = new HashMap();
	
	private TableColumn infoColumn;
	private TableColumn checkColumn;
	
	public static Vector defaultColumns = new Vector();
		
	protected int extraHeight = 100;//everything in the frame other than the table itself
	protected int tableHeight = 0;
	
	protected int[] selectedRows = { -1 };
	
//	protected RequestSet requestSet = null;
	
	protected Manager parent;
	
	//filter fields
	private JLabel filterLabel = new JLabel("Filter:");
	private JTextField idField = new JTextField(10);
	private JTextField nameField = new JTextField(20);
	private JTextField infoField = new JTextField(30);
	
	private JButton loadButton = new JButton("Load");
		
	private Book book;
	public PageFormat mPageFormat;
	
	private boolean tooltips = false;
	private boolean commonSelection = false;
	private String target;
	
	//this is used to display different types of data whose format and sorting are identical to
	//those defined by default, e.g. name -> pdb_code
	private HashMap replacement;//default column title -> new column title
	
	private HashMap columnRef = new HashMap();//contains references from fields to the corresponding table titles
	
	static{
	
		defaultColumns.add("Entry ID");
		defaultColumns.add("Name");
		defaultColumns.add("Description");
	}
	
	/**
	 * Constructs the <code>ResultPanel</code> when created as in a frame typically storing search results.
	 * @param p parent frame
	 * @param data <code>DataSet</code> to initialize the table
	 * @param callable Manager object that will handle entry loading
	 * @param pp 
	 */
	public ResultPanel(JFrame p, DataSet data, Manager callable, int type){
		parentFrame = p;
		dataSet = data;
		parent = callable;
		tableType = type;
		initialize();
	}
	
	/**
	 * Constructs the <code>ResultPanel</code> when created as in a dialog typically as a browsing and loading interface.
	 * @param p parent frame
	 * @param d <code>DataSet</code> to initialize the table
	 * @param pa Manager object that will handle entry loading
	 */
	public ResultPanel(JDialog jd, DataSet d, Manager pa, int type){
		
		parentDialog = jd;
		dataSet = d;
		parent = pa;
		tableType = type;
	
		extraHeight += 40;//the buttons
		initialize();
	}
	
	
	public ResultPanel(JDialog jd, DataSet d, Manager pa, int type, Vector columns, HashMap replacement){
		
		parentDialog = jd;
		dataSet = d;
		parent = pa;
		tableType = type;
		this.replacement = replacement;
		
		extraHeight += 40;//the buttons
		if (columns != null){
			defaultColumns = columns;
		}
		initialize();
	}

	/**
	 * Initialization code common to both constructors.
	 */
	private void initialize(){
		
		if (dataSet == null){
			return;
		}

		setBackground(Color.YELLOW);
		//set layout
		setLayout(new BorderLayout());
		
		//get the list of search terms
//		searchTypes = SearchRegistry.getTypeNames();
		searchTypes.add("Entry ID");
		searchTypes.add("Name");
		searchTypes.add("Description");
		
		columnWidths.put("Entry ID", new Integer(80));
		columnWidths.put("Name", new Integer(60));
		columnWidths.put("Description", new Integer(220));
		
		columnAlign.put("Entry ID", new Integer(JLabel.LEFT));
		columnAlign.put("Name", new Integer(JLabel.LEFT));
		columnAlign.put("Description", new Integer(JLabel.LEFT));
		
		columnRef.put("Entry ID", idField);
		columnRef.put("Name", nameField);
		columnRef.put("Description", infoField);
		
		//create the table
		m_data = new ResultData(dataSet, searchTypes, columnWidths, columnAlign, currentColumns, currentIndices, replacement);
		m_table = new ResultTable();
		m_table.setBackground(Color.RED);
		
		m_table.addKeyListener(new KeyAdapter(){
			public void keyPressed(KeyEvent e){
				char key = e.getKeyChar();
				if (key == 'f'){
					String input = (String)JOptionPane.showInputDialog(null, "Enter search string", 
							"Entry search", JOptionPane.INFORMATION_MESSAGE, null, null, "");
					
					//walk through the entries and highlight the hits
					for (int i = 0; i < dataSet.size(); i++){
						DataBlock b = dataSet.getData(i);
						if (b.getDescription().startsWith(input)){
							m_table.addRowSelectionInterval(i,i);
							break;
						}
					}
					m_table.revalidate();
					m_table.repaint();
				}
			}
		});


		m_table.setAutoCreateColumnsFromModel(false);
		m_table.setModel(m_data);
		
		DefaultListSelectionModel selectionModel = new DefaultListSelectionModel();
		//TODO a little hack - we take it to be a loading dialog if replacements are present
		if (replacement == null){
			selectionModel.setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
		}
		else{
			selectionModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		}
		m_table.setSelectionModel(selectionModel);
		
		m_table.setRowSelectionAllowed(true);
//		m_table.setCellSelectionEnabled(true);
		
//		m_table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		m_table.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
//		m_table.setAutoResizeMode(JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);

		TableCellRenderer renderer = null;
		TableCellEditor editor = null;
		columnCount = m_data.m_columns.length;
		
		//take care of the renderers
		for (int k = 0; k < columnCount; k++){
			TableColumn column = null;
			renderer = new ColorRenderer(tableType);
			((ColorRenderer)renderer).setHorizontalAlignment(m_data.m_columns[k].m_alignment);
			column = new TableColumn(k, m_data.m_columns[k].m_width, renderer, null);
			
			//set minimum size for each column to be equal the original size
			String tag = m_data.m_columns[k].m_title;
			column.setMinWidth(((Integer)columnWidths.get(tag)).intValue() - 10);//initial width - 10 pixels
			
			final String label = m_data.m_columns[k].m_title;
			columns.put(label, column);
			columnOrder.add(label);
			
			if ((defaultColumns.contains(label))||(infoColumn == column)||(checkColumn == column)){
				m_table.addColumn(column);
				currentColumns.add(column);
				currentIndices.add(new Integer(k));
			}
		}
		
		JTableHeader header = m_table.getTableHeader();
		header.setUpdateTableInRealTime(true);
		header.setReorderingAllowed(false);
		
//		header.addMouseListener(m_data.new ColumnListener(m_table));
		
		m_table.getColumnModel().addColumnModelListener(new ColumnMovementListener());
		m_table.setMinimumWidth(m_table.getSize().width);
		
		scrollPane = new JScrollPane(m_table);
		
		m_table.setSize(new Dimension(200,200));
		
		scrollBar = scrollPane.getHorizontalScrollBar();
		scrollBar.addAdjustmentListener(new AdjustmentListener( ){
			public void adjustmentValueChanged( AdjustmentEvent ae ) {
				viewOffset = ae.getValue();
				paintFilterPanel();
			}
		});
		
		//add key listener to the table to respond to Enter key events
		m_table.addKeyListener(new KeyAdapter(){
			public void keyPressed(KeyEvent k){
/*				if (k.getKeyCode() == KeyEvent.VK_ENTER){
					//do the same thing as with OK button
					loadSelection();
					m_table.clearSelection();
				}
*/				if(k.getKeyCode() == KeyEvent.VK_ESCAPE){
					//do the same thing as with Cancel button
					m_table.clearSelection();
				}
			}
		});
		
		
		//printing stuff
		PrinterJob pj = PrinterJob.getPrinterJob();
		mPageFormat = pj.defaultPage();
		mPageFormat.setOrientation(PageFormat.LANDSCAPE);
		pj = null;

		setVisible(true);

		vp = scrollPane.getViewport();
		
		add(scrollPane, BorderLayout.CENTER);
		
		//take care of the filter panel if necessary
		if (tableType != NAVIGATION_TABLE){
			filterPanel.setLayout(null);
			if (parentFrame != null){
				filterPanel.setPreferredSize(new Dimension(parentFrame.getSize().width, 35));
				filterPanel.setSize(parentFrame.getSize().width, 35);
			}
			else{
				filterPanel.setPreferredSize(new Dimension(parentDialog.getSize().width, 35));
				filterPanel.setSize(parentDialog.getSize().width, 35);
			}
			filterPanel.setVisible(true);
			
			filterLabel.setLocation(new Point(3,10));
			filterLabel.setFont(new Font("Dialog", Font.PLAIN, 12));
			filterLabel.setPreferredSize(new Dimension(40,20));
			filterLabel.setBounds(3,10,40,20);
			filterLabel.setVisible(true);
			
			filterPanel.add(filterLabel);
			
			loadButton.setMargin(new Insets(0,1,0,1));
	    	loadButton.setFont(new Font("Dialog", Font.PLAIN, 10));
/*	    		loadButton.addActionListener(new ActionListener(){
	    			public void actionPerformed(ActionEvent e){
	    			
	    				loadSelection();
	    			
	    			}
	    		});
*/	    		
	    		//add key listeners to every filter field
	    	addFieldListeners();
	
			//go over currently loaded columns and add their corresponding fields
			int offset = 43;//from the left where the label is located (added to all measurements)
			for (int i = 0; i < currentIndices.size(); i++){
				int currentIndex = ((Integer)currentIndices.elementAt(i)).intValue();
				String title = (String)m_data.m_columns[currentIndex].m_title;
	
				TableColumn current = m_table.getColumnModel().getColumn(i);
				
				if (columnRef.containsKey(title)){
					final JTextField pointer = (JTextField)columnRef.get(title);//reference to the associated field
					
					
					//get the column size
					int w = current.getWidth();
					pointer.setLocation(new Point(offset, 10));
					pointer.setPreferredSize(new Dimension(w-6, 20));
					pointer.setSize(w-6, 20);
					pointer.setBounds(offset, 10, w - 6, 20);
					pointer.setVisible(true);
					
					offset += w;//5 pixel margins
					
					filterPanel.add(pointer);
				}
				if (title.equals("Clear")){
				
					//it's the last column
					//get the column size
					int w = current.getWidth();
					loadButton.setLocation(new Point(offset, 5));
					
					//check if the vertical scrollbar is visible and adjust button width
					
					loadButton.setPreferredSize(new Dimension(w+10, 20));
					loadButton.setBounds(offset, 10, w+10, 20);
					loadButton.setVisible(true);
					
					if (parentFrame != null){
						filterPanel.add(loadButton);
					}
				}
			}
			
//			add(filterPanel, BorderLayout.SOUTH);
		}
		else{
			extraHeight -= 35;//the height difference due to the filter panel
		}
		
		//make sure the blank lines are painted if necessary
		if (parentFrame != null){
			resizePanel(parentFrame.getSize());
		}
		else if (parentDialog != null){
			resizePanel(parentDialog.getSize());
		}
		else{
			this.getSize();
		}
		
		
		setVisible(true);
	}
	
	public void setSelection(boolean selected, int row){
		LineData line = (LineData)m_data.getRow(row);
		line.selected = selected;
		repaint();
	}
	
	/**
	 * Removes selection of the rows.
	 */
	public void clearRowSelection(){
		m_table.clearSelection();
	}
	
	public int getSelectedRowCount(){
		return m_table.getSelectedRows().length;
	}
	
	public int[] getSelectedRows(){
		return m_table.getSelectedRows();
	}
	
	public void displayColumn(String title, boolean show){
		
		//show or hide column identified by its title
		TableColumnModel model = m_table.getColumnModel();
		
		//first, remove the columns to reinsert them later in the correct order
		for (int k = 0; k < currentColumns.size(); k++){
			TableColumn c = (TableColumn)currentColumns.elementAt(k);
			model.removeColumn(c);
		}
		
		currentIndices.clear();
		
		//now go over the columns and add the right ones
		for (int k = 0; k < columnOrder.size(); k++){
			String cTitle = (String)columnOrder.elementAt(k);
			TableColumn c = (TableColumn)columns.get(cTitle);
			if ((cTitle).equals(title)){
				if (show){
					currentColumns.add(c);
					currentIndices.add(new Integer(k));
					model.addColumn(c);
					if (m_data.getSortColumn() != k){
						//reset this column's header
						c.setHeaderValue(m_data.getColumnName(c.getModelIndex()));
					}
				}
				else{
					currentColumns.remove(c);
				}
			}
			else{
				if (currentColumns.contains(c)){
					model.addColumn(c);
					currentIndices.add(new Integer(k));
				}
			}
			int[] rows = m_table.getSelectedRows();
			
			m_table.tableChanged(new TableModelEvent(m_data));
			m_table.getTableHeader().repaint();
			m_table.repaint();
			
			for (int i = 0; i < rows.length; i++){
//				System.out.println("Selecting row " + rows[i]);
				m_table.addRowSelectionInterval(rows[i],rows[i]);
			}
		}
	}
	
	public int getColumnCount(){
		return columnCount;
	}
	
	class ColumnMovementListener implements TableColumnModelListener {

		public void columnAdded(TableColumnModelEvent e){
			columnCount++;
		}
		
		public void columnRemoved(TableColumnModelEvent e){
			columnCount--;
		}
		
		public void columnMarginChanged(ChangeEvent e){
			paintFilterPanel();
		}
		public void columnMoved(TableColumnModelEvent e){}
		public void columnSelectionChanged(ListSelectionEvent e){}
		
	}
	
	
	private void paintFilterPanel(){
		
		filterPanel.removeAll();
		
		//use viewOffset to shift the painted elements accordingly
		
		//go over currently loaded columns and add their corresponding fields
		int offset = 3 - viewOffset;//from the left where the label is located (added to all measurements)
		filterLabel.setLocation(new Point(offset, 10));
		filterLabel.setPreferredSize(new Dimension(40,20));
		filterLabel.setBounds(offset,10,40,20);
		filterLabel.setVisible(true);
		
		filterPanel.add(filterLabel);
		
		offset += 40;
		
		for (int i = 0; i < currentIndices.size(); i++){
			int currentIndex = ((Integer)currentIndices.elementAt(i)).intValue();
			String title = (String)m_data.m_columns[currentIndex].m_title;

			TableColumn current = m_table.getColumnModel().getColumn(i);
			
			if (columnRef.containsKey(title)){
				final JTextField pointer = (JTextField)columnRef.get(title);//reference to the associated field
				
				
				//get the column size
				int w = current.getWidth();
				pointer.setLocation(new Point(offset, 10));
				pointer.setPreferredSize(new Dimension(w-4, 20));
				pointer.setSize(w-6, 20);
				pointer.setBounds(offset, 10, w-6, 20);
				pointer.setVisible(true);
				
				offset += w;
				filterPanel.add(pointer);
			}
			if (title.equals("Clear")){
				//it's the last column
				//get the column size
				int w = current.getWidth();
				
				loadButton.setLocation(new Point(offset, 10));
				loadButton.setPreferredSize(new Dimension(w, 20));
				loadButton.setBounds(offset, 10, w, 20);
				loadButton.setVisible(true);
				
				if (parentDialog == null){
					filterPanel.add(loadButton);
				}
			}
		}
		filterPanel.repaint();
	}
	
	/**
	 * Handles repainting of the table panel in response to resizing of the container frame.
	 * Adds filler lines if table height is less than the height of the view.
	 * @param size new size of the frame as a <code>Dimension</code>
	 */
	public void resizePanel(Dimension size){
		
		tableHeight = m_table.getPreferredSize().height;
		int factor = extraHeight;//may need to be increased if the scrollbar is present
        
		//get the dimensions of the viewport
		Dimension extent = vp.getExtentSize();
		if (extent.width < m_table.getPreferredSize().width){
			factor += 15;
		}
		int height = size.height - factor;
		
    	if (height > tableHeight){
			//need to add more blank rows
			//calculate how many
			int rows = ((height - tableHeight)/(m_table.getRowHeight())) + 2;
			for (int i = 0; i < rows; i++){
				m_data.insertBlankRow();
			}
			
			m_table.revalidate();
			tableHeight = m_table.getSize().height;
			m_table.tableChanged(new TableModelEvent(m_data));
			scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER);
			m_table.repaint();
		}
		else{
			//there are some blanks
			int blanks = m_data.getBlankRowCount();
			int rows = (tableHeight - height)/(m_table.getRowHeight());

			for (int i = 0; i < rows; i++){
				if (i >= blanks){
					break;
				}
				m_data.removeBlankRow();
			}
			
			if (m_data.getBlankRowCount() == 0){
				scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);	
			}
			
			tableHeight = m_table.getSize().height;
			
			m_table.tableChanged(new TableModelEvent(m_data));
			m_table.repaint();
		}
    	
    	
    	
	}
	
	private boolean testLine(LineData line){
		
		//this method tests whether the given line should be displayed withg currently shown columns
		//and any field values

		//first, get the content of the line in question
		String idContent = line.id;
		String nameContent = line.name;
		String infoContent = line.description;
		
		//now go through the map
		if (currentIndices.contains(new Integer(1))){
			
			String id = idField.getText().toUpperCase();
			if (id.length() != 0){

				//remove zero padding from the content to allow simple number searching
				char[] set = idContent.toCharArray();
				StringBuffer buffer = new StringBuffer();
				boolean past = false;
				for (int i = 0; i < set.length; i++){
					if (!past){
						if (set[i] != '0'){
							past = true;
							buffer.append(set[i]);
						}
					}
					else{
						buffer.append(set[i]);
					}
				}
				
				String alt = buffer.toString();
				
				if (!idContent.startsWith(id) && !alt.startsWith(id)){
					return false;
				}
			}
		
		}
		if (currentIndices.contains(new Integer(2))){
			String name = nameField.getText();
			if (!name.equals("")){
				if (nameContent != null){
					if ((nameContent.toUpperCase()).indexOf(name.toUpperCase()) == -1){
						return false;
					}
					if (nameContent.startsWith("null")){
						return false;
					}
				}
				else{
					return false;
				}
			}
		}
		if (currentIndices.contains(new Integer(3))){
			String info = infoField.getText();
			if (info.equals("null")){
				if ((infoContent.toUpperCase()).indexOf(info.toUpperCase()) == -1){
					return false;
				}
			}
			else{
				if (infoContent.toUpperCase().indexOf(info.toUpperCase()) == -1){
					return false;
				}
			}
		}
		
		return true;
	
	}
	
	public void updateRow(int row){
		LineData line = m_data.getRow(row);
		line.updateData();
	}
	
	private void processField(){
	
		HashMap rows = new HashMap();//row -> boolean
		Vector order = new Vector();
		
		//now go through the rows and see which ones don't match this text
		for (int i = 0; i < m_data.getFullRowCount(); i++){
			LineData line = m_data.getRow(i);
			if (testLine(line)){
				rows.put(line, new Boolean(true));
				order.add(line);
			}
			else{
				rows.put(line, new Boolean(false));
				order.add(line);
			}
		}
		
		//run the rows
		//in the process, record what index each of the lines will have in the subset
		int index = 0;
		for (int j = 0; j < order.size(); j++){
			LineData d = (LineData)order.elementAt(j);
			boolean flag = ((Boolean)rows.get(d)).booleanValue();
			if (flag){
				m_data.displayLine(d, index);
				index++;
			}
			else{
				m_data.hideLine(d);
			}
		}
//		System.out.println("index = " + index);
		if (parentFrame != null){
			((ResultPanelCallable)parentDialog).setNumberOfHits(index);
		}
		else{
//			System.out.println(parentDialog);
			((ResultPanelCallable)parentDialog).setNumberOfHits(index);
		}
		m_data.resort();
		
		//force blank row generation if necessary
		if (parentFrame != null){
			resizePanel(parentFrame.getSize());
		}
		else{
			resizePanel(parentDialog.getSize());
		}
		m_table.tableChanged(new TableModelEvent(m_data));
		m_table.repaint();
		
	}
	
	private void addFieldListeners(){
	
		idField.addKeyListener(new KeyAdapter(){
			public void keyReleased(KeyEvent e){
				processField();
			}
		});

		nameField.addKeyListener(new KeyAdapter(){
			public void keyReleased(KeyEvent e){
				processField();
			}
		});

		infoField.addKeyListener(new KeyAdapter(){
			public void keyReleased(KeyEvent e){
				processField();
			}
		});
		
	}
	
	/**
	 * Method that handles mouse clicks depending on the row where they occurred.
	 */
	public void processClick(int nRow){
		m_table.clearSelection();
		m_table.repaint();
//		System.out.println("Received click at row " + nRow);
		//identify id by row number
//		parent.displayInfo((m_data.getValueAt(nRow, 1)).toString());
	}
	
	public void printSetupHandler(){

		//this method is called from Viewer when Print Setup is selected
		//currently active frame is found and this method is called

		Thread t = new Thread(){
			public void run(){
				PrinterJob pj = PrinterJob.getPrinterJob();
				mPageFormat = pj.pageDialog(mPageFormat);
			}
		};
		t.start();
	}
	
	
	public void printData() {

		//use the current page format to get the printable area
		PrinterJob pj = PrinterJob.getPrinterJob();
		
		//set the cursor, 'cause the wait will be long
		setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		book = new Book();
		
		int pageWidth = 0;
		int pageHeight = 0;
		if (mPageFormat.getOrientation() == PageFormat.PORTRAIT) {
			pageWidth = (int)mPageFormat.getImageableWidth();
			pageHeight = (int)mPageFormat.getImageableHeight();
		}
		else {
			pageWidth = (int)mPageFormat.getImageableWidth();
			pageHeight = (int)mPageFormat.getImageableHeight();
		}
		
		boolean newPage = true;
		BufferedImage m_image = null;
		Graphics2D g = null;
		FontMetrics fm = null;
		int y = 20;//start a bit lower
		
		int imageWidth = (int)(pageWidth*1.7);
		int imageHeight = (int)(pageHeight*1.7);
		int nRow = 0;
		while (true){
			
			if (newPage){
				m_image = new BufferedImage(imageWidth, imageHeight, BufferedImage.TYPE_INT_RGB);
				g = m_image.createGraphics();
//				g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				newPage = false;
				g.setColor(Color.white);
				g.fillRect(0,0, imageWidth, imageHeight);
				
				//draw the header
				g.setColor(Color.black);
				Font headerFont = m_table.getFont().deriveFont(Font.BOLD);
//				int size = headerFont.getSize();
//				System.out.println("header font = " + size);
				g.setFont(headerFont);
				fm = g.getFontMetrics();

				TableColumnModel colModel = m_table.getColumnModel();
				int nColumns = colModel.getColumnCount();
				int x[] = new int[nColumns];
				x[0] = 5;
        
				int h = fm.getAscent();
				y += h;//add ascent of header font
 
				for (int nCol=0; nCol < nColumns; nCol++){
					TableColumn tk = colModel.getColumn(nCol);
					int width = tk.getWidth();
					if (x[nCol] + width > imageWidth) {
						nColumns = nCol;
						break;
					}
					if (nCol+1<nColumns){
						x[nCol+1] = x[nCol] + width;
					}
					String title = (String)tk.getIdentifier();
					g.drawString(title, x[nCol], y);
				}
				
				y += h*2;
			}
			
			//start iterating over the table and drawing the data on one line
			g.setFont(m_table.getFont());
			fm = g.getFontMetrics();
			int h = fm.getHeight();
			int lineX = 5;
			for (int nCol=0; nCol < m_table.getColumnCount(); nCol++) {
				TableColumn tk = m_table.getColumnModel().getColumn(nCol);
				int width = tk.getWidth();
				int col = m_table.getColumnModel().getColumn(nCol).getModelIndex();
				Object obj = m_data.getValueAt(nRow, col);
				String str = null;
				if (obj == null){
					str = "null";
				}
				else{
					str = obj.toString();
				}

				int w = fm.stringWidth(str);
				if (w > (width - 5)){
					g.drawString(str, lineX, y);
					//and cover the rest of the string with a white rectangle
					g.setColor(Color.white);
					g.fillRect(lineX+width-7, y-h, w, h+2);
					g.setColor(Color.black);
				}
				else{
					g.drawString(str, lineX, y);
				}
				lineX += width;
			}
			y += h;
			
			if (y > imageHeight){
				PagePrintable p = new PagePrintable(m_image, pageWidth, pageHeight, imageWidth, imageHeight);
				book.append(p, mPageFormat);
//				savePrint(m_image, "1");
					
				//create a new image
				m_image = new BufferedImage(imageWidth, imageHeight, BufferedImage.TYPE_INT_RGB);
				g = m_image.createGraphics();
				g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				newPage = false;
				g.setColor(Color.white);
				g.fillRect(0,0, imageWidth, imageHeight);
				
				y = 20;
				//draw the header
				g.setColor(Color.black);
				Font headerFont = m_table.getFont().deriveFont(Font.BOLD);
				g.setFont(headerFont);
				fm = g.getFontMetrics();

				TableColumnModel colModel = m_table.getColumnModel();
				int nColumns = colModel.getColumnCount();
				int x[] = new int[nColumns];
				x[0] = 5;
        
				y += fm.getAscent();//add ascent of header font
 
				for (int nCol=0; nCol < nColumns; nCol++){
					TableColumn tk = colModel.getColumn(nCol);
					int width = tk.getWidth();
					if (x[nCol] + width > imageWidth) {
						nColumns = nCol;
						break;
					}
					if (nCol+1<nColumns){
						x[nCol+1] = x[nCol] + width;
					}
					String title = (String)tk.getIdentifier();
					g.drawString(title, x[nCol], y);
				}
				
				y += h;
			}
			
			nRow++;
			if (nRow > m_table.getRowCount()-1){
				PagePrintable p = new PagePrintable(m_image, pageWidth, pageHeight, imageWidth, imageHeight);
				book.append(p, mPageFormat);
//				savePrint(m_image, "2");
				break;
			}
			
			
		}
		
		pj.setPageable(book);
		if (pj.printDialog()){
			try{
				pj.print();
			}
			catch (PrinterException pe){
				System.out.println("Exception " + pe);
				JOptionPane.showMessageDialog(this, "Printing error: " + pe, "Error", JOptionPane.ERROR_MESSAGE);
			}
		}
			
		setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		book = null;
	}
	
	public void enableTooltips(boolean enable){
		tooltips = enable;
	}
	
	public void enableCommonSelection(boolean enable){
		commonSelection = enable;
	}
	
	class ColorRenderer extends JLabel implements TableCellRenderer {
	
		private int type = ResultPanel.SEARCH_TABLE;
		protected String tooltip = null;

		protected Graphics2D g = null;
		protected FontMetrics fm = null;
	
		public ColorRenderer(int t){
			super();
			setOpaque(true);
			type = t;
		
			BufferedImage m_image = new BufferedImage(1, 1, BufferedImage.TYPE_INT_RGB);
			g = m_image.createGraphics();
		}

		public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column){
		
			//get the object from the row number
			LineData line = m_data.getRow(row);
			
			if (isSelected){
				setBackground(new Color(120,120,255));
			}
			else{
				if ((row%2) == 0){
					if (type == ResultPanel.PATTERN_TABLE){
						setBackground(new Color(210,220,255));
					}
					else if (type == ResultPanel.SEARCH_TABLE){
						setBackground(new Color(225,225,180));
					}
					else if (type == ResultPanel.NAVIGATION_TABLE){
						setBackground(new Color(210,230,230));
					}
					else{
						//default if nothing else matches
						setBackground(new Color(220,220,220));
					}
				}
				else{
					setBackground(Color.white);
				}
			}
			if (line.selected){
				setForeground(Color.red);
			}
			else{
				setForeground(Color.BLACK);
			}
			setFont(new Font("Dialog", Font.PLAIN, 12));
			if (value == null){
				value = "null";
			}
			setText(" " + value.toString() + " ");
		
			if (tooltips){
				//get width of the text and the column and see whether it fits
				int columnWidth = table.getColumnModel().getColumn(column).getWidth();
				g.setFont(table.getFont());
				fm = g.getFontMetrics();
				int stringWidth = (int)(fm.stringWidth(value.toString())*1.1);
			
				if (stringWidth > columnWidth){
//					tooltip = "<html><body bgcolor=\"cdd7ed\">&nbsp " + value.toString() + "&nbsp</html>";
					tooltip = "<html><body>&nbsp " + value.toString() + "&nbsp</html>";
				}
				else{
					tooltip = null;
				}
			}
			else{
				tooltip = null;
			}
			return this;
		}
	
		public String getToolTipText(){
				return tooltip;
		}
		

	}


}



