//Copyright (c) 2000-2003  San Diego Supercomputer Center (SDSC),
//a facility operated by the University of California, San Diego (UCSD)
//
//Users and possessors of this source code are hereby granted a
//nonexclusive, royalty-free copyright and design patent license to
//use this code in individual software.  License is not granted for
//commercial resale, in whole or in part, without prior written
//permission from SDSC.  This source is provided "AS IS" without express
//or implied warranty of any kind.
//
//For further information, please see:  http://mbt.sdsc.edu
//

package edu.sdsc.sirius.dialogs.TableImpl;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.table.*;
import java.awt.print.*;
import java.awt.image.*;

import edu.sdsc.sirius.util.Manager;
import edu.sdsc.sirius.util.PagePrintable;

/**
 * This class implements a panel used to display sequence data retrieved from a database. Each entry
 * is contained within a DataBlock as a part of a DataSet.
 * This class permits extended functionality including sorting, dynamic adding and removal of columns,
 * filtering of entries by content of any of the fields, custom buttons to display entry-specific information, 
 * as well as checkboxes in each line to trigger loading of the corresponding entry.
 * 
 * @see edu.sdsc.mbt.util.DataBlock
 * @see edu.sdsc.mbt.DataSet
 * @see edu.sdsc.mbt.util.Entry
 * 
 * @author Oleksandr V. Buzko
 */
public class TableViewPanel extends JPanel {
	
	//different types of tables that differ in their background colors, etc.
	public static final int SEARCH_TABLE = 0;
	public static final int PATTERN_TABLE = 1;
	public static final int NAVIGATION_TABLE = 2;
	
	private int tableType = NAVIGATION_TABLE;
	
	public static final int FIRST_COLUMN_NARROW = 0;
	public static final int SECOND_COLUMN_NARROW = 1;
	public static final int THREE_COLUMNS = 2;
	public static final int AUTO_COLUMNS = 3;//use it for non-specific tables where column count is derived from the number of fields
	public static final int DOCKING_JOBS = 4;
	public static final int DOCKING_COMPOUNDS = 5;
	public static final int DOCKING_COMPOUNDS_INDIVIDUAL = 6;

	private DataSet dataSet;
	
	private JScrollPane scrollPane;
	private JViewport vp;
	protected int viewOffset = 0;//offset in pixels when part of the table is scrolled out of the view
	private JPanel filterPanel = new JPanel();
	
	private ResultData m_data;
	private ResultTable m_table;
	private int columnCount = 0;
	private HashMap columns = new HashMap();
	private Vector columnOrder = new Vector();
	private Vector currentColumns = new Vector();
	public Vector currentIndices = new Vector();//stores indices of columns that are displayed
	
	private Vector fieldNames = new Vector();
	
	public HashMap columnWidths = new HashMap();
	public HashMap columnAlign = new HashMap();
	
	private TableColumn infoColumn;
	private TableColumn checkColumn;
	
	public static Vector defaultColumns = new Vector();
		
	protected int tableHeight = 0;
	
	protected int[] selectedRows = { -1 };
	
	protected Manager callable;
	private Container parent;
	
	private Book book;
	public PageFormat mPageFormat;
	
	private boolean tooltips = false;
	private boolean commonSelection = false;
	private String target;
	
	public int MIN_WIDTH = 150;
	
	private ResultPanelCallable resultCallable;
	
	static{
	
		defaultColumns.add("Component");
		defaultColumns.add("Name");
	}
	
	/**
	 * Constructs the <code>ResultPanel</code> when created as in a frame typically storing search results.
	 * @param p parent frame
	 * @param data <code>DataSet</code> to initialize the table
	 * @param callable Manager object that will handle entry loading
	 * @param pp 
	 */
	public TableViewPanel(Container parent, DataSet data, Manager callable, Vector fieldNames, int ratio, boolean multiselect){
		
		dataSet = data;
		this.callable = callable;
		this.parent = parent;
		
		if (dataSet == null){
			return;
		}
		
		//set layout
		setLayout(new BorderLayout());
		
		if (ratio == FIRST_COLUMN_NARROW){
			columnWidths.put((String)fieldNames.get(0), new Integer(70));
			columnWidths.put((String)fieldNames.get(1), new Integer(80));
			if (fieldNames.size() > 2){
				columnWidths.put((String)fieldNames.get(1), new Integer(80));
				columnWidths.put((String)fieldNames.get(2), new Integer(100));
			}
			
			columnAlign.put((String)fieldNames.get(0), new Integer(JLabel.LEFT));
			columnAlign.put((String)fieldNames.get(1), new Integer(JLabel.LEFT));
			if (fieldNames.size() > 2){
				columnAlign.put((String)fieldNames.get(2), new Integer(JLabel.LEFT));
			}
		}
		else if (ratio == SECOND_COLUMN_NARROW){
			columnWidths.put((String)fieldNames.get(0), new Integer(70));
			if (fieldNames.size() == 2){
				columnWidths.put((String)fieldNames.get(1), new Integer(25));
			}
			
			if (fieldNames.size() > 2){
				columnWidths.put((String)fieldNames.get(2), new Integer(30));
			}
			
			columnAlign.put((String)fieldNames.get(0), new Integer(JLabel.LEFT));
			if (fieldNames.size() == 2){
				columnAlign.put((String)fieldNames.get(1), new Integer(JLabel.LEFT));
			}
			if (fieldNames.size() > 2){
				columnAlign.put((String)fieldNames.get(2), new Integer(JLabel.LEFT));
			}
		}
		else if (ratio == THREE_COLUMNS){
			columnWidths.put((String)fieldNames.get(0), new Integer(80));
			columnWidths.put((String)fieldNames.get(1), new Integer(70));
			columnWidths.put((String)fieldNames.get(2), new Integer(80));
			
			columnAlign.put((String)fieldNames.get(0), new Integer(JLabel.LEFT));
			columnAlign.put((String)fieldNames.get(1), new Integer(JLabel.CENTER));
			columnAlign.put((String)fieldNames.get(2), new Integer(JLabel.LEFT));
		}
		else if (ratio == DOCKING_JOBS){
			columnWidths.put((String)fieldNames.get(0), new Integer(30));
			columnWidths.put((String)fieldNames.get(1), new Integer(70));
			columnWidths.put((String)fieldNames.get(2), new Integer(130));
			
			columnAlign.put((String)fieldNames.get(0), new Integer(JLabel.LEFT));
			columnAlign.put((String)fieldNames.get(1), new Integer(JLabel.CENTER));
			columnAlign.put((String)fieldNames.get(2), new Integer(JLabel.LEFT));
		}
		else if (ratio == DOCKING_COMPOUNDS){
			columnWidths.put((String)fieldNames.get(0), new Integer(70));
			columnWidths.put((String)fieldNames.get(1), new Integer(70));
			if (fieldNames.size() > 2)columnWidths.put((String)fieldNames.get(2), new Integer(70));
			
			columnAlign.put((String)fieldNames.get(0), new Integer(JLabel.LEFT));
			columnAlign.put((String)fieldNames.get(1), new Integer(JLabel.CENTER));
			if (fieldNames.size() > 2) columnAlign.put((String)fieldNames.get(2), new Integer(JLabel.CENTER));
		}
		else if (ratio == DOCKING_COMPOUNDS_INDIVIDUAL){
			columnWidths.put((String)fieldNames.get(0), new Integer(20));
			columnWidths.put((String)fieldNames.get(1), new Integer(120));
			
			columnAlign.put((String)fieldNames.get(0), new Integer(JLabel.LEFT));
			columnAlign.put((String)fieldNames.get(1), new Integer(JLabel.CENTER));
		}
		else if (ratio == AUTO_COLUMNS){
			
			int fieldWidth = MIN_WIDTH/fieldNames.size();
			
			for (int i = 0; i < fieldNames.size(); i++){
				if (((String)fieldNames.get(i)).equals("Cat. #")){
					columnWidths.put((String)fieldNames.get(i), new Integer(fieldWidth*3));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.LEFT));
				}
				else if (((String)fieldNames.get(i)).equals("Source id")){
					columnWidths.put((String)fieldNames.get(i), new Integer(fieldWidth*3));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.LEFT));
				}
				else if (((String)fieldNames.get(i)).equals("Mol. wt.")){
					columnWidths.put((String)fieldNames.get(i), new Integer((int)(fieldWidth*0.7)));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.CENTER));
				}
				else if (((String)fieldNames.get(i)).equals("Quantity")){
					columnWidths.put((String)fieldNames.get(i), new Integer((int)(fieldWidth*0.7)));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.CENTER));
				}
				else if (((String)fieldNames.get(i)).equals("Description")){
					columnWidths.put((String)fieldNames.get(i), new Integer((int)(fieldWidth*5)));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.LEFT));
				}
				else if (((String)fieldNames.get(i)).equals("Project")){
					columnWidths.put((String)fieldNames.get(i), new Integer((int)(fieldWidth*0.5)));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.CENTER));
				}
				else if (((String)fieldNames.get(i)).equals("Comment")){
					columnWidths.put((String)fieldNames.get(i), new Integer((int)(fieldWidth*5)));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.LEFT));
				}
				else if (((String)fieldNames.get(i)).equals("Timestamp")){
					columnWidths.put((String)fieldNames.get(i), new Integer((int)(fieldWidth)));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.CENTER));
				}
				else if (((String)fieldNames.get(i)).equals("Concentration")){
					columnWidths.put((String)fieldNames.get(i), new Integer((int)(fieldWidth*2)));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.CENTER));
				}
				else if (((String)fieldNames.get(i)).equals("Value")){
					columnWidths.put((String)fieldNames.get(i), new Integer((int)(fieldWidth)));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.CENTER));
				}
				else if (((String)fieldNames.get(i)).equals("Screen")){
					columnWidths.put((String)fieldNames.get(i), new Integer((int)(fieldWidth*3)));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.LEFT));
				}
				else{
					columnWidths.put((String)fieldNames.get(i), new Integer(fieldWidth));
					columnAlign.put((String)fieldNames.get(i), new Integer(JLabel.LEFT));
				}
			}
			
		}
		
		//create the table
		m_data = new ResultData(dataSet, fieldNames, columnWidths, columnAlign, currentColumns, currentIndices, null);
		m_data.setSortingMode(false);
		m_table = new ResultTable();
		
		m_table.setAutoCreateColumnsFromModel(false);
		m_table.setModel(m_data);
		
		DefaultListSelectionModel selectionModel = new DefaultListSelectionModel();
		//TODO a little hack - we take it to be a loading dialog if replacements are present
		if (multiselect){
			selectionModel.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		}
		else{
			selectionModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		}
		
		m_table.setSelectionModel(selectionModel);
		
		m_table.setRowSelectionAllowed(true);
		
		m_table.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
		
		TableCellRenderer renderer = null;
		TableCellEditor editor = null;
		columnCount = m_data.m_columns.length;
		
		//take care of the renderers
		for (int k = 0; k < columnCount; k++){
			TableColumn column = null;
			renderer = new ColorRenderer(tableType);
			((ColorRenderer)renderer).setHorizontalAlignment(m_data.m_columns[k].m_alignment);
			column = new TableColumn(k, m_data.m_columns[k].m_width, renderer, null);
			
			//set minimum size for each column to be equal the original size
			String tag = m_data.m_columns[k].m_title;
			column.setMinWidth(((Integer)columnWidths.get(tag)).intValue() - 10);//initial width - 10 pixels
			
			final String label = m_data.m_columns[k].m_title;
			columns.put(label, column);
			columnOrder.add(label);
			
			m_table.addColumn(column);
			currentColumns.add(column);
			currentIndices.add(new Integer(k));
		}
		
		JTableHeader header = m_table.getTableHeader();
		header.setUpdateTableInRealTime(true);
		header.setReorderingAllowed(false);
		
		m_table.setMinimumWidth(MIN_WIDTH);
		
		scrollPane = new JScrollPane(m_table);
		
		//add key listener to the table to respond to Enter key events
		m_table.addKeyListener(new KeyAdapter(){
			public void keyPressed(KeyEvent k){
/*				if (k.getKeyCode() == KeyEvent.VK_ENTER){
					//do the same thing as with OK button
					loadSelection();
					m_table.clearSelection();
				}
*/				if(k.getKeyCode() == KeyEvent.VK_ESCAPE){
					//do the same thing as with Cancel button
					m_table.clearSelection();
				}
			}
		});
		
		m_table.addMouseListener(new MouseAdapter(){
			public void mouseClicked(MouseEvent e){
				//get the selection and pass it back to the parent
				int[] rows = getSelectedRows();
				Vector out = new Vector();
				//check whether these rows are within the limits of the data set
				for (int i = 0; i < rows.length; i++){
					int r = rows[i];
					if (r >= dataSet.size()){
						//clear this row selection
						setSelection(false, r);
///						if (resultCallable != null) resultCallable.setSelectedIndices(null);
					}
					else{
						if (resultCallable != null){
							out.add(new Integer(r));
						}
					}
				}
				if (resultCallable != null){
					resultCallable.setSelectedIndices(out);
				}
				
			}
		});
		
		
		//printing stuff
		PrinterJob pj = PrinterJob.getPrinterJob();
		mPageFormat = pj.defaultPage();
		mPageFormat.setOrientation(PageFormat.LANDSCAPE);
		pj = null;

		setVisible(true);

		vp = scrollPane.getViewport();
		
		add(scrollPane, BorderLayout.CENTER);
		
		resizePanel(parent.getSize());
		setVisible(true);
	}
	
	public void printTable() throws PrinterException {
//		MessageFormat header = new MessageFormat("Page {0,number,integer}");
		m_table.print(JTable.PrintMode.FIT_WIDTH, null, null);
	}
	
	public void setResultPanelCallable(ResultPanelCallable c){
		resultCallable = c;
	}
	
	public void setSelection(boolean selected, int row){
		LineData line = (LineData)m_data.getRow(row);
		line.selected = selected;
		m_table.removeRowSelectionInterval(row, row);
		m_table.repaint();
	}
	
	public void deselectAll(){
		for (int i = 0; i < m_data.getRowCount(); i++){
			LineData line = (LineData)m_data.getRow(i);
			line.selected = false;
		}
		m_table.clearSelection();
		m_table.repaint();
	}
	
	/**
	 * Removes selection of the rows.
	 */
	public void clearRowSelection(){
		m_table.clearSelection();
	}
	
	public int getSelectedRowCount(){
		return m_table.getSelectedRows().length;
	}
	
	public int[] getSelectedRows(){
		return m_table.getSelectedRows();
	}
	
	
	public void setSelectedRows(int[] rows){
		for (int i = 0; i < rows.length; i++){
			m_table.setRowSelectionInterval(rows[i], rows[i]);
		}
		repaint();
	}
	
	public void scrollToRow(int row){
		
	    // get the parent viewport
	    Component parent = m_table.getParent();
	    while (parent != null && !(parent instanceof JViewport))
	    {
	        parent = parent.getParent();
	    }
	    Rectangle rec = null;
	    int height = m_table.getRowHeight();
	    if (parent == null)
	    {
	        // no parent so use 0 and 1 as X and width
	        rec = new Rectangle(0, row * height, 1, height);
	    }
	    else
	    {
	        // use the X pos and width of the current viewing rectangle 
	        rec = ((JViewport)parent).getViewRect();
	        rec.y = row * height;
	        rec.height = height;
	    }
	    m_table.scrollRectToVisible(rec);
	    m_table.revalidate();
	    m_table.repaint();
	    
	    repaint();
	}
	
	public void setRowSelection(int row, boolean update){
		m_table.addRowSelectionInterval(row, row);
		if (update) repaint();
	}
	
	public int getColumnCount(){
		return columnCount;
	}
	
	public void updateRow(int row){
		LineData line = m_data.getRow(row);
		line.updateData();
	}
	
	/**
	 * Handles repainting of the table panel in response to resizing of the container frame.
	 * Adds filler lines if table height is less than the height of the view.
	 * @param size new size of the frame as a <code>Dimension</code>
	 */
	public void resizePanel(Dimension size){
		
		tableHeight = m_table.getPreferredSize().height;
		int height = size.height;
		
    	if (height > tableHeight){
			//need to add more blank rows
			//calculate how many
			int rows = ((height - tableHeight)/(m_table.getRowHeight())) + 2;
			for (int i = 0; i < rows; i++){
				m_data.insertBlankRow();
			}
			
			m_table.revalidate();
			tableHeight = m_table.getSize().height;
			m_table.tableChanged(new TableModelEvent(m_data));
			scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);
			m_table.repaint();
		}
		else{
			//there are some blanks
			int blanks = m_data.getBlankRowCount();
			int rows = (tableHeight - height)/(m_table.getRowHeight());

			for (int i = 0; i < rows; i++){
				if (i >= blanks){
					break;
				}
				m_data.removeBlankRow();
			}
			
			if (m_data.getBlankRowCount() == 0){
				scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);	
			}
			
			tableHeight = m_table.getSize().height;
			
			m_table.tableChanged(new TableModelEvent(m_data));
			m_table.repaint();
		}
    	
    	revalidate();
    	repaint();
	}

	public void processClick(int nRow){
		m_table.clearSelection();
		m_table.repaint();
//		System.out.println("Received click at row " + nRow);
		//identify id by row number
//		parent.displayInfo((m_data.getValueAt(nRow, 1)).toString());
	}
	
	public void printSetupHandler(){

		//this method is called from Viewer when Print Setup is selected
		//currently active frame is found and this method is called

		Thread t = new Thread(){
			public void run(){
				PrinterJob pj = PrinterJob.getPrinterJob();
				mPageFormat = pj.pageDialog(mPageFormat);
			}
		};
		t.start();
	}
	
	
	public void printData() {

		//use the current page format to get the printable area
		PrinterJob pj = PrinterJob.getPrinterJob();
		
		//set the cursor, 'cause the wait will be long
		setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		book = new Book();
		
		int pageWidth = 0;
		int pageHeight = 0;
		if (mPageFormat.getOrientation() == PageFormat.PORTRAIT) {
			pageWidth = (int)mPageFormat.getImageableWidth();
			pageHeight = (int)mPageFormat.getImageableHeight();
		}
		else {
			pageWidth = (int)mPageFormat.getImageableWidth();
			pageHeight = (int)mPageFormat.getImageableHeight();
		}
		
		boolean newPage = true;
		BufferedImage m_image = null;
		Graphics2D g = null;
		FontMetrics fm = null;
		int y = 20;//start a bit lower
		
		int imageWidth = (int)(pageWidth*1.7);
		int imageHeight = (int)(pageHeight*1.7);
		int nRow = 0;
		while (true){
			
			if (newPage){
				m_image = new BufferedImage(imageWidth, imageHeight, BufferedImage.TYPE_INT_RGB);
				g = m_image.createGraphics();
//				g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				newPage = false;
				g.setColor(Color.white);
				g.fillRect(0,0, imageWidth, imageHeight);
				
				//draw the header
				g.setColor(Color.black);
				Font headerFont = m_table.getFont().deriveFont(Font.BOLD);
//				int size = headerFont.getSize();
//				System.out.println("header font = " + size);
				g.setFont(headerFont);
				fm = g.getFontMetrics();

				TableColumnModel colModel = m_table.getColumnModel();
				int nColumns = colModel.getColumnCount();
				int x[] = new int[nColumns];
				x[0] = 5;
        
				int h = fm.getAscent();
				y += h;//add ascent of header font
 
				for (int nCol=0; nCol < nColumns; nCol++){
					TableColumn tk = colModel.getColumn(nCol);
					int width = tk.getWidth();
					if (x[nCol] + width > imageWidth) {
						nColumns = nCol;
						break;
					}
					if (nCol+1<nColumns){
						x[nCol+1] = x[nCol] + width;
					}
					String title = (String)tk.getIdentifier();
					g.drawString(title, x[nCol], y);
				}
				
				y += h*2;
			}
			
			//start iterating over the table and drawing the data on one line
			g.setFont(m_table.getFont());
			fm = g.getFontMetrics();
			int h = fm.getHeight();
			int lineX = 5;
			for (int nCol=0; nCol < m_table.getColumnCount(); nCol++) {
				TableColumn tk = m_table.getColumnModel().getColumn(nCol);
				int width = tk.getWidth();
				int col = m_table.getColumnModel().getColumn(nCol).getModelIndex();
				Object obj = m_data.getValueAt(nRow, col);
				String str = null;
				if (obj == null){
					str = "null";
				}
				else{
					str = obj.toString();
				}

				int w = fm.stringWidth(str);
				if (w > (width - 5)){
					g.drawString(str, lineX, y);
					//and cover the rest of the string with a white rectangle
					g.setColor(Color.white);
					g.fillRect(lineX+width-7, y-h, w, h+2);
					g.setColor(Color.black);
				}
				else{
					g.drawString(str, lineX, y);
				}
				lineX += width;
			}
			y += h;
			
			if (y > imageHeight){
				PagePrintable p = new PagePrintable(m_image, pageWidth, pageHeight, imageWidth, imageHeight);
				book.append(p, mPageFormat);
//				savePrint(m_image, "1");
					
				//create a new image
				m_image = new BufferedImage(imageWidth, imageHeight, BufferedImage.TYPE_INT_RGB);
				g = m_image.createGraphics();
				g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				newPage = false;
				g.setColor(Color.white);
				g.fillRect(0,0, imageWidth, imageHeight);
				
				y = 20;
				//draw the header
				g.setColor(Color.black);
				Font headerFont = m_table.getFont().deriveFont(Font.BOLD);
				g.setFont(headerFont);
				fm = g.getFontMetrics();

				TableColumnModel colModel = m_table.getColumnModel();
				int nColumns = colModel.getColumnCount();
				int x[] = new int[nColumns];
				x[0] = 5;
        
				y += fm.getAscent();//add ascent of header font
 
				for (int nCol=0; nCol < nColumns; nCol++){
					TableColumn tk = colModel.getColumn(nCol);
					int width = tk.getWidth();
					if (x[nCol] + width > imageWidth) {
						nColumns = nCol;
						break;
					}
					if (nCol+1<nColumns){
						x[nCol+1] = x[nCol] + width;
					}
					String title = (String)tk.getIdentifier();
					g.drawString(title, x[nCol], y);
				}
				
				y += h;
			}
			
			nRow++;
			if (nRow > m_table.getRowCount()-1){
				PagePrintable p = new PagePrintable(m_image, pageWidth, pageHeight, imageWidth, imageHeight);
				book.append(p, mPageFormat);
//				savePrint(m_image, "2");
				break;
			}
			
			
		}
		
		pj.setPageable(book);
		if (pj.printDialog()){
			try{
				pj.print();
			}
			catch (PrinterException pe){
				System.out.println("Exception " + pe);
				JOptionPane.showMessageDialog(this, "Printing error: " + pe, "Error", JOptionPane.ERROR_MESSAGE);
			}
		}
			
		setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		book = null;
	}
	
	class ColorRenderer extends JLabel implements TableCellRenderer {
	
		private int type = ResultPanel.SEARCH_TABLE;
		protected String tooltip = null;

		protected Graphics2D g = null;
		protected FontMetrics fm = null;
	
		public ColorRenderer(int t){
			super();
			setOpaque(true);
			type = t;
		
			BufferedImage m_image = new BufferedImage(1, 1, BufferedImage.TYPE_INT_RGB);
			g = m_image.createGraphics();
		}

		public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column){
		
			//get the object from the row number
			LineData line = m_data.getRow(row);
			
			if (isSelected){
				setBackground(new Color(120,120,255));
			}
			else{
				if ((row%2) == 0){
					if (type == ResultPanel.PATTERN_TABLE){
						setBackground(new Color(210,220,255));
					}
					else if (type == ResultPanel.SEARCH_TABLE){
						setBackground(new Color(225,225,180));
					}
					else if (type == ResultPanel.NAVIGATION_TABLE){
						setBackground(new Color(210,230,230));
					}
					else{
						//default if nothing else matches
						setBackground(new Color(220,220,220));
					}
				}
				else{
					setBackground(Color.white);
				}
			}
			if (line.selected){
				setForeground(Color.red);
			}
			else if (line.marked){
				setForeground(Color.blue);
			}
			else{
				setForeground(Color.BLACK);
			}
			
			if (line.pending && line.bold){
				setFont(new Font("Dialog", Font.ITALIC + Font.BOLD, 12));
			}
			else if (line.pending && !line.bold){
				setFont(new Font("Dialog", Font.ITALIC, 12));
			}
			else if (line.bold){
				setFont(new Font("Dialog", Font.BOLD, 12));
			}
			else{
				setFont(new Font("Dialog", Font.PLAIN, 12));
			}
			
			if (value == null){
				value = "null";
			}
			setText(" " + value.toString() + " ");
		
			if (tooltips){
				//get width of the text and the column and see whether it fits
				int columnWidth = table.getColumnModel().getColumn(column).getWidth();
				g.setFont(table.getFont());
				fm = g.getFontMetrics();
				int stringWidth = (int)(fm.stringWidth(value.toString())*1.1);
			
				if (stringWidth > columnWidth){
//					tooltip = "<html><body bgcolor=\"cdd7ed\">&nbsp " + value.toString() + "&nbsp</html>";
					tooltip = "<html><body>&nbsp " + value.toString() + "&nbsp</html>";
				}
				else{
					tooltip = null;
				}
			}
			else{
				tooltip = null;
			}
			return this;
		}
	
		public String getToolTipText(){
				return tooltip;
		}
		

	}


	public void setCallable(ResultPanelCallable c){
		resultCallable = c;
	}
	
	public void removeCallable(){
		resultCallable = null;
	}
}



