package edu.sdsc.sirius.export;

import edu.sdsc.mbt.viewables.*;
import edu.sdsc.mbt.util.*;
import edu.sdsc.mbt.*;
import edu.sdsc.sirius.dialogs.*;
import edu.sdsc.sirius.io.*;
import edu.sdsc.sirius.monitors.MonitorDialog;
import edu.sdsc.sirius.util.Manager;
import edu.sdsc.sirius.util.IOHandler;
import edu.sdsc.sirius.viewers.*;
import edu.sdsc.sirius.viewers.SequenceViewerImpl.*;

import java.util.*;
import java.io.*;
import java.util.zip.*;
import java.awt.*;

/**
 * This class is intended as a utility to save state of the current display to an external
 * file. It works with multiple structures loaded, as well as with any structure styles.
 * The coordinates are saved to Strings in mol2 format to preserve bonding information and
 * are stored in binary format in a hash with structure url strings as keys.
 * @author Oleksandr V. Buzko
 *
 */
public class StateHandler {
	
	public static void exportState(StructureViewer viewer, SequenceViewer sequenceViewer, String filename){
		
		StateObject out = createStateObject(viewer, sequenceViewer);
		
		//write it out to the given file as a compressed stream
		try{
			ObjectOutputStream dataStream = new ObjectOutputStream(new GZIPOutputStream(new FileOutputStream(filename)));
			dataStream.writeObject(out);
			dataStream.close();
			dataStream = null;

		}
		catch (Exception ex){
			System.out.println("Unable to write: " + ex);
		}
		
	}
	
	public static StateObject createStateObject(StructureViewer viewer, SequenceViewer sequenceViewer){
		
		//take care of the coordinates first
		Vector coordinates = new Vector();//same order of structures as in the structures vector
		Vector sequences = new Vector();//sequences of these structures in the same order (in case there have been dashes inserted)
		
		Vector styling = new Vector();//Structure -> composite styles object
		
		HashMap atomSets = viewer.getAtomSets();
		Vector atomSetNames = viewer.getAtomSetNames();

		Structure independentStructure = viewer.getIndependentStructure();
		Vector independentComponents = viewer.getIndependentComponents();
		int independentStructureIndex = -1;
		
		HashMap stylesToIndex = new HashMap();//mapping of StructureStyles to current index in the SequenceViewer
		HashMap stylesToSequence = new HashMap();
		
		HashMap sequenceStyleMap = sequenceViewer.getStyleMap();//to find out which sequence the structure is associated with and get its index

		Set set = sequenceStyleMap.keySet();
		Iterator it = set.iterator();
		while (it.hasNext()){
			Sequence sequence = (Sequence)it.next();
			StructureStyles st = (StructureStyles)sequenceStyleMap.get(sequence);
			stylesToSequence.put(st, sequence);
			int index = sequenceViewer.getSequenceObjects().indexOf(sequence);
			stylesToIndex.put(st, new Integer(index));
		}
		
		int[] independentAtoms = null;
		int[] independentResidues = null;

		
		//walk through the structures and get the atoms
		Vector structures = viewer.getLoadedStructures();
		int[] ramaViewers = new int[structures.size()];//if structure has a Rama viewer = 1, if not = 0
		
		HashMap reference = new HashMap();//Structure -> vector of old numbers (index is the new number)
		
		HashMap atomToLabels = viewer.getAtomToLabels();
		HashMap atomToResidueLabels = viewer.getAtomToResidueLabels();
		
		//labels are first collected in a vector, since it's not known how many labels will be in each individual structure
		//later the vector is converted to an array for a more compact and reference-independent storage
		
		for (int i = 0; i < structures.size(); i++){
			Structure structure = (Structure)structures.get(i);
			
			HashMap atomMap = new HashMap();
			String coords = IOHandler.convertMol2ToString(structure, false, viewer, atomMap);
			
			//if there are any conformation records, add them to the String to have a complete record
			int coilCount = structure.getStructureComponentCount(
					StructureComponentRegistry.TYPE_COIL );
			int helixCount = structure.getStructureComponentCount(
					StructureComponentRegistry.TYPE_HELIX );
			int strandCount = structure.getStructureComponentCount(
					StructureComponentRegistry.TYPE_STRAND );
			int turnCount = structure.getStructureComponentCount(
					StructureComponentRegistry.TYPE_TURN );
			
			if (coilCount > 0 || helixCount > 0 || strandCount > 0 || turnCount > 0){
				StructureMap structureMap = structure.getStructureMap();
				//add a conformation record. name, chain, compound are given as strings, residue is given as index in structureMap
				StringBuffer buffer = new StringBuffer();
				buffer.append("@<TRIPOS>CONFORMATION\n");
				
				//go over helices
				for (int j = 0; j < helixCount; j++){
					Conformation h = (Conformation)structure.getStructureComponentByIndex(StructureComponentRegistry.TYPE_HELIX, j);
//					System.out.println("h = " + h + ": name = " + h.name + ", start compound = " + h.start_compound + ", start chain = " +
//							h.start_chain + ", start residue = " + h.start_residue);
					String chain = h.start_chain;
					if (chain.length() == 0){
						chain = "_";
					}
					buffer.append("HELIX");
					buffer.append("\t");
					buffer.append(h.start_compound);
					buffer.append("\t");
					buffer.append(chain);
					buffer.append("\t");
					buffer.append(h.start_residue);
					buffer.append("\t");
					buffer.append(h.end_compound);
					buffer.append("\t");
					buffer.append(chain);
					buffer.append("\t");
					buffer.append(h.end_residue);
					buffer.append("\n");
					
				}
				
				for (int j = 0; j < strandCount; j++){
					Conformation h = (Conformation)structure.getStructureComponentByIndex(StructureComponentRegistry.TYPE_STRAND, j);
//					System.out.println("h = " + h + ": name = " + h.name + ", start compound = " + h.start_compound + ", start chain = " +
//							h.start_chain + ", start residue = " + h.start_residue);
					String chain = h.start_chain;
					if (chain.length() == 0){
						chain = "_";
					}
					buffer.append("SHEET");
					buffer.append("\t");
					buffer.append(h.start_compound);
					buffer.append("\t");
					buffer.append(chain);
					buffer.append("\t");
					buffer.append(h.start_residue);
					buffer.append("\t");
					buffer.append(h.end_compound);
					buffer.append("\t");
					buffer.append(chain);
					buffer.append("\t");
					buffer.append(h.end_residue);
					buffer.append("\n");
					
				}

				for (int j = 0; j < turnCount; j++){
					Conformation h = (Conformation)structure.getStructureComponentByIndex(StructureComponentRegistry.TYPE_TURN, j);
//					System.out.println("h = " + h + ": name = " + h.name + ", start compound = " + h.start_compound + ", start chain = " +
//							h.start_chain + ", start residue = " + h.start_residue);
					String chain = h.start_chain;
					if (chain.length() == 0){
						chain = "_";
					}
					buffer.append("TURN");
					buffer.append("\t");
					buffer.append(h.start_compound);
					buffer.append("\t");
					buffer.append(chain);
					buffer.append("\t");
					buffer.append(h.start_residue);
					buffer.append("\t");
					buffer.append(h.end_compound);
					buffer.append("\t");
					buffer.append(chain);
					buffer.append("\t");
					buffer.append(h.end_residue);
					buffer.append("\n");
					
				}
				
				coords = coords + buffer;
			}
			
//			System.out.println(coords);
			
			reference.put(structure, atomMap);
			
			coordinates.add(coords);
			
			//work on the appearance
			StructureMap map = structure.getStructureMap();
			StructureStyles styles = map.getStructureStyles();
			
			if (structure == independentStructure){
				independentStructureIndex = i;
				
				//walk over the components and get their indices from the structure map
				//store in arrays
				independentResidues = new int[map.getResidueCount()];
				independentAtoms = new int[map.getAtomCount()];
				
				//initialize the arrays with zeros
				for (int n = 0; n < independentResidues.length; n++){
					independentResidues[n] = 0;
				}
				for (int n = 0; n < independentAtoms.length; n++){
					independentAtoms[n] = 0;
				}
				
				if (independentComponents != null && independentComponents.size() > 0){
					
					for (int j = 0; j < independentComponents.size(); j++){
						try{
							Residue r = (Residue)independentComponents.get(j);
							independentResidues[map.getResidueIndex(r)] = 1;
						}
						catch (ClassCastException e){
							try{
								Atom a = (Atom)independentComponents.get(j);
								independentResidues[map.getAtomIndex(a)] = 1;
							}
							catch (Exception ex){
								continue;
							}
						}
					}
				}
				
			}
			
			

			
			StyleMap styleMap = new StyleMap();
			
			//find out about ribbons displayed for this structure
			int[] ribbons = new int[map.getChainCount()];
			float[] ribbonQuality = new float[map.getChainCount()];//quality
			boolean[] ribbonVis = new boolean[map.getResidueCount()];//visibility of ribbon
			
			for (int j = 0; j < map.getChainCount(); j++){
				Chain chain = map.getChain(j);
				ribbons[j] = viewer.hasRibbon(chain);
				ribbonQuality[j] = styles.getRibbonQuality(chain);
			}
			
			//check visibility of structure components in this structure
			//atoms
			Vector atoms = map.getAtoms();
			boolean[] atomVisibility = new boolean[atoms.size()];
			boolean[] atomSelection = new boolean[atoms.size()];
			short[] atomRenderingStyle = new short[atoms.size()];
			short[] atomRenderingQuality = new short[atoms.size()];
			
			int[] labels = new int[atoms.size()];

			
			int[] atomSetAssignment = new int[atoms.size()];
			int[] residueSetAssignment = new int[map.getResidueCount()];
			
			//populate the assignment arrays with -1
			for (int j = 0; j < atomSetAssignment.length; j++){
				atomSetAssignment[j] = -1;
			}
			for (int j = 0; j < residueSetAssignment.length; j++){
				residueSetAssignment[j] = -1;
			}
			
			Vector residues = map.getResidues();
			String[] labelText = new String[atoms.size()];
			int[] labelSize =  new int[atoms.size()];
			float[][]labelColor = new float[atoms.size()][3];
			boolean[] labelSelection = new boolean[atoms.size()];
			
			float[][] atomColors = new float[atoms.size()][3];
			float[][] residueColors = new float[residues.size()][3];
			float[][] ribbonColors = new float[residues.size()][3];
			
			for (int j = 0; j < atoms.size(); j++){
				Atom a = (Atom)atoms.get(j);
				atomVisibility[j] = a.visible;
				atomSelection[j] = styles.isSelected(a);
				atomRenderingStyle[j] = a.render;
				atomRenderingQuality[j] = a.quality;
				
				if (atomToLabels.containsKey(a)){
					LabelComponent lc = (LabelComponent)atomToLabels.get(a);
					labels[j] = lc.getLabelType();
					labelSize[j] = lc.getFont();
					labelSelection[j] = lc.getSelected();
					
					float[] temp = lc.getSetColor();
					labelColor[j][0] = temp[0];
					labelColor[j][1] = temp[1];
					labelColor[j][2] = temp[2];
					
					if (lc.getLabelType() == LabelDialog.TYPE_CUSTOM){
						//record the label text, as well
						labelText[j] = lc.getText();
					}
					/**
					 * TODO check for garbage output of uninitialized array
					 */
				}
				else{
					labels[j] = -1;
				}
				
				//look at the color
				AtomColor c = styles.getAtomColor(a);
				if (c instanceof AtomColorByElement){
					atomColors[j] = null;
				}
				else{
					//custom color. get the hashcode of the custom color and its corresponding
					//rgb values
					float[] temp = new float[3];
					c.getAtomColor(a, temp);
					atomColors[j][0] = temp[0];
					atomColors[j][1] = temp[1];
					atomColors[j][2] = temp[2];
				}
			}
			
			
			int[] residueLabels = new int[atoms.size()];
			String[] residueLabelText = new String[residues.size()];
			int[] residueLabelSize =  new int[residues.size()];
			float[][]residueLabelColor = new float[residues.size()][3];
			boolean[] residueLabelSelection = new boolean[residues.size()];

			for (int j = 0; j < residues.size(); j++){
				Residue r = (Residue)residues.get(j);
				
				//look at the color
				ResidueColor c = styles.getResidueColor(r);
				if (c instanceof ResidueColorByElement){
					residueColors[j] = null;
				}
				else{
					//custom color. get the hashcode of the custom color and its corresponding
					//rgb values
					float[] temp = new float[3];
					c.getResidueColor(r, temp);
					residueColors[j][0] = temp[0];
					residueColors[j][1] = temp[1];
					residueColors[j][2] = temp[2];
				}
				
				//if there is a ribbon, remember the colors
				if (viewer.hasRibbon(map.getChain(r.getAtom(0))) > -1){
					RibbonColor rc = styles.getRibbonColor(r);
					if (rc instanceof RibbonColorDefault){
						ribbonColors[j] = null;
					}
					else{
						//custom color. get the hashcode of the custom color and its corresponding
						//rgb values
						float[] temp = new float[3];
						rc.getRibbonColor(r, temp);
						ribbonColors[j][0] = temp[0];
						ribbonColors[j][1] = temp[1];
						ribbonColors[j][2] = temp[2];
					}
				}
				
				ribbonVis[j] = r.ribbon;
				
				//check for labels
				Atom ref = r.centerAtom;
				if (ref == null) ref = r.getAtom(0);
				
				if (atomToResidueLabels.containsKey(ref)){
					ResidueLabelComponent lc = (ResidueLabelComponent)atomToResidueLabels.get(ref);
					residueLabels[j] = lc.getLabelType();
					residueLabelSize[j] = lc.getFont();
					residueLabelSelection[j] = lc.getSelected();
					
					float[] temp = lc.getSetColor();
					residueLabelColor[j][0] = temp[0];
					residueLabelColor[j][1] = temp[1];
					residueLabelColor[j][2] = temp[2];
					
					if (lc.getLabelType() == LabelDialog.TYPE_CUSTOM){
						//record the label text, as well
						residueLabelText[j] = lc.getText();
					}
					/**
					 * TODO check for garbage output of uninitialized array
					 */
				}
				else{
					residueLabels[j] = -1;
				}
				
			}

			//check in the sequence style map, which Sequence corresponds to this Structure
			
			//size of the hash will be greater than zero only if the sequence viewer is open
			if (stylesToIndex.size() > 0){
				int index = ((Integer)stylesToIndex.get(styles)).intValue();
				styleMap.setSequenceIndex(index);
				
				//check whether this entry is selected in the sequence viewer
				Sequence sss = (Sequence)stylesToSequence.get(styles);
				styleMap.setSequenceSelected(sequenceViewer.isSelected(sss));
				sequences.add(sss.getSequence());
			}
			
				
			/**
			 * TODO add monitors, such as distance
			 */
			
			
			//check if the structure has a Ramachandran viewer associated with it
			if (viewer.getStructureDocument().parent.getRamachandranViewer(structure) != null){
				ramaViewers[i] = 1;
			}
			else{
				ramaViewers[i] = 0;
			}
			
			styleMap.setAtomVisibility(atomVisibility);
			styleMap.setAtomSelection(atomSelection);
			styleMap.setAtomRenderingStyle(atomRenderingStyle);
			styleMap.setAtomRenderingQuality(atomRenderingQuality);
			styleMap.setAtomColors(atomColors);
			styleMap.setResidueColors(residueColors);
			styleMap.setRibbonColors(ribbonColors);
			styleMap.setAtomSetAssignment(atomSetAssignment);
			styleMap.setResidueSetAssignment(residueSetAssignment);
			
			styleMap.setLabels(labels);
			styleMap.setResidueLabels(residueLabels);
			styleMap.setLabelText(labelText);
			styleMap.setLabelSize(labelSize);
			styleMap.setLabelSelection(labelSelection);
			styleMap.setLabelColor(labelColor);
			
			styleMap.setResidueLabelText(residueLabelText);
			styleMap.setResidueLabelSize(residueLabelSize);
			styleMap.setResidueLabelSelection(residueLabelSelection);
			styleMap.setResidueLabelColor(residueLabelColor);

			styleMap.setRibbons(ribbons);
			styleMap.setRibbonQuality(ribbonQuality);
			styleMap.setRibbonVis(ribbonVis);
			
			styling.add(styleMap);
		}
		
		//get cell colors
		Vector seqs = sequenceViewer.getSequences();
		int max = sequenceViewer.getResidueMax();
		float[][][] cellColors = new float[seqs.size()][max][3];
		for (int i = 0; i < seqs.size(); i++){
			Cell[] cells = (Cell[])seqs.get(i);
//			System.out.println("i = " + i + ", length = " + cells.length);
			for (int j = 0; j < cells.length; j++){
				try{
					Color c = cells[j].color;
					float[] temp = new float[3];
					c.getRGBColorComponents(temp);
					cellColors[i][j][0] = temp[0];
					cellColors[i][j][1] = temp[1];
					cellColors[i][j][2] = temp[2];
				}
				catch (Exception e){}
			}
			
		}
		
		//now walk over the sets and populate the style as needed
		for (int i = 0; i < atomSetNames.size(); i++){
			Vector as = (Vector)atomSets.get(atomSetNames.get(i));
			for (int j = 0; j < as.size(); j++){
				try{
					Atom aa = (Atom)as.get(j);
					int index = viewer.getLoadedStructures().indexOf(aa.structure);
					int[] array = ((StyleMap)styling.get(index)).getAtomSetAssignment();
					array[aa.structure.getStructureMap().getAtomIndex(aa)] = i;
				}
				catch (ClassCastException e){
					Residue rr = (Residue)as.get(j);
					int index = viewer.getLoadedStructures().indexOf(rr.structure);
					int[] array = ((StyleMap)styling.get(index)).getResidueSetAssignment();
					array[rr.structure.getStructureMap().getResidueIndex(rr)] = i;
				}
			}
		}
		
		//save distance monitors
		HashMap monitors = viewer.getMonitors();
		int[][] distances = new int[monitors.size()][];
		Set keys = monitors.keySet();
		it = keys.iterator();
		int counter = 0;
		while (it.hasNext()){
			MonitorComponent mc = (MonitorComponent)it.next();
			Atom a1 = mc.atom1;
			Atom a2 = mc.atom2;
			
			HashMap map1 = (HashMap)reference.get(a1.structure);
			int index1 = ((Integer)map1.get(a1)).intValue();
			HashMap map2 = (HashMap)reference.get(a2.structure);
			int index2 = ((Integer)map2.get(a2)).intValue();
			
			distances[counter] = new int[]{ viewer.getLoadedStructures().indexOf(a1.structure), index1,
					viewer.getLoadedStructures().indexOf(a2.structure), index2};
			
			counter++;
		}
		
		//get the viewer parameters to reproduce the view and fog
		ViewerParameters view = viewer.getGeometryViewer().getViewerParameters();
		
		StateObject out = new StateObject( coordinates, sequences, styling, view, cellColors, atomSetNames);
		out.setRamaViewers(ramaViewers);

		
		HashMap hb = viewer.getHBonds();
		if (hb.size() > 0){
			int[][] hBonds = new int[hb.size()][];
			keys = hb.keySet();
			it = keys.iterator();
			int hCounter = 0;
			while (it.hasNext()){
				HydrogenBondComponent hc = (HydrogenBondComponent)it.next();
				Atom a1 = hc.atom1;
				Atom a2 = hc.atom2;
				
				HashMap map1 = (HashMap)reference.get(a1.structure);
				int index1 = ((Integer)map1.get(a1)).intValue();
				HashMap map2 = (HashMap)reference.get(a2.structure);
				int index2 = ((Integer)map2.get(a2)).intValue();

				hBonds[hCounter] = new int[]{ viewer.getLoadedStructures().indexOf(a1.structure), index1,
						viewer.getLoadedStructures().indexOf(a2.structure), index2 };
				
				hCounter++;
			}
			
			out.setHBonds(hBonds);
		}
		
		out.setDistanceMonitors(distances);
		
		HashMap b = viewer.getBumps();
		if (b.size() > 0){
			int[][] bumps = new int[b.size()][];
			keys = b.keySet();
			it = keys.iterator();
			int bCounter = 0;
			while (it.hasNext()){
				StericBumpComponent hc = (StericBumpComponent)it.next();
				Atom a1 = hc.atom1;
				Atom a2 = hc.atom2;
				
				HashMap map1 = (HashMap)reference.get(a1.structure);
				int index1 = ((Integer)map1.get(a1)).intValue();
				HashMap map2 = (HashMap)reference.get(a2.structure);
				int index2 = ((Integer)map2.get(a2)).intValue();

				bumps[bCounter] = new int[]{ viewer.getLoadedStructures().indexOf(a1.structure), index1,
						viewer.getLoadedStructures().indexOf(a2.structure), index2 };
				
				bCounter++;
			}
			
			out.setBumps(bumps);
		}
		if (independentStructureIndex > -1){
			out.setIndependentStructureIndex(independentStructureIndex);
			out.setIndependentAtoms(independentAtoms);
			out.setIndependentResidues(independentResidues);
			out.setTrackBumps(viewer.isTrackBumps());
		}
		
		return out;
		
	}
	
	public static void importState(Manager parent, StructureViewer viewer, SequenceViewer sequenceViewer, String filename){
		
		try{
			ObjectInputStream input = new ObjectInputStream(new GZIPInputStream(new FileInputStream(filename)));
			StateObject in = (StateObject)input.readObject();
			if (in == null){
				return;
			}
			
			restoreState(in, parent, viewer, sequenceViewer);
			
			parent.getApplicationFrame().setTitle("Sirius 1.2: " + filename);
			viewer.updateAppearance();
		}
		catch (Exception ex){
			parent.displayExceptionMessage("Unable to restore state",ex);
			return;
		}
		
	}
	
	public static void restoreState(StateObject in, Manager parent, StructureViewer viewer, SequenceViewer sequenceViewer){
			
		try{
			
			Vector styling = in.getStyleMap();
			
			float[][][] cellColors = in.getCellColors();
			
			//get the mapping of Sequence and StructureStyles
			HashMap stylesToSequence = new HashMap();
			
			HashMap sequenceStyleMap = sequenceViewer.getStyleMap();//to find out which sequence the structure is associated with and get its index
		
			Vector sequences = in.getSequences();//strings of the corresponding sequence viewer entries
			
			Vector atomSetNames = in.getAtomSetNames();
			
			HashMap atomSets = new HashMap();//will rebuild the actual object map from the saved data
			if (atomSetNames.size() > 0){
				for (int i = 0; i < atomSetNames.size(); i++){
					Vector temp = new Vector();
					atomSets.put(atomSetNames.get(i), temp);//prepare the data structure
				}
			}
			
			int independentStructureIndex = in.getIndependentStructureIndex();
			int[] independentAtoms = in.getIndependentAtoms();
			int[] independentResidues = in.getIndependentResidues();
			
			Vector independentComponents = null;
			
			Vector structureObjects = new Vector();
			Vector structures = in.getStructures();
			for (int i = 0; i < structures.size(); i++){
				String content = (String)structures.get(i);
				
//				System.err.println(content);
				
				//convert the String into Entry using mol2 loader
				StructureEntry entry = (StructureEntry)IOHandler.loadMol2FromString(content);
				
				if (entry == null){
					parent.displayErrorMessage("Unable to convert string to entry");
					return;
				}
				
				parent.addEntry(entry, false, false);
				
				//now take care of the styles
				Structure s = entry.getStructure();
				StructureMap structureMap = s.getStructureMap();
				StructureStyles styles = structureMap.getStructureStyles();
				
				structureObjects.add(s);
				
				//check for independent motion
				if (independentStructureIndex == i){
					//this is the structure 
					//populate the independent components
					independentComponents = new Vector();
					for (int j = 0; j < independentResidues.length; j++){
						if (independentResidues[j] == 1){
							independentComponents.add(structureMap.getResidue(j));
						}
					}
					
					for (int j = 0; j < independentAtoms.length; j++){
						if (independentAtoms[j] == 1){
							independentComponents.add(structureMap.getAtom(j));
						}
					}
					
					
				}
				
				
				
				StyleMap map = (StyleMap)styling.get(i);//same order as coordinates in their vector
				
				if (map == null) continue;

				int[] ribbons = map.getRibbons();
				float[] ribbonQuality = map.getRibbonQuality();
				boolean[] ribbonVis = map.getRibbonVis();
				
				//find out whether this sequence is selected in the sequence viewer
				boolean selected = map.getSequenceSelected();
				int index = map.getSequenceIndex();
//				System.out.println("index = " + index);
				
				//if the sequence viewer is open
				try{
					if (sequenceStyleMap.size() > 0){
						Set set = sequenceStyleMap.keySet();
						Iterator it = set.iterator();
						while (it.hasNext()){
							Sequence sequence = (Sequence)it.next();
							stylesToSequence.put(sequenceStyleMap.get(sequence), sequence);
						}
		
						
						sequenceViewer.updateSequence((Sequence)stylesToSequence.get(styles), (String)sequences.get(i));
						sequenceViewer.movePanel((Sequence)stylesToSequence.get(styles), index);
						sequenceViewer.setSelected((Sequence)stylesToSequence.get(styles), selected);
					}
				}
				catch (Exception exx){
					System.out.println("exception in updating sequence in undo: " + exx);
				}
				
				
				
				//walk through the map
				boolean[] av = map.getAtomVisibility();
				boolean[] as = map.getAtomSelection();
				short[] ar = map.getAtomRenderingStyle();
				short[] aq = map.getAtomRenderingQuality();
				
				int[] atomSetAssignment = map.getAtomSetAssignment();
				int[] residueSetAssignment = map.getResidueSetAssignment();
				
				int[] labels = map.getLabels();
				String[] labelText = map.getLabelText();
				int[] labelSize = map.getLabelSize();
				boolean[] labelSelection = map.getLabelSelection();
				float[][] labelColor = map.getLabelColor();
				
				int[] residueLabels = map.getResidueLabels();
				String[] residueLabelText = map.getResidueLabelText();
				int[] residueLabelSize = map.getResidueLabelSize();
				boolean[] residueLabelSelection = map.getResidueLabelSelection();
				float[][] residueLabelColor = map.getResidueLabelColor();

				
				float[][] colors = map.getAtomColors();
				float[][] residueColors = map.getResidueColors();
				float[][] ribbonColors = map.getRibbonColors();
				
				if (s.getStructureMap().getAtomCount() != av.length || av.length != as.length || av.length != ar.length || av.length != aq.length){
//					System.out.println("mismatch of atom styles");
					continue;
				}


				for (int j = 0; j < structureMap.getAtomCount(); j++){
					Atom a = structureMap.getAtom(j);
					styles.setVisible(a, av[j], true, true);
					styles.setSelected(a, as[j], true, true);
					styles.setRenderingStyle(a, ar[j], aq[j], true);
					
					//color
					float[] value = colors[j];
					if (value != null){
						//it's a custom color, so fire an event
						AtomColor atomColor = AtomColorFactory.getAtomColor(new Color(value[0], value[1], value[2]));
						styles.setAtomColor(a, atomColor, true, true);
					}
					
					if (atomSetNames.size() > 0){
						int n = atomSetAssignment[j];//set name
						if (n != -1){
							((Vector)atomSets.get(atomSetNames.get(n))).add(a);
						}
					}
					
					//get labels, if any
					int label = labels[j];
					if (label > -1){
						//create a label in the StructureViewer
						if (label == LabelDialog.TYPE_CUSTOM){
							viewer.createLabel(a, label, labelSize[j], labelText[j], labelSelection[j], labelColor[j], false);
						}
						else{
							viewer.createLabel(a, label, labelSize[j], null, labelSelection[j], labelColor[j], false);
						}
					}
				}
				
				for (int j = 0; j < structureMap.getResidueCount(); j++){
					Residue r = structureMap.getResidue(j);
					r.ribbon = ribbonVis[j];
					
					//color
					float[] value = residueColors[j];
					if (value != null){
						//it's a custom color, so fire an event
						ResidueColor residueColor = ResidueColorFactory.getResidueColor(new Color(value[0], value[1], value[2]));
						styles.setResidueColor(r, residueColor, true, true);
					}
					
					//same for ribbon colors
					if (r.ribbon){
						float[] v = ribbonColors[j];
						if (v != null){
							//it's a custom color, so fire an event
							RibbonColor ribbonColor = RibbonColorFactory.getRibbonColor(new Color(v[0], v[1], v[2]));
							styles.setRibbonColor(r, ribbonColor, true, true);
						}
						
					}
					
					//get labels, if any
					int label = residueLabels[j];
					if (label > -1){
						//create a label in the StructureViewer
						if (label == LabelDialog.TYPE_CUSTOM){
							viewer.createLabel(r, label, residueLabelSize[j], residueLabelText[j], residueLabelSelection[j], residueLabelColor[j], false);
						}
						else{
							viewer.createLabel(r, label, residueLabelSize[j], null, residueLabelSelection[j], residueLabelColor[j], false);
						}
					}
					
					//check atoms of this residue to determine whether it needs to be declared invisible
					boolean vis = false;
					for (int k = 0; k < r.getAtomCount(); k++){
						Atom aa = r.getAtom(k);
						if (aa.visible){
							vis = true;
							break;//residue is visible by default
						}
					}
					
					if (!vis) r.visible = false;

				}
				
				//check the ribbons
				for (int j = 0; j < structureMap.getChainCount(); j++){
					Chain chain = structureMap.getChain(j);
					
					if (ribbons.length <= j || ribbons[j] == -1) continue;
					
					//otherwise, create a ribbon of type ribbons[j]
					viewer.createRibbon(chain, ribbons[j], ribbonQuality[j], 1.0f, false);
				}


				//also walk through the residues
				if (atomSetNames.size() > 0){
					for (int j = 0; j < structureMap.getResidueCount(); j++){
						Residue r = structureMap.getResidue(j);
						int n = residueSetAssignment[structureMap.getResidueIndex(r)];//set name
						if (n != -1){
							((Vector)atomSets.get(atomSetNames.get(n))).add(r);
						}
					}
					
				}
				
				viewer.checkDummyAtoms(s, false);
			}
			
			//walk through the structures again and set their order in the sequence viewer, as well as selection state
			
			
			sequenceViewer.setCellColors(cellColors);
			viewer.setAtomSets(atomSets);
			viewer.setAtomSetNames(atomSetNames);
			
			//take care of distance monitors
			int[][] distance = in.getDistanceMonitors();
			for (int i = 0; i < distance.length; i++){
				try{
					Structure s1 = (Structure)structureObjects.get(distance[i][0]);
					Structure s2 = (Structure)structureObjects.get(distance[i][2]);
					
					Atom a1 = s1.getStructureMap().getAtom(distance[i][1]);
					Atom a2 = s2.getStructureMap().getAtom(distance[i][3]);
					
					double d = Algebra.distance(a1.coordinate, a2.coordinate);
					
					if (d == 0.0){
						continue;
					}
					
					String res = (new Double(d)).toString();
					int index = res.indexOf(".");
					String result = null;
					try{
						result = res.substring(0, index+3);//two decimal places after the dot
					}
					catch (Exception ex){
						//if it has only one digit after the dot to begin with
						result = res.substring(0, index+2);
					}
					
					//create the monitor
					viewer.createMonitor(a1, a2, result, null, false);

				}
				catch (Exception ex){
					ex.printStackTrace();
				}
			}
			
			//take care of hydrogen bonds
			int[][] hBonds = in.getHBonds();
			if (hBonds != null){
				for (int i = 0; i < hBonds.length; i++){
					try{
						Structure s1 = (Structure)structureObjects.get(hBonds[i][0]);
						Structure s2 = (Structure)structureObjects.get(hBonds[i][2]);
						
						Atom a1 = s1.getStructureMap().getAtom(hBonds[i][1]);
						Atom a2 = s2.getStructureMap().getAtom(hBonds[i][3]);
						
						
						//create the monitor
						viewer.createHydrogenBond(a1, a2, false);
					}
					catch (Exception ex){
						ex.printStackTrace();
					}
				}
			}
			
			//take care of bumps
			int[][] bumps = in.getBumps();
			if (bumps != null){
				for (int i = 0; i < bumps.length; i++){
					try{
						Structure s1 = (Structure)structureObjects.get(bumps[i][0]);
						Structure s2 = (Structure)structureObjects.get(bumps[i][2]);
						
						Atom a1 = s1.getStructureMap().getAtom(bumps[i][1]);
						Atom a2 = s2.getStructureMap().getAtom(bumps[i][3]);
						
						//create the monitor
						viewer.createBump(a1, a2, false);
					}
					catch (Exception ex){
						ex.printStackTrace();
					}
				}
			}
			
			MonitorDialog md = parent.getMonitorDialog();
			if (md != null) md.updateTree();


			if (independentComponents != null && independentComponents.size() > 0){
				//set up independent structure motion for this set
				viewer.setTrackBumps(in.isTrackBumps());
				
				Vector atoms = new Vector();
				for (int i = 0; i < independentComponents.size(); i++){
					try{
						Atom a = (Atom)independentComponents.get(i);
						if (!atoms.contains(a)) atoms.add(a);
					}
					catch (ClassCastException ex){
						try{
							Residue r = (Residue)independentComponents.get(i);
							for (int j = 0; j < r.getAtomCount(); j++){
								Atom a = r.getAtom(j);
								if (!atoms.contains(a)) atoms.add(a);
							}
						}
						catch (Exception e){}
					}
				}
				
				viewer.setIndependentMotion(atoms, false);
			}
			
			ViewerParameters view = in.getView();
			viewer.getGeometryViewer().setViewerParameters(view);
			
			parent.clearHoldDisplayUpdate();
			
		}
		catch (Exception ex){
//			System.out.println("ex = " + ex);
			ex.printStackTrace();
		}
		
	}
	
	
	
}