package edu.sdsc.sirius.rasmol;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.filechooser.FileFilter;

import edu.sdsc.sirius.util.*;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import edu.sdsc.mbt.*;


public class CommandPanel extends JDialog {
	
	private Manager parent;
	private JFrame frame;
	
	private int width = 350;
	private int height = 500;
	
	private CommandPanel thisObject;
	
	private JScrollPane scrollPane;
	
	private JEditorPane editorPane = new JEditorPane();
	private JTextField text = new JTextField(50);
	
	private JButton execButton = new JButton("Execute");
	
	private RasmolReader reader;
	private StringBuffer buffer;
	
	private Font font = new Font("Courier", 0, 14);
	private int letterWidth = 9;//compute it in the constructor
	
	private boolean rasmol = false;//whether the loaded script is imported from Rasmol, so that rotation transformation is needed
	private JCheckBoxMenuItem menuDataRasmol;
	
	/**
	 * This panel contains command-line and text panel for entering and running scripting commands
	 * that control appearance of the loaded structures
	 * @param frame
	 * @param parent
	 */
	public CommandPanel(JFrame frame, Manager p){
		
		super(frame, "Command Panel", false);
		this.parent = p;
		this.frame = frame;
		
		
		thisObject = this;
		
		reader = parent.getRasmolReader();
		
		if (parent.getStructureViewer().getLoadedStructures().size() == 1){
			reader.setStructure((Structure)parent.getStructureViewer().getLoadedStructures().get(0));
		}
		
		getContentPane().setLayout(new BorderLayout());
		
		editorPane.setFont(font);
		scrollPane = new JScrollPane(editorPane);

		getContentPane().add(scrollPane, BorderLayout.CENTER);
		getContentPane().add(createMenuBar(), BorderLayout.NORTH);
		
		JPanel bottom = new JPanel();
		bottom.setLayout(new BorderLayout());
		
		text.setFont(font);
		text.addKeyListener(new KeyAdapter(){
			public void keyPressed(KeyEvent k){
				if (k.getKeyCode() == KeyEvent.VK_ENTER){
					//do the same thing as with Execute button
					String line = text.getText().trim();
					if (line.length() == 0) return;
					if (line.startsWith("clear")){
						//clear the editor pane
						editorPane.setText("");
						text.setText("");
						resizeEditorPane();
						return;
					}
					
					try{
						String result = reader.readLine(parent, line, null, false);
						if (result != null){
							text.setText("");
							parent.displayErrorMessage("Error running command: " + result);
							return;
						}
					}
					catch (NullPointerException ex){
//						ex.printStackTrace();
						//a structure may have been loaded outside the interpreter
						Vector structures = parent.getStructureViewer().getLoadedStructures();
						if (structures.size() != 1){
							text.setText("");
							parent.displayErrorMessage("Either no structures or more than one are loaded. To connect interpreter to a particular structure, use connect command with structure name.");
							return;
						}
						
						//there is only one structure
						try{
							reader.setStructure((Structure)structures.get(0));
							String result = reader.readLine(parent, line, null, false);
							if (result != null){
								text.setText("");
								parent.displayErrorMessage("Error running command: " + result);
								return;
							}
						}
						catch (Exception e){
							text.setText("");
							parent.displayExceptionMessage("Exception running command", e);
							return;
						}
					}
					buffer = new StringBuffer();
					String content = editorPane.getText();
					buffer.append(content);
					buffer.append(line);
					buffer.append("\n");
					editorPane.setText(buffer.toString());
					
					text.setText("");
					
					resizeEditorPane();
				}
			}
		});
		
		
		bottom.add(text, BorderLayout.CENTER);
		
		JPanel temp = new JPanel();
		temp.setLayout(null);
		temp.setPreferredSize(new Dimension(100,20));
		execButton.setBounds(10, 0, 90, 25);
		execButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				//run the line entered in the field
				String line = text.getText().trim();
				if (line.length() == 0) return;
				if (line.startsWith("clear")){
					//clear the editor pane
					editorPane.setText("");
					text.setText("");
					resizeEditorPane();
					return;
				}
				try{
					String result = reader.readLine(parent, line, null, false);
					if (result != null){
						text.setText("");
						parent.displayErrorMessage("Error running command: " + result);
						return;
					}
				}
				catch (NullPointerException ex){
					//a structure may have been loaded outside the interpreter
					Vector structures = parent.getStructureViewer().getLoadedStructures();
					if (structures.size() != 1){
						text.setText("");
						parent.displayErrorMessage("Either no structures or more than one are loaded. To connect interpreter to a particular structure, use attach command with structure name.");
						return;
					}
					
					//there is only one structure
					try{
						reader.setStructure((Structure)structures.get(0));
						String result = reader.readLine(parent, line, null, false);
						if (result != null){
							text.setText("");
							parent.displayErrorMessage("Error running command: " + result);
							return;
						}
					}
					catch (Exception exx){
						text.setText("");
						parent.displayExceptionMessage("Exception running command", exx);
						return;
					}
				}
				buffer = new StringBuffer();
				String content = editorPane.getText();
				buffer.append(content);
				buffer.append(line);
				buffer.append("\n");
				editorPane.setText(buffer.toString());
				
				text.setText("");
				
				resizeEditorPane();
			}
		});
		
		
		temp.add(execButton);
		bottom.add(temp, BorderLayout.EAST);
		bottom.setBorder(new EmptyBorder(5,5,5,5));
		
		getContentPane().add(bottom, BorderLayout.SOUTH);
		
		WindowAdapter closer = new WindowAdapter(){
			public void windowClosing( WindowEvent event ){
				setVisible(false);
				thisObject.parent.closeCommandPanel();
			}
		};
		addWindowListener( closer );
		
		this.addComponentListener(new ComponentAdapter(){
			public void componentResized(ComponentEvent e){
				resizeEditorPane();
			}
		});
		
		//position the window
		Dimension dim = getToolkit().getScreenSize();
		
		//position the panel next to the main window on the right
		//if it ends up being wider than screen, fit it to the width of the screen
		if ((frame.getX() + frame.getWidth() + width) > dim.getWidth()){
			setBounds(dim.width - width, frame.getY(), width, height);
		}
		else{
			setBounds(frame.getX() + frame.getWidth(), frame.getY(), width, height);
		}
		setSize(width, height);

		setVisible(true);
		
		text.requestFocus();
	}
	
	private void resizeEditorPane(){
		//get the length of the longest line to avoid wrapping
		int max = width-10;
		String content = editorPane.getText();
		StringTokenizer tok = new StringTokenizer(content, "\n", false);
		while (tok.hasMoreTokens()){
			String token = tok.nextToken();
			
			//get the length of the string in pixels
			int a = token.length()*letterWidth;
			if (a > max) max = a;
		}
		
		
		editorPane.setSize(Math.max(width-10, max), editorPane.getHeight());
		scrollPane.revalidate();
		scrollPane.repaint();
	}
	
	private JMenuBar createMenuBar(){

		
		JMenuBar menuBar = new JMenuBar();
		
		JMenu menuFile = new JMenu("File");
		JMenuItem menuFileOpen = new JMenuItem("Open command script...");
		menuFileOpen.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				Thread runner = new Thread(){
					public void run(){
						try{
							JFileChooser chooser = new JFileChooser(parent.getLastUsedDirectory());
							chooser.setDialogTitle("Open command script");
							
							chooser.setDialogType(JFileChooser.OPEN_DIALOG);

							FileFilter filter2 = new FileNameFilter("ras", "RasMol script");
							chooser.setFileFilter(filter2);
							FileFilter filter = new FileNameFilter("scs", "Sirius command script");
							chooser.setFileFilter(filter);
							

							chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
							int result = chooser.showOpenDialog(parent.getApplicationFrame());
							if (result == JFileChooser.CANCEL_OPTION){
								return;
							}
							File filename = chooser.getSelectedFile();
							
							String n = filename.toString();
							if (n.toUpperCase().endsWith("RAS")){
								rasmol = true;
							}
							else{
								rasmol = false;
							}

							menuDataRasmol.setSelected(rasmol);

							StringBuffer buffer = new StringBuffer();
							BufferedReader bufferedReader = new BufferedReader(new FileReader(filename));
							String line = null;
							while ((line = bufferedReader.readLine()) != null){
								buffer.append(line);
								buffer.append("\n");
							}
							
							bufferedReader.close();
							
							editorPane.setText(buffer.toString());
							
							//save the used directory
							String dir = filename.getParentFile().toString();
							parent.setLastUsedDirectory(dir);
						}
						catch (Exception e){
							parent.displayExceptionMessage("Exception opening command script", e);
						}
					}
				};
				runner.start();

			}
		});
		
		JMenuItem menuFileSave = new JMenuItem("Save command script...");
		menuFileSave.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				saveCommandScript();
			}
		});

		
		JMenuItem menuFileExport = new JMenuItem("Export as RasMol script...");
		menuFileExport.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				Thread runner = new Thread(){
					public void run(){
						String content = editorPane.getText();
						//check content for Sirius-specific keywords
						if (content.indexOf("render") > 0 || content.indexOf("connect") > 0 || content.indexOf("addselect") > 0){
							if (JOptionPane.showConfirmDialog(
									parent.getApplicationFrame(), "The script contains keywords that are not readable by RasMol. Continue?", "Confirmation",
									JOptionPane.YES_NO_OPTION) == JOptionPane.NO_OPTION){
								return;
							}

						}
						
						try{
							IOHandler.saveFile(parent, content, "ras", "RasMol script");
						}
						catch (Exception e){
							parent.displayExceptionMessage("Exception trying to save RasMol script", e);
						}
					}
				};
				runner.start();
			}
		});

		JMenuItem menuFileClose = new JMenuItem("Close panel");
		menuFileClose.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				Thread runner = new Thread(){
					public void run(){
						parent.closeCommandPanel();
					}
				};
				runner.start();
			}
		});
		
		menuFile.add(menuFileOpen);
		menuFile.add(menuFileSave);
		menuFile.add(new JSeparator());
		menuFile.add(menuFileExport);
		menuFile.add(new JSeparator());
		menuFile.add(menuFileClose);
		
		JMenu menuData = new JMenu("Script");
		
		JMenuItem menuDataRun = new JMenuItem("Run script");
		menuDataRun.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				Thread runner = new Thread(){
					public void run(){
						runScript();
					}
				};
				runner.start();

			}
		});
		
		menuDataRasmol = new JCheckBoxMenuItem("RasMol script");
		menuDataRasmol.setSelected(rasmol);
		menuDataRasmol.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				rasmol = menuDataRasmol.isSelected();
			}
		});

		JMenuItem menuDataClear = new JMenuItem("Clear");
		menuDataClear.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				text.setText("");
				editorPane.setText("");
			}
		});
		
		
		menuData.add(menuDataRun);
		menuData.add(new JSeparator());
		menuData.add(menuDataRasmol);
		menuData.add(menuDataClear);
		
		
		menuBar.add(menuFile);
		menuBar.add(menuData);
		
		return menuBar;
		
	
	}
	
	private void saveCommandScript(){
		Thread runner = new Thread(){
			public void run(){
				String content = editorPane.getText();
				try{
					IOHandler.saveFile(parent, content, "scs", "Sirius command script");
				}
				catch (Exception e){
					parent.displayExceptionMessage("Exception trying to save command script", e);
				}
			}
		};
		runner.start();

	}
	
	/**
	 * This method runs commands in the editor panel. If any commands are selected, the selection is
	 * executed, and deselected lines are ignored
	 *
	 */
	private void runScript(){
		
		String content = editorPane.getText();
		StringBuffer result = new StringBuffer();
		StringTokenizer tok = new StringTokenizer(content, "\n", false);
		parent.getStructureViewer().getGeometryViewer().hold = true;
		while (tok.hasMoreTokens()){
			String token = tok.nextToken().trim();
			String outcome = reader.readLine(parent, token, null, rasmol);
			if (outcome != null){
				result.append(outcome);
				result.append("\n");
				
				if (outcome.startsWith("load")){
					parent.displayErrorMessage("Errors loading the structure specified in the script");
					break;
				}
			}
			
		}
		
//		reader.clearSelection();
//		reader.clear();
		
//		System.err.println("updating appearance: " + parent.getStructureViewer().getSelectedChains().size());
		parent.getStructureViewer().getGeometryViewer().hold = false;
		parent.getStructureViewer().updateAppearance();
		
		if (result.length() > 0){
			parent.displayTextMessage("RasMol script import errors", result.toString());
		}
	}
	
}