package edu.sdsc.sirius.rasmol;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.io.*;
import java.awt.*;

import edu.sdsc.sirius.dialogs.VisibilityDialog;
import edu.sdsc.sirius.io.*;
import edu.sdsc.mbt.viewables.*;
import edu.sdsc.sirius.viewers.*;
import edu.sdsc.mbt.*;
import edu.sdsc.mbt.util.*;
import edu.sdsc.sirius.util.*;


/**
 * This class provides a static method for reading rasmol input scripts with one line being
 * input at a time. Then it calls the appropriate methods in the viewers and styles object
 * to perform the requested operation.
 * @author Sasha Buzko
 *
 */
public class RasmolReader {
	
	public static HashMap colorMap = new HashMap();
	
	private static final Object value = new Object();
	private final HashMap groups = new HashMap();//groups within parentheses
	
	private StructureViewer viewer = null;
	
	static {
		
		colorMap.put("black", new Color(0,0,0));
		colorMap.put("blue", new Color(0,0,255));
		colorMap.put("bluetint", new Color(175, 214, 255));
		colorMap.put("brown", new Color(175, 117, 89));
		colorMap.put("cyan", new Color(0, 255, 255));
		colorMap.put("gold", new Color(255, 156, 0));
		colorMap.put("gray", new Color(125, 125, 125));
		colorMap.put("grey", new Color(125, 125, 125));
		colorMap.put("green", new Color(0, 255, 0));
		colorMap.put("greenblue", new Color(46, 139, 87));
		colorMap.put("greentint", new Color(152, 255, 179));
		colorMap.put("hotpink", new Color(255, 0, 101));
		colorMap.put("magenta", new Color(255, 0, 255));
		colorMap.put("orange", new Color(255, 165, 0));
		colorMap.put("pink", new Color(255, 101, 117));
		colorMap.put("pinktint", new Color(255, 171, 187));
		colorMap.put("purple", new Color(160, 32, 240));
		colorMap.put("red", new Color(255,0,0));
		colorMap.put("redorange", new Color(255, 69, 0));
		colorMap.put("seagreen", new Color(0, 250, 109));
		colorMap.put("skyblue", new Color(58, 144, 255));
		colorMap.put("violet", new Color(238, 130, 238));
		colorMap.put("white", new Color(255, 255, 255));
		colorMap.put("yellow", new Color(255, 255, 0));
		colorMap.put("yellowtint", new Color(246, 246, 117));
		
	}
	
	private static Vector predefinedSets = new Vector();
	
	static{
		
		predefinedSets.add("at");
		predefinedSets.add("acidic");
		predefinedSets.add("acyclic");
		predefinedSets.add("aliphatic");
		predefinedSets.add("alpha");
		predefinedSets.add("amino");
		predefinedSets.add("aromatic");
		predefinedSets.add("backbone");
		predefinedSets.add("basic");
		predefinedSets.add("bonded");
		predefinedSets.add("buried");
		predefinedSets.add("cg");
		predefinedSets.add("charged");
		predefinedSets.add("cyclic");
		predefinedSets.add("cystine");
		predefinedSets.add("helix");
		predefinedSets.add("hetero");
		predefinedSets.add("hydrogen");
		predefinedSets.add("hydrophobic");
		predefinedSets.add("ions");
		predefinedSets.add("large");
		predefinedSets.add("ligand");
		predefinedSets.add("medium");
		predefinedSets.add("neutral");
		predefinedSets.add("nucleic");
		predefinedSets.add("polar");
		predefinedSets.add("protein");
		predefinedSets.add("purine");
		predefinedSets.add("pyrimidine");
		predefinedSets.add("selected");
		predefinedSets.add("sheet");
		predefinedSets.add("sidechain");
		predefinedSets.add("small");
		predefinedSets.add("solvent");
		predefinedSets.add("surface");
		predefinedSets.add("turn");
		predefinedSets.add("water");
		
	}
	
	//current variables that are set at each load call
	private Structure structure = null;
	private StructureMap map = null;
	private StructureStyles styles = null;
	
	private boolean trace = false;
	private boolean cartoon = false;
	
	public RasmolReader(StructureViewer v){
		this.viewer = v;
	}
	
	public void setStructure(Structure s){
		this.structure = s;
		map = structure.getStructureMap();
		styles = map.getStructureStyles();
	}
	
	public final String readLine(Manager manager, String line, String path, boolean rasmol){
		
		if (viewer == null) viewer = manager.getStructureViewer();
		
		if (!line.startsWith("load") && line.indexOf("background") < 0 && line.indexOf("attach") < 0 && structure == null){
			Vector structures = viewer.getLoadedStructures();
			if (structures.size() == 0){
				return "Nothing is loaded. Either open a file interactively or run a load command.";
			}
			if (structures.size() != 1 && structure == null){
				return "Use the attach command to connect the interpreter to one of the loaded structures.";
			}
			
			//there is only one structure
			try{
				structure = (Structure)structures.get(0);
				map = structure.getStructureMap();
				styles = map.getStructureStyles();
			}
			catch (Exception exx){
				return "Unable to attach to loaded structure";
			}

		}
		
//		System.out.println("line = " + line);
		
		if (line.startsWith("load")){
			String[] args = new String[3];
			StringTokenizer tok = new StringTokenizer(line, " ", false);
			int count = 0;
			while (tok.hasMoreTokens()){
				args[count++] = tok.nextToken();
			}
			
			//there may be no third argument (file type skipped). in this case, treat second argument as file name
			
			if (count < 3){
				args[2] = args[1];
			}
			
			//check whether file name contains full path or only the file name
			if (args[2].indexOf("/") < 0 && args[2].indexOf("\\") < 0){
				//it's just the name
				//get the directory where the ras file is located and add the full path
				String dir = null;
				if (path == null){
					dir = StylesPreferences.workingDirectory;
				}
				else{
					int in = path.lastIndexOf(File.separator);
					dir = path.substring(0, in);
				}
				
				args[2] = dir + File.separator + args[2];
				
			}
			
			//token 2 describes the file format, and token 3 contains file location
			File f = new File(args[2]);
			if (!f.exists()){
//				manager.displayErrorMessage("File specified in the RasMol script does not exist");
				return line + ": file does not exist";
			}
			
			//read the file and load the entry
			Entry entry = null;
			try{
				entry = EntryFactory.load( f, -1 );
			}
			catch (NumberFormatException e){
//				manager.displayErrorMessage("Incorrect file format: " + f.toString());
				return line + ": incorrect file format";
			}
			
			if (entry == null){
//				manager.displayErrorMessage("Unable to load specified file: " + f.toString());
				return line + ": unable to load file";
			}
	
			manager.addEntry(entry, true, true);
			
			if (entry.TYPE == Entry.STRUCTURE_ENTRY){
				structure = ((StructureEntry)entry).getStructure();
				map = structure.getStructureMap();
				styles = map.getStructureStyles();
				
				if (rasmol){
					//rotate the structure
					viewer.getGeometryViewer().rotate(map.getCenter(), Algebra.X_AXIS, 180);
				}
			}
					
		}
		else if (line.startsWith("attach")){
			//connect the interpreter to a different structure
			String name = line.substring(line.indexOf("attach")+6).trim();
			
			Vector structures = viewer.getLoadedStructures();
			for (int i = 0; i < structures.size(); i++){
				Structure s = (Structure)structures.get(i);
				String n = viewer.getStructureName(s);
				if (n.toUpperCase().equals(name.toUpperCase())){
					//this is it
					this.structure = s;
					this.map = s.getStructureMap();
					this.styles = map.getStructureStyles();
					break;
				}
			}
			
			if (structure == null){
				return "Structure not found";
			}
		}
		else if (line.startsWith("set strands")){
			//get the value after the keywords
			String num = line.substring(11).trim();
			if (num.equals("1")){
				trace = true;
			}
			else{
				trace = false;
			}
			
//			System.out.println("set trace to " + trace);
		}
		else if (line.startsWith("cartoon on")){
			cartoon = true;
		}
		else if (line.startsWith("zoom")){
			//zoom into current selection
			viewer.zoomSelection();
		}
		else if (line.startsWith("strands")){
			//selected part of the structure needs a ribbon
			//add the ribbon by default
			for (int i = 0; i < map.getChainCount(); i++){
				Chain c = map.getChain(i);
				//check if this chain is in the selection
				boolean selected = false;
				for (int j = 0; j < c.getResidueCount(); j++){
					Residue r = c.getResidue(j);
					if (!r.getClassification().equals(Residue.COMPOUND_AMINO_ACID) && 
							!r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)) continue;
					if (styles.isSelected(r)){
						selected = true;
						break;
					}
					//if the residue itself is not selected, check individual atoms
					//it's likely that just the backbone atoms are selected
					for (int k = 0; k < r.getAtomCount(); k++){
						Atom a = r.getAtom(k);
						if (styles.isSelected(a)){
							if (a.backbone){
								selected = true;
								break;
							}
						}
					}
				}
				if (selected){
//					System.out.println("strands command: chain " + c.getChainId() + " selection = " + selected + ", trace = " + trace);
					if (trace){
						StylesPreferences.smoothRibbon = false;
						viewer.createRibbonInSelection(StylesPreferences.RIBBON_UNIFORM_ROUND, 2.0f, 0.4f, true);
					}
					else{
						StylesPreferences.smoothRibbon = true;//just in case
						viewer.createRibbonInSelection(StylesPreferences.RIBBON_RENDERED, 2.0f, 1.0f, true);
						StylesPreferences.smoothRibbon = false;
					}
					
				}
			}
			
			viewer.getGeometryViewer().updateView();
		}
		else if (line.startsWith("trace")){
			//selected part of the structure needs a ribbon
			//add the ribbon by default
//			viewer.getSelectedChains().clear();
			for (int i = 0; i < map.getChainCount(); i++){
				Chain c = map.getChain(i);
				//check if this chain is in the selection
				boolean selected = false;
				for (int j = 0; j < c.getResidueCount(); j++){
					Residue r = c.getResidue(j);
					if (!r.getClassification().equals(Residue.COMPOUND_AMINO_ACID) && 
							!r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)) continue;
					
					if (styles.isSelected(r)){
						selected = true;
						break;
					}
					
					//if the residue itself is not selected, check individual atoms
					//it's likely that just the backbone atoms are selected
					for (int k = 0; k < r.getAtomCount(); k++){
						Atom a = r.getAtom(k);
						if (a.backbone && styles.isSelected(a)){
							selected = true;
							break;
						}
					}
				}

				if (selected){
					if (cartoon){
						StylesPreferences.smoothRibbon = true;
						viewer.createRibbonInSelection(StylesPreferences.RIBBON_RENDERED, 2.0f, 0.8f, true);
						StylesPreferences.smoothRibbon = false;
						cartoon = false;
					}
					else{
						StylesPreferences.smoothRibbon = false;
						viewer.createRibbonInSelection(StylesPreferences.RIBBON_UNIFORM_ROUND, 2.0f, 0.8f, true);

					}
//					styles.updateChain(c, true);
				}
			}
			
			viewer.getGeometryViewer().updateView();

			
		}
		else if (line.startsWith("ribbons")){
			String flag = line.substring(line.indexOf("ribbons")+7).trim();
			processRibbon(flag);
		}
		else if (line.startsWith("ribbon")){
			String flag = line.substring(line.indexOf("ribbon")+6).trim();
			processRibbon(flag);
		}
		else if (line.startsWith("cartoon")){
			String flag = line.substring(line.indexOf("cartoon")+7).trim();
			processRibbon(flag);
		}
		else if (line.startsWith("translate")){
			try{
				String flag = line.substring(line.indexOf("translate")+9).trim();
				return processMove(flag, rasmol);
			}
			catch (Exception e){
				return null;
			}
		}
		else if (line.startsWith("move")){
			try{
				String flag = line.substring(line.indexOf("move")+4).trim();
				return processMove(flag, rasmol);
			}
			catch (Exception ex){
				return null;
			}
		}
		else if (line.startsWith("rotate")){
			try{
				String flag = line.substring(line.indexOf("rotate")+6).trim();
				return processRotation(flag, rasmol);
			}
			catch (Exception e){
				e.printStackTrace();
				return null;
			}
		}
		else if (line.indexOf("background") > -1){
			int index = line.indexOf("background");
			String color = line.substring(index+10).trim();
			
			Color c = getColor(color);
			if (c == null) return line + ": Incorrect color syntax";
			
			float[] temp = new float[3];
			
			temp[0] = (float)c.getRed()/(float)255.0;
			temp[1] = (float)c.getGreen()/(float)255.0;
			temp[2] = (float)c.getBlue()/(float)255.0;
			
			viewer.getGeometryViewer().setBackgroundColor(temp);
			viewer.getGeometryViewer().updateView();
		}
		else if (line.indexOf("wireframe") > -1){
			String flag = line.substring(line.indexOf("wireframe")+9).trim();
			
			//if something is selected, apply the change to the selection only
			//if nothing is selected, apply to the entire structure
			if (styles.getSelectionFlag() == StructureStyles.FLAG_NONE){
				if (flag.startsWith("off")){
					//hide atomic structure
					styles.hideAll(false);
				}
				else if (flag.startsWith("on")){
					//show atomic structure
					styles.showAll(false);
				}
			}
			else{
				//something is selected
				if (flag.startsWith("off")){
					//hide atomic structure of the selection
					viewer.setVisibilityInSelection(VisibilityDialog.VIS_HIDE);
				}
				else if (flag.startsWith("on")){
					//show atomic structure of the selection
					viewer.setVisibilityInSelection(VisibilityDialog.VIS_SHOW);
				}
				else if (flag.length() == 0){
					viewer.setVisibilityInSelection(VisibilityDialog.VIS_SHOW);//assume it's on
				}
				else{
					//get the number that follows: if it's less than 200, draw stick, else draw CPK
					int size = Integer.parseInt(line.substring(9).trim());
					Enumeration selectedAtoms = styles.getSelection().keys();
					if (size < 80){
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_STICK, StylesPreferences.RENDERING_FAST, false, true);
					}
					if (size >= 80 && size < 200){
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_BALL_STICK, StylesPreferences.RENDERING_FAST, false, true);
					}
					else{
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_CPK, StylesPreferences.RENDERING_FAST, false, true);

					}
				}
			}
			
			viewer.getGeometryViewer().updateView();
		}
		else if (line.indexOf("render") > -1){
			String flag = line.substring(line.indexOf("render")+6).trim();

			//if something is selected, apply the change to the selection only
			//if nothing is selected, do not apply to the entire structure, to avoid unintended long waits for rendering to complete
			if (styles.getSelectionFlag() != StructureStyles.FLAG_NONE){
				//something is selected
				Enumeration selectedAtoms = styles.getSelection().keys();
				
				if (flag.startsWith("lines")){
					String quality = flag.substring(flag.indexOf("lines")+5).trim();
					if (quality.startsWith("best")){
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_LINES, StylesPreferences.RENDERING_BEST, true, true);
					}
					else{
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_LINES, StylesPreferences.RENDERING_FAST, true, true);
					}
				}
				else if (flag.startsWith("stick")){
					String quality = flag.substring(flag.indexOf("stick")+5).trim();
					if (quality.startsWith("best")){
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_STICK, StylesPreferences.RENDERING_BEST, true, true);
					}
					else{
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_STICK, StylesPreferences.RENDERING_FAST, true, true);
					}
				}
				else if (flag.startsWith("ballstick")){
					String quality = flag.substring(flag.indexOf("ballstick")+9).trim();
					if (quality.startsWith("best")){
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_BALL_STICK, StylesPreferences.RENDERING_BEST, true, true);
					}
					else{
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_BALL_STICK, StylesPreferences.RENDERING_FAST, true, true);
					}
				}
				else if (flag.startsWith("spacefill")){
					String quality = flag.substring(flag.indexOf("spacefill")+9).trim();
					if (quality.startsWith("best")){
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_CPK, StylesPreferences.RENDERING_BEST, true, true);
					}
					else{
						styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_CPK, StylesPreferences.RENDERING_FAST, true, true);
					}
				}

			}
			
			viewer.getGeometryViewer().updateView();
			
		}
		else if (line.indexOf("spacefill") > -1){
			//get the number that follows: if it's less than 200, draw stick, else draw CPK
			int size = Integer.parseInt(line.substring(9).trim());
			Enumeration selectedAtoms = styles.getSelection().keys();
			if (size < 80){
				styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_STICK, StylesPreferences.RENDERING_BEST, false, true);
			}
			if (size >= 80 && size < 200){
				styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_BALL_STICK, StylesPreferences.RENDERING_BEST, false, true);
			}
			else{
				styles.setRenderingStyle(selectedAtoms, StylesPreferences.RENDERING_CPK, StylesPreferences.RENDERING_BEST, false, true);

			}
			
			viewer.getGeometryViewer().updateView();

			
		}
		else if (line.indexOf("define") > -1){
			//define an atom set
			String content = line.substring(line.indexOf("define")+6).trim();
			int index = content.indexOf(" ");
			String name = content.substring(0, index).trim();
			String expression = content.substring(index+1);
			
			Pattern pattern = Pattern.compile(" and ");
			Matcher matcher = pattern.matcher(expression);
			String temp = matcher.replaceAll(" & ");

			pattern = Pattern.compile(" not ");
			matcher = pattern.matcher(temp);
			String temp2 = matcher.replaceAll(" ! ");
			
			pattern = Pattern.compile(" or ");
			matcher = pattern.matcher(temp2);
			String next = matcher.replaceAll(" | ").trim();
						
			Hashtable finalSet = parseExpression(next);//set of atoms and residues that is defined by the expression
			
			Vector list = new Vector();
			Enumeration selectedAtoms = finalSet.keys();

			//run residues first, then leftover atoms
			while (selectedAtoms.hasMoreElements()){
				try{
					Residue r = (Residue)selectedAtoms.nextElement();
					if (!list.contains(r)) list.add(r);
				}
				catch (Exception ex){
					continue;
				}
			}
			
			//run through the lone atoms
			selectedAtoms = finalSet.keys();
			while (selectedAtoms.hasMoreElements()){
				try{
					
					Atom a = (Atom)selectedAtoms.nextElement();
					if (list.contains(a.structure.getStructureMap().getResidue(a))) continue;
					if (!list.contains(a)) list.add(a);
				}
				catch (ClassCastException ex){}
			}

			viewer.createSet(name, list);
			
		}
		else if (line.indexOf("dots") > -1){
			String next = line.substring(line.indexOf("dots")+4).trim();
			
			int qualityFlag = StylesPreferences.SURFACE_QUALITY_MEDIUM;//default

			if (next.startsWith("off")){
				//delete all existing surfaces
				viewer.deleteSurfaces(false);
			}
			else{
				
				if (next.startsWith("on")){
					//delete all surfaces and add a new one for the current selection
					viewer.deleteSurfaces(true);
				}
				
				
				//add a new surface to the selected atoms
				
				String n = null;
				if (next.indexOf(" ") > 0){
					n = next.substring(0, next.indexOf(" "));
				}
				else{
					n = next;
				}
				
				int num = 100;
				try{
					num = Integer.parseInt(n);
				}
				catch (NumberFormatException ex){}
				
				if (num <= 200){
					qualityFlag = StylesPreferences.SURFACE_QUALITY_MEDIUM;
				}
				else if (num > 200 && num < 600){
					qualityFlag = StylesPreferences.SURFACE_QUALITY_HIGH;
				}
				else{
					qualityFlag = StylesPreferences.SURFACE_QUALITY_ULTRA;
				}
				
				//just in case, make sure the involved atoms are visible (otherwise, the surface is invisible)
				viewer.setVisibilityInSelection(VisibilityDialog.VIS_SHOW);
				
				//get the list of currently selected atoms
				Vector list = new Vector();
				Enumeration selectedAtoms = styles.getSelection().keys();

				//run through the atoms
				selectedAtoms = styles.getSelection().keys();
				while (selectedAtoms.hasMoreElements()){
					try{
						
						Atom a = (Atom)selectedAtoms.nextElement();
						if (!list.contains(a)) list.add(a);
					}
					catch (ClassCastException ex){}
				}
				
				if (list.size() == 0){
					return line + ": no selection for the surface";
				}
				
				viewer.createSurface(list, qualityFlag, true);

			}
			
			viewer.getGeometryViewer().updateView();

			
		}
		else if (line.indexOf("color") > -1){
//			System.out.println(line);
			
			//apply specified color to whatever is selected
			String c = line.substring(line.indexOf("color")+5).trim();
			Color color = getColor(c);
			
			if (color == null){
				//check named color groups
				ColorScheme scheme = ColorManager.getColorScheme(c.toUpperCase());
				
				if (scheme == null) return line + ": unable to read";
				
				//scan all residues and if a residue is not fully selected, check the backbone,
				//since it may influence whether the ribbon (if any) gets colored as well
				for (int i = 0 ; i < map.getResidueCount(); i++){
					Residue r = map.getResidue(i);
					ResidueColor rc = null;
					if (styles.isSelected(r)){
						if (scheme.getName().equals("AMINO")){
							//per residue coloring
							rc = ResidueColorFactory.getResidueColor(scheme.getColor(r.getCompoundCode()));
							styles.setResidueColor(r, rc, true, true);
							if (StylesPreferences.commonRibbonColoring) styles.setRibbonColor(r, rc, false, true);
						}
						else if (scheme.getName().equals("STRUCTURE")){
							rc = ResidueColorFactory.getResidueColor(scheme.getColor(r.getConformationType()));
							styles.setResidueColor(r, rc, true, true);
							if (StylesPreferences.commonRibbonColoring) styles.setRibbonColor(r, rc, false, true);
						}
						else if (scheme.getName().equals("CPK")){
							for (int j = 0; j < r.getAtomCount(); j++){
								Atom a = r.getAtom(j);
								AtomColor ac = AtomColorFactory.getAtomColor(scheme.getColor(a.getElement()));
								styles.setAtomColor(a, ac, true, true);
							}
						}
					}
					else{
						//check individual atoms
						boolean backbone = true;//is reset to false if at least one backbone atom is not selected
						for (int j = 0; j < r.getAtomCount(); j++){
							Atom a = r.getAtom(j);
							
							if (styles.isSelected(a)){
								AtomColor ac = null;
								if (scheme.getName().equals("CPK")){
									ac = AtomColorFactory.getAtomColor(scheme.getColor(a.getElement()));
								}
								else if (scheme.getName().equals("AMINO")){
									ac = AtomColorFactory.getAtomColor(scheme.getColor(r.getCompoundCode()));
								}
								else if (scheme.getName().equals("STRUCTURE")){
									ac = AtomColorFactory.getAtomColor(scheme.getColor(r.getConformationType()));
								}
								styles.setAtomColor(a, ac, true, true);
							}
							else{
								if (!backbone) continue;//already skipped
								//if the current atom is not selected, check whether it's a part of the backbone.
								//if yes, then ribbon is not colored
								if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID) ||
										r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)){
									if (a.backbone){
										backbone = false;
									}
								}
							}
						}
						
						if (backbone){
							//color the ribbon
							rc = null;
							if (scheme.getName().equals("AMINO")){
								//per residue coloring
								rc = ResidueColorFactory.getResidueColor(scheme.getColor(r.getCompoundCode()));
								styles.setResidueColor(r, rc, true, true);
								if (StylesPreferences.commonRibbonColoring) styles.setRibbonColor(r, rc, false, true);
							}
							else if (scheme.getName().equals("STRUCTURE")){
								rc = ResidueColorFactory.getResidueColor(scheme.getColor(r.getConformationType()));
								styles.setResidueColor(r, rc, true, true);
								if (StylesPreferences.commonRibbonColoring) styles.setRibbonColor(r, rc, false, true);
							}
							
							styles.setRibbonColor(r, rc, false, true);
						}
					}
				}
				
				viewer.getGeometryViewer().updateView();
				
				return null;
			}
			
			AtomColor atomColor = AtomColorFactory.getAtomColor(color);
			ResidueColor residueColor = ResidueColorFactory.getResidueColor(color);
			RibbonColor ribbonColor = RibbonColorFactory.getRibbonColor(color);
			
			//color whatever is selected in the display
			//quick check: whether the entire structure is selected
			if (styles.getSelectionFlag() == StructureStyles.FLAG_ALL){
				styles.selectNone(false, false);
				styles.setStructureColor(residueColor, atomColor);
			}
			else{
				//scan all residues and if a residue is not fully selected, check the backbone,
				//since it may influence whether the ribbon (if any) gets colored as well
				for (int i = 0 ; i < map.getResidueCount(); i++){
					Residue r = map.getResidue(i);
					if (styles.isSelected(r)){
						styles.setResidueColor(r, residueColor, true, true);
						styles.setRibbonColor(r, ribbonColor, false, true);
					}
					else{
						//check individual atoms
						boolean backbone = true;//is reset to false if at least one backbone atom is not selected
						for (int j = 0; j < r.getAtomCount(); j++){
							Atom a = r.getAtom(j);
							if (styles.isSelected(a)){
								styles.setAtomColor(a, atomColor, true, true);
							}
							else{
								if (!backbone) continue;//already skipped
								//if the current atom is not selected, check whether it's a part of the backbone.
								//if yes, then ribbon is not colored
								if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID) || 
										r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)){
									if (a.backbone){
										backbone = false;
									}
								}
							}
						}
						
						if (backbone){
							//color the ribbon
							styles.setRibbonColor(r, ribbonColor, false, true);
						}
					}
				}
			}
			
			viewer.getGeometryViewer().updateView();
		}
		else if (line.startsWith("zap")){
			//clear all
			manager.clearEntries(false);
		}
		else if (line.startsWith("close")){
			//clear all
			manager.clearEntries(false);
		}
		else if (line.startsWith("reset")){
			//restore original coordinates of all loaded structures and reset view
			for (int i = 0; i < viewer.getLoadedStructures().size(); i++){
				Structure s = (Structure)viewer.getLoadedStructures().get(i);
				viewer.restoreInitialCoordinates(s);
			}
			//reset view
			viewer.resetView();
		}
		else if (line.startsWith("addselect")){
			//selection without removing previous
			String nn = line.substring(9);
			return select(nn, false);
			
		}
		else if (line.startsWith("select")){
			//remove any prior selection
			String nn = line.substring(6);
			return select(nn, true);
		}
			
		return null;
	}
	
	private Color getColor(String c){
		Color color = null;
		if (c.indexOf("[") > -1){
			//rgb values are given
			int start = c.indexOf("[")+1;
			int end = c.indexOf("]");
			String value = c.substring(start, end);
			
			StringTokenizer tok = new StringTokenizer(value, ",", false);
			int[] comps = new int[3];
			int i = 0;
			while (tok.hasMoreElements()){
				comps[i++] = Integer.parseInt(tok.nextToken());
			}
			
			color = new Color(comps[0], comps[1], comps[2]);
		}
		else{
			color = (Color)colorMap.get(c);
		}
		
		return color;
	}
	
	private String select(String expression, boolean clear){
		if (expression.trim().startsWith("off")){
			styles.selectNone(true, true);
			viewer.getGeometryViewer().updateView();
			return null;
		}
					
		if (expression.trim().startsWith("all")){
			styles.selectAll();
			viewer.getGeometryViewer().updateView();
			return null;
		}
		
		if (clear){
			styles.selectNone(true, true);
		}

		
		//replace all worded logical operators with symbols
		Pattern pattern = Pattern.compile(" and ");
		Matcher matcher = pattern.matcher(expression);
		String temp = matcher.replaceAll(" & ");

		pattern = Pattern.compile(" not ");
		matcher = pattern.matcher(temp);
		String temp2 = matcher.replaceAll(" ! ");
		
		pattern = Pattern.compile(" or ");
		matcher = pattern.matcher(temp2);
		String next = matcher.replaceAll(" | ").trim();
		
		Hashtable finalSet = parseExpression(next);
//		System.out.println("finalSet = " + finalSet.size());
		Enumeration keys = finalSet.keys();
		
		//select it
		//along the way, check which chains are affected, and ribbons may need to be updated
		Vector chains = new Vector();
		
		Vector residues = new Vector();
		while (keys.hasMoreElements()){
			try{
				Residue r = (Residue)keys.nextElement();
				System.out.println("THIS SHOULD NEVER BE EXECUTED: selecting residue = " + r.getCompoundCode());
				styles.setSelected(r, true, true, false);
				residues.add(r);
			}
			catch (ClassCastException ex){
				continue;
			}
		}
		
		//the overall scheme: the expression parsing generates only Atoms in the selection
		//once the set is finalized, the full residues are detected in the atom set, and selected accordingly
		//this makes life easier for selecting ribbons and keeping track of events (especially related to ribbons)
		
/*		Vector finalSelection = new Vector();//a mix of Residue and Atom objects (non-redundant - 
		//if a Residue is selected, its atoms are not part of the Vector
		
		for (int i = 0; i < map.getResidueCount(); i++){
			Residue r = map.getResidue(i);
			boolean sel = true;//whether the residue should be selected
			Vector aa = new Vector();//temporary holder for atoms
			for(int j = 0; j < r.getAtomCount(); j++){
				Atom a = r.getAtom(j);
				if (finalSet.containsKey(a)){
					aa.add(a);
				}
				else{
					sel = false;
				}
			}
			
			if (sel){
				finalSelection.add(r);
			}
			else{
				finalSelection.addAll(aa);
			}
		}
		
		//go through finalSelection and perform setSelection
		
		for (int i = 0; i < finalSelection.size(); i++){
			try{
				Residue r = (Residue)finalSelection.get(i);
				styles.setSelected(r, true, true, true);
			}
			catch (ClassCastException e){
				Atom a = (Atom)finalSelection.get(i);
				styles.setSelected(a, true, true, true);
			}
		}
*/		
		
		//run through the lone atoms
		keys = finalSet.keys();
		while (keys.hasMoreElements()){
			try{
				
				Atom a = (Atom)keys.nextElement();
				if (residues.contains(a.residue)) continue;//this should not happen anyway
				styles.setSelected(a, true, true, false);
				Chain c = map.getChain(a);
				if (!chains.contains(c)) chains.add(c);
//				System.out.println("selecting atom = " + a.name);
			}
			catch (ClassCastException ex){}
		}

		//update the chains if necessary
//		System.out.println("Affected chains = " + chains.size());
		for (int i = 0; i < chains.size(); i++){
			Chain c = (Chain)chains.get(i);
//			System.out.println("c = " + c.getChainId());
			styles.updateChain(c, true);
		}
		
		viewer.getGeometryViewer().updateView();

			
		return null;
	}
	
	private Hashtable parseExpression(String next){
		
		Hashtable out = new Hashtable();
		
		//replace expressions in parentheses by tags pointing at the content in the hash
		int count = 0;
		while (true){
			if (next.indexOf("(") < 0) break;
			
			int start = next.indexOf("(");
			int end = next.indexOf(")");
			
			String exp = next.substring(start+1, end);//without parentheses
			String label = "block" + (new Integer(count)).toString();
			groups.put(label, exp);
			
			//replace the group with its label in the original expression
			Pattern pattern = Pattern.compile(exp);
			Matcher matcher = pattern.matcher(next);
			String temp = matcher.replaceAll(label);
			pattern = Pattern.compile("\\(");
			matcher = pattern.matcher(temp);
			String temp2 = matcher.replaceAll("");
			pattern = Pattern.compile("\\)");
			matcher = pattern.matcher(temp2);
			next = matcher.replaceAll("");
			count++;
			
			if (count == 10) break;
		}
		
		next = next.trim();
		
//		System.out.println("next = " + next);
		
		if (next.indexOf("|") > -1){
			StringTokenizer tok = new StringTokenizer(next, "|", false);
			while (tok.hasMoreTokens()){
				String token = tok.nextToken();
				if (out.size() == 0){
					//this is the first item on the line
					out = parseExpression(token);
//					System.out.println("first or expression = " + out.size());
				}
				else{
					out.putAll(parseExpression(token));//add this item to the existing set
//					System.out.println("subsequent or expression = " + out.size());
				}
			}
		}
		else if (next.indexOf("&") > -1){
			StringTokenizer tok = new StringTokenizer(next, "&", false);
			Hashtable hash = new Hashtable();
			int counter = 0;
			while (tok.hasMoreTokens()){
				String token = tok.nextToken().trim();
				counter++;
//				System.out.println("AND: " + token);
				if (hash.size() == 0){
					//this is the first item on the line
					if (counter > 1){
						//something went wrong in the first iteration
						out.clear();
						return out;
					}
					hash = parseExpression(token);
//					System.out.println("first hash = " + hash.size());
				}
				else{
					if (token.startsWith("!")){
//						System.out.println("and not case: " + token);
						//exclude the new set from the previous one
						Hashtable s = parseExpression(token.substring(1).trim());
//						System.out.println("s = " + s.size());
//						System.out.println("incoming hash = " + s.size());
						Enumeration keys = hash.keys();
						while (keys.hasMoreElements()){
							try{
								Object v = keys.nextElement();
								if (s.containsKey(v)){
									hash.remove(v);
								}
							}
							catch (Exception ex){
								ex.printStackTrace();
								continue;
							}
						}
						
//						System.out.println("resulting hash = " + hash.size());
					}
					else{
						//determine overlap between existing set and the new one
						Hashtable s = parseExpression(token);
						
						Hashtable output = new Hashtable();
						Enumeration keys = hash.keys();
						while (keys.hasMoreElements()){
							try{
								Object v = keys.nextElement();
								if (s.containsKey(v)){
									output.put(v, value);
								}
							}
							catch (Exception ex){
								continue;
							}
						}
						hash = output;//reassign the hash reference to the new one

					}
//					System.out.println("> 0: " + hash.size());
				}
			}
			
			out = hash;
		}
		else if (groups.containsKey(next)){
//			System.out.println("next is a group");
			//the current expression is a group in parentheses
			String expr = (String)groups.get(next);
//			System.out.println("actual expression = " + expr);
			out = parseExpression(expr);
		}
		else if (next.startsWith("!")){//special case with selecting all but a certain set
			
//			System.out.println("starts with not");
			String exp = next.substring(1).trim();
			
			//check whether this is a group
			if (groups.containsKey(exp)){
				out = parseExpression(exp);
				
			}
			else{
//				System.out.println("all but a unary set = " + exp);
				out = parseUnarySet(exp);
				
				//now, output set should contain everything except what's in the out hash
				Hashtable output = new Hashtable();
				
				Vector sc = new Vector();
				//select everything except the content of the finalSet
				for (int i = 0; i < map.getResidueCount(); i++){
					Residue r = map.getResidue(i);
					if (out.containsKey(r)) continue;
					boolean selected = true;//all atoms of this residue are selected
					for (int j = 0; j < r.getAtomCount(); j++){
						Atom a = r.getAtom(j);
						if (out.containsKey(a)){
							selected = false;
							continue;
						}
						output.put(a, value);
					}
					
//					if (selected) output.put(r, value);
					
				}
				
				out = output;
			}
		}
		else{
			//there are no logical operators, so get the set directly
			out = parseUnarySet(next);

		}

		return out;
	}
	
	
	private Hashtable parseUnarySet(String set){
		
		//get rid of the parentheses, if any
		Pattern pattern = Pattern.compile("\\(");
		Matcher matcher = pattern.matcher(set);
		String temp = matcher.replaceAll("");
		
		pattern = Pattern.compile("\\)");
		matcher = pattern.matcher(set);
		set = matcher.replaceAll("");
		
		pattern = Pattern.compile(" ");
		matcher = pattern.matcher(set);
		set = matcher.replaceAll("");

//		System.out.println("parsing unary set = #" + set + "#");

		Hashtable out = new Hashtable();
		if (set.indexOf(",") > -1){
			//comma-separated list
			StringTokenizer tok = new StringTokenizer(set, ",", false);
			while (tok.hasMoreTokens()){
				String token = tok.nextToken();
				out.putAll(parseUnarySet(token));
			}
			
		}
		else if (set.indexOf("-") > -1){
			//a range of residue numbers
			int k = set.indexOf("-");
			String start = set.substring(0,k);
			String end = set.substring(k+1);
			
			try{
				int s = Integer.parseInt(start);
				int e = Integer.parseInt(end);
				
				for (int i = 0; i < map.getResidueCount(); i++){
					Residue r = map.getResidue(i);
					if (r.getResidueId() >= s && r.getResidueId() <= e){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			
			}
			catch (NumberFormatException ex){
				
			}
		}
		else if (set.startsWith("*")){
			/**
			 * TODO if just a residue number is given, it will match residue number in all chains
			 */
			String match = set.substring(1).toUpperCase();//residue name + chain name should end with match to be removed
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				String n = (new Integer(r.getResidueId())).toString() + r.getChainId();
				if (n.endsWith(match)){
//					out.put(r, value);
					for (int j = 0; j < r.getAtomCount(); j++){
						out.put(r.getAtom(j), value);
					}
				}
			}
		}
		else if (predefinedSets.contains(set)){
			out = getPredefinedSet(set);
		}
		else{
			//check whether this name is a set name
			
			Vector sets = viewer.getAtomSetNames();
			if (sets.contains(set)){
				Vector components = (Vector)viewer.getAtomSets().get(set);
				for (int i = 0; i < components.size(); i++){
					out.put(components.get(i), value);
				}
				
				return out;
			}
			
			//since it's not a set, it's something else
			
			//residue name, atom name or element name
			//first, check if it's an element name
//			System.out.println("a named item = " + set);
			Element element = null;
			for (int i = 1; i < PeriodicTable.getElementCount(); i++){
				Element e = PeriodicTable.getElement(i);
				if (e.name.toUpperCase().equals(set.toUpperCase()) || e.symbol.toUpperCase().equals(set.toUpperCase())){
					element = e;
					break;
				}
			}
			
//			System.out.println("element = " + element);
			
			if (element != null){
				//select element atoms
				for (int i = 0; i < map.getAtomCount(); i++){
					Atom a = map.getAtom(i);
					if (a.element == (short)element.atomic_number){
						out.put(a, value);
					}
				}
				
				return out;
			}
			
			//check if it starts with a digit
			Pattern aanum = Pattern.compile("(([0-9]+)([a-zA-Z]+))");
			Matcher m = aanum.matcher(set);
			boolean res = m.find();
			if (res){
				String num = m.group(2);
				String name = m.group(3);
				try{
					int a = Integer.parseInt(num);//residue id
//					System.out.println("residue id = " + a);
					//name is chain id
					//find the right residue
					Residue residue = null;
					for (int i = 0; i < map.getResidueCount(); i++){
						Residue r = map.getResidue(i);
//						System.out.println("chain = " + map.getChain(r.getAtom(0)).getChainId());
						if (map.getChain(r.getAtom(0)).getChainId().equals(name.toUpperCase())){
							//this is the correct chain
							if (r.getResidueId() == a){
								residue = r;
								break;
							}
						}
					}
					
					if (residue != null){
						if (set.indexOf(".") > -1){
							//it's an atom name
							String an = set.substring(set.indexOf(".")+1).toUpperCase();
							for (int j = 0; j < residue.getAtomCount(); j++){
								Atom at = residue.getAtom(j);
								if (at.name.toUpperCase().equals(an)){
									out.put(at, value);
									break;
								}
							}
						}
						else{
//							out.put(residue, value);
							for (int j = 0; j < residue.getAtomCount(); j++){
								out.put(residue.getAtom(j), value);
							}
						}
					}
					else{
						//residue is null
//						System.out.println("residue is null");
						out.clear();
					}
				}
				catch (NumberFormatException exx){
					exx.printStackTrace();
				}
			}
			else{
				//try another possibility: e.g., phe347.o (without the chain id)
				aanum = Pattern.compile("(([a-zA-Z]+)([0-9]+))");
				m = aanum.matcher(set);
				res = m.find();
				if (res){
					String name = m.group(2);//residue compound code
					String num = m.group(3);//residue number
					
					try{
						int a = Integer.parseInt(num);//residue id
						//find the right residue
						Residue residue = null;
						
						if (name.equals("HOH") || name.equals("WAT")){
							//we need to pick up the water molecule with the number given by num
							//keep in mind that the number refers to the order, not to the actual residue id
							for (int i = 0; i < map.getResidueCount(); i++){
								Residue r = map.getResidue(i);
								if (r.getCompoundCode().equals("HOH") || r.getCompoundCode().equals("WAT")){
									if (r.getResidueId() == a){
										residue = r;
										break;
									}
								}
							}
							
						}
						else{
							for (int i = 0; i < map.getResidueCount(); i++){
								Residue r = map.getResidue(i);
								if (r.getResidueId() == a){
									residue = r;
									break;
								}
							}
						}
						
						
						if (residue != null){
							if (set.indexOf(".") > -1){
								//it's an atom name
								String an = set.substring(set.indexOf(".")+1).toUpperCase();
								for (int j = 0; j < residue.getAtomCount(); j++){
									Atom at = residue.getAtom(j);
									if (at.name.toUpperCase().equals(an)){
										out.put(at, value);
										break;
									}
								}
							}
							else{
//								out.put(residue, value);
								for (int j = 0; j < residue.getAtomCount(); j++){
									out.put(residue.getAtom(j), value);
								}
							}
						}
					}
					catch (NumberFormatException exx){
						exx.printStackTrace();
					}
				
				}
				else{
					//decide whether it's a simple residue number or a residue name
					aanum = Pattern.compile("([0-9]+)");
					m = aanum.matcher(set);
					res = m.find();
					if (res){
						String num = m.group(1);//residue number
						try{
							int N = Integer.parseInt(num);
							for (int i = 0; i < map.getResidueCount(); i++){
								Residue r = map.getResidue(i);
								if (r.getResidueId() == N){
									for (int j = 0; j < r.getAtomCount(); j++){
										out.put(r.getAtom(j), value);
									}
								}
							}
							
							return out;
						}
						catch (NumberFormatException e){}
					}
					
					//this is the last option, and it's executed only if residue # either fails or is not a match
					for (int i = 0; i < map.getResidueCount(); i++){
						Residue r = map.getResidue(i);
						if (r.getCompoundCode().toUpperCase().equals(set.toUpperCase())){
//							out.put(r, value);
							for (int j = 0; j < r.getAtomCount(); j++){
								out.put(r.getAtom(j), value);
							}
						}
					}
				}
				
			}
			
			
		}
		
		return out;
	}
	
	private void processRibbon(String flag){
		if (flag.startsWith("off")){
			//remove ribbon in selection
			//if nothing selected, remove all ribbons in the structure
			Hashtable selection = styles.getSelection();
			if (selection.size() == 0){
				//remove all ribbons
				for (int i = 0; i < map.getChainCount(); i++){
					Chain c = map.getChain(i);
					viewer.removeRibbon(c, false);
				}
				
			}
			else{
				//for selected residues, set ribbon flag to false, and rerender the ribbon
				if (cartoon){
					StylesPreferences.smoothRibbon = true;
					viewer.setRibbonInSelection(false, StylesPreferences.RIBBON_RENDERED, 0.8f, 2.0f);
					StylesPreferences.smoothRibbon = false;
					cartoon = false;
				}
				else{
					StylesPreferences.smoothRibbon = false;
					viewer.setRibbonInSelection(false, StylesPreferences.RIBBON_UNIFORM_ROUND, 0.8f, 2.0f);
				}
			}
		}
		else{
			//selected part of the structure needs a ribbon
			//add the ribbon by default
			for (int i = 0; i < map.getChainCount(); i++){
				Chain c = map.getChain(i);
				//check if this chain is in the selection
				boolean selected = false;
				for (int j = 0; j < c.getResidueCount(); j++){
					Residue r = c.getResidue(j);
					if (!r.getClassification().equals(Residue.COMPOUND_AMINO_ACID) && 
							!r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)) continue;
					if (styles.isSelected(r)){
						selected = true;
						break;
					}
					
					//if the resdidue itself is not selected, check individual atoms
					//it's likely that just the backbone atoms are selected
					for (int k = 0; k < r.getAtomCount(); k++){
						Atom a = r.getAtom(k);
						if (styles.isSelected(a)){
							if (a.backbone){
								selected = true;
								break;
							}
						}
					}
				}

				if (selected){
					if (cartoon){
						StylesPreferences.smoothRibbon = true;
						viewer.createRibbonInSelection(StylesPreferences.RIBBON_RENDERED, 2.0f, 0.8f, true);
						StylesPreferences.smoothRibbon = false;
						cartoon = false;
					}
					else{
						StylesPreferences.smoothRibbon = false;
						viewer.createRibbonInSelection(StylesPreferences.RIBBON_UNIFORM_ROUND, 2.0f, 0.8f, true);
					}
				}
			}
		}
		
		viewer.getGeometryViewer().updateView();
	}
	
	private Hashtable getPredefinedSet(String set){
		
		set = set.trim();
		
		Hashtable out = new Hashtable();
		if (set.equals("at")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)){
					if (r.getCompoundCode().equals("A") || r.getCompoundCode().equals("T")){
						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("cg")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)){
					if (r.getCompoundCode().equals("C") || r.getCompoundCode().equals("G")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("acidic")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("ASP") || r.getCompoundCode().equals("GLU")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("acyclic")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (!r.getCompoundCode().equals("HIS") && !r.getCompoundCode().equals("PHE") &&
							!r.getCompoundCode().equals("PRO") && !r.getCompoundCode().equals("TYR")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("aliphatic")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("ALA") || r.getCompoundCode().equals("GLY") || r.getCompoundCode().equals("VAL") ||
							r.getCompoundCode().equals("ILE") || r.getCompoundCode().equals("LEU")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("alpha")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					out.put(r.getAlphaAtom(), value);
				}
			}
		}
		else if (set.equals("amino")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
//					out.put(r, value);
					for (int j = 0; j < r.getAtomCount(); j++){
						out.put(r.getAtom(j), value);
					}
				}
			}
		}
		else if (set.equals("aromatic")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("HIS") || r.getCompoundCode().equals("PHE") ||
							r.getCompoundCode().equals("TRP") || r.getCompoundCode().equals("TYR")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("backbone")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					for (int j = 0; j < r.getAtomCount(); j++){
						Atom a = r.getAtom(j);
						if (a.backbone) out.put(a, value);
					}
				}
				else if (r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)){
					for (int j = 0; j < r.getAtomCount(); j++){
						Atom a = r.getAtom(j);
						if (a.backbone) out.put(a, value);
					}
				}
			}
		}
		else if (set.equals("basic")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("ARG") || r.getCompoundCode().equals("HIS") ||
							r.getCompoundCode().equals("LYS")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("bonded")){
			for (int i = 0; i < map.getAtomCount(); i++){
				Atom a = map.getAtom(i);
				Vector bonds = map.getBonds(a);
				if (bonds == null || bonds.size() == 0) continue;
				out.put(a, value);
			}
		}
		else if (set.equals("buried")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("ALA") || r.getCompoundCode().equals("CYS") ||
							r.getCompoundCode().equals("ILE") || r.getCompoundCode().equals("LEU")||
							r.getCompoundCode().equals("MET") || r.getCompoundCode().equals("PHE")||
							r.getCompoundCode().equals("TRP") || r.getCompoundCode().equals("VAL")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("charged")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("ARG") || r.getCompoundCode().equals("ASP") ||
							r.getCompoundCode().equals("GLU") || r.getCompoundCode().equals("HIS")||
							r.getCompoundCode().equals("LYS")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("cyclic")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("HIS") || r.getCompoundCode().equals("PHE") ||
							r.getCompoundCode().equals("PRO") || r.getCompoundCode().equals("TRP")||
							r.getCompoundCode().equals("TYR")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("cystine")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("CYS")){
						//and its S is less than 3 angstroms away from another cystein's S
						/**
						 * TODO 
						 */
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("helix")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getConformationType().equals(StructureComponentRegistry.TYPE_HELIX)){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("hetero")){
			for (int i = 0; i < map.getAtomCount(); i++){
				Atom a = map.getAtom(i);
				if (a.residue.getClassification().equals(Residue.COMPOUND_LIGAND) ||
						a.residue.getCompoundCode().equals("HOH") || a.residue.getCompoundCode().equals("WAT")){
					out.put(a, value);
					if (!out.contains(a.residue)) out.put(a.residue, value);
				}
				
			}
		}
		else if (set.equals("hydrogen")){
			for (int i = 0; i < map.getAtomCount(); i++){
				Atom a = map.getAtom(i);
				if (a.element == 1){
					out.put(a, value);
				}
				
			}
		}
		else if (set.equals("hydrophobic")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("ALA") || r.getCompoundCode().equals("GLY") ||
							r.getCompoundCode().equals("ILE") || r.getCompoundCode().equals("LEU")||
							r.getCompoundCode().equals("MET") || r.getCompoundCode().equals("PHE")||
							r.getCompoundCode().equals("PRO") || r.getCompoundCode().equals("TRP")||
							r.getCompoundCode().equals("TYR") || r.getCompoundCode().equals("VAL")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("large")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("ARG") || r.getCompoundCode().equals("GLU") ||
							r.getCompoundCode().equals("GLN") || r.getCompoundCode().equals("HIS")||
							r.getCompoundCode().equals("ILE") || r.getCompoundCode().equals("LEU")||
							r.getCompoundCode().equals("LYS") || r.getCompoundCode().equals("MET")||
							r.getCompoundCode().equals("PHE") || r.getCompoundCode().equals("TRP")||
							r.getCompoundCode().equals("TYR")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("ligand")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_LIGAND)){
					if (r.getCompoundCode().equals("HOH") || r.getCompoundCode().equals("WAT")) continue;
//					out.put(r, value);
					for (int j = 0; j < r.getAtomCount(); j++){
						out.put(r.getAtom(j), value);
					}
				}
			}
			
		}
		else if (set.equals("medium")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("ASP") || r.getCompoundCode().equals("ASN") ||
							r.getCompoundCode().equals("CYS") || r.getCompoundCode().equals("PRO")||
							r.getCompoundCode().equals("THR") || r.getCompoundCode().equals("VAL")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("neutral")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (!r.getCompoundCode().equals("ARG") && !r.getCompoundCode().equals("ASP") &&
							!r.getCompoundCode().equals("GLU") && !r.getCompoundCode().equals("LYS")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("nucleic")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)){
//					out.put(r, value);
					for (int j = 0; j < r.getAtomCount(); j++){
						out.put(r.getAtom(j), value);
					}
				}
			}
		}
		else if (set.equals("polar")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getCompoundCode().equals("ARG") || r.getCompoundCode().equals("ASN") ||
							r.getCompoundCode().equals("ASP") || r.getCompoundCode().equals("CYS")||
							r.getCompoundCode().equals("GLU") || r.getCompoundCode().equals("GLN")||
							r.getCompoundCode().equals("HIS") || r.getCompoundCode().equals("LYS")||
							r.getCompoundCode().equals("SER") || r.getCompoundCode().equals("THR")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("protein")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
//					out.put(r, value);
					for (int j = 0; j < r.getAtomCount(); j++){
						out.put(r.getAtom(j), value);
					}
				}
			}
		}
		else if (set.equals("purine")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)){
					if (r.getCompoundCode().equals("A") || r.getCompoundCode().equals("G")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("pyrimidine")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)){
					if (r.getCompoundCode().equals("T") || r.getCompoundCode().equals("C")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("sheet")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getConformationType().equals(StructureComponentRegistry.TYPE_STRAND)){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("sidechain")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					for (int j = 0; j < r.getAtomCount(); j++){
						Atom a = r.getAtom(j);
						if (a.name.equals("N") || a.name.equals("CA") || a.name.equals("C") || a.name.equals("O") 
								||a.name.equals("O\"") || a.name.equals("OX")) continue;
						out.put(r.getAtom(j), value);
					}
				}
				else if (r.getClassification().equals(Residue.COMPOUND_NUCLEIC_ACID)){
					for (int j = 0; j < r.getAtomCount(); j++){
						Atom a = r.getAtom(j);
						if (a.name.equals("P")||a.name.equals("O3'")||a.name.equals("C3'")
								||a.name.equals("C4'")||a.name.equals("C5'")||a.name.equals("O5'")
								||a.name.indexOf("'") > -1) continue;
						out.put(a, value);
					}

					
				}
			}
		}
		else if (set.equals("small")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (!r.getCompoundCode().equals("ALA") && !r.getCompoundCode().equals("GLY") &&
							!r.getCompoundCode().equals("SER")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("solvent") || set.equals("water")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getCompoundCode().equals("HOH") || r.getCompoundCode().equals("WAT")){
//					out.put(r, value);
					for (int j = 0; j < r.getAtomCount(); j++){
						out.put(r.getAtom(j), value);
					}
				}
			}
		}
		else if (set.equals("surface")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (!r.getCompoundCode().equals("ALA") && !r.getCompoundCode().equals("CYS") &&
							!r.getCompoundCode().equals("ILE") && !r.getCompoundCode().equals("LEU") && 
							!r.getCompoundCode().equals("MET") && !r.getCompoundCode().equals("PHE") &&
							!r.getCompoundCode().equals("TRP") && !r.getCompoundCode().equals("VAL")){
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		else if (set.equals("turn")){
			for (int i = 0; i < map.getResidueCount(); i++){
				Residue r = map.getResidue(i);
				if (r.getClassification().equals(Residue.COMPOUND_AMINO_ACID)){
					if (r.getConformationType().equals(StructureComponentRegistry.TYPE_COIL) ||
							r.getConformationType().equals(StructureComponentRegistry.TYPE_TURN)){
						
//						out.put(r, value);
						for (int j = 0; j < r.getAtomCount(); j++){
							out.put(r.getAtom(j), value);
						}
					}
				}
			}
		}
		
		else if (set.equals("ions")){
			/**
			 * TODO 
			 */
		}
		
		
		return out;
	}
	
	private String processMove(String flag, boolean rasmol){
		//get the axis and the extent of translation
		StringTokenizer tok = new StringTokenizer(flag, " ", false);
		String axis = tok.nextToken().trim();
		String value = tok.nextToken().trim();
		
		if (viewer.getInitialCoordinates(structure) == null){
			viewer.saveInitialCoordinates(structure);
		}
		
		int v = 10;
		try{
			v = (int)Double.parseDouble(value);
		}
		catch (ClassCastException e){
			return "Invalid value for structure translation: " + value;
		}
		
		int ax = Algebra.X_AXIS;
		if (axis.toUpperCase().startsWith("Y")){
			ax = Algebra.Y_AXIS;
			if (rasmol) v = -v;
		}
		else if (axis.toUpperCase().startsWith("Z")){
			ax = Algebra.Z_AXIS;
		}
		
		//check whether there is only one structure or several
		if (viewer.getLoadedStructures().size() > 1){
			//transform coordinates
			if (ax == Algebra.X_AXIS){
				double[] look = viewer.getGeometryViewer().getLookAt();
				double[] up = viewer.getGeometryViewer().getUpVector();
				double[] vector = Algebra.crossProduct(look, up);
				Algebra.normalizeVector(vector);
				Algebra.scalarMultiply(vector, v);
				
				//the motion is applied to the entire structure the interpreter is attached to
				//first, get the transformation matrix
				double[] matrix = StructureMath.getLinearTranslationMatrix(vector);
				
				for (int i = 0; i < map.getAtomCount(); i++){
					Atom a = map.getAtom(i);
					StructureMath.transformCoordinates(a.coordinate, matrix);
				}
				
				StructureMath.transformCoordinates(map.getCenter(), matrix);
			}
			else if (ax == Algebra.Y_AXIS){
				double[] up = viewer.getGeometryViewer().getUpVector();
				double[] vector = new double[]{ up[0], up[1], up[2] };
				Algebra.normalizeVector(vector);
				Algebra.scalarMultiply(vector, v);
				
				//the motion is applied to the entire structure the interpreter is attached to
				//first, get the transformation matrix
				double[] matrix = StructureMath.getLinearTranslationMatrix(vector);
				
				for (int i = 0; i < map.getAtomCount(); i++){
					Atom a = map.getAtom(i);
					StructureMath.transformCoordinates(a.coordinate, matrix);
				}
				StructureMath.transformCoordinates(map.getCenter(), matrix);
			}
			else if (ax == Algebra.Z_AXIS){
				double[] look = viewer.getGeometryViewer().getLookAt();
				double[] vector = new double[]{ look[0], look[1], look[2] };
				Algebra.normalizeVector(vector);
				Algebra.scalarMultiply(vector, v);
				
				//the motion is applied to the entire structure the interpreter is attached to
				//first, get the transformation matrix
				double[] matrix = StructureMath.getLinearTranslationMatrix(vector);
				
				for (int i = 0; i < map.getAtomCount(); i++){
					Atom a = map.getAtom(i);
					StructureMath.transformCoordinates(a.coordinate, matrix);
				}
				StructureMath.transformCoordinates(map.getCenter(), matrix);
			}
			
			viewer.updateAppearance();
		}
		else{
			//translate the camera
			viewer.getGeometryViewer().translate(ax, v);
		}
	
		return null;
	}
	
	private String processRotation(String flag, boolean rasmol){
		//get the axis and the extent of translation
		StringTokenizer tok = new StringTokenizer(flag, " ", false);
		String axis = tok.nextToken().trim();
		String value = tok.nextToken().trim();
		
		if (viewer.getInitialCoordinates(structure) == null){
			viewer.saveInitialCoordinates(structure);
		}

		int v = 10;
		try{
			v = (int)Double.parseDouble(value);
		}
		catch (ClassCastException e){
			return "Invalid value for structure rotation: " + value;
		}
		
		int ax = Algebra.X_AXIS;
		if (axis.toUpperCase().startsWith("Y")){
			ax = Algebra.Y_AXIS;
		}
		else if (axis.toUpperCase().startsWith("Z")){
			ax = Algebra.Z_AXIS;
			if (rasmol) v = -v;
		}
		
		double[] toOrigin = StructureMath.getToOriginMatrix(map.getCenter());
		double[] fromOrigin = StructureMath.getFromOriginMatrix(map.getCenter());

		
		//check whether there is only one structure or several
		if (viewer.getLoadedStructures().size() > 1){
			//transform coordinates
			if (ax == Algebra.X_AXIS){
				double[] look = viewer.getGeometryViewer().getLookAt();
				double[] up = viewer.getGeometryViewer().getUpVector();
				double[] vector = Algebra.crossProduct(look, up);
				
				//the motion is applied to the entire structure the interpreter is attached to
				//first, get the transformation matrix
				double[] matrix = StructureMath.getLinearRotationMatrix(v, vector);
				
				for (int i = 0; i < map.getAtomCount(); i++){
					Atom a = map.getAtom(i);
					StructureMath.transformCoordinates(a.coordinate, toOrigin);
					StructureMath.transformCoordinates(a.coordinate, matrix);
					StructureMath.transformCoordinates(a.coordinate, fromOrigin);
				}
			}
			else if (ax == Algebra.Y_AXIS){
				double[] vector = viewer.getGeometryViewer().getUpVector();
				
				//the motion is applied to the entire structure the interpreter is attached to
				//first, get the transformation matrix
				double[] matrix = StructureMath.getLinearRotationMatrix(v, vector);
				
				for (int i = 0; i < map.getAtomCount(); i++){
					Atom a = map.getAtom(i);
					StructureMath.transformCoordinates(a.coordinate, toOrigin);
					StructureMath.transformCoordinates(a.coordinate, matrix);
					StructureMath.transformCoordinates(a.coordinate, fromOrigin);
				}
			}
			else if (ax == Algebra.Z_AXIS){
				double[] vector = viewer.getGeometryViewer().getLookAt();
				
				//the motion is applied to the entire structure the interpreter is attached to
				//first, get the transformation matrix
				double[] matrix = StructureMath.getLinearRotationMatrix(-v, vector);
				
				for (int i = 0; i < map.getAtomCount(); i++){
					Atom a = map.getAtom(i);
					StructureMath.transformCoordinates(a.coordinate, toOrigin);
					StructureMath.transformCoordinates(a.coordinate, matrix);
					StructureMath.transformCoordinates(a.coordinate, fromOrigin);
				}
			}
			
			viewer.updateAppearance();
		}
		else{
			//rotate the camera
			viewer.getGeometryViewer().rotate(map.getCenter(), ax, v);
		}
	
		return null;
	}


	public void clear(Structure s){
		if (s == structure){
			structure = null;
			map = null;
			styles = null;
			trace = false;
		}
	}
	
	public void clear(){
		structure = null;
		map = null;
		styles = null;
		trace = false;
	}
	
	public void clearSelection(){
		if (styles != null) styles.selectNone(false, false);
	}
	
}